#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2005 Free Software Foundation
#
# FILE:
# PopupMenu.py
#
# DESCRIPTION:
#
# NOTES:

import sys, os, time, string
from wxPython.wx import *
from gnue.common.apps import GDebug
from gnue.common.definitions import GObjects
from gnue.common.logic import GTrigger
from gnue.forms import GFForm, GFObjects, GFLibrary



class ActionMenu(wxMenu):
  def __init__(self, instance, object, parent=None, text=None, parentText=None):
    wxMenu.__init__(self, text)
    self.instance = instance
    self.__parent = parent
    self._object = object
    self._newMap = {}
    if parent:
      parent.AppendMenu(wxNewId(), parentText, self)


    if object == None:

      self.addCreateTag(_('datasource'),_('New Data Source'))
      self.addCreateTag(_('trigger'),_('New Shared Trigger'))
      self.addCreateTag(_('page'),_('New Page'))

    else:

      # New >
      CreateObjectMenu(instance, object, self)

      # Properties
      id = wxNewId()
      self.Append(id, _("&Properties..."))
      EVT_MENU(self, id, self.OnEditProperties)

      if not isinstance (object, GFForm.GFForm):
        # Delete
        id = wxNewId()
        self.Append(id, _("&Delete %s") % object._type[2:])
        EVT_MENU(self, id, self.OnDeleteObject)

      self.AppendSeparator()

      # Events >
      if isinstance (object, GTrigger.GTriggerExtension):
        menu = wxMenu(_("Events..."))
        self._triggerMap = {}
        keys = object._validTriggers.keys()
        keys.sort()
        for trigger in (keys):
          tid = wxNewId()
          self._triggerMap[tid] = trigger
          menu.Append(tid, object._validTriggers[trigger] + "...")
          EVT_MENU(menu, tid, self.OnEditEvent)
        self.AppendMenu(wxNewId(), _("&Events"), menu)


      if isinstance(object, GFObjects.GFPage) or \
         isinstance(object, GFObjects.GFBlock):
        self.AppendSeparator()
        # Reorder
        id = wxNewId()
        self.Append(id, _("&Reorder by Location"))
        EVT_MENU(self, id, self.OnReorderLayout)


  def addChildObjectMenu(self):
    object = self._object
    if len(object._children):
      self.AppendSeparator()
      for child in object._children:
        if isinstance (child, GObjects.GObj):
          self.AppendMenu(wxNewId(), hasattr(child,'name') and child.name or \
            string.upper(child._type[2]) + \
            string.lower(child._type[3:]),
            PageMenu(self.instance, child, self.x, self.y))


  def addCreateTag(self, tag, text):
    tid = wxNewId()
    self._newMap[tid] = tag
    self.Append(tid, text)
    EVT_MENU(self.instance.uiinstance, tid, self.OnCreateObject)


  def OnEditProperties(self, event):
    self.instance.propertyEditorWindow.Show(1)
    self.instance.propertyEditorWindow.Raise()
    self.instance.propertyEditorWindow.SetFocus()

  def OnEditEvent(self, event):
    pass

  def OnCreateObject(self, event):
    tag = self._newMap[event.GetId()]
    GDebug.printMesg(4,'Creating object %s with id %s'%(tag, event.GetId()))
    attributes=hasattr(self,'Char__x') and {'Char:x': self.Char__x, 'Char:y': self.Char__y} or {}
    if tag == 'trigger':
      attributes['type']='named'

    self.instance.incubator.createObject(self.instance.rootObject, tag,
      self._object, attributes)

  def OnCreateDataSource(self, event):
    self.instance.incubator.createObject(self.instance.rootObject, 'datasource')

  def OnCreateNamedTrigger(self, event):
    self.instance.incubator.createObject(self.instance.rootObject, 'trigger')

  def OnCreatePage(self, event):
    self.instance.incubator.createObject(self.instance.rootObject, 'page')

  def OnDeleteObject(self, event):
    self.instance.dispatchEvent('ObjectDeleted', object=self._object,
                           originator=self)

  def OnReorderLayout(self, event):
    unqId = 0
    triggerObjects = []
    hiddenObjects = []
    boxObjects = {}
    navObjects = {}
    blockObjects = {}
    self._object._upperX = 99999
    self._object._upperY = 99999

    # Handle each child (note we are working from a copy
    # because self.oject._children will be morphing
    for object in self._object._children[:]:
      if hasattr(object, 'hidden') and object.hidden:
        hiddenObjects.append(object)
      elif object._type == 'GCTrigger':
        triggerObjects.append(object)
      elif object._type == 'GFBlock':
        object._popupMenu.OnReorderLayout(event)
        object.__savedChildren = object._children[:]
        for o2 in object.__savedChildren:
          self.instance.onDeleteObject(o2, __name__)

        oid = "%5d.%5d.%5d" % (object._upperY, object._upperX, unqId)
        unqId = unqId + 1
        blockObjects[oid] = object

      else:
        if hasattr(object,'Char__y') and object.Char__y < self._object._upperY:
          self._object._upperY = object.Char__y
        if hasattr(object,'Char__x') and object.Char__x < self._object._upperX:
          self._object._upperX = object.Char__x

        oid = "%5d.%5d.%5d" % (object.y, object.x, unqId)
        unqId  = unqId + 1
        if object._type == 'GFBox':
          boxObjects[oid] = object
        else:
          navObjects[oid] = object

      # Notify everyone that this object was deleted
      self.instance.onDeleteObject(object, __name__)

    self._object._children = []

    # First, add back triggers
    for object in triggerObjects:
      self._object._children.append(object)
      self.instance.onCreateObject(object, __name__)


    # Next, add back hidden objects
    for object in hiddenObjects:
      self._object._children.append(object)
      self.instance.onCreateObject(object, __name__)

    # Now, add hidden objects
    boxes = boxObjects.keys()
    boxes.sort()
    for box in boxes:
      self._object._children.append(boxObjects[box])
      self.instance.onCreateObject(boxObjects[box], __name__)

    # The rest of the objects (other than blocks) go here
    nav = navObjects.keys()
    nav.sort()
    for n in nav:
      self._object._children.append(navObjects[n])
      self.instance.onCreateObject(navObjects[n], __name__)

    # And, finally, any child blocks
    blocks = blockObjects.keys()
    blocks.sort()
    for block in blocks:
      children = blockObjects[block].__savedChildren[:]
      self._object._children.append(blockObjects[block])
      self.instance.onCreateObject(blockObjects[block], __name__)
      for child in children:
        blockObjects[block]._children.append(child)
        self.instance.onCreateObject(child,__name__)


class CreateObjectMenu(ActionMenu):
  def __init__(self, instance, object, parent):
    ActionMenu.__init__(self, instance, None, parent, _('New...'),_('&New'))

    self._object = object

    self.AppendSeparator()
    tag = string.lower(object._type[2:])
    if self.instance.incubator.elementMapping.has_key(tag):
      self.instance.incubator.elementMapping[tag].sort()
      for element in self.instance.incubator.elementMapping[tag]:
        descr = string.upper(element[0])+string.lower(element[1:])
        if descr == 'Import':
          descr = 'Import Library'
        elif len(descr) > 7 and descr[:7] == 'Import-':
          descr = 'Imported ' + string.upper(descr[7])+ \
            string.lower(descr[8:])

        self.addCreateTag(element, descr)



# This menu is the popup menu on the object tree
class ObjectMenu(ActionMenu):
  def __init__(self, instance, object):
    ActionMenu.__init__(self, instance, object, None, _("Object Tasks..."))



# This menu is the popup menu on the layout screen
class PageMenu(ActionMenu):
  def __init__(self, instance, object, x, y):
    ActionMenu.__init__(self, instance, object, None, hasattr(object,'name') and \
       object.name or string.upper(object._type[2]) + \
       string.lower(object._type[3:]) )

    self.x = x
    self.y = y

    self.addChildObjectMenu()




