#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2002-2005 Free Software Foundation
#
# FILE:
# GRFilters.py
#
# DESCRIPTION:
# Class that handles report filters
#
# NOTES:
#

from gnue.common.apps import GConfig, GDebug
from gnue.common.utils.FileUtils import openResource, dyn_import
from ConfigParser import *
import os, sys, string


class FilterError(StandardError):
  pass


#
#
#
class GRFilters:

  def __init__(self):
    self._config = GRFilterConfig()

  def loadFilter(self, namespace, filter, destination, filter_options):

    lastNamespace = namespace and namespace.split(':')[-1] or 'Raw'

    try:

      if not namespace:
        raise KeyError

      parameters = self._config.filters[namespace + ":" + filter]
    except KeyError:
      if filter == 'raw':
        parameters = {'engine': 'raw'}
      else:
        raise FilterError, \
          "Unable to find a filter definition for %s:%s" % (namespace,filter)

    try:
      module = dyn_import ('gnue.reports.adapters.filters.%s.%s' % (lastNamespace,parameters['engine']))
      GDebug.printMesg(5,'Import of Filter at gnue.reports.adapters.filters.'+\
                       '%s.%s successful.' % (lastNamespace,parameters['engine']))
    except ImportError:
      module = dyn_import ('gnue.reports.adapters.filters.general.%s' % parameters['engine'])
      GDebug.printMesg(5,'Import of Filter at gnue.reports.adapters.filters.'+\
                       'general.%s successful.' %  parameters['engine'])
    except ImportError:
      raise
      raise FilterError, "Invalid filter specified: %s" % filter
    except KeyError:
      raise FilterError, "Filter configuration file does not contain an 'engine=' for %s" % filter

    parameters ['_filterbase'] = self._config.filterBase

    try:
      adapter = module.TransformAdapter(destination, parameters, filter_options)
    except AttributeError:      
      raise FilterError, "Plugin for Filter %s does not contain a TransformAdapter." % filter
    return adapter

#
# Handles the processing of the
# report-filters.conf file
#
class GRFilterConfig:

  def __init__(self):

    self.mappings = {}
    self.filters = {}


    # Load the report-filters file
    location = gConfigReports('filters')
    if location:
      if location[0] != '/':
        location = os.path.join(
            GConfig.getInstalledBase('reports_etc','common_etc','install_prefix'),
            location )
      self.readConfig(location)



  def readConfig(self,location):
    self._parser = ConfigParser()

    GDebug.printMesg(1,'Filters Configuration File: "%s"' % location)

    fileHandle = openResource(location)

    try:
      self._parser.readfp(fileHandle)

      try:
        fileHandle.close()
      except AttributeError:
        pass

    except DuplicateSectionError:
      raise InvalidFormatError, 'The file has duplicate source definitions.'
    except MissingSectionHeaderError:
      raise InvalidFormatError, 'The file has no source definitions.'
    except:
      print'The file cannot be parsed. %s :: %s' % (sys.exc_type, sys.exc_value)
      raise InvalidFormatError, 'The file cannot be parsed.'


    # Pre-load the mappings section, as it's
    # needed for interpreting the other sections.
    for option in self._parser.options('mappings'):
      self.mappings[string.lower(option)] = self._parser.get('mappings',option)

    # Get the base directory for filters
    filterBase = gConfigReports('FiltersDir')
    if filterBase[0] != '/':
      filterBase = os.path.join(GConfig.getInstalledBase('reports_shared', 'common_shared'), filterBase)

    self.filterBase = filterBase

    # Pull in all the other sections
    for section in self._parser.sections():
      if section != 'mappings':

        # First, translate any mappings
        transparts = string.split(section,':')
        translookup = string.join(transparts[:-1],':')
        if len(transparts) > 1 and \
           self.mappings.has_key(transparts[0]):
          translated = "%s:%s" % (self.mappings[translookup], transparts[-1])
        else:
          translated = section
        self.filters[translated] = {}

        # Then pull in the config variables
        for option in self._parser.options(section):
          val = self._parser.get(section, option)

          # If this is the template name, prepend the path
          if option == 'template' and val[0] != '/':
            if translated[:13] == 'GNUe:Reports:':
              val = "%s/%s/%s" % (filterBase,
                       string.split(translated,':')[2],
                       val)
            else:
              val = filterBase + val

          self.filters[translated][option] = val


  def getMappings(self):
    return tuple(self.mappings)


  def getFilters(self, namespace=""):

    filters = []

    if namespace != "":
      namespace += ':'

    for param in self.filters.keys():
      if namespace == "" or param[:len(namespace)] == namespace:
        filters.append((param,self.filters[param]))

    return filters

