#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2005 Free Software Foundation
#
# FILE:
# GRRun.py
#
# DESCRIPTION:
# Class that loads and populates a report
#
# NOTES:
#


from gnue.reports import VERSION
from gnue.common.apps.GClientApp import GClientApp
from gnue.common.utils.FileUtils import openResource, dyn_import
from gnue.reports.base import GRFilters, GRExceptions
from gnue.reports.base.GRConfig import ConfigOptions
import string, os, sys
import tempfile

class PostProcessor(GClientApp):

  #
  # GClientApp() overrides
  #
  VERSION = VERSION
  COMMAND = "gnue-reports-filter"
  NAME = "GNUe Reports Post-Processor"
  USAGE = "%s file filter-name ..." % GClientApp.USAGE
  SUMMARY = \
      _("The GNUe Reports Post-Processor takes the XML output of gnue-reports and applies filters to generate formatted output.")
  USE_DATABASE_OPTIONS = False
  USE_CONNECTIONS = False

  #
  # Initialize the class
  #
  def __init__(self):
    self.addCommandOption ('destination', 'd', 'destination',
        argument = 'dest', default = '-',
        help = _("Where should the report be output to?  The value of this "
            "depends on the destination type (e.g., if sending to printer, "
            "then -d specifies the printer name; if sending via email, then "
            "-d specifies the email address.) If <dest> is \"-\", then output is "
            "sent to stdout. "))

    self.addCommandOption ('destination_type', 'D', 'destination-type',
        argument = 'type', default = 'file',
        help = _("This specifies how the report should be output. The currently "
            "supported values for <type> are file [default], printer, email, "
            "and fax. Note that printer, email, and fax are sent via the "
            "server\'s machine, not the client\'s machine.  To "
            "\nNOTE: Only file, printer, and email are currently implemented!"))

    self.addCommandOption ('destination_options', None, 'destination-options',
        argument = 'opts',
        help = _("Options to pass to the destination process. Available options are "
            "specific to the type of destination. "
            "Example: \'--destination-options \"-o nobanner\" \'"))

    self.addCommandOption ('filter_options', 'F', 'filter-options',
        argument = 'opts',
        help = _("Options to pass to the filter process. Available options are "
            "specific to the filter.  --list-filters will list available filters "
            "and their options.  Example: \'--filter-options \"paper=letter margin=1,1,1,1\" \'"))

    GClientApp.__init__(self, application='reports',defaults=ConfigOptions)
    self.configurationManager.registerAlias('gConfigReports', 'reports')


  def run(self):

    #
    # Are we doing a simple help-like operation?
    #

    # assign report file from 1st free argument
    try:
      reportfile = self.ARGUMENTS[0]
    except:
      self.handleStartupError ("No Report Definition File Specified.")

    try:
      filter = self.ARGUMENTS[1]
    except:
      self.handleStartupError ("No Filter Specified.")


    destination = self.OPTIONS["destination"]
    destinationType = self.OPTIONS["destination_type"]
    destinationOptions = self.OPTIONS["destination_options"]
    filterOptions = self.OPTIONS['filter_options']


    #
    # Now, run an engine instance
    #

    filters = GRFilters.GRFilters()

    if reportfile == '-':
      fileHandle = sys.stdin
    else:
      try:
        fileHandle = openResource(reportfile)
      except IOError, msg:
        self.handleStartupError(
          "Unable to read the requested report output\n  (%s)\n\n" % reportFile \
          + "Additional Information:\n  %s" % msg)

    if destinationType == 'file' and destination == '-':
      # Handle the special case of stdout
      destAdapter = dyn_import('gnue.reports.adapters.destinations.file.stdout')
    else:
      # Load a destination adapter
      destAdapter = dyn_import('gnue.reports.adapters.destinations.%s.%s' % \
                      (destinationType, gConfigReports('%sAdapter' % destinationType)))

    try:
      # Create a Destination Adapter
      dest = destAdapter.DestinationAdapter(destination, destinationOptions)

      # Get the namespace from the xml file (we're cheating here,
      # as we don't really want to load the entire XML file at this point)
      namespace = None

      prereadLines = []

      line = fileHandle.readline()
      prereadLines.append(line)
      while line and not namespace:
        if 'xmlns="GNUe:Reports:' in line:
          try:
            namespace=line.split('xmlns="')[1].split('"')[0]
          except:
            continue
        line = fileHandle.readline()
        prereadLines.append(line)

      if not namespace:
        fileHandle.close()
        self.handleStartupError(_("The file does not appear to be output from gnue-reports or is in an unsupported document type. (Perhaps gnue-reports was run with the -X option?)"))


      # Create a Filter Adapter
      filtAdapter = filters.loadFilter(namespace,
                                       filter,
                                       dest,
                                       filterOptions)

      dest = filtAdapter.open()
      line = string.join(prereadLines)
      while line:
        dest.write(line)
        sys.stdout.write(line)
        line = fileHandle.readline()

      # Cleanup
      filtAdapter.close()


    except GRExceptions.ReportMarkupException, msg:
      self.handleStartupError (msg)
    except GRExceptions.ReportUnreadable, msg:
      self.handleStartupError (msg)
    except IOError, msg:
      self.handleStartupError (msg)

    sys.exit()


if __name__ == '__main__':
  GRRun().run()

