/*
 *  Copyright (C) 2004 Morten Fjord-Larsen
 *  Copyright (C) 2005 Kouji TAKAO <kouji@netlab.jp>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <glib/gi18n.h>

#include "util.h"
#include "general-password.h"

static GPassPasswordClass *parent_class = NULL;

static void
general_password_instance_init(GTypeInstance *instance, gpointer g_class)
{
    GPassGeneralPassword *self = GPASS_GENERAL_PASSWORD(instance);

    GPASS_ENTRY(self)->type = "general";
    self->hostname = g_strdup("");
}

enum {
    PROP_0,
    PROP_HOSTNAME,
};

static void
general_password_instance_set_property(GObject *object, guint prop_id,
                                       const GValue *value, GParamSpec *pspec)
{
    GPassGeneralPassword *self = GPASS_GENERAL_PASSWORD(object);
  
    switch (prop_id) {
    case PROP_HOSTNAME:
        g_free(self->hostname);
        self->hostname = g_value_dup_string(value);
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
        break;
    }
}

static void
general_password_instance_get_property(GObject *object, guint prop_id,
                                       GValue *value, GParamSpec *pspec)
{
    GPassGeneralPassword *self = GPASS_GENERAL_PASSWORD(object);

    switch (prop_id) {
    case PROP_HOSTNAME:
        g_value_set_static_string(value, self->hostname);
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
        break;
    }
}

static void
general_password_instance_finalize(GObject *object)
{
    GPassGeneralPassword *self = GPASS_GENERAL_PASSWORD(object);
    
    g_free(self->hostname);
    G_OBJECT_CLASS(parent_class)->finalize(object);
}

static const gchar *
general_password_class_nick(void)
{
    return _("General");
}

static const gchar *
general_password_class_default_launcher(void)
{
    return "gnome-open @hostname@";
}

static void
general_password_class_attributes(GPassEntryClass *entry_class,
                                  GPassAttributeList *attributes)
{
    GObjectClass *g_object_class = G_OBJECT_CLASS(entry_class);
    GPassAttribute *attr;
    GParamSpec *spec;
    
    GPASS_ENTRY_CLASS(parent_class)->attributes(entry_class, attributes);
    
    spec = g_object_class_find_property(g_object_class, "hostname");
    attr = gpass_param_spec_to_attribute(spec, GPASS_ATTRIBUTE_TYPE_URL);
    gpass_attribute_list_append(attributes, attr);
    g_object_unref(attr);
}

static void
general_password_instance_set_attributes(GPassEntry *entry,
                                         GPassAttributeList *attributes)
{
    GPassGeneralPassword *self = GPASS_GENERAL_PASSWORD(entry);
    GPassAttribute *attr;

    GPASS_ENTRY_CLASS(parent_class)->set_attributes(entry, attributes);
    
    attr = gpass_attribute_list_lookup(attributes, "hostname");
    if (attr != NULL) {
        g_object_set_property(G_OBJECT(self), "hostname", attr->value);
    }
}

static void
general_password_instance_get_attributes(GPassEntry *entry,
                                         GPassAttributeList *attributes)
{
    GPassGeneralPassword *self = GPASS_GENERAL_PASSWORD(entry);
    GPassAttribute *attr;
    
    GPASS_ENTRY_CLASS(parent_class)->get_attributes(entry, attributes);
    
    attr = gpass_attribute_list_lookup(attributes, "hostname");
    if (attr != NULL) {
        gpass_attribute_set(attr, self->hostname);
    }
}

static gboolean
general_password_instance_include(GPassEntry *entry, const gchar *string)
{
    GPassGeneralPassword *self = GPASS_GENERAL_PASSWORD(entry);
    
    if (GPASS_ENTRY_CLASS(parent_class)->include(entry, string)) {
        return TRUE;
    }
    if (strstr(self->hostname, string)) {
        return TRUE;
    }
    return FALSE;
}

static gboolean
general_password_instance_equal(GPassEntry *entry, GPassEntry *target)
{
    GPassGeneralPassword *self = GPASS_GENERAL_PASSWORD(entry);
    GPassGeneralPassword *general;

    if (!GPASS_ENTRY_CLASS(parent_class)->equal(entry, target)) {
        return FALSE;
    }
    general = GPASS_GENERAL_PASSWORD(target);
    if (gpass_strcmp(self->hostname, general->hostname) != 0) {
        return FALSE;
    }
    return TRUE;
}

static void
general_password_class_init(gpointer g_class, gpointer g_class_data)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS(g_class);
    GPassEntryClass *entry_class = GPASS_ENTRY_CLASS(g_class);

    parent_class = g_type_class_peek_parent(g_class);

    entry_class->nick = general_password_class_nick;
    entry_class->default_launcher = general_password_class_default_launcher;
    entry_class->attributes = general_password_class_attributes;
    
    gobject_class->set_property = general_password_instance_set_property;
    gobject_class->get_property = general_password_instance_get_property;
    gobject_class->finalize = general_password_instance_finalize;
    entry_class->set_attributes = general_password_instance_set_attributes;
    entry_class->get_attributes = general_password_instance_get_attributes;
    entry_class->include = general_password_instance_include;
    entry_class->equal = general_password_instance_equal;
    
    g_object_class_install_property
        (gobject_class, PROP_HOSTNAME,
         g_param_spec_string("hostname", _("Hostname"),
                             _("Hostname"),
                             NULL, G_PARAM_READWRITE));
}

GType
gpass_general_password_get_type(void)
{
    static GType type = 0;
    
    if (type == 0) {
        static const GTypeInfo info = {
            sizeof(GPassGeneralPasswordClass),
            NULL,
            NULL,
            general_password_class_init,
            NULL,
            NULL,
            sizeof(GPassGeneralPassword),
            0,
            general_password_instance_init
        };
        
        type = g_type_register_static(GPASS_TYPE_PASSWORD,
                                      "GPassGeneralPassword", &info, 0);
    }
    return type;
}
