/***************************************************************************
                          gpc_app.c  -  description
                             -------------------
    begin                : Wed May 17 2000
    copyright            : (C) 2000 by Thierry Florac
    email                : tflorac@free.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <gdk-pixbuf/gnome-canvas-pixbuf.h>

#include "pg_connection.h"
#include "pg_query.h"

#include "gpc_app.h"
#include "gpc_app_intf.h"

#include "gpc_options.h"
#include "gpc_options_intf.h"

#include "gpc_print.h"

#include "gpc_query.h"
#include "gpc_query_intf.h"

#include "gpc_film.h"
#include "gpc_film_infos.h"
#include "gpc_film_infos_intf.h"

#include "gpc_image.h"
#include "gpc_image_infos.h"
#include "gpc_image_infos_intf.h"

#include "gpc_utils.h"

#include "about_dialog_intf.h"

#include "support.h"


 /*
 **
 **  User internal variables, callbacks and functions
 **
 */

void
select_new_directory_callback (GtkFileSelection *selector, gpointer user_data);

void
select_new_filename_callback (GtkFileSelection *selector, gpointer user_data);

void
gpc_app_refresh_thumbnails (void);

void
gpc_app_resize_image (void);


 /*
 **
 **  Interface callbacks
 **
 */
 
/* Retrieve database param of integer type */
gint 
gpc_app_get_int_param (pgConnection *connection, gchar *param, gint defvalue)
{
  gchar *value;
  gint   result;

  value = gpc_app_get_string_param (connection, param, "0");
  if ((value == "0") && gpc_error)
    result = defvalue;
  else
    result = atoi (value);
  g_free(value);
  return result;
}


/* Set database param value of integer type */
gboolean 
gpc_app_set_int_param (pgConnection *connection, gchar *param, gchar *label, gint value)
{
  gchar    *svalue;
  gboolean  result;

  svalue = g_strdup_printf ("%d", value);
  result = gpc_app_set_string_param (connection, param, label, svalue);
  g_free (svalue);
  return result;
}


/* Retrieve database param of string type */
gchar* 
gpc_app_get_string_param (pgConnection *connection, gchar *param, gchar *defvalue)
{
  gchar   *sql;
  gchar   *sql_param;
  pgQuery *query;
  int      tuples;
  gchar   *result;

  result = 0;
  gpc_error = 0;
  sql_param = get_sql_string (param);
  sql = g_strdup_printf ("select * from GPC_PARAM where CODE = %s", sql_param);
  query = (pgQuery*)  pg_connection_query (connection, sql);
  tuples = pg_query_open (query);
  if (tuples < 0) {
    gpc_error = GPC_ERROR_QUERY_NOT_OPENED;
    result = g_strdup (defvalue);
  }
  else
  if (tuples == 0) {
    gpc_error = GPC_ERROR_PARAM_NOT_EXISTS;
    result = g_strdup (defvalue);
  }
  else
    result = g_strdup (pg_query_get_fieldvalue_byname (query, "VALUE"));
  g_free (sql_param);
  g_free (sql);
  pg_query_free (query);
  return result;
}


/* Set database param value of string type */
gboolean 
gpc_app_set_string_param (pgConnection *connection, gchar *param, gchar *label, gchar *value)
{
  gchar    *sql;
  gchar    *sql_param;
  gchar    *sql_label;
  gchar    *sql_value;
  gchar    *exists;
  gboolean result;

  exists = gpc_app_get_string_param (connection, param, "-NO-PARAM-");
  sql_param = get_sql_string (param);
  sql_label = get_sql_string (label);
  sql_value = get_sql_string (value);
  if (!g_strcasecmp (exists, "-NO-PARAM-"))
    sql = g_strdup_printf ("insert into GPC_PARAM (CODE, LABEL, VALUE) "
                           "values (%s, %s, %s) ", sql_param, sql_label, sql_value);
  else
    sql = g_strdup_printf ("update GPC_PARAM "
                           "set LABEL = %s, VALUE = %s "
                           "where (CODE = %s) ", sql_label, sql_value, sql_param);
  result = pg_connection_execute (connection, sql);
  g_free (exists);
  g_free (sql_value);
  g_free (sql_label);
  g_free (sql_param);
  g_free (sql);
  return result;
}


/* Check database for correct version
   Can create or update objects if needed */
gboolean 
gpc_app_check_database (pgConnection *connection)
{
  GtkWidget   *dialog = NULL;
  pgQuery     *query = NULL;
  gchar       *product = NULL;
  gint        version;

  if (!connection || !pg_connection_is_active (connection)) {
    gnome_dialog_run_and_close (GNOME_DIALOG (gnome_error_dialog (_("Connection is not active !"))));
    return FALSE;
  }
  product = gpc_app_get_string_param (connection, "PRODUCT_NAME", "-NO-VALUE-");
  version = gpc_app_get_int_param (connection, "DATABASE_VERSION", 0);
  if (g_strcasecmp (product, "gpc") ||        // PRODUCT_NAME param does not exists
      (version != GPC_DATABASE_VERSION)) {    // Database is not up-to-date
    g_free (product);
    if (version > GPC_DATABASE_VERSION) {
      gnome_dialog_run_and_close (GNOME_DIALOG (gnome_error_dialog (_("Your current version of GPC is older than the last one which accessed this database.\n"
                                                                      "Database format has been modified between the two releases, "
                                                                      "so using GPC with this database could handle data incorrectly !!\n"
                                                                      "You MUST upgrade your GPC product before going on..."))));
      return FALSE;
    }
    modal_result = GPC_MODAL_RESULT_UNKNOWN;
    dialog = gnome_question_dialog (_("Your Postgres database is not up-to-date according to current GPC version.\n"
                                      "Would you like to update current Postgres database to create/update GPC tables ?"),
                                    gpc_app_check_reply_callback, NULL);
    gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
    if (modal_result == GPC_MODAL_RESULT_OK) {       // OK button pressed
      if (!pg_connection_start (connection)) {
        pg_connection_show_error (connection, _("An error occured :"), _("Can't start transaction on GPC database...!"));
        pg_query_free (query);
        return FALSE;
      }
      query = pg_query_new (connection);
      while (version != GPC_DATABASE_VERSION) {
        switch (version) {
          case 0 : pg_query_set_sql (query, "create table GPC_PARAM "
                                            "(ID serial,"
                                            " CODE varchar(30),"
                                            " LABEL varchar(50),"
                                            " VALUE varchar(50),"
                                            " primary key (CODE))");
                   if (!pg_query_execute (query)) {
                     pg_connection_show_error (connection, _("An error occured"), _("Can't create GPC_PARAM table...!"));
                     pg_query_free (query);
                     return FALSE;
                   }
                   pg_query_set_sql (query, "create table GPC_KEY "
                                            "(ID serial,"
                                            " CODE varchar(30),"
                                            " primary key (CODE))");
                   if (!pg_query_execute (query)) {
                     pg_connection_show_error (connection, _("An error occured"), _("Can't create GPC_KEY table...!"));
                     pg_query_free (query);
                     return FALSE;
                   }
                   pg_query_set_sql (query, "create table GPC_FILM_KIND "
                                            "(ID serial,"
                                            " CODE varchar(30),"
                                            " primary key (CODE))");
                   if (!pg_query_execute (query)) {
                     pg_connection_show_error (connection, _("An error occured"), _("Can't create GPC_FILM_KIND table...!"));
                     pg_query_free (query);
                     return FALSE;
                   }
                   pg_query_set_sql (query, "create table GPC_FILM_MAKER "
                                            "(ID serial,"
                                            " CODE varchar(30),"
                                            " primary key (CODE))");
                   if (!pg_query_execute (query)) {
                     pg_connection_show_error (connection, _("An error occured"), _("Can't create GPC_FILM_MAKER table...!"));
                     pg_query_free (query);
                     return FALSE;
                   }
                   pg_query_set_sql (query, "create table GPC_FILM_MODEL "
                                            "(ID serial,"
                                            " MAKER varchar(30),"
                                            " CODE varchar(30),"
                                            " primary key (MAKER, CODE))");
                   if (!pg_query_execute (query)) {
                     pg_connection_show_error (connection, _("An error occured"), _("Can't create GPC_FILM_MODEL table...!"));
                     pg_query_free (query);
                     return FALSE;
                   }
                   pg_query_set_sql (query, "create table GPC_CAMERA_MAKER "
                                            "(ID serial,"
                                            " CODE varchar(30),"
                                            " primary key (CODE))");
                   if (!pg_query_execute (query)) {
                     pg_connection_show_error (connection, _("An error occured"), _("Can't create GPC_CAMERA_MAKER table...!"));
                     pg_query_free (query);
                     return FALSE;
                   }
                   pg_query_set_sql (query, "create table GPC_CAMERA_MODEL "
                                            "(ID serial,"
                                            " MAKER varchar(30),"
                                            " CODE varchar(30),"
                                            " primary key (MAKER, CODE))");
                   if (!pg_query_execute (query)) {
                     pg_connection_show_error (connection, _("An error occured"), _("Can't create GPC_CAMERA_MODEL table...!"));
                     pg_query_free (query);
                     return FALSE;
                   }
                   pg_query_set_sql (query, "create table GPC_LENS_MAKER "
                                            "(ID serial,"
                                            " CODE varchar(30),"
                                            " primary key (CODE))");
                   if (!pg_query_execute (query)) {
                     pg_connection_show_error (connection, _("An error occured"), _("Can't create GPC_LENS_MAKER table...!"));
                     pg_query_free (query);
                     return FALSE;
                   }
                   pg_query_set_sql (query, "create table GPC_LENS_MODEL "
                                            "(ID serial,"
                                            " MAKER varchar(30),"
                                            " CODE varchar(30),"
                                            " primary key (MAKER, CODE))");
                   if (!pg_query_execute (query)) {
                     pg_connection_show_error (connection, _("An error occured"), _("Can't create GPC_LENS_MODEL table...!"));
                     pg_query_free (query);
                     return FALSE;
                   }
                   pg_query_set_sql (query, "create table GPC_FILM "
                                            "(ID serial,"
                                            " PATHNAME varchar(255),"
                                            " KIND varchar(30),"
                                            " NUMBER varchar(30),"
                                            " MAKER varchar(30),"
                                            " MODEL varchar(30),"
                                            " ISO int2,"
                                            " SIZE int2,"
                                            " CAMERA_MAKER varchar(30),"
                                            " CAMERA_MODEL varchar(30),"
                                            " PERIOD varchar(30),"
                                            " DESCRIPTION text,"
                                            " primary key (PATHNAME))");
                   if (!pg_query_execute (query)) {
                     pg_connection_show_error (connection, _("An error occured"), _("Can't create GPC_FILM table...!"));
                     pg_query_free (query);
                     return FALSE;
                   }
                   pg_query_set_sql (query, "create table GPC_IMAGE "
                                            "(ID serial,"
                                            " FILM_ID int4,"
                                            " FILENAME varchar(255),"
                                            " WIDTH int4,"
                                            " HEIGHT int4,"
                                            " MIME_TYPE varchar(255),"
                                            " NUMBER varchar(30),"
                                            " LENS_MAKER varchar(30),"
                                            " LENS_MODEL varchar(30),"
                                            " FOCAL int2,"
                                            " APERTURE float4,"
                                            " SPEED varchar(6),"
                                            " INTEREST_NOTE int2,"
                                            " QUALITY_NOTE int2,"
                                            " DESCRIPTION text,"
                                            " primary key (FILM_ID, FILENAME))");
                   if (!pg_query_execute (query)) {
                     pg_connection_show_error (connection, _("An error occured"), _("Can't create GPC_IMAGE table...!"));
                     pg_query_free (query);
                     return FALSE;
                   }
                   pg_query_set_sql (query, "create table GPC_KEYWORD "
                                            "(ID serial,"
                                            " CODE varchar(30),"
                                            " FILM_ID int4,"
                                            " IMAGE_ID int4,"
                                            " primary key (CODE, FILM_ID, IMAGE_ID))");
                   if (!pg_query_execute (query)) {
                     pg_connection_show_error (connection, _("An error occured"), _("Can't create GPC_KEYWORD table...!"));
                     pg_query_free (query);
                     return FALSE;
                   }
                   pg_query_set_sql (query, "create rule ON_GPC_FILM_DELETE_IMAGE as "
                                            "on delete to GPC_FILM "
                                            "do delete from GPC_IMAGE where FILM_ID = old.ID");
                   if (!pg_query_execute (query)) {
                     pg_connection_show_error (connection, _("An error occured"), _("Can't create ON_GPC_FILM_DELETE rule...!"));
                     pg_query_free (query);
                     return FALSE;
                   }
                   pg_query_set_sql (query, "create rule ON_GPC_FILM_DELETE_KEYWORD as "
                                            "on delete to GPC_FILM "
                                            "do delete from GPC_KEYWORD where FILM_ID = old.ID");
                   if (!pg_query_execute (query)) {
                     pg_connection_show_error (connection, _("An error occured"), _("Can't create ON_GPC_FILM_DELETE rule...!"));
                     pg_query_free (query);
                     return FALSE;
                   }
                   pg_query_set_sql (query, "create rule ON_GPC_IMAGE_DELETE_KEYWORD as "
                                            "on delete to GPC_IMAGE "
                                            "do delete from GPC_KEYWORD where IMAGE_ID = old.ID");
                   if (!pg_query_execute (query)) {
                     pg_connection_show_error (connection, _("An error occured"), _("Can't create ON_GPC_FILM_DELETE rule...!"));
                     pg_query_free (query);
                     return FALSE;
                   }
                   break;
          case 1 : pg_query_set_sql (query, "alter table GPC_IMAGE "
                                            "add LATITUDE float4");
                   if (!pg_query_execute (query)) {
                     pg_connection_show_error (connection, _("An error occured"), _("Can't add field LATITUDE in GPC_IMAGE table...!"));
                     pg_query_free (query);
                     return FALSE;
                   }
                   pg_query_set_sql (query, "alter table GPC_IMAGE "
                                            "add LONGITUDE float4");
                   if (!pg_query_execute (query)) {
                     pg_connection_show_error (connection, _("An error occured"), _("Can't add field LONGITUDE in GPC_IMAGE table...!"));
                     pg_query_free (query);
                     return FALSE;
                   }
                   pg_query_set_sql (query, "alter table GPC_IMAGE "
                                            "add ELEVATION float4");
                   if (!pg_query_execute (query)) {
                     pg_connection_show_error (connection, _("An error occured"), _("Can't add field ELEVATION in GPC_IMAGE table...!"));
                     pg_query_free (query);
                     return FALSE;
                   }
                   pg_query_set_sql (query, "alter table GPC_IMAGE "
                                            "add DIRECTION float4");
                   if (!pg_query_execute (query)) {
                     pg_connection_show_error (connection, _("An error occured"), _("Can't add field DIRECTION in GPC_IMAGE table...!"));
                     pg_query_free (query);
                     return FALSE;
                   }
                   pg_query_set_sql (query, "alter table GPC_IMAGE "
                                            "add PITCH float4");
                   if (!pg_query_execute (query)) {
                     pg_connection_show_error (connection, _("An error occured"), _("Can't add field PITCH in GPC_IMAGE table...!"));
                     pg_query_free (query);
                     return FALSE;
                   }
                   pg_query_set_sql (query, "alter table GPC_IMAGE "
                                            "add ROLL float4");
                   if (!pg_query_execute (query)) {
                     pg_connection_show_error (connection, _("An error occured"), _("Can't add field ROLL in GPC_IMAGE table...!"));
                     pg_query_free (query);
                     return FALSE;
                   }
                   break;
        }
        version++;
      }
      gpc_app_set_string_param (connection, "PRODUCT_NAME", "GNOME Photo Collector", "gpc");
      gpc_app_set_int_param (connection, "DATABASE_VERSION", "GPC Database Version Level", version);
      if (pg_connection_commit (connection))
        gnome_dialog_run_and_close (GNOME_DIALOG (gnome_error_dialog (_("Postgres database was updated successfully...!\n"
                                                                        "You should now grant access to authorised users..."))));
      else {
        pg_connection_show_error (connection, _("An error occured :"), _("Can't validate modifications to database tables..."));
        if (!pg_connection_rollback (connection))
          pg_connection_show_error (connection, _("An error occured :"), _("Can't rollback modifications to database tables...\n"
                                                                           "Your database should be in an unstable state...!!"));
      }
      pg_query_free (query);
    }
    else { // Cancel button or "close box"
      gnome_dialog_run_and_close (GNOME_DIALOG (gnome_error_dialog (_("Postgres database was not updated to store GPC tables...!"))));
      return FALSE;
    }
  }
  else
    g_free (product);
  return TRUE;
}


/* Check for a keyword in a reference list */
gboolean
gpc_app_check_keyword (pgConnection *connection, GList *list, gchar *value, gchar *table)
{
  if (!connection)
    return FALSE;
  if (!list)
    list = gpc_app_get_keywords (connection, table);
  return (g_list_find_custom (list, value, (GCompareFunc) g_strcasecmp) != NULL);
}


/* Retrieve a list of keywords extracted from a given database table */
GList*
gpc_app_get_keywords (pgConnection *connection, gchar *table)
{
  pgQuery *query;
  gchar   *sql;
  GList   *result = NULL;

  if (!connection)
    return NULL;
  result = g_list_append (result, "");
  sql = g_strdup_printf ("select distinct CODE from %s order by CODE", table);
  query = pg_query_new_with_sql (connection, sql);
  if (pg_query_open (query) > 0) {
    pg_query_move_first (query);
    while (!pg_query_at_EOF (query)) {
      result = g_list_append (result, g_strdup (pg_query_get_fieldvalue_byname (query, "CODE")));
      pg_query_move_next (query);
    }
  }
  pg_query_free (query);
  g_free (sql);
  return result;
}


/* Store a list of keywords in a database table */
gboolean
gpc_app_store_keywords (pgConnection *connection, gchar *table, GList *list)
{
  pgQuery  *query;
  gchar    *sql;
  gchar    *sql_code;
  guint    index;
  guint    length;
  gboolean result = TRUE;

  if (!connection || !list || !table)
    return FALSE;
  query = pg_query_new (connection);
  length = g_list_length (list);
  for (index = 0; (index < length) && result; index++) {
    sql_code = get_sql_string (g_list_nth_data (list, index));
    sql = g_strdup_printf ("insert into %s (CODE) values (%s)", table, sql_code);
    pg_query_set_sql (query, sql);
    result = result && pg_query_execute (query);
    g_free (sql);
    g_free (sql_code);
  }
  return result;
}


/* Retrieve a list of codes in a given table, relatives to a given maker */
GList*
gpc_app_get_maker_codes (pgConnection *connection, gchar *table, gchar *maker)
{
  pgQuery *query;
  gchar   *sql;
  gchar   *sql_maker;
  GList   *result = NULL;

  if (!connection)
    return NULL;
  result = g_list_append (result, "");
  sql_maker = get_sql_string (maker);
  sql = g_strdup_printf ("select distinct CODE from %s where MAKER = %s order by CODE", table, sql_maker);
  query = pg_query_new_with_sql (connection, sql);
  if (pg_query_open (query) > 0) {
    pg_query_move_first (query);
    while (!pg_query_at_EOF (query)) {
      result = g_list_append (result, g_strdup (pg_query_get_fieldvalue_byname (query, "CODE")));
      pg_query_move_next (query);
    }
  }
  pg_query_free (query);
  g_free (sql);
  g_free (sql_maker);
  return result;
}


/* Store a list of codes in a given table, corresponding to a given maker */
gboolean
gpc_app_store_maker_codes (pgConnection *connection, gchar *table, gchar *maker, GList *list)
{
  pgQuery  *query;
  gchar    *sql;
  gchar    *sql_maker;
  gchar    *sql_code;
  guint    index;
  guint    length;
  gboolean result = TRUE;

  if (!connection || !list || !table || !maker)
    return FALSE;
  query = pg_query_new (connection);
  length = g_list_length (list);
  sql_maker = get_sql_string (maker);
  for (index = 0; (index < length) && result; index++) {
    sql_code = get_sql_string (g_list_nth_data (list, index));
    sql = g_strdup_printf ("insert into %s (MAKER, CODE) values (%s, %s)", table, sql_maker, sql_code);
    pg_query_set_sql (query, sql);
    result = result && pg_query_execute (query);
    g_free (sql);
    g_free (sql_code);
  }
  g_free (sql_maker);
  return result;
}


/* Read GPC preferences from configuration file */
void 
gpc_app_read_options ()
{
  gpc_repository = gnome_config_get_string ("/gphotocoll/options/repository");
  gpc_zoom_mode = gnome_config_get_int ("/gphotocoll/options/zoom_mode=0");
  gpc_confirm_insert_refs = gnome_config_get_bool ("/gphotocoll/options/confirm_insert_refs=true");
  gpc_delete_removed_files = gnome_config_get_bool ("/gphotocoll/options/delete_removed_files=false");
  gpc_get_hidden_files = gnome_config_get_bool ("/gphotocoll/options/get_hidden_files=false");
  gpc_store_password = gnome_config_get_bool ("/gphotocoll/options/store_password=true");
  gpc_auto_connect = gnome_config_get_bool ("/gphotocoll/options/auto_connect=false");
  gpc_store_thumbnails = gnome_config_get_bool ("/gphotocoll/thumbnails/store=false");
}


/* Write GPC preferences in configuration file */
void 
gpc_app_write_options ()
{
  gint x;
  gint y;
  gint width;
  gint height;

  gnome_config_set_int ("/gphotocoll/options/zoom_mode", gpc_zoom_mode);
  gnome_config_set_bool ("/gphotocoll/options/confirm_insert_refs", gpc_confirm_insert_refs);
  gnome_config_set_bool ("/gphotocoll/options/delete_removed_files", gpc_delete_removed_files);
  gnome_config_set_bool ("/gphotocoll/options/get_hidden_files", gpc_get_hidden_files);
  gnome_config_set_bool ("/gphotocoll/options/store_password", gpc_store_password);
  gnome_config_set_bool ("/gphotocoll/options/auto_connect", gpc_auto_connect);
  gnome_config_set_bool ("/gphotocoll/thumbnails/store", gpc_store_thumbnails);
  if (gpc_app->window) {
    gdk_window_get_root_origin (GTK_WIDGET (gpc_app)->window, &x, &y);
    gdk_window_get_size (GTK_WIDGET (gpc_app)->window, &width, &height);
    gnome_config_set_int ("/gphotocoll/geometry/x", x);
    gnome_config_set_int ("/gphotocoll/geometry/y", y);
    gnome_config_set_int ("/gphotocoll/geometry/width", width);
    gnome_config_set_int ("/gphotocoll/geometry/height", height);
  }
  gnome_config_sync ();
}


/* Display a list of films, extracted from a given query, in GPC main window */
void
gpc_app_update_films_list (pgQuery *query)
{
  gchar     *fields[4];

  gtk_clist_clear (films_clist);
  if (pg_query_get_recordcount (query) > 0) {
    pg_query_move_first (query);
    while (!pg_query_at_EOF (query)) {
      fields[0] = g_strdup (pg_query_get_fieldvalue_byname (query, "ID"));
      fields[1] = g_strdup (pg_query_get_fieldvalue_byname (query, "NUMBER"));
      fields[2] = g_strdup (pg_query_get_fieldvalue_byname (query, "DESCRIPTION"));
      fields[3] = NULL;
      gtk_clist_append (films_clist, fields);
      pg_query_move_next (query);
    }
  }
  gpc_film_selection = -1;
}


/* Display a list of images, extracted from a given query, in GPC main window */
void
gpc_app_update_images_list (pgQuery *query)
{
  gchar     *fields[4];

  gtk_clist_clear (images_clist);
  if (pg_query_get_recordcount (query) > 0) {
    pg_query_move_first (query);
    while (!pg_query_at_EOF (query)) {
      fields[0] = g_strdup (pg_query_get_fieldvalue_byname (query, "ID"));
      fields[1] = g_strdup (pg_query_get_fieldvalue_byname (query, "NUMBER"));
      fields[2] = g_strdup (pg_query_get_fieldvalue_byname (query, "DESCRIPTION"));
      fields[3] = NULL;
      gtk_clist_append (images_clist, fields);
      pg_query_move_next (query);
    }
  }
  gpc_image_selection = -1;
}


/* Frees all internal lists used in GPC */
void
gpc_app_free_lists (void)
{
  gpc_list_free (gpc_keywords);
  gpc_list_free (gpc_film_kinds);
  gpc_list_free (gpc_film_makers);
  gpc_list_free (gpc_film_maker_codes);
  gpc_list_free (gpc_camera_makers);
  gpc_list_free (gpc_camera_maker_codes);
  gpc_list_free (gpc_lens_makers);
  gpc_list_free (gpc_lens_maker_codes);
  gpc_list_free (gpc_film_models);
  gpc_list_free (gpc_camera_models);
  gpc_list_free (gpc_lens_models);
  gpc_keywords = NULL;
  gpc_film_kinds = NULL;
  gpc_film_makers = NULL;
  gpc_film_maker_codes = NULL;
  gpc_camera_makers = NULL;
  gpc_camera_maker_codes = NULL;
  gpc_lens_makers = NULL;
  gpc_lens_maker_codes = NULL;
  gpc_film_models = NULL;
  gpc_camera_models = NULL;
  gpc_lens_models = NULL;
}


/* When a thumbnail is clicked, update selection in images list */
gint
gpc_app_image_item_event (GnomeCanvasItem *item, GdkEvent *event, gpointer data)
{
  GtkArg  args[1];
  gint    nbrows;
  gint    index;
  gchar  *ID;

  if (!gpc_images)
    return FALSE;
  args[0].name = "GnomeCanvasText::text";
  gtk_object_getv (GTK_OBJECT (data), 1, args);
  switch (event->type) {
    case GDK_BUTTON_PRESS:
      nbrows = pg_query_get_recordcount (gpc_images);
      index = 0;
      while (index < nbrows) {
        gtk_clist_get_text (images_clist, index, 0, &ID);
        if (!g_strcasecmp (ID, GTK_VALUE_STRING (args[0])))
          break;
        index++;
      }
      if (index < nbrows) {
        gtk_clist_select_row (images_clist, index, 0);
        gtk_clist_moveto (images_clist, index, 0, 0.5, 0);
      }
      if (((GdkEventButton *) event)->button == 3) {
        if (!gpc_images_popup_menu)
          gpc_images_popup_menu = create_images_popup_menu ();
        gtk_menu_popup (GTK_MENU (gpc_images_popup_menu), NULL, NULL, NULL, NULL, ((GdkEventButton *) event)->button, ((GdkEventButton *) event)->time);
        gtk_widget_set (lookup_widget (gpc_images_popup_menu, "zoom_in_popup_menu"),
                        "sensitive", FALSE,
                        NULL);
        gtk_widget_set (lookup_widget (gpc_images_popup_menu, "zoom_out_popup_menu"),
                        "sensitive", FALSE,
                        NULL);
        gtk_widget_set (lookup_widget (gpc_images_popup_menu, "zoom_normal_popup_menu"),
                        "sensitive", FALSE,
                        NULL);
        gtk_widget_set (lookup_widget (gpc_images_popup_menu, "zoom_fit_popup_menu"),
                        "sensitive", FALSE,
                        NULL);
      }
      break;

    case GDK_2BUTTON_PRESS:
      gtk_notebook_set_page (GTK_NOTEBOOK (lookup_widget (gpc_app, "right_notebook")), 1);
      on_images_clist_select_row (images_clist, gpc_image_selection, 0, NULL, NULL); 
      break;

    default:
      break;
  }
  return FALSE;
}


/* Refresh thumbnails list for selected film */
void
gpc_app_refresh_thumbnails (void)
{
  GpcImageInfos   *infos;
  GdkPixbuf       *image_pixbuf;
  GdkPixbuf       *thumb_pixbuf;
  GnomeCanvasItem *image_item;
  GnomeCanvasItem *text_item;
  gchar           *text;
  gchar           *begin_text;
  GtkWidget       *canvas_window;
  gint             ID;
  gchar           *pathname;
  gchar           *thumbname;
  gchar           *pathlib;
  gchar           *filename;
  gchar           *fullname;
  gchar           *fontname;
  gchar           *nbimages;
  gint             x;
  gint             y;
  gint             cols;
  gint             rows;
  gint             image_width;
  gint             image_height;
  gint             canvas_width;
  gint             canvas_height;
  gint             thumb_image_width;
  gint             thumb_image_height;
  gint             thumb_width;
  gint             thumb_height;
  gfloat           coeff_x;
  gfloat           coeff_y;
  gfloat           coeff;
  double           item_left;
  double           item_right;
  double           item_top;
  double           item_bottom;
  
  if (!gpc_connection || !pg_connection_is_active (gpc_connection))
    return;
  if (gpc_thumbnails_canvas) {
    canvas_window = lookup_widget (gpc_app, "thumbnails_scrolledwindow");
    gtk_object_destroy (GTK_OBJECT (gpc_thumbnails_canvas));
    gtk_widget_push_visual (gdk_imlib_get_visual ());
    gtk_widget_push_colormap (gdk_imlib_get_colormap ());
    gpc_thumbnails_canvas = gnome_canvas_new ();
    gtk_widget_pop_colormap ();
    gtk_widget_pop_visual ();
    gtk_widget_set_name (gpc_thumbnails_canvas, "thumbnails_canvas");
    gtk_widget_ref (gpc_thumbnails_canvas);
    gtk_object_set_data_full (GTK_OBJECT (gpc_app), "thumbnails_canvas", gpc_thumbnails_canvas,
                              (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (gpc_thumbnails_canvas);
    gtk_container_add (GTK_CONTAINER (canvas_window), gpc_thumbnails_canvas);
    gnome_canvas_set_scroll_region (GNOME_CANVAS (gpc_thumbnails_canvas), 0, 0, 1000, 1000);
    while (gtk_events_pending ())
      gtk_main_iteration ();
  }
  pathname = gpc_film_get_full_pathname (gpc_film_infos);
  thumbname = g_strconcat (pathname, ".gpc/", NULL);
  pathlib = g_strdup_printf (_("Pathname: %s"), pathname);
  gtk_label_set_text (GTK_LABEL (lookup_widget (gpc_app, "film_directory_label")), pathlib);
  g_free (pathlib);
  if (gpc_images) {
    nbimages = g_strdup_printf (_("%d images"), pg_query_get_recordcount (gpc_images));
    gtk_label_set_text (GTK_LABEL (lookup_widget (gpc_app, "film_images_label")), nbimages);
    g_free (nbimages);
    fontname = gnome_config_get_string ("/gphotocoll/thumbnails/font=-adobe-helvetica-medium-r-normal-*-*-100-*-*-p-*-iso8859-1");
    thumb_image_width = gnome_config_get_int ("/gphotocoll/thumbnails/size=100");
    thumb_image_height = gnome_config_get_int ("/gphotocoll/thumbnails/size=100");
    thumb_width = thumb_image_width + 10;
    thumb_height = thumb_image_height + 40;
    gdk_window_get_size (GTK_WIDGET (gpc_thumbnails_canvas)->window, &canvas_width, &canvas_height);
    cols = (gint) (canvas_width / thumb_width);
    if (cols == 0)
      cols = 1;
    rows = (gint) pg_query_get_recordcount (gpc_images) == 1 ? 1 : ((pg_query_get_recordcount (gpc_images)-1) / cols)+1;
    gnome_canvas_set_scroll_region (GNOME_CANVAS (gpc_thumbnails_canvas), 0, 0, thumb_width * cols, thumb_height * rows);
    gnome_canvas_scroll_to (GNOME_CANVAS (gpc_thumbnails_canvas), 0, 0);
    x = 0;
    y = 0;
    pg_query_move_first (gpc_images);
    while (!pg_query_at_EOF (gpc_images)) {
      ID = atoi (pg_query_get_fieldvalue_byname (gpc_images, "ID"));
      infos = gpc_image_get_infos (gpc_connection, ID);
      filename = gpc_image_get_filename (infos);
      if (gpc_store_thumbnails && g_file_test (thumbname, G_FILE_TEST_ISDIR)) {
        fullname = g_strconcat (thumbname, filename, NULL);
        if (!g_file_test (fullname, G_FILE_TEST_ISFILE)) {
          g_free (fullname);
          fullname = g_strconcat (pathname, filename, NULL);
        }
      }
      else
        fullname = g_strconcat (pathname, filename, NULL);
      image_pixbuf = gdk_pixbuf_new_from_file (fullname);
      if (gpc_thumbnails_canvas && image_pixbuf) {
        image_width = gdk_pixbuf_get_width (image_pixbuf);
        image_height = gdk_pixbuf_get_height (image_pixbuf);
        coeff_x = (gfloat) thumb_image_width / image_width;
        coeff_y = (gfloat) thumb_image_height / image_height;
        (coeff_x > coeff_y) ? (coeff = coeff_y) : (coeff = coeff_x);
        if (coeff > 1)
          coeff = 1.0;
        thumb_pixbuf = gdk_pixbuf_scale_simple (image_pixbuf, image_width * coeff, image_height * coeff, GDK_INTERP_NEAREST);
        if (thumb_pixbuf) {
          gdk_pixbuf_unref (image_pixbuf);
          image_item = gnome_canvas_item_new (gnome_canvas_root (GNOME_CANVAS (gpc_thumbnails_canvas)),
                                              gnome_canvas_pixbuf_get_type (),
                                              "pixbuf", thumb_pixbuf,
                                              "x", (gdouble) (x * thumb_width) + 5 + (thumb_width - (image_width * coeff))/2,
                                              "x_in_pixels", TRUE,
                                              "y", (gdouble) (y * thumb_height) + 5 + (thumb_height - (image_height * coeff))/2,
                                              "y_in_pixels", TRUE,
                                              NULL);
          gdk_pixbuf_unref (thumb_pixbuf);
          text = g_strdup_printf ("%d", ID);
          text_item = gnome_canvas_item_new (gnome_canvas_root (GNOME_CANVAS (gpc_thumbnails_canvas)),
                                             gnome_canvas_text_get_type (),
                                             "text", text,
                                             NULL);
          gtk_signal_connect (GTK_OBJECT (image_item), "event", (GtkSignalFunc) gpc_app_image_item_event, text_item);
          g_free (text);
          text_item = gnome_canvas_item_new (gnome_canvas_root (GNOME_CANVAS (gpc_thumbnails_canvas)),
                                             gnome_canvas_text_get_type (),
                                             "text", filename,
                                             "x", (gdouble) (x * thumb_width) + 5 + (thumb_width / 2),
                                             "y", (gdouble) (y * thumb_height) + 28 + thumb_image_height,
                                             "anchor", GTK_ANCHOR_CENTER | GTK_ANCHOR_NORTH,
                                             "font", fontname,
                                             "fill_color", "black",
                                             NULL);
          gnome_canvas_item_get_bounds (text_item, &item_left, &item_top, &item_right, &item_bottom);
          while ((item_right - item_left) > thumb_image_width) {
            begin_text = g_strndup (filename, strlen (filename)-4);
            g_free (filename);
            text = g_strconcat (begin_text, "...", NULL);
            g_free (begin_text);
            filename = text;
            gnome_canvas_item_set (text_item, "text", filename, NULL);
            gnome_canvas_item_get_bounds (text_item, &item_left, &item_top, &item_right, &item_bottom);
          }
        }
        if (++x == cols) {
          x = 0; 
          y++;
        }
        while (gtk_events_pending ())
          gtk_main_iteration ();
      }
      g_free (fullname);
      g_free (filename);
      pg_query_move_next (gpc_images);
    }
    g_free (fontname);
  }
  g_free (thumbname);
  g_free (pathname);
}


/* Resize currently selected image's canvas, corresponding to the current zoom factor */
void
gpc_app_resize_image (void)
{
  gchar *zoom;

  if (gpc_image_canvas && gpc_image_canvas_item) {
    gnome_canvas_set_scroll_region (GNOME_CANVAS (gpc_image_canvas),
                                    0, 0,
                                    gpc_image_width * gpc_image_zoom_factor,
                                    gpc_image_height * gpc_image_zoom_factor);
    gnome_canvas_item_set (gpc_image_canvas_item, 
                           "width_set", TRUE,
                           "width", gpc_image_width * gpc_image_zoom_factor,
                           "height_set", TRUE,
                           "height", gpc_image_height * gpc_image_zoom_factor,
                           NULL);
    zoom = g_strdup_printf (_("Zoom factor: %.2f"), gpc_image_zoom_factor);
    gtk_label_set_text (GTK_LABEL (lookup_widget (gpc_app, "image_zoom_label")), zoom);
    g_free (zoom);
  }
}


 /*
 **
 ** Start of main window callback functions
 **
 */


/* On REALIZE, update columns, reference global components and update the list of films */
void
on_gpc_app_realize_event               (GtkWidget       *widget,
                                        gpointer         user_data)
{
  films_clist = GTK_CLIST (lookup_widget (widget, "films_clist"));
  if (films_clist)
    gtk_clist_set_column_justification (films_clist, 0, GTK_JUSTIFY_RIGHT);

  images_clist = GTK_CLIST (lookup_widget (widget, "images_clist"));
  if (images_clist)
    gtk_clist_set_column_justification (images_clist, 0, GTK_JUSTIFY_RIGHT);

  gpc_thumbnails_canvas = lookup_widget (widget, "thumbnails_canvas");
  gpc_image_canvas = lookup_widget (widget, "image_canvas");
  gpc_image_zoom_factor = 1.0;

  gpc_app_read_options ();

  gpc_query_mode = FALSE;  
  if (!gpc_connection || !pg_connection_is_active (gpc_connection))
    return;
  if (gpc_films)
    pg_query_free (gpc_films);
  gpc_films = gpc_film_get_list (gpc_connection);
  gpc_app_update_films_list (gpc_films);
}


/* On MAP, restore previously saved image's position and size */
void
on_gpc_app_map                         (GtkWidget       *widget,
                                        gpointer         user_data)
{
  gdk_window_move_resize (GTK_WIDGET (gpc_app)->window, 
                          gnome_config_get_int ("/gphotocoll/geometry/x=20"),
                          gnome_config_get_int ("/gphotocoll/geometry/y=20"),
                          gnome_config_get_int ("/gphotocoll/geometry/width=600"),
                          gnome_config_get_int ("/gphotocoll/geometry/height=400"));
}


/* On DELETE, free a few global pointers, close connection and save preferences */ 
gboolean
on_gpc_app_delete_event                (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
  if (gpc_image_infos)
    gpc_image_infos_free (gpc_image_infos);
  if (gpc_images) {
    pg_query_free (gpc_images);
    gpc_images = NULL;
  }
  if (gpc_film_infos)
    gpc_film_infos_free (gpc_film_infos);
  if (gpc_films) {
    pg_query_free (gpc_films);
    gpc_films = NULL;
  }
  if (gpc_connection) {
    pg_connection_free (gpc_connection);
    gpc_connection = NULL;
  }

  if (gpc_repository)
    g_free (gpc_repository);
  if (gpc_last_directory)
    g_free (gpc_last_directory);  
  gpc_app_free_lists ( );

  gpc_app_write_options ();

  gtk_widget_destroy (GTK_WIDGET (gpc_film_dialog));
  gtk_widget_destroy (GTK_WIDGET (gpc_image_dialog));
  gtk_main_quit ();
  return FALSE;
}


/* On 'connect', close previously active connection and display new connection dialog */
void
on_connect_file_menu_activate          (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  on_close_file_menu_activate (NULL, NULL);
  if (gpc_connection)
    return;
  gpc_connection = pg_connection_new_with_dialog ();
  if (pg_connection_open (gpc_connection) && gpc_app_check_database (gpc_connection)) {
    if (gpc_query_mode)
      gpc_films = gpc_query_get_films (gpc_connection);
    else
      gpc_films = gpc_film_get_list (gpc_connection);
    gpc_app_update_films_list (gpc_films);
  }
  else {
    gnome_dialog_run_and_close (GNOME_DIALOG (gnome_error_dialog (_("Postgres database not available !"))));
    pg_connection_free (gpc_connection);
    gpc_connection = NULL;
  }
}


void
select_new_directory_callback (GtkFileSelection *selector, gpointer user_data)
{
  gchar *path;
  
  path = g_strdup (gtk_file_selection_get_filename (GTK_FILE_SELECTION (user_data)));
  gtk_widget_destroy (GTK_WIDGET (user_data));
  while (gtk_events_pending ())
    gtk_main_iteration ();
  gpc_film_append (gpc_connection, path);
  if (gpc_last_directory)
    g_free (gpc_last_directory);
  gpc_last_directory = g_strdup (path);
  g_free (path);
}


void
on_add_new_film_menu_activate          (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWidget   *dialog;

  if (!gpc_connection || !pg_connection_is_active (gpc_connection))
    return;
  if (!gpc_repository) {
    gnome_dialog_run_and_close (GNOME_DIALOG (gnome_error_dialog (_("GPC repository is not defined...!"))));
    return;
  }
  dialog = gtk_file_selection_new (_("New film directory"));
  gtk_file_selection_hide_fileop_buttons (GTK_FILE_SELECTION (dialog));
  gtk_widget_hide (GTK_WIDGET (GTK_FILE_SELECTION (dialog)->file_list)->parent);
  gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (dialog)->ok_button), "clicked", GTK_SIGNAL_FUNC (select_new_directory_callback), (gpointer) dialog);
  gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (dialog)->ok_button), "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy), (gpointer) dialog);
  gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (dialog)->cancel_button), "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy), (gpointer) dialog);
  if (gpc_last_directory)
    gtk_file_selection_set_filename (GTK_FILE_SELECTION (dialog), gpc_last_directory);
  else
    gtk_file_selection_set_filename (GTK_FILE_SELECTION (dialog), gpc_repository);
  gtk_widget_show (dialog);
}


void
select_new_filename_callback (GtkFileSelection *selector, gpointer user_data)
{
  gchar *filename;
  
  filename = gtk_file_selection_get_filename (GTK_FILE_SELECTION (user_data));
  gpc_image_append (gpc_connection, filename);
  if (gpc_last_directory)
    g_free (gpc_last_directory);
  gpc_last_directory = g_dirname (filename);
  gtk_widget_destroy (GTK_WIDGET (user_data));
}


void
on_add_new_image_menu_activate         (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWidget   *dialog;

  if (!gpc_connection || !pg_connection_is_active (gpc_connection))
    return;
  if (!gpc_repository) {
    gnome_dialog_run_and_close (GNOME_DIALOG (gnome_error_dialog (_("GPC repository is not defined...!"))));
    return;
  }
  dialog = gtk_file_selection_new (_("New image file"));
  gtk_file_selection_hide_fileop_buttons (GTK_FILE_SELECTION (dialog));
  gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (dialog)->ok_button), "clicked", GTK_SIGNAL_FUNC (select_new_filename_callback), (gpointer) dialog);
  gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (dialog)->ok_button), "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy), (gpointer) dialog);
  gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (dialog)->cancel_button), "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy), (gpointer) dialog);
  if (gpc_last_directory)
    gtk_file_selection_set_filename (GTK_FILE_SELECTION (dialog), gpc_last_directory);
  gtk_widget_show (dialog);
}


void
on_update_current_film_menu_activate   (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  gint ID;
  
  if (!gpc_connection || !pg_connection_is_active (gpc_connection) || !gpc_film_infos)
    return;
  if (gpc_image_infos && gpc_image_modified) {
    gpc_image_infos_store (gpc_connection, gpc_image_infos);
    gpc_image_infos_free (gpc_image_infos);
    gpc_image_infos = NULL;
  }
  ID = gpc_film_get_id (gpc_film_infos);
  gpc_film_update (gpc_connection, ID);
  on_films_clist_select_row (films_clist, gpc_film_selection, 0, NULL, NULL);
}


void
on_delete_current_image_menu_activate  (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWidget   *dialog;
  gint         ID;
  
  if (!gpc_connection || !pg_connection_is_active (gpc_connection) || !gpc_image_infos)
    return;
  modal_result = GPC_MODAL_RESULT_UNKNOWN;
  dialog = gnome_question_dialog (_("Do you really want to delete current image from database ?"),
                                    gpc_app_check_reply_callback, NULL);
  gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
  if (modal_result == GPC_MODAL_RESULT_OK) {       // OK button pressed
    ID = gpc_image_get_id (gpc_image_infos);
    gpc_image_infos_free (gpc_image_infos);
    gpc_image_infos = NULL;
    gpc_image_delete (gpc_connection, ID);
    gtk_clist_remove (images_clist, gpc_image_selection);
  }
  gtk_object_destroy (GTK_OBJECT (dialog));
}


void
on_delete_current_film_menu_activate   (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWidget   *dialog;
  gint         ID;
  
  if (!gpc_connection || !pg_connection_is_active (gpc_connection) || !gpc_film_infos)
    return;
  modal_result = GPC_MODAL_RESULT_UNKNOWN;
  dialog = gnome_question_dialog (_("Do you really want to delete current film from database ?"),
                                    gpc_app_check_reply_callback, NULL);
  gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
  if (modal_result == GPC_MODAL_RESULT_OK) {       // OK button pressed
    if (gpc_image_infos && gpc_image_modified) {
      gpc_image_infos_store (gpc_connection, gpc_image_infos);
      gpc_image_infos_free (gpc_image_infos);
      gpc_image_infos = NULL;
    }
    if (gpc_film_infos && gpc_film_modified) {
      gpc_film_infos_store (gpc_connection, gpc_film_infos);
      gpc_film_infos_free (gpc_film_infos);
      gpc_film_infos = NULL;
    }
    ID = gpc_film_get_id (gpc_film_infos);
    gpc_film_delete (gpc_connection, ID);
    gtk_clist_remove (films_clist, gpc_film_selection);
    gtk_clist_clear (images_clist);
  }
}


void
on_print_thumbnails_menu_activate      (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  if (gpc_film_infos)
    gpc_print_index (gpc_film_infos);
}


void
on_print_image_menu_activate           (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  if (gpc_image_infos)
    gpc_print_image (gpc_image_infos);
}


void
on_close_file_menu_activate            (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWidget  *dialog;
  
  if (!gpc_connection || !pg_connection_is_active (gpc_connection))
    return;
  modal_result = GPC_MODAL_RESULT_UNKNOWN;
  dialog = gnome_question_dialog (_("Do you really want to close current connection to PostgreSQL database ?"),
                                    gpc_app_check_reply_callback, NULL);
  gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
  if (modal_result != GPC_MODAL_RESULT_OK)       // OK button not pressed
    return;
  if (gpc_image_infos && gpc_image_modified) {
    gpc_image_infos_store (gpc_connection, gpc_image_infos);
    gpc_image_infos_free (gpc_image_infos);
    gpc_image_infos = NULL;
  }
  if (gpc_film_infos && gpc_film_modified) {
    gpc_film_infos_store (gpc_connection, gpc_film_infos);
    gpc_film_infos_free (gpc_film_infos);
    gpc_film_infos = NULL;
  }
  if (gpc_images) {
    pg_query_free (gpc_images);
    gpc_images = NULL;
  }
  if (gpc_films) {
    pg_query_free (gpc_films);
    gpc_films = NULL;
  }
  gtk_clist_clear (films_clist);
  gtk_clist_clear (images_clist);
  pg_connection_close (gpc_connection);
  pg_connection_free (gpc_connection);
  gpc_connection = NULL;
  gpc_app_free_lists ( );
}


void
on_exit_file_menu_activate             (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  on_gpc_app_delete_event (NULL, NULL, NULL);
}


void
on_preferences_edit_menu_activate      (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  if (!gpc_options_dialog)
    gpc_options_dialog = create_gpc_options_dialog ();
  gnome_dialog_close_hides (GNOME_DIALOG (gpc_options_dialog), TRUE);
  gtk_widget_show (GTK_WIDGET (gpc_options_dialog));
  gtk_window_set_transient_for (GTK_WINDOW (gpc_options_dialog), GTK_WINDOW (gpc_app));
}


void
on_zoom_in_view_menu_activate          (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  gpc_image_zoom_factor *= 1.2;
  gpc_app_resize_image ();
}


void
on_zoom_out_view_menu_activate         (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  gpc_image_zoom_factor /= 1.2;
  gpc_app_resize_image ();
}


void
on_zoom_normal_view_menu_activate      (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  gpc_image_zoom_factor = 1.0;
  gpc_app_resize_image ();
}


void
on_zoom_fit_view_menu_activate         (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  gfloat     x_ratio;
  gfloat     y_ratio;
  
  if (gpc_image_canvas && gpc_image_canvas_item) {
    x_ratio = (gfloat) gpc_image_canvas_width / gpc_image_width;
    y_ratio = (gfloat) gpc_image_canvas_height / gpc_image_height;
    gpc_image_zoom_factor = MIN (x_ratio, y_ratio);
    gpc_app_resize_image ();
  }
}


void
on_film_properties_view_menu_activate  (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  if (!gpc_film_dialog) {
    gpc_film_dialog = create_gpc_film_dialog();
    gtk_widget_show (GTK_WIDGET (gpc_film_dialog));
    gtk_window_set_transient_for (GTK_WINDOW (gpc_film_dialog), GTK_WINDOW (gpc_app));
    gpc_film_dialog_update (gpc_film_infos);
  }
  else {
    on_film_hide_button_clicked (NULL, NULL);
  }
}


void
on_image_properties_view_menu_activate (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  if (!gpc_image_dialog) {
    gpc_image_dialog = create_gpc_image_dialog ();
    gtk_widget_show (GTK_WIDGET (gpc_image_dialog));
    gtk_window_set_transient_for (GTK_WINDOW (gpc_image_dialog), GTK_WINDOW (gpc_app));
    gpc_image_dialog_update (gpc_image_infos);
  }
  else {
    on_image_hide_button_clicked (NULL, NULL);
  }
}


void
on_query_images_view_menu_activate     (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GpcQueryParam   index;
  GList          *params = NULL;
  GList          *operators = NULL;
  
  if (!gpc_query_dialog) {
    gpc_query_dialog = create_gpc_query_dialog ();
    for (index = GPC_FILM_ID; index < GPC_QUERY_END; index++)
      params = g_list_append (params, _(GpcQueryParamLib[index]));
    for (index = 0; index < GPC_QUERY_OPERATORS; index++)
      operators = g_list_append (operators, GpcQueryOperator[index]);
    gtk_combo_set_popdown_strings (GTK_COMBO (lookup_widget (gpc_query_dialog, "query_field_combo")), params);
    gtk_combo_set_popdown_strings (GTK_COMBO (lookup_widget (gpc_query_dialog, "query_operator_combo")), operators);
  }
  gnome_dialog_close_hides (GNOME_DIALOG (gpc_query_dialog), TRUE);
  gtk_widget_show (GTK_WIDGET (gpc_query_dialog));
  gtk_window_set_transient_for (GTK_WINDOW (gpc_query_dialog), GTK_WINDOW (gpc_app));
}


void
on_acquire_new_image_tools_menu_activate  (GtkMenuItem     *menuitem,
                                           gpointer         user_data)
{
  gchar *command;
  gchar *params;
  gchar *arg[3];
  
  command = gnome_config_get_string ("/gphotocoll/tools/acquisition_command=xsane");
  params = gnome_config_get_string ("/gphotocoll/tools/acquisition_params");
  arg[0] = command;
  arg[1] = params;
  arg[2] = NULL;
  gpc_app_launch (arg[0], arg);
  g_free (command);
  g_free (params);
}


void
on_edit_with_gimp_tools_menu_activate  (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  gchar *pathname;
  gchar *filename;
  gchar *command;
  gchar *params;
  gchar *fullparams;
  gchar *arg[3];
  
  if (!(gpc_film_infos && gpc_image_infos))
    return;
  pathname = gpc_film_get_full_pathname (gpc_film_infos);
  filename = g_strconcat (pathname, gpc_image_get_filename (gpc_image_infos), NULL);
  command = gnome_config_get_string ("/gphotocoll/tools/gimp_command=gimp");
  params = gnome_config_get_string ("/gphotocoll/tools/gimp_params=%s");
  fullparams = g_strdup_printf (params, filename);
  arg[0] = command;
  arg[1] = fullparams;
  arg[2] = NULL;
  gpc_app_launch (arg[0], arg);
  g_free (pathname);
  g_free (filename);
  g_free (command);
  g_free (params);
  g_free (fullparams);
}


void
on_edit_with_ee_tools_menu_activate    (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  gchar *pathname;
  gchar *filename;
  gchar *command;
  gchar *params;
  gchar *fullparams;
  gchar *arg[3];
  
  if (!(gpc_film_infos && gpc_image_infos))
    return;
  pathname = gpc_film_get_full_pathname (gpc_film_infos);
  filename = g_strconcat (pathname, gpc_image_get_filename (gpc_image_infos), NULL);
  command = gnome_config_get_string ("/gphotocoll/tools/ee_command=eeyes");
  params = gnome_config_get_string ("/gphotocoll/tools/ee_params=%s");
  fullparams = g_strdup_printf (params, filename);
  arg[0] = command;
  arg[1] = fullparams;
  arg[2] = NULL;
  gpc_app_launch (arg[0], arg);
  g_free (pathname);
  g_free (filename);
  g_free (command);
  g_free (params);
  g_free (fullparams);
}


void
on_about_help_menu_activate            (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  gnome_dialog_run_and_close (GNOME_DIALOG (create_gpc_about_dialog ()));
}


gboolean
on_films_clist_button_press_event      (GtkWidget       *widget,
                                        GdkEventButton  *event,
                                        gpointer         user_data)
{
  if ((widget == NULL) ||
      (GTK_IS_MENU (widget)) ||
      (event == NULL))
    return FALSE;
  
  if (event->button == 3) {
    if (!gpc_films_popup_menu)
      gpc_films_popup_menu = create_films_popup_menu ();
    gtk_menu_popup (GTK_MENU (gpc_films_popup_menu), NULL, NULL, NULL, NULL, event->button, event->time);
    return TRUE;
  }
  return FALSE;
}


void
on_films_clist_select_row              (GtkCList        *clist,
                                        gint             row,
                                        gint             column,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
  gchar *str;

  if (!gpc_connection || !pg_connection_is_active(gpc_connection))
    return;
  gtk_label_set_text (GTK_LABEL (lookup_widget (gpc_app, "image_filename_label")), NULL);
  gtk_label_set_text (GTK_LABEL (lookup_widget (gpc_app, "image_zoom_label")), NULL);
  if (gpc_image_canvas_item) {
    gtk_object_destroy (GTK_OBJECT (gpc_image_canvas_item));
    gpc_image_canvas_item = NULL;
  }
  if (gpc_image_infos) {
    gpc_image_infos_store (gpc_connection, gpc_image_infos);
    gpc_image_infos_free (gpc_image_infos);
    gpc_image_infos = NULL;
    gpc_image_dialog_update (gpc_image_infos);
  }
  if (gpc_film_infos) {
    gpc_film_infos_store (gpc_connection, gpc_film_infos);
    gpc_film_infos_free (gpc_film_infos);
  }
  if (gpc_images)
    pg_query_free (gpc_images);
  gpc_film_selection = row;
  if (gtk_clist_get_text(films_clist, gpc_film_selection, 0, &str)) {
    gpc_film_infos = gpc_film_get_infos (gpc_connection, atoi (str));
    gpc_film_dialog_update (gpc_film_infos);
    if (gpc_query_mode)
      gpc_images = gpc_query_get_images (gpc_connection, atoi (str));
    else
      gpc_images = gpc_image_get_list_from_film (gpc_connection, atoi (str));
    gpc_app_update_images_list (gpc_images);
    gtk_notebook_set_page (GTK_NOTEBOOK (lookup_widget (gpc_app, "right_notebook")), 0);
    gpc_app_refresh_thumbnails ();
  }
}


void
on_films_clist_scroll_vertical         (GtkCList        *clist,
                                        GtkScrollType    scroll_type,
                                        gfloat           position,
                                        gpointer         user_data)
{
  switch (scroll_type) {
    case GTK_SCROLL_STEP_BACKWARD : gtk_clist_select_row (films_clist, gpc_film_selection - 1, 0);
                                    break;
    case GTK_SCROLL_STEP_FORWARD  : gtk_clist_select_row (films_clist, gpc_film_selection + 1, 0);
                                    break;
    default                       : break;
  }
}


void
on_films_clist_click_column            (GtkCList        *clist,
                                        gint             column,
                                        gpointer         user_data)
{
  gtk_clist_set_sort_column (clist, column);
  gtk_clist_sort (clist);
}


void
on_images_clist_select_row             (GtkCList        *clist,
                                        gint             row,
                                        gint             column,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
  GdkPixbuf *gpc_image_pixbuf;
  gchar     *str;
  gchar     *pathname;
  gchar     *filename;
  
  if (!gpc_connection || !pg_connection_is_active(gpc_connection))
    return;
  if (gpc_image_canvas_item) {
    gtk_object_destroy (GTK_OBJECT (gpc_image_canvas_item));
    gpc_image_canvas_item = NULL;
  }
  if (gpc_image_infos) {
    gpc_image_infos_store (gpc_connection, gpc_image_infos);
    gpc_image_infos_free (gpc_image_infos);
  }
  gpc_image_selection = row;
  if (gtk_clist_get_text (images_clist, gpc_image_selection, 0, &str)) {
    gpc_image_infos = gpc_image_get_infos (gpc_connection, atoi (str));
    gpc_image_dialog_update (gpc_image_infos);
    if ((gtk_notebook_get_current_page (GTK_NOTEBOOK (lookup_widget (gpc_app, "right_notebook"))) > 0) ||
        (user_data && (GPOINTER_TO_INT (user_data) == 1))) {
      pathname = gpc_film_get_full_pathname (gpc_film_infos);
      filename = gpc_image_get_filename (gpc_image_infos);
      gpc_image_pixbuf = gdk_pixbuf_new_from_file (g_strconcat (pathname, filename, NULL));
      if (gpc_image_canvas && gpc_image_pixbuf) {
        gpc_image_width = gdk_pixbuf_get_width (gpc_image_pixbuf);
        gpc_image_height = gdk_pixbuf_get_height (gpc_image_pixbuf);
        gnome_canvas_set_scroll_region (GNOME_CANVAS (gpc_image_canvas),
                                        0, 0,
                                        gpc_image_width * gpc_image_zoom_factor,
                                        gpc_image_height * gpc_image_zoom_factor);
        gpc_image_canvas_item = gnome_canvas_item_new (gnome_canvas_root (GNOME_CANVAS (gpc_image_canvas)),
                                                       gnome_canvas_pixbuf_get_type (),
                                                       "pixbuf", gpc_image_pixbuf,
                                                       "width_set", TRUE,
                                                       "width", gpc_image_width * gpc_image_zoom_factor,
                                                       "height_set", TRUE,
                                                       "height", gpc_image_height * gpc_image_zoom_factor,
                                                       NULL);
        switch (gpc_zoom_mode) {
          case GPC_ZOOM_CURRENT      : break;
          case GPC_ZOOM_NORMAL       : on_zoom_normal_view_menu_activate (NULL, NULL);
       	                               break;
          case GPC_ZOOM_FIT          : on_zoom_fit_view_menu_activate (NULL, NULL);
                                       break;
      	  case GPC_ZOOM_FIT_IF_LARGE : if ((gpc_image_width > gpc_image_canvas_width) ||
                                           (gpc_image_height > gpc_image_canvas_height))
                                         on_zoom_fit_view_menu_activate (NULL, NULL);
                                       else
                                         on_zoom_normal_view_menu_activate (NULL, NULL);
                                       break;
        }
        gdk_pixbuf_unref (gpc_image_pixbuf);
        gpc_image_pixbuf = NULL;
      }
      gtk_label_set_text (GTK_LABEL (lookup_widget (gpc_app, "image_filename_label")), g_strdup_printf (_("Filename: %s%s"), pathname, filename));
      g_free (pathname);
    }
  }
}


void
on_images_clist_scroll_vertical        (GtkCList        *clist,
                                        GtkScrollType    scroll_type,
                                        gfloat           position,
                                        gpointer         user_data)
{
  switch (scroll_type) {
    case GTK_SCROLL_STEP_BACKWARD : gtk_clist_select_row (images_clist, gpc_image_selection - 1, 0);
                                    break;
    case GTK_SCROLL_STEP_FORWARD  : gtk_clist_select_row (images_clist, gpc_image_selection + 1, 0);
                                    break;
    default                       : break;
  }
}


void
on_images_clist_click_column           (GtkCList        *clist,
                                        gint             column,
                                        gpointer         user_data)
{
  gtk_clist_set_sort_column (clist, column);
  gtk_clist_sort (clist);
}


void
on_image_canvas_size_allocate          (GtkWidget       *widget,
                                        GtkAllocation   *allocation,
                                        gpointer         user_data)
{
  gpc_image_canvas_width = allocation->width;
  gpc_image_canvas_height = allocation->height;
}


gboolean
on_images_clist_button_press_event     (GtkWidget       *widget,
                                        GdkEventButton  *event,
                                        gpointer         user_data)
{
  if ((widget == NULL) ||
      (GTK_IS_MENU (widget)) ||
      (event == NULL))
    return FALSE;
  
  if (event->button == 3) {
    if (!gpc_images_popup_menu)
      gpc_images_popup_menu = create_images_popup_menu ();
    gtk_menu_popup (GTK_MENU (gpc_images_popup_menu), NULL, NULL, NULL, NULL, event->button, event->time);
    return TRUE;
  }
  return FALSE;
}


gboolean
on_image_canvas_button_press_event     (GtkWidget       *widget,
                                        GdkEventButton  *event,
                                        gpointer         user_data)
{
  if ((widget == NULL) ||
      (GTK_IS_MENU (widget)) ||
      (event == NULL))
    return FALSE;
  
  if (event->button == 3) {
    if (!gpc_images_popup_menu)
      gpc_images_popup_menu = create_images_popup_menu ();
    gtk_menu_popup (GTK_MENU (gpc_images_popup_menu), NULL, NULL, NULL, NULL, event->button, event->time);
    gtk_widget_set (lookup_widget (gpc_images_popup_menu, "zoom_in_popup_menu"),
                    "sensitive", TRUE,
                    NULL);
    gtk_widget_set (lookup_widget (gpc_images_popup_menu, "zoom_out_popup_menu"),
                    "sensitive", TRUE,
                    NULL);
    gtk_widget_set (lookup_widget (gpc_images_popup_menu, "zoom_normal_popup_menu"),
                    "sensitive", TRUE,
                    NULL);
    gtk_widget_set (lookup_widget (gpc_images_popup_menu, "zoom_fit_popup_menu"),
                    "sensitive", TRUE,
                    NULL);
    return TRUE;
  }
  return FALSE;
}


gboolean gpc_page_changed = FALSE;

void
on_right_notebook_switch_page          (GtkNotebook     *notebook,
                                        GtkNotebookPage *page,
                                        gint             page_num,
                                        gpointer         user_data)
{
  gpc_page_changed = TRUE;
  if (page_num == 1)
    on_images_clist_select_row (images_clist, gpc_image_selection, 0, NULL, GINT_TO_POINTER (1)); 
}


void
on_left_panel_size_allocate            (GtkWidget       *widget,
                                        GtkAllocation   *allocation,
                                        gpointer         user_data)
{
  if ((!gpc_page_changed) && gpc_images) {
    if (gtk_notebook_get_current_page (GTK_NOTEBOOK (lookup_widget (gpc_app, "right_notebook"))) == 0)
      gpc_app_refresh_thumbnails ();
  }
  gpc_page_changed = FALSE;
}

