// -*- c++ -*-
//------------------------------------------------------------------------------
//                              Deck.h
//------------------------------------------------------------------------------
// $Id: Deck.h,v 1.25 2006/09/07 02:34:04 vlg Exp $
//
//  Copyright (c) 2004 by Vladislav Grinchenko
//
//  This program is free software; you can redistribute it and/or 
//  modify it under the terms of the GNU General Public License   
//  as published by the Free Software Foundation; either version  
//  2 of the License, or (at your option) any later version.      
//
// Date: Jan 8, 2004
//------------------------------------------------------------------------------
#ifndef DECK_H
#define DECK_H

#include "VDeck.h"
#include <libxml/tree.h>		// xmlDocPtr

class Card;

class Deck : public VDeck
{
public:
	Deck (DeckEditStatus status_);
	Deck (const string& fname_, DeckEditStatus status_);
	~Deck ();

	virtual int  load (const string& fname_, DeckAction = INSTALL_DECK);
	void         insert (const string& from_file_);
	bool         import_from_csv (const string& file_);
	virtual int  save ();
	virtual int  save_as (const string& fname_);

	virtual cardlist_const_iterator begin () const { return m_cards.begin (); }
	virtual cardlist_const_iterator end   () const { return m_cards.end   (); }

	virtual cardlist_iterator begin () { return m_cards.begin (); }
	virtual cardlist_iterator end   () { return m_cards.end   (); }

	virtual size_t size  () const { return m_cards.size ();  }
	virtual bool   empty () const { return m_cards.empty (); }

	virtual void push_back (VCard* card_) { m_cards.push_back (card_); }
	virtual void erase (cardlist_iterator iter_) { m_cards.erase (iter_); }
	virtual bool erase (VCard* card_);

	void swap_with_next (VCard* vcard_);
	void swap_with_prev (VCard* vcard_);

	virtual const string&  get_name (void) const { return m_fname; }
	virtual void get_progress (float& pct_done_, std::string& msg_, int idx_);
	virtual void play_card (Deck::cardlist_iterator& iter_);

	VCard*               find_card_by_id (long id_);
	DeckEditStatus       get_status   ();
	const string&        get_snd_path () const { return m_snd_path; }
	const Glib::ustring& get_author   () const { return m_author; }
	const Glib::ustring& get_desc     () const { return m_desc; }

	void set_snd_path (const string& spath_)    { m_snd_path = spath_; }
	void set_author   (const Glib::ustring& s_) { m_author = s_; }
	void set_desc     (const Glib::ustring& s_) { m_desc = s_; }

	void mark_as_modified     ()       { m_status = MODIFIED_DECK; }
	bool needs_renaming       () const { return m_needs_renaming; }
	DeckEditStatus get_status () const { return m_status; }

	void dump () const;
	void dump_deck_status () const;

private:
	/** Parse the Deck file card by card.
		@return 0 on success; -1 on error with error_ explaining the reason.
	*/
	int  parse_xml_tree (xmlDocPtr parser_, std::string& error_);

	void reset_snd_setup ();

private:
	string         m_fname;		// full XML file path name
	cardlist_type  m_cards;
	string         m_indent;	// for pretty printing

	Glib::ustring  m_author;
	Glib::ustring  m_desc;
	SideSelection  m_side_selection; // Tells which side to play next time

	string         m_snd_path; // The location of WyabdcRealPeopleTTS package
	bool           m_with_snd;

	DeckEditStatus m_status;
	bool           m_needs_renaming;
};

/** C'tor for existing deck which would be
	loaded later on.
 */
inline 
Deck::
Deck (DeckEditStatus status_) 
	: 
	m_side_selection (FRONT),
	m_with_snd (true),
	m_status (status_), 
	m_needs_renaming (false)
{
	trace_with_mask("Deck::Deck",GUITRACE|DECK);
}

inline 
Deck::
~Deck ()
{
	trace_with_mask("Deck::~Deck",GUITRACE|DECK);
	dump_deck_status ();
}

#endif /* DECK_H */
