// -*- c++ -*-
//------------------------------------------------------------------------------
// $Id: DeckPlayer.cpp,v 1.102 2006/10/08 02:46:30 vlg Exp $
//------------------------------------------------------------------------------
//                            DeckPlayer.cpp
//------------------------------------------------------------------------------
//  Copyright (c) 2004-2006 by Vladislav Grinchenko 
//
//  This program is free software; you can redistribute it and/or 
//  modify it under the terms of the GNU General Public License   
//  as published by the Free Software Foundation; either version  
//  2 of the License, or (at your option) any later version.      
//------------------------------------------------------------------------------
//
// Date   : Fri Feb  6 23:37:11 EST 2004
//
//------------------------------------------------------------------------------

#include <gtkmm/stock.h>
#include <gtkmm/separator.h>
#include <gtkmm/spinbutton.h>
#include <gtkmm/box.h>
#include <gtkmm/handlebox.h>
#include <gtkmm/buttonbox.h>
#include <gtkmm/radiobutton.h>
#include <gtkmm/frame.h>
#include <gtkmm/toolbar.h>
#include <gtkmm/textattributes.h>
#include <gtkmm/main.h>
#include <gtkmm/progressbar.h>
#include <gtkmm/eventbox.h>

using sigc::bind;
using sigc::mem_fun;

#include "ButtonWithImageLabel.h"

#include "Granule-main.h"
#include "Granule.h"
#include "MainWindow.h"
#include "Card.h"
#include "Deck.h"
#include "CardBox.h"
#include "DeckPlayer.h"
#include "DeckView.h"
#include "CardView.h"
#include "Intern.h"

static const Glib::ustring clabels [] = { 
	N_("Check"), N_("Next "), N_("Correct  "), N_("Incorrect"), 
	"               "
};

enum { CHECK_ANSWER, NEXT, CORRECT, INCORRECT, EMPTY };

static const gchar* states [] = { "VC_START", "VC_ARMED", "VC_VALIDATED" };

/*******************************************************************************
 *                       VerifyControl Member Functions                        *
 *******************************************************************************
 */
VerifyControl::
VerifyControl (DeckPlayer& parent_) 
	: 
	Gtk::Table  (2, 2, false),
	m_parent    (parent_),
	m_state     (VC_START),
	m_red       ("red"),
	m_green     ("MediumSeaGreen"),
	m_white     ("white"),
	m_black     ("black"),
	m_test_line (NULL),
	m_line_num  (1)
	
{
    trace_with_mask("VerifyControl::VerifyControl", GUITRACE);

	Gtk::Table* table = this;

	m_report = Gtk::manage (new Gtk::Label);
	m_report->set_alignment  (0, 0.5);
	m_report->set_padding    (0,0);
	m_report->set_justify    (Gtk::JUSTIFY_FILL);
	m_report->set_line_wrap  (false);
	m_report->set_use_markup (false);
	m_report->set_selectable (false);
	m_report->set_size_request (80, -1); // width, height
	m_report->set_text (clabels [EMPTY]);
	m_report->set_sensitive (false);

	m_report_box = Gtk::manage (new Gtk::EventBox);
	
    m_entry = Gtk::manage (new Gtk::Entry);
    m_entry->set_flags       (Gtk::CAN_FOCUS);
    m_entry->set_visibility  (true);
    m_entry->set_editable    (true);
    m_entry->set_max_length  (0);
    m_entry->set_text        ("");
    m_entry->set_has_frame   (true);
    m_entry->set_width_chars (30);
    m_entry->set_activates_default (false);
	m_entry->modify_font     (CONFIG->get_input_font ());

	for (int i = 0; i < LABELS_SZ; i++) {
		m_label [i] = Gtk::manage (new Gtk::Label (clabels [i]));
		m_label [i]->set_alignment  (0, 0.5);
		m_label [i]->set_padding    (0,0);
		m_label [i]->set_justify    (Gtk::JUSTIFY_FILL);
		m_label [i]->set_line_wrap  (false);
		m_label [i]->set_use_markup (false);
		m_label [i]->set_selectable (false);
	}
	m_button = Gtk::manage  (new Gtk::Button);
    m_button->set_flags     (Gtk::CAN_FOCUS);
    m_button->set_relief    (Gtk::RELIEF_NORMAL);
	m_button->set_label     (clabels [CHECK_ANSWER]);
	m_button->set_sensitive (false);

	Gtk::HBox* hbox1  = manage (new Gtk::HBox (false, 0));
	Gtk::HBox* hbox2  = manage (new Gtk::HBox (false, 0));

	Gtk::Label* blank1 = manage (new Gtk::Label);
	Gtk::Label* blank2 = manage (new Gtk::Label);

	blank1->set_text ("            ");
	blank2->set_text ("            ");

	m_report_box->add  (*m_report);
	hbox1->pack_start (*m_report_box, Gtk::PACK_SHRINK, 0);
	hbox1->pack_start (*blank1,   Gtk::PACK_SHRINK, 0);

	hbox2->pack_start (*m_entry,  Gtk::PACK_EXPAND_WIDGET, 3);
	hbox2->pack_start (*m_button, Gtk::PACK_SHRINK, 0);
	hbox2->pack_start (*blank2,   Gtk::PACK_SHRINK, 0);

	table->set_row_spacings (0);
	table->set_col_spacings (2);
	table->attach (*hbox1, 1, 2, 0, 1, 
				   Gtk::EXPAND|Gtk::FILL, Gtk::FILL, 0, 0);
	table->attach (*hbox2, 0, 1, 0, 1, 
				   Gtk::EXPAND|Gtk::SHRINK|Gtk::FILL, Gtk::FILL, 0, 0);

	m_entry->signal_changed ().connect (
		mem_fun (*this, &VerifyControl::on_entry_changed));

    m_button->signal_clicked ().connect (
        mem_fun (m_parent, &DeckPlayer::on_verify_clicked));
}

void
VerifyControl::
set_focus_on_entry () 
{ 
    trace_with_mask("VerifyControl::set_focus_on_entry", KEYIN);
	m_entry->grab_focus (); 
}

void
VerifyControl::
on_test_line_changed ()
{
    trace_with_mask("VerifyControl::on_test_line_changed", GUITRACE);

	if (m_test_line != NULL) {
		m_line_num = m_test_line->get_value_as_int ();
		DL ((GRAPP,"test line = %d\n", m_line_num));
	}
}

/**
 * Here is the tricky part. The answer_ can be marked up with
 * Pango markup - so strip it before comparing. Strip all well-known
 * prefixes as well and reduce multiple white spaces to a single one.
 */
void
VerifyControl::
compare (const ustring& answer_)
{
    trace_with_mask("VerifyControl::compare", GUITRACE);

	m_button->set_label (clabels [NEXT]);

	gchar* pc1;
	gchar* text1;
	gchar* pc2;
	gchar* text2;
	pc1 = text1 = pc2 = text2 = NULL;

	Glib::ustring tmp (answer_);

    /** If we are comparing the back of the card and
		the m_line_num > 1, then we fetch that line instead.
		If we are short of lines, that's a failed match.
	*/
	if (m_parent.get_selected_size () == BACK && m_line_num > 1) 
	{
		std::vector<Glib::ustring> vec;
		std::istringstream input (tmp.c_str ());
		Glib::ustring token;
		while (input >> token) {
			vec.push_back (token);
		}
		if (vec.size () >= m_line_num) {
			tmp = vec [m_line_num-1];
		}
		else {
			tmp = "";			// Selected answer line is missing
		}
	}

	DeckPlayer::shrink_to_single_line (tmp);

	DL ((GRAPP,"tmp = \"%s\"\n", tmp.c_str ()));

	text1 = pc1 = Granule::strip_pango_markup (tmp.c_str ());
	Granule::remove_common_prefixes (text1);

	text2 = pc2 = Granule::strip_pango_markup (m_entry->get_text ().c_str ());
	Granule::remove_common_prefixes (text2);

	DL ((GRAPP,"m_entry = \"%s\"\n", text2));
	DL ((GRAPP,"answer  = \"%s\"\n", text1));
	m_was_equal = true;

	if (!strcmp (text1, text2)) {
		m_report_box->modify_fg (Gtk::STATE_INSENSITIVE, m_green);
		m_report->modify_fg     (Gtk::STATE_INSENSITIVE, m_green);
		m_report->set_label     (clabels [CORRECT]);
	}
	else {
		m_report_box->modify_fg (Gtk::STATE_INSENSITIVE, m_red);
		m_report->modify_fg     (Gtk::STATE_INSENSITIVE, m_red);
		m_report->set_label     (clabels [INCORRECT]);

		m_was_equal= false;

		/** Remember, you are comparing UTF-8 string and simple
			gchar* arithmetic won't work.
		*/
		Glib::ustring lhs (text1);
		Glib::ustring rhs (text2);

		for (int idx = 0; idx < lhs.size () && idx < rhs.size (); idx++) 
		{
			if (lhs [idx] != rhs [idx]) 
			{
				m_entry->select_region (idx, -1);
				break;
			}
		}
	}

	m_entry->set_editable (false);
	m_parent.flip_card ();
	set_state (VC_VALIDATED);
	g_free (pc1);
	g_free (pc2);
}

void
VerifyControl::
on_entry_changed ()
{
    trace_with_mask("VerifyControl::on_entry_changed", KEYIN);

	m_button->set_sensitive (true);

	if (get_state () != VC_ARMED     &&
		m_entry->get_text () != " "  && 
		m_entry->get_text ().size ()) 
	{
		m_parent.disable_answer_controls ();
		set_state (VC_ARMED);
		Glib::ustring str ("<span size=\"medium\">");
		str += m_entry->get_text ();
		str += "</span>";
		m_entry->get_layout ()->set_markup (str);
		DL ((KEYIN,"Gtk::Entry markup enabled\n"));
	}
}

void
VerifyControl::
clear ()
{
	set_state (VC_START);

	m_report->set_text (clabels [EMPTY]);
	m_report->set_sensitive (false);
	m_report_box->modify_fg   (Gtk::STATE_INSENSITIVE, m_black);

	m_entry->set_text ("");
	m_entry->set_editable (true);

	m_button->set_label (clabels [CHECK_ANSWER]);
	m_button->set_sensitive (false);

	set_focus_on_entry ();
	m_parent.enable_answer_controls ();
}

VCState 
VerifyControl::
get_state () const
{
//	DL ((GRAPP,"current state = %s\n", states [m_state]));
	return (m_state);
}
void 
VerifyControl::
set_state (VCState state_)
{
	if (m_state != state_) {
		DL ((GRAPP,"change of state: %s -> %s\n", 
			 states [m_state], states [state_]));
		m_state = state_;
	}
}

/*******************************************************************************
 *                         DeckPlayer Member Functions                         *
 *******************************************************************************
 */
Gtk::StockID DeckPlayer::SAY_IT     ("say-it"    );
Gtk::StockID DeckPlayer::THUMB_UP   ("thumb-up"  );
Gtk::StockID DeckPlayer::THUMB_DOWN ("thumb-down");

DeckPlayer::
DeckPlayer (VDeck& vdeck_) 
		:

#ifdef IS_HILDON
	Hildon::AppView ("Deck Player"),
	m_hildon_fullscreen (false),
#endif

    m_close_button      (manage (new ButtonWithImageLabel (
									 Gtk::Stock::CLOSE, _("Close")))),
    m_shuffle_button    (manage (new ButtonWithImageLabel (
									 Gtk::Stock::REFRESH, _("Shuffle")))),
    m_edit_button       (manage (new ButtonWithImageLabel (
									 Gtk::Stock::DND_MULTIPLE,
									 _("Edit Deck")))),
    m_vcard_edit_button (manage (new ButtonWithImageLabel (
									 Gtk::Stock::DND, _("Edit Card")))),
    m_selected_side     (FRONT),
    m_notebook          (manage (new Gtk::Notebook)),
    m_play_progress     (manage (new Gtk::ProgressBar)),
    m_answer_box_width  (0),
	m_verify_control    (manage (new VerifyControl (*this))),
    m_bg_white          ("white"),
    m_bg_pink           ("pink"),
    m_bg_black          ("black"),
    m_deck              (vdeck_),
    m_deck_iter         (m_deck.end ()),         /* empty deck */
    m_is_lesson_learned (false),
    m_is_real_deck      (true),
	m_tab_activated     (false),
	m_initialized       (false)
{
    trace_with_mask("DeckPlayer::DeckPlayer", GUITRACE);

    set_title ("DeckPlayer");

#ifdef IS_HILDON
	/**
	 * Not a dialog any more - it is a Bin (AppView) instead.
	 * We handle packaging ourselves and mute all attributes of a 
	 * Dialog everywhere down the line.
	 */
    m_vbox = manage (new Gtk::VBox (false, 2));
	add (*m_vbox);

#else
	/**
	 * A desktop Dialog - we can resize and move the window around
	 */	
    set_resizable (true);
	set_position (Gtk::WIN_POS_CENTER_ALWAYS);
	Gdk::Rectangle geom = CONFIG->get_deck_player_geometry ();
	DL ((GRAPP,"Setting size to width=%d, height=%d\n",
		 geom.get_width (), geom.get_height ()));
	set_size_request (geom.get_width (), geom.get_height ());
	set_modal (true);
	set_transient_for (*MAINWIN);

#endif

    /** Are we dealing with CardDeck?
     */
    if (dynamic_cast<Deck*>(&m_deck) == NULL) {
		m_is_real_deck = false;
		DL((GRAPP,"Deck isn't REAL!\n"));
    }
    else {
		DL((GRAPP,"Yep, this IS a real deck\n"));
    }

    /** HBox holds both the notebook and controls/info side panel
     */
    Gtk::HBox* nb_box = manage (new Gtk::HBox (false, 2));
    nb_box->pack_start (*m_notebook, true, true);

    /********************
	 * Add the notebook *
	 ********************
     */
    m_notebook->set_tab_pos     (Gtk::POS_BOTTOM);
    m_notebook->set_show_border (true);
    m_notebook->modify_bg       (Gtk::STATE_NORMAL, m_bg_white);
    m_notebook->modify_base     (Gtk::STATE_NORMAL, m_bg_white);

#ifndef IS_HILDON

	/** "The other useful fields are the min_aspect and max_aspect fields; 
	 *   these contain a width/height ratio as a floating point number. 
	 *   If a geometry widget is set, the aspect applies to the geometry 
	 *   widget rather than the entire window. The most common use of these 
	 *   hints is probably to set min_aspect and max_aspect to the same value, 
	 *   thus forcing the window to keep a constant aspect ratio. 
	 */
	if (CONFIG->keep_deck_player_aspect ()) 
	{
		Gdk::Rectangle& rec = CONFIG->get_deck_player_aspect ();
		gdouble aspect = rec.get_width () * 1.0 / rec.get_height ();

		Gdk::Geometry answer_box_geometry = 
		{ 
			geom.get_width(),/*min_width*/ geom.get_height (), /* min_height  */
			10000,	         /* max_width; */	10000,	       /* max_height  */
			-1,	             /* base_width */	-1,	           /* base_height */
			-1,	             /* width_inc  */	-1,	           /* height_inc  */
			aspect,	         /* min_aspect (width/height) */
			aspect	         /* max_aspect (width/height) */
		};

		set_geometry_hints (*this, answer_box_geometry, 
							Gdk::HINT_ASPECT | Gdk::HINT_MIN_SIZE);
	}
#endif

    /***********************************************
	 *         Fill up notebook with elements      *
     ***********************************************
     */

	/*********************
	 * Front of the card *
	 *********************
	 */
#ifdef IS_HILDON
    m_front.set_alignment (Gtk::ALIGN_LEFT, Gtk::ALIGN_CENTER);
    m_front.set_justify   (Gtk::JUSTIFY_CENTER);

#else
    m_front.set_alignment (0.46, 0.5);
    m_front.set_justify    (Gtk::JUSTIFY_LEFT);

    m_front.set_selectable ();
#endif

    m_front.set_use_markup ();
	m_front.set_line_wrap  (true);
    m_front.modify_font    (CONFIG->get_question_font ());

    Gtk::Frame* question_frame = manage (new Gtk::Frame());
    question_frame->set_label_align (0, 0);
    question_frame->set_shadow_type (Gtk::SHADOW_ETCHED_IN);
    question_frame->modify_bg       (Gtk::STATE_NORMAL, m_bg_black);

    question_frame->add (m_front);

	Gtk::EventBox* question_evbox = Gtk::manage (new Gtk::EventBox);
	question_evbox->modify_bg (Gtk::STATE_NORMAL, m_bg_white);
	question_evbox->add (*question_frame);

    m_notebook->pages ().push_back (
		Gtk::Notebook_Helpers::TabElem (*question_evbox, _("Front")));

    /********************
	 * Back of the card *
	 ********************
     */

    /** VBox holds front, separator, example.
     */
    m_answer_box = manage (new Gtk::VBox (false, 5));
    m_answer_box->set_border_width (5); // How far box is placed from the edges.

    /* -Back (the answer)- 
	 */
#ifdef IS_HILDON
    m_back.set_justify    (Gtk::JUSTIFY_CENTER);
	m_back.set_alignment  (Gtk::ALIGN_LEFT, Gtk::ALIGN_CENTER);
#else
    m_back.set_justify    (Gtk::JUSTIFY_FILL);
    m_back.set_selectable ();
#endif

    m_back.set_use_markup ();
    m_back.modify_font    (CONFIG->get_answer_font ());
    m_back.set_line_wrap  ();

	Gtk::EventBox* answer_evbox = Gtk::manage (new Gtk::EventBox);
	answer_evbox->modify_bg (Gtk::STATE_NORMAL, m_bg_white);
	answer_evbox->add (m_back);

    /* -Separator- 
	 */
    Gtk::HSeparator* separator = manage (new Gtk::HSeparator);
    separator->modify_fg   (Gtk::STATE_NORMAL, m_bg_pink);
    separator->modify_bg   (Gtk::STATE_NORMAL, m_bg_pink);
    separator->modify_base (Gtk::STATE_NORMAL, m_bg_pink);

    /* -Example- 
	 */
    m_example.set_padding    (4, 0);  /* x - distance between text and 
										 right/left edges */
    m_example.set_justify    (Gtk::JUSTIFY_LEFT);
#ifndef IS_HILDON
    m_example.set_selectable ();
#endif
    m_example.set_use_markup ();
    m_example.modify_font    (CONFIG->get_example_font ());
    m_example.set_line_wrap  ();

	Gtk::EventBox* example_evbox = Gtk::manage (new Gtk::EventBox);
	example_evbox->modify_bg (Gtk::STATE_NORMAL, m_bg_white);
	example_evbox->add (m_example);

    m_answer_box->pack_start (*answer_evbox,  Gtk::PACK_SHRINK, 5);
    m_answer_box->pack_start (*separator,     Gtk::PACK_SHRINK, 5);
    m_answer_box->pack_start (*example_evbox, Gtk::PACK_SHRINK, 5); 

	Gtk::Viewport *viewport =
		Gtk::manage (new Gtk::Viewport (*manage (new Gtk::Adjustment(0,0,1)),
										*manage (new Gtk::Adjustment(0,0,1))));

	viewport->set_shadow_type(Gtk::SHADOW_IN);
	viewport->modify_bg (Gtk::STATE_NORMAL, m_bg_white);
	
	viewport->add (*m_answer_box);

	/** Without horizontal AUTOMATIC policy, dialog's resizing
	 *  by the user doesn't function properly. Instead, the window
	 *  keeps growing in size every time there is more text to fit then
	 *  size allows.
	 */
	Gtk::ScrolledWindow* scrollw = Gtk::manage (new Gtk::ScrolledWindow);

	scrollw->set_flags (Gtk::CAN_FOCUS);
	scrollw->set_shadow_type (Gtk::SHADOW_NONE);
	scrollw->property_window_placement ().set_value (Gtk::CORNER_TOP_LEFT);

#ifdef IS_HILDON
	scrollw->set_policy (Gtk::POLICY_NEVER , Gtk::POLICY_AUTOMATIC);
#else
	scrollw->set_policy (Gtk::POLICY_AUTOMATIC , Gtk::POLICY_AUTOMATIC);
#endif

	scrollw->add (*viewport);

    m_notebook->pages ().push_back (
		Gtk::Notebook_Helpers::TabElem (*scrollw, _("Back")));

    /***********************
	 * Side-panel controls *
	 ***********************
     */
    Gtk::VBox* vbox = manage (new Gtk::VBox (false, 3));

    Gtk::RadioButton::Group group;
    Gtk::RadioButton* radio_fb = NULL;
    Gtk::RadioButton* radio_bb = NULL;
    Gtk::VButtonBox*  vbuttonbox = manage (new Gtk::VButtonBox);

#ifndef IS_HILDON
    vbuttonbox->set_homogeneous ();
    vbuttonbox->set_spacing (1);
#endif	
                                                                                
    radio_fb = manage (new Gtk::RadioButton (group, _("Ask front")));
    radio_fb->signal_clicked ().connect (
		bind<SideSelection>(
			mem_fun (*this, &DeckPlayer::on_radio_selected), FRONT));
    vbuttonbox->pack_end (*radio_fb, false, false, 1); 
                                                                                
    radio_bb = manage (new Gtk::RadioButton (group, _("Ask back")));
    radio_bb->signal_clicked ().connect (
		bind<SideSelection>(
			mem_fun (*this, &DeckPlayer::on_radio_selected), BACK));
    vbuttonbox->pack_end (*radio_bb, false, false, 1); 
    vbox->pack_start (*vbuttonbox, false, false, 0);

	/** Add Line selection spin button
	 */
	Gtk::HBox*       sb_hbox   = manage (new Gtk::HBox);
	Gtk::Adjustment* sb_adjust = manage (new Gtk::Adjustment (1,1,10));
	Gtk::Label*      sb_label  = manage (new Gtk::Label (_("Test line: ")));
	m_test_line_button = manage (new Gtk::SpinButton (*sb_adjust));

	sb_hbox->pack_start (*sb_label, false, false, 0);
	sb_hbox->pack_start (*m_test_line_button, false, false, 0);
	vbox->pack_start (*sb_hbox, false, false, 0);
	m_test_line_button->signal_value_changed ().connect (
		mem_fun (*m_verify_control, &VerifyControl::on_test_line_changed));
	m_verify_control->attach (m_test_line_button);
	m_verify_control->set_focus_on_entry ();

	/** Fetch out the side history and adjust accordingly
	 */
	if ((m_selected_side = vdeck_.get_side_selection ()) == FRONT) {
		radio_fb->set_active ();
		m_test_line_button->set_sensitive (false);
	}
	else {
		radio_bb->set_active ();
		m_test_line_button->set_sensitive (true);
	}

    /** Side-panel buttons
     */
    m_shuffle_button->signal_clicked ().connect (
		mem_fun (*this, &DeckPlayer::on_shuffle_clicked));
    vbox->pack_start (*m_shuffle_button, false, false, 0);

    m_edit_button->signal_clicked ().connect (
		mem_fun (*this, &DeckPlayer::on_edit_clicked));
    vbox->pack_start (*m_edit_button, false, false, 0);

    m_vcard_edit_button->signal_clicked ().connect (
		bind<bool>(mem_fun (*this, &DeckPlayer::on_vcard_edit_clicked), false));
    vbox->pack_start (*m_vcard_edit_button, false, false, 0);

    m_count_label.set_alignment (0.5, 0.5);
    m_count_label.set_padding   (0, 0);
    m_count_label.set_justify   (Gtk::JUSTIFY_FILL);

    vbox->pack_end (*m_play_progress, false, false, 4);
    vbox->pack_end (m_count_label, false, false, 4);
	
    nb_box->pack_start (*vbox, Gtk::PACK_SHRINK, 5);
    get_vbox ()->pack_start (*nb_box);
    nb_box->show_all ();

    /** Bottom playback controls toolbar panel
     */
    m_playit_xpm = Gdk::Pixbuf::create_from_file (
		GRAPPDATDIR "/pixmaps/speaker_24.png");
	
    Glib::RefPtr<Gtk::IconFactory> icon_factory = Gtk::IconFactory::create ();

    Gtk::IconSet spkr_set;
    Gtk::IconSource spkr_source;
    spkr_source.set_pixbuf (m_playit_xpm);
    spkr_source.set_size (Gtk::ICON_SIZE_BUTTON);
    spkr_set.add_source (spkr_source);
    icon_factory->add (DeckPlayer::SAY_IT, spkr_set);
    Gtk::Stock::add (Gtk::StockItem (DeckPlayer::SAY_IT, _("Say It!")));

	Glib::RefPtr<Gdk::Pixbuf> thumbup_xpm = 
		Gdk::Pixbuf::create_from_file (GRAPPDATDIR "/pixmaps/thumbup.png");
    Gtk::IconSet thumbup_set;
    Gtk::IconSource thumbup_source;
    thumbup_source.set_pixbuf (thumbup_xpm);
    thumbup_source.set_size (Gtk::ICON_SIZE_BUTTON);
    thumbup_set.add_source (thumbup_source);
    icon_factory->add (DeckPlayer::THUMB_UP, thumbup_set);
    Gtk::Stock::add (Gtk::StockItem (DeckPlayer::THUMB_UP, _("I know it!")));

	Glib::RefPtr<Gdk::Pixbuf> thumbdown_xpm = 
		Gdk::Pixbuf::create_from_file (GRAPPDATDIR "/pixmaps/thumbdown.png");
    Gtk::IconSet thumbdown_set;
    Gtk::IconSource thumbdown_source;
    thumbdown_source.set_pixbuf (thumbdown_xpm);
    thumbdown_source.set_size (Gtk::ICON_SIZE_BUTTON);
    thumbdown_set.add_source (thumbdown_source);
    icon_factory->add (DeckPlayer::THUMB_DOWN, thumbdown_set);
    Gtk::Stock::add (Gtk::StockItem (DeckPlayer::THUMB_DOWN, _("No clue")));

    icon_factory->add_default ();

    m_actgroup = Gtk::ActionGroup::create ();

    m_actgroup->add (Gtk::Action::create ("Say_It", DeckPlayer::SAY_IT),
					 mem_fun (*this, &DeckPlayer::on_sayit_clicked));

    m_actgroup->add (Gtk::Action::create ("First", Gtk::Stock::GOTO_FIRST),
					 mem_fun (*this, &DeckPlayer::go_first_cb));

    m_actgroup->add (Gtk::Action::create ("Prev", Gtk::Stock::GO_BACK),
					 mem_fun (*this, &DeckPlayer::go_prev_cb));

    m_actgroup->add (Gtk::Action::create ("Next", Gtk::Stock::GO_FORWARD),
					 mem_fun (*this, &DeckPlayer::go_next_cb));

    m_actgroup->add (Gtk::Action::create ("Last", Gtk::Stock::GOTO_LAST),
					 mem_fun (*this, &DeckPlayer::go_last_cb));

    m_actgroup->add (Gtk::Action::create ("IKnowIt", DeckPlayer::THUMB_UP),
					 mem_fun (*this, &DeckPlayer::know_answer_cb));

    m_actgroup->add (Gtk::Action::create ("NoClue", DeckPlayer::THUMB_DOWN),
					 mem_fun (*this, &DeckPlayer::no_clue_cb));

    m_uimanager = Gtk::UIManager::create ();
    m_uimanager->insert_action_group (m_actgroup);

#ifndef IS_HILDON
    add_accel_group (m_uimanager->get_accel_group ());
#endif

    try {
		Glib::ustring ui_info =
			"<ui>"
			"  <toolbar name='Controls'>"
			"     <toolitem action='Say_It'/>"
			"     <separator name='1'/>"
			"     <toolitem action='First'/>"
			"     <toolitem action='Prev'/>"
			"     <toolitem action='Next'/>"
			"     <toolitem action='Last'/>"
			"     <separator name='2'/>"
			"     <toolitem action='IKnowIt'/>"
			"     <toolitem action='NoClue'/>"
			"  </toolbar>"
			"</ui>";
		m_uimanager->add_ui_from_string (ui_info);
    }
    catch (const Glib::Error& ex_) {
		DL((GRAPP,"Building menues failed (%s)\n"));
		std::cerr << "Failed: " << ex_.what ();
    }

    m_toolbar_go_first = m_uimanager->get_widget ("/Controls/First");
    m_toolbar_go_prev  = m_uimanager->get_widget ("/Controls/Prev");
    m_toolbar_go_next  = m_uimanager->get_widget ("/Controls/Next");
    m_toolbar_go_last  = m_uimanager->get_widget ("/Controls/Last");
    m_toolbar_know_it  = m_uimanager->get_widget ("/Controls/IKnowIt");
    m_toolbar_no_clue  = m_uimanager->get_widget ("/Controls/NoClue");

	/**
	 *  Add input verification control
	 */
	Gtk::HSeparator* hsep = manage (new Gtk::HSeparator);
	get_vbox ()->pack_start (*hsep, Gtk::PACK_SHRINK, 4);
	get_vbox ()->pack_start (*m_verify_control, false, false, 0);
	m_verify_control->show_all ();

    /** 
	 * Add play controls' bar and <Close> button
     */
    Gtk::Widget* controls = m_uimanager->get_widget ("/Controls");
    Gtk::Toolbar* ctrl_toolbar = dynamic_cast<Gtk::Toolbar*> (controls);
    ctrl_toolbar->set_orientation (Gtk::ORIENTATION_HORIZONTAL);
    ctrl_toolbar->set_toolbar_style (Gtk::TOOLBAR_ICONS);
    ctrl_toolbar->set_tooltips (true);

#ifndef IS_HILDON
    set_has_separator (false);
#endif

    Gtk::HBox* hbox = manage (new Gtk::HBox);
    hbox->pack_start (*ctrl_toolbar, true, true, 2);
    hbox->pack_start (*m_close_button, false, false, 0);
    get_vbox ()->pack_start (*hbox, false, false, 0);
    hbox->show_all ();

    m_close_button->signal_clicked ().connect (
		mem_fun (*this, &DeckPlayer::on_close_clicked));

	/**
	 * Setting resize callbacks. The answer box holds 
	 * Back/Separator/Example triplet trapped in the scrollwindow.
	 */
    m_answer_box->signal_size_allocate ().connect (
		mem_fun (*this, &DeckPlayer::answer_box_size_allocate_cb));
	
    signal_size_allocate ().connect (
		mem_fun (*this, &DeckPlayer::size_allocate_cb));

	add_events (Gdk::BUTTON_PRESS_MASK|Gdk::EXPOSURE_MASK);

	/** 'false' argument tells Gtk+ to call on_key_pressed() BEFORE
		keypress event handling. Returning 'true' from on_key_pressed()
		will stop event processing.
	*/
	if (!CONFIG->disable_key_shortcuts ()) {
		signal_key_press_event ().connect(
			mem_fun (*this, &DeckPlayer::on_key_pressed), false);
	}
	
	/* For maemo, reduce overall font size
	 */
	Pango::FontDescription fd ("Bitstream Vera Sans 8");
	m_shuffle_button->modify_font (fd);
	m_edit_button->modify_font (fd);
	m_vcard_edit_button->modify_font (fd);

	/** Grand finale
	 */
#ifndef IS_HILDON
	set_icon_from_file (GRAPPDATDIR "/pixmaps/deckplayer_32x32.png");
#endif
    set_sensitivity ();
	sensitize_controls ();
    m_deck.reset_progress ();
    set_win_name ();
    show_deck ();
    switch_side ();
    show_all_children ();	

	m_initialized = true;
}

//==============================================================================
//                              Utilities
//==============================================================================
//
#ifdef IS_HILDON

gint
DeckPlayer::
run ()
{
    trace_with_mask("DeckPlayer::run",GUITRACE);

	GRANULE->register_appview (this);
	show ();
	m_verify_control->set_focus_on_entry ();

	DL ((APP,"Enter Main::run()\n"));
	Gtk::Main::run ();

	DL ((APP,"Exit Main::run()\n"));
	return (Gtk::RESPONSE_NONE);
}
#endif

void
DeckPlayer::
on_close_clicked ()
{
    trace_with_mask("DeckPlayer::on_close_clicked",GUITRACE);

#ifdef IS_HILDON
	GRANULE->unregister_appview (this);

	DL ((GRAPP,"Call Main::quit()\n"));
	Gtk::Main::quit ();
#else
	hide ();
#endif
}

void
DeckPlayer::
show_deck ()
{
    trace_with_mask("DeckPlayer::show_deck",GUITRACE);

    if (m_deck.empty ()) {
		DL((GRAPP,"The deck is empty!\n"));
		return;
    }
    reset ();
}

void
DeckPlayer::
reset ()
{
    trace_with_mask("DeckPlayer::reset",GUITRACE);

    m_deck_iter = m_deck.begin ();
	DL ((APP,"m_deck_iter = begin()\n"));
    show_card ();
}

/**
 * Always clear all fields first. Then, try to fill them up.
 * If we were given an invalid markup text, then set_markup()
 * would fail with 
 * "Gtk-WARNING **: Failed to set label from markup due to error parsing markup"
 *
 * I'd like ultimately to catch the error and show it in a warning dialog 
 * to the user, but for now I don't see any way of doing it with GTK+.
 */
void
DeckPlayer::
show_card (bool with_adjust_count_)
{
    trace_with_mask("DeckPlayer::show_card",GUITRACE);

	clear_text ();

    if (m_deck_iter != m_deck.end () || m_deck.size () > 0) {
		while (!markup_is_valid ((*m_deck_iter))) {
			Gtk::MessageDialog em (
				_("The card you are about to see has invalid markup.\n"
				  "To proceed further, first, fix the broken markup.\n"
				  "If unsure about the correct syntax, consult Help."), 
				Gtk::MESSAGE_ERROR);
			em.run ();
			em.hide ();
			on_vcard_edit_clicked (true);
		}
		m_front.set_markup ((*m_deck_iter)->get_question ());
		set_answer_field ((*m_deck_iter)->get_answer ());
		m_example.set_markup (markup_tildas((*m_deck_iter)->get_example()));

		if ((*m_deck_iter)->get_reversed ()) {
			m_notebook->set_current_page (BACK);
		}
    }

	if (with_adjust_count_) {
		adjust_count ();
	}

#ifndef IS_HILDON
	/** Calling set_focus_on_entry() causes crash with Maemo 2.0 when
	 *  you open CardFile twice in the row. 
	 *  No clue - investigate later.
     */
	m_verify_control->set_focus_on_entry ();
#endif

}

void
DeckPlayer::
set_answer_field (Glib::ustring answer_)
{
	trace_with_mask("DeckPlayer::set_answer_field",GUITRACE);

//	DL ((GRAPP,"answer: \"%s\"\n", answer_.c_str ()));

	m_back.set_markup (answer_);

/*
  Multi-line right alignment works, but perhaps it should be
  optional via Preferences

	if (answer_.find ("\n") != Glib::ustring::npos) {
		m_back.set_alignment (Gtk::ALIGN_LEFT, Gtk::ALIGN_CENTER);
	}
	else {
		m_back.set_alignment (Gtk::ALIGN_CENTER, Gtk::ALIGN_CENTER);
	}
*/
}

void
DeckPlayer::
adjust_count ()
{
    trace_with_mask("DeckPlayer::adjust_count",GUITRACE);

    if (m_deck.empty ()) {
		DL((GRAPP,"The deck is empty!\n"));
		m_count_label.set_text ("0/0");
		return;
    }
    string pmsg;
    float pcnt_done;
    m_deck.get_progress (pcnt_done, pmsg, m_deck_iter - m_deck.begin ());
    m_count_label.set_text (pmsg);
    m_play_progress->set_fraction (pcnt_done);
}

void
DeckPlayer::
set_sensitivity ()
{
    trace_with_mask("DeckPlayer::set_sensitivity", GUITRACE);

    if (m_is_real_deck) {
		m_toolbar_know_it ->set_sensitive (false);
		m_toolbar_no_clue ->set_sensitive (false);
    }
    else {
		m_toolbar_go_first->set_sensitive (false);
		m_toolbar_go_prev ->set_sensitive (false);
		m_toolbar_go_next ->set_sensitive (false);
		m_toolbar_go_last ->set_sensitive (false);
    }
}

void
DeckPlayer::
sensitize_controls ()
{
	trace_with_mask("DeckPlayer::set_sensitize", GUITRACE);

    if (!m_is_real_deck) {
		return;
	}
	DL ((GRAPP,"Deck is empty\n"));

	if (m_deck.empty ()) {
		m_shuffle_button    ->set_sensitive (false);
		m_vcard_edit_button ->set_sensitive (false);

		m_toolbar_go_first  ->set_sensitive (false);
		m_toolbar_go_prev   ->set_sensitive (false);
		m_toolbar_go_next   ->set_sensitive (false);
		m_toolbar_go_last   ->set_sensitive (false);
    }
	else {
		m_shuffle_button    ->set_sensitive (true);
		m_vcard_edit_button ->set_sensitive (true);

		m_toolbar_go_first  ->set_sensitive (true);
		m_toolbar_go_prev   ->set_sensitive (true);
		m_toolbar_go_next   ->set_sensitive (true);
		m_toolbar_go_last   ->set_sensitive (true);
	}
}

void 
DeckPlayer::
enable_answer_controls ()
{
    if (!m_is_real_deck) {
		m_toolbar_know_it->set_sensitive (true);
		m_toolbar_no_clue->set_sensitive (true);
	}
}

void 
DeckPlayer::
disable_answer_controls ()
{
	m_toolbar_know_it->set_sensitive (false);
	m_toolbar_no_clue->set_sensitive (false);
}

//==============================================================================
//                          Action Buttons callbacks
//==============================================================================
void
DeckPlayer::
go_last_cb ()
{
    if ((m_deck_iter + 1) != m_deck.end ()) {
		m_deck_iter = m_deck.end ();
		m_deck_iter--;
		DL ((APP,"m_deck_iter--\n"));
		show_card ();
    }
    switch_side ();
	m_verify_control->clear ();
}

void
DeckPlayer::
go_first_cb ()
{
    if (m_deck_iter != m_deck.begin ()) {
		m_deck_iter = m_deck.begin ();
		DL ((APP,"m_deck_iter = begin()\n"));
		show_card ();
    }
    switch_side ();
	m_verify_control->clear ();
}

void
DeckPlayer::
go_prev_cb ()
{
    if (m_deck_iter != m_deck.begin ()) {
		m_deck_iter--;
		DL ((APP,"m_deck_iter--\n"));
		show_card ();
    }
    switch_side ();
	m_verify_control->clear ();
}

void
DeckPlayer::
go_next_cb ()
{
    trace_with_mask("DeckPlayer::go_next_cb",GUITRACE);

    if (m_deck_iter != m_deck.end () &&	         // empty deck
		(m_deck_iter + 1) != m_deck.end ())      // at the last element
	{
	    DL((GRAPP,"Go next card=>\n"));
	    m_deck_iter++;
		DL ((APP,"m_deck_iter++\n"));
	    show_card ();
	}
    /** We flip the side only if user has gone (hopefully)
     *  through an entire Deck (for *real* Decks only).
     */
    if (m_is_real_deck) {
		if (m_deck_iter + 1 == m_deck.end () && !m_is_lesson_learned) {	
			DL((GRAPP,"Reached the last card!\n"));
			CONFIG->set_flipside_history (m_deck.get_name (), 
										  m_selected_side==FRONT ? BACK:FRONT);
			m_is_lesson_learned = true;
		}
    }
    switch_side ();
	m_verify_control->clear ();
}

/** This callback is activated only when learning flashcards in the
    Card Box. We are always at the beginning of the list, moving
    elements to the next box or in case of the last box, moving 
    them to the back of the same list. We proceed in this manner until 
    all cards in the box are exhausted.
*/
void
DeckPlayer::
know_answer_cb ()
{
    trace_with_mask("DeckPlayer::know_answer_cb",GUITRACE);

    if (m_deck_iter == m_deck.end ()) {	// empty deck
		DL((GRAPP,"The deck is empty!\n"));
		return;
    }
    int old_idx = m_deck.get_index ();
    DL((GRAPP,"Moving card \"%s\" : deck %d => %d\n",
		(*m_deck_iter)->get_question ().c_str (), old_idx+1, old_idx+2));

    CARDBOX->move_card_to_box (m_deck_iter, old_idx, old_idx + 1, 
					   (dynamic_cast<CardRef*> (*m_deck_iter))->is_expired ());
    switch_side ();
	m_verify_control->clear ();
}

/** Move the card to the first box or if already in the first box,
    move it to the back of the list.
*/
void
DeckPlayer::
no_clue_cb ()
{
    trace_with_mask("DeckPlayer::no_clue_cb",GUITRACE);

    if (m_deck_iter == m_deck.end ()) {	// empty deck
		DL((GRAPP,"The deck is empty!\n"));
		return;
    }
    int old_idx = m_deck.get_index ();
    DL((GRAPP,"Moving card \"%s\" : deck %d => 1\n",
		(*m_deck_iter)->get_question ().c_str (), old_idx+1));

    CARDBOX->move_card_to_box (m_deck_iter, old_idx, 0,
					   (dynamic_cast<CardRef*> (*m_deck_iter))->is_expired ());
    switch_side ();
	m_verify_control->clear ();
}


void
DeckPlayer::
on_radio_selected (SideSelection selected_)
{
    trace_with_mask("DeckPlayer::on_radio_selected",GUITRACE);

    m_selected_side = selected_;
	m_deck.set_side_selection (m_selected_side);
    switch_side ();
	m_test_line_button->set_sensitive ((m_selected_side == BACK));

	DL ((GRAPP,"selected = %s\n",
		 (m_selected_side == BACK ? "BACK" : "FRONT")));
}

void
DeckPlayer::
on_sayit_clicked ()
{
    trace_with_mask("DeckPlayer::on_sayit_clicked",GUITRACE);
    if (m_deck_iter != m_deck.end ()) {
		m_deck.play_card (m_deck_iter);
    }
}

void
DeckPlayer::
on_shuffle_clicked ()
{
    trace_with_mask("DeckPlayer::on_shuffle_clicked",GUITRACE);

    if (m_is_real_deck) {
		random_shuffle (m_deck.begin (), m_deck.end ());
		/* Gtk::MessageDialog qm (*this,  */
		Gtk::MessageDialog qm (
				   _("Would you like to make cards\norder changes permanent?"),
							   false,
                               Gtk::MESSAGE_QUESTION, 
							   Gtk::BUTTONS_NONE,
                               true);
        qm.add_button (Gtk::Stock::NO,  Gtk::RESPONSE_NO);
        qm.add_button (Gtk::Stock::YES, Gtk::RESPONSE_YES);
		if (qm.run () == Gtk::RESPONSE_YES) {
			Deck& real_deck = dynamic_cast<Deck&>(m_deck);
			real_deck.mark_as_modified ();
		}
    }
	/** For CardDeck, shuffle all subsets.
	 */
	else {
		CardDeck& card_deck = dynamic_cast<CardDeck&>(m_deck);
		card_deck.shuffle_subsets ();
	}

	/** Refresh display, but don't adjust progress bar
	 */
    m_deck_iter = m_deck.begin ();
	DL ((APP,"m_deck_iter = begin()\n"));
    show_card (false);
}

void
DeckPlayer::
on_edit_clicked ()
{
    trace_with_mask("DeckPlayer::on_edit_clicked",GUITRACE);
    DL((GRAPP,"Bring up deck viewer dialog\n"));

	hide (); // hide DeckPlayer while DeckView is visible.

	int idx = m_deck_iter == m_deck.end() ? -1 : (m_deck_iter - m_deck.begin());
	bool dv_modified = false;

	{
		DeckView dv (m_deck, idx);
		dv.run ();
		dv_modified = dv.is_modified ();
	}

    DL((GRAPP,"Deck size after edit = %d cards (modified = %s)\n", 
		m_deck.size (), (dv_modified ? "TRUE" : "FALSE")));

	sensitize_controls ();

    /** 
     * Update viewer - iterators might have become invalid.
     */
    if (m_deck.size () == 0) 	// Last card removed - clean up
	{
		m_deck_iter = m_deck.end ();
		CARDBOX->mark_as_modified ();
		show_card ();
		
#ifndef IS_HILDON
		/** Under Hildon, this messes up return chain.
		 */
		on_close_clicked ();
#endif
    }
    else 
	{
		if (dv_modified) {
			DL ((APP,"Deck's been modified - go to the beginning\n"));
			CARDBOX->mark_as_modified ();
			show_deck ();
		}
		else {
			DL ((APP,"Deck's intact - stay where we were\n"));
			show_card (false);
		}
    }

	show ();					// Restore DeckPlayer
}

void
DeckPlayer::
on_vcard_edit_clicked (bool disable_cancel_)
{
    trace_with_mask("DeckPlayer::on_vcard_edit_clicked",GUITRACE);

	if (m_deck_iter != m_deck.end () || m_deck.size () > 0) {
		VCard* vc = *m_deck_iter;
		CardView cview (vc);
		int ret = cview.run (disable_cancel_);

		if (ret == Gtk::RESPONSE_OK) {
			show_card (false);
		}
	}
	else {
        Gtk::MessageDialog em (_("The deck is empty!"), Gtk::MESSAGE_ERROR);
        em.run ();
        return;
	}
}

void
DeckPlayer::
on_verify_clicked ()
{
    trace_with_mask("DeckPlayer::on_verify_clicked",GUITRACE);

	if (m_verify_control->get_state () == VC_ARMED) 
	{
		m_verify_control->compare (get_answer ());
		auto_pronounce_card (BACK);
	}
	else if (m_verify_control->get_state () == VC_VALIDATED) 
	{
		if (m_is_real_deck) {
			go_next_cb ();
		}
		else { /* CardBox Player*/ 
			if (m_verify_control->matched ()) {
				know_answer_cb ();
			}
			else {
				no_clue_cb ();
			}
		}
		m_verify_control->clear ();
	}
}

/** 
  Key values can be found in /usr/include/gtk-2.0/gdk/gdkkeysyms.h

  Key bindings:

  Deck mode:
    /------------------------+------------------------.
    |     Action             |        Key(s)          |
    +------------------------+------------------------+
    | Play the sound         | UpArrow                |
    | Previous card          | LeftArrow              |
    | Next card              | RigthArrow, Enter      |
    | Flip the card          | DownArrow, Spacebar    |
    | Iconize windows        | 'u'                    |
    `------------------------+------------------------/

  CardBox mode:
    /------------------------+------------------------.
    |     Action             |        Key(s)          |
    +------------------------+------------------------+
    | Play the sound         | UpArrow                |
    | 'NO' answer            | LeftArrow, 'n'         |
    | 'YES' answer           | RigthArrow, Enter, 'y' |
    | Flip the card          | DownArrow, Spacebar    |
    | Iconize windows        | 'u'                    |
    `------------------------+------------------------/

  Engaging Verification Control (any mode):

    /-------------------------+------------------------.
	| Activate [Check] button | Type in the answer     |
    +-------------------------+------------------------+
	| Check the answer,       | [Check], Enter         |
	| flip the card,          |                        |
	| activate [Next] button  |                        |
    `-------------------------+------------------------/
*/
/*
  GDK_space     0x0020

  GDK_Left      0xFF51
  GDK_Up        0xFF52
  GDK_Right     0xFF53
  GDK_Down      0xFF54

  GDK_KP_Left   0xFF96
  GDK_KP_Up     0xFF97
  GDK_KP_Right  0xFF98
  GDK_KP_Down   0xFF99

  GDK_Tab       0xFF09
  GDK_Return    0xFF0D
  GDK_KP_Enter  0xFF8D
  GDK_F1        0xFFBE
  GDK_F6        0xFFC3
  GDK_F7        0xFFC4
  GDK_F8        0xFFC5
*/

bool 
DeckPlayer::
on_key_pressed (GdkEventKey* evt_)
{
    trace_with_mask("DeckPlayer::on_key_pressed",KEYIN);

#ifndef IS_HILDON
	if (evt_->keyval == GDK_F1) {
		MAINWIN->iconify ();
	}
#endif

	if (evt_->keyval == GDK_space || evt_->keyval > 0xFF51) {
		DL ((KEYIN,"Key pressed: 0x%04X, real_deck: %s, state: %s\n", 
			 evt_->keyval, (m_is_real_deck ? "Yes" : "No"),
			 (m_verify_control->get_state() == VC_START ? "START" :
			  m_verify_control->get_state() == VC_ARMED ? "ARMED" : "VALIDATED")
				));
	}
	
	if (evt_->keyval == GDK_space && m_tab_activated) {
		m_tab_activated = false;
		return false;
	}
	
	if (evt_->keyval == GDK_Tab) {
		m_tab_activated = true;
		return false;
	}

	if (check_F6_key (evt_->keyval)) {
		return true;
	}

	/*********************************************
	 * Real Deck                                 *
	 *********************************************/

	if (m_is_real_deck) {                         
		switch (m_verify_control->get_state ()) {

		case VC_START:
		case VC_VALIDATED:
			if (evt_->keyval == GDK_Right    || 
				evt_->keyval == GDK_KP_Right ||
				evt_->keyval == GDK_Return   || 
				evt_->keyval == GDK_KP_Enter ||
				check_F7_key (evt_->keyval)) 
			{
				go_next_cb ();
			}
			else if (evt_->keyval == GDK_Left    || 
					 evt_->keyval == GDK_KP_Left ||
					 check_F8_key (evt_->keyval))
			{
				go_prev_cb ();
			}
			break;

		case VC_ARMED:
			if (evt_->keyval == GDK_Return || 
				evt_->keyval == GDK_KP_Enter) 
			{
				m_verify_control->compare (get_answer ());
				auto_pronounce_card (BACK);
			}
			else if (evt_->keyval == GDK_space) {
				return false;
			}
			break;
		}
	}
	/*********************************************
	 * CardBox Player                            *
	 *********************************************/

	else {
		switch (m_verify_control->get_state ()) 
		{
		case VC_START:
			if (evt_->keyval == GDK_Right    || 
				evt_->keyval == GDK_KP_Right ||
				check_F7_key (evt_->keyval))
			{
				know_answer_cb ();
			}
			else if (evt_->keyval == GDK_Left    || 
					 evt_->keyval == GDK_KP_Left ||
					 check_F8_key (evt_->keyval))
			{
				no_clue_cb ();
			}
			break;

		case VC_ARMED:
			if (evt_->keyval == GDK_Return || 
				evt_->keyval == GDK_KP_Enter) 
			{
				m_verify_control->compare (get_answer ());
				auto_pronounce_card (BACK);
			}
			else if (evt_->keyval == GDK_space) {
				return false;
			}
			break;

		case VC_VALIDATED:
			if (evt_->keyval == GDK_Return || 
				evt_->keyval == GDK_KP_Enter) 
			{
				if (m_verify_control->matched ()) {
					know_answer_cb ();
				}
				else {
					no_clue_cb ();
				}
				m_verify_control->clear ();
			}
			break;
		}

	} // end
	
	switch (evt_->keyval)
		{
		case GDK_Up:          
		case GDK_KP_Up:
			on_sayit_clicked ();
			break;
			
		case GDK_space: 
		{ 
			m_tab_activated = false; 
		}
		case GDK_Down:        
		case GDK_KP_Down:
			flip_card ();
			break;

		default:
			return false;
		}
	
	return true;
}	

void
DeckPlayer::
set_win_name ()
{
    string title = m_deck.get_name ();
    string::size_type idx = title.rfind (G_DIR_SEPARATOR);
    if (idx != string::npos) {
        title.replace (0, idx+1, "");
    }
    set_title ("Deck Player : " + title);
}

/**
   Remember the size of the window if user resized it.
*/
void
DeckPlayer::
size_allocate_cb (Gtk::Allocation& allocation_)
{
    trace_with_mask("DeckPlayer::size_allocate_cb",GEOM);

	DL ((GEOM,"New size change requested: w=%d, h=%d\n", 
		 allocation_.get_width (), allocation_.get_height ()));

	CONFIG->set_deck_player_geometry (allocation_.get_width (),
									  allocation_.get_height ());
}

/** 
	Adjust the size of the 'example' label when size of 
	the enclosing VBox is changed by the user via resize.
*/
void
DeckPlayer::
answer_box_size_allocate_cb (Gtk::Allocation& allocation_)
{
    trace_with_mask("DeckPlayer::answer_box_size_allocate_cb",GEOM);

    DL((GEOM,"new answer_box size (w=%d; h=%d)\n", 
		allocation_.get_width (), allocation_.get_height ()));

    if (m_answer_box_width != allocation_.get_width ()) {
		m_example.set_size_request (allocation_.get_width () - 20,  -1);
		m_answer_box_width = allocation_.get_width ();
	}
}

void
DeckPlayer::
repaint ()
{
   trace_with_mask("DeckPlayer::repaint",GUITRACE);
   
   m_front  .modify_font (CONFIG->get_question_font ());
   m_back   .modify_font (CONFIG->get_answer_font ());
   m_example.modify_font (CONFIG->get_example_font ());
   m_verify_control->repaint ();

   update_geometry ();
}

/** Resize only if geometry stored in Configurator differs
 *  from the actual size;
 */
void
DeckPlayer::
update_geometry ()
{
#ifndef IS_HILDON
	int actual_height = 0;
	int actual_width = 0;

	Gdk::Rectangle geom = CONFIG->get_deck_player_geometry ();
	get_size (actual_width, actual_height);

	if (actual_width != geom.get_width () || 
		actual_height != geom.get_height ())
	{
		resize (geom.get_width (), geom.get_height ());
	}
#endif
}

/**
 * Replace all newline characters first.
 * Then remove all leading and following extra spaces.
 * Then shrink all  whitespaces that are left to a single space.
 */
void
DeckPlayer::
shrink_to_single_line (Glib::ustring& txt_)
{
   trace_with_mask("DeckPlayer::shrink_to_single_line",GUITRACE);

   Glib::ustring::size_type idx;

   idx = txt_.find_first_of ("\n");
   if (idx != Glib::ustring::npos) {
	   txt_.replace (idx, txt_.size (), "");
   }

   while ((idx = txt_.find ("  ")) != Glib::ustring::npos) {
	   txt_.replace (idx, 2, " ");
   }

   idx = txt_.find_first_not_of (" \t");
   if (idx != Glib::ustring::npos) {
	   txt_.replace (0, idx, "");
   }
   
   idx = txt_.find_last_not_of (" \t");
   if (idx != Glib::ustring::npos) {
	   txt_.replace (idx + 1, txt_.size (), "");
   }
}


Glib::ustring
DeckPlayer::
markup_tildas (const Glib::ustring& src_)
{
//    trace_with_mask("DeckPlayer::markup_tildas", GUITRACE);

	static const string tilda = "<span foreground=\"red\">~</span>";
	Glib::ustring ret;

	if (src_.size () == 0) {
		return ret;
	}

	u_int i;
	u_int b;

	for (i = 0, b = 0; i < src_.size (); i++) {
		if (src_[i] == '~') {
			if (i == 0) {
				ret = tilda;
				b++;
				continue;
			}
			ret += src_.substr (b, i-b) + tilda;
			b = i+1;
		}
	}
	ret += src_.substr (b);

	return ret;
}

bool
DeckPlayer::
markup_is_valid (VCard* card_)
{
	return (Granule::check_markup (card_->get_question ()) &&
			Granule::check_markup (card_->get_answer   ()) &&
			Granule::check_markup (card_->get_example  ()));
}

/** If card we view is reversed, disregard Deck's orientation. 
 *  In this case, each card carries its own *sticky* orientation history
 *  which makes the whole experience of coding and testing maddening!!!
 */
void
DeckPlayer::
switch_side ()
{
    trace_with_mask("DeckPlayer::switch_side", GUITRACE);

	if (m_deck_iter == m_deck.end ()) {
		m_notebook->set_current_page (m_selected_side);
		return;
	}

	if (!m_is_real_deck && (*m_deck_iter)->get_reversed ())
	{
		m_notebook->set_current_page (m_selected_side == FRONT ? BACK : FRONT);
	}
	else {
		m_notebook->set_current_page (m_selected_side);
	}

	auto_pronounce_card (!(*m_deck_iter)->get_reversed () ? FRONT : BACK);
}

/** We want to have card already painted in the window.
 */
void
DeckPlayer::
auto_pronounce_card (SideSelection side_)
{
    trace_with_mask("DeckPlayer::auto_pronounce_card", GUITRACE);

	if (!m_initialized || m_deck_iter == m_deck.end ()) {
		return;
	}

	if (m_selected_side == side_ && CONFIG->auto_pronounce ())
    {
		while (Gtk::Main::instance ()->events_pending ()) {
			Gtk::Main::instance ()->iteration ();
		}

		m_deck.play_card (m_deck_iter);
	}
}

Glib::ustring
DeckPlayer::
get_answer ()
{
    trace_with_mask("DeckPlayer::get_answer", GUITRACE);

	if ((*m_deck_iter)->get_reversed () || m_selected_side == BACK) {
		return (m_front.get_text ());
	}
	else {
		return (m_back.get_text ());
	}
}

