/*
 * Display bar graphs aranged by unique id and by non-unique group id.
 * Continuously reads a new data set and updates the display.
 * 
 * Very slow on a 486Dx2/66 with 16MB and a LB Tseng ET4000 (not a W32).
 *
 * Based on a cut up of the GLUT example/scube.c which rotates a cube 
 * above a checker board with shadows, lighting etc.
 *
 * I reused the scube code for lighting and colors.
 *
 * (c) Copyright 1999, Michael Hamilton, Gentoo South Pacific Limited
 *
 * You have the same rights as stated in the original SGI copyright.
 * (see below).
 *
 * See gr_monitor.1 man page for usage.
 */

/* ---- ORIGINAL scube.c stuff: */

/* Copyright (c) Mark J. Kilgard, 1994. */

/**
 * (c) Copyright 1993, 1994, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED 
 * Permission to use, copy, modify, and distribute this software for 
 * any purpose and without fee is hereby granted, provided that the above
 * copyright notice appear in all copies and that both the copyright notice
 * and this permission notice appear in supporting documentation, and that 
 * the name of Silicon Graphics, Inc. not be used in advertising
 * or publicity pertaining to distribution of the software without specific,
 * written prior permission. 
 *
 * THE MATERIAL EMBODIED ON THIS SOFTWARE IS PROVIDED TO YOU "AS-IS"
 * AND WITHOUT WARRANTY OF ANY KIND, EXPRESS, IMPLIED OR OTHERWISE,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY OR
 * FITNESS FOR A PARTICULAR PURPOSE.  IN NO EVENT SHALL SILICON
 * GRAPHICS, INC.  BE LIABLE TO YOU OR ANYONE ELSE FOR ANY DIRECT,
 * SPECIAL, INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY
 * KIND, OR ANY DAMAGES WHATSOEVER, INCLUDING WITHOUT LIMITATION,
 * LOSS OF PROFIT, LOSS OF USE, SAVINGS OR REVENUE, OR THE CLAIMS OF
 * THIRD PARTIES, WHETHER OR NOT SILICON GRAPHICS, INC.  HAS BEEN
 * ADVISED OF THE POSSIBILITY OF SUCH LOSS, HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE
 * POSSESSION, USE OR PERFORMANCE OF THIS SOFTWARE.
 * 
 * US Government Users Restricted Rights 
 * Use, duplication, or disclosure by the Government is subject to
 * restrictions set forth in FAR 52.227.19(c)(2) or subparagraph
 * (c)(1)(ii) of the Rights in Technical Data and Computer Software
 * clause at DFARS 252.227-7013 and/or in similar or successor
 * clauses in the FAR or the DOD or NASA FAR Supplement.
 * Unpublished-- rights reserved under the copyright laws of the
 * United States.  Contractor/manufacturer is Silicon Graphics,
 * Inc., 2011 N.  Shoreline Blvd., Mountain View, CA 94039-7311.
 *
 * OpenGL(TM) is a trademark of Silicon Graphics, Inc.
 */

/*
 * 1992 David G Yu -- Silicon Graphics Computer Systems
 */

/* ---- End of original scube.c stuff. */

#define VERSION_STRING "Gr_Monitor Version 0.80"

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <time.h>
#include <sys/ioctl.h>
#include <pwd.h>
#ifdef linux
#include <termcap.h>
#endif
#include <termios.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <ctype.h>
#include <setjmp.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include <assert.h>
#include <stdarg.h>

#include <GL/gl.h>
#include <GL/glut.h>

#include "dataset.h"

#define DEBUG_MIN 1
#define DEBUG_MAX 3

#define ROTATION      6
#define PER_ROW      12
#define START_ANGLE  50
#define START_X      0
#define START_Y      -3
#define START_DOLLY  -9
#define FONT_SCALE   0.002
#define ROW_GAP      3.0
#define CRAM_FACTOR  0.25

#define GR_GATHER_COMMAND "GR_GATHER_COMMAND"

#define GATHER_PROC BINDIR "/gr_gather"
#define GATHER_FREE BINDIR "/gr_free"
#define GATHER_RUP  BINDIR "/gr_rup"

#define MAX_LINE        10000
#define MAX_VALUES        200
#define MAX_TEXT           79
#define MAX_CONTROLLABLE   10

#define WIN_WIDTH	  600	/* Default */
#define WIN_HEIGHT	  400	/* Default */

				/* glFrustum settings for perspective */
#define CLIP_LEFT        -1.0	/* Maps to window edges */
#define CLIP_RIGHT        1.0   /* Ditto */
#define CLIP_BOTTOM      -0.6   /* Ditto */
#define CLIP_TOP          1.26  /* Ditto */
#define CLIP_NEAR         1.0   /* Front clipping plane */
#define CLIP_FAR        500.0   /* Back clipping plane */

static char key_help[][50] = { 
  "exit:               ESC",
  "echo help to tty:   ?",
  "graphs per row +/-: p/P",
  "adjust zoom +/-:    a/A",
  "rotate +/-:         x/X y/Y z/Z",
  "translate +/-:      i/I j/J k/K",
  "toggle rotation:    r",
  "select bar type:    0..9",
  "scale selected +/-: s/S",
  "scale selected 1.0: n",
  "toggle selected:    o",
  "toggle lighting:    l or L",
  "toggle fog:         f",
  "toggle halt:        h",
  "step thru halt:     SPACE"
  ""
};

static int key_menu;

static int debug = 0;

static int num_items;

static int useRGB = 1;
static int useLighting = 1;
static int useFog = 0;
static int useDB = 1;

static int isvisible = 1;
static int halted = 0;

static int skip = 1;

static int cram = 0;
static int per_row = 0;
static int gap = ROW_GAP;
static float cram_factor = CRAM_FACTOR;

static int rotate = 0;
static int rotate_y = 0;
static int rotate_x = 1;
static int rotate_z = 0;
static int rotate_angle = START_ANGLE;

static int trans_y = START_Y;
static int trans_x = START_X;
static int trans_z = START_DOLLY;

static float zoom_horz = 1.0;
static float zoom_vert = 1.0;

static int hide_value[MAX_CONTROLLABLE] = {
  0,0,0,0,0, 0,0,0,0,0 
};

static float scale_value[MAX_CONTROLLABLE] = {
  0.8, 0.8, 0.8, 0.8, 0.8, 
  0.8, 0.8, 0.8, 0.8, 0.8 
};

static int value_index = 0;

static int use_stdin = 0;

static int sleep_seconds = 5;

static FILE *inputfp = NULL;

void *font = GLUT_STROKE_ROMAN;
void *fonts[] = {GLUT_STROKE_ROMAN, GLUT_STROKE_MONO_ROMAN};

#define GREY	0
#define RED	1
#define GREEN	2
#define BLUE	3
#define CYAN	4
#define MAGENTA	5
#define YELLOW	6
#define BLACK	7

static float materialColor[8][4] =
{
  {0.8, 0.8, 0.8, 1.0},
  {0.8, 0.0, 0.0, 1.0},
  {0.0, 0.8, 0.0, 1.0},
  {0.0, 0.0, 0.8, 1.0},
  {0.0, 0.8, 0.8, 1.0},
  {0.8, 0.0, 0.8, 1.0},
  {0.8, 0.8, 0.0, 1.0},
  {0.0, 0.0, 0.0, 0.4},
};

static float lightPos[4] =
{2.0, 4.0, 2.0, 1.0};
static float lightDir[4] =
{-2.0, -4.0, -2.0, 1.0};
static float lightAmb[4] =
{0.2, 0.2, 0.2, 1.0};
static float lightDiff[4] =
{0.8, 0.8, 0.8, 1.0};
static float lightSpec[4] =
{0.4, 0.4, 0.4, 1.0};

static float groundPlane[4] =
{0.0, 1.0, 0.0, 1.499};
static float backPlane[4] =
{0.0, 0.0, 1.0, 0.899};

static float fogColor[4] =
{0.0, 0.0, 0.0, 0.0};
static float fogIndex[1] =
{0.0};

static char *windowNameRGBDB = "Display RGB Double buffered";
static char *windowNameRGB = "Display RGB Single buffered";
static char *windowNameIndexDB = "Display Color DB Double buffered";
static char *windowNameIndex = "Display Color DB Single buffered";

static float cube_vertexes[6][4][4] =
{
  {
    {-1.0, -1.0, -1.0, 1.0},
    {-1.0, -1.0, 1.0, 1.0},
    {-1.0, 1.0, 1.0, 1.0},
    {-1.0, 1.0, -1.0, 1.0}},

  {
    {1.0, 1.0, 1.0, 1.0},
    {1.0, -1.0, 1.0, 1.0},
    {1.0, -1.0, -1.0, 1.0},
    {1.0, 1.0, -1.0, 1.0}},

  {
    {-1.0, -1.0, -1.0, 1.0},
    {1.0, -1.0, -1.0, 1.0},
    {1.0, -1.0, 1.0, 1.0},
    {-1.0, -1.0, 1.0, 1.0}},

  {
    {1.0, 1.0, 1.0, 1.0},
    {1.0, 1.0, -1.0, 1.0},
    {-1.0, 1.0, -1.0, 1.0},
    {-1.0, 1.0, 1.0, 1.0}},

  {
    {-1.0, -1.0, -1.0, 1.0},
    {-1.0, 1.0, -1.0, 1.0},
    {1.0, 1.0, -1.0, 1.0},
    {1.0, -1.0, -1.0, 1.0}},

  {
    {1.0, 1.0, 1.0, 1.0},
    {-1.0, 1.0, 1.0, 1.0},
    {-1.0, -1.0, 1.0, 1.0},
    {1.0, -1.0, 1.0, 1.0}}
};

static float cube_normals[6][4] =
{
  {-1.0, 0.0, 0.0, 0.0},
  {1.0, 0.0, 0.0, 0.0},
  {0.0, -1.0, 0.0, 0.0},
  {0.0, 1.0, 0.0, 0.0},
  {0.0, 0.0, -1.0, 0.0},
  {0.0, 0.0, 1.0, 0.0}
};


void projection(float, float);
void display(Dataset *);

int is_data_ready(FILE *inputfp) 
{
  /* For select(2). */
  struct timeval tv;
  fd_set in;

  /* Non-blocking read */

  tv.tv_sec = 0;
  tv.tv_usec = 0;
  FD_ZERO(&in);
  FD_SET(fileno(inputfp), &in);
  return select(16, &in, 0, 0, &tv) > 0 && !feof(inputfp);
}

Dataset *get_data(FILE *inputfp,
		  int prev_values) {

  static Dataset *data_set = NULL;
  Dataset *new_set = NULL;

  int i, j;
  int id, group_size;

  char group[MAX_TEXT];
 
  static char line[MAX_LINE], *upto;

  if ((prev_values || !is_data_ready(inputfp)) && data_set) {
    if (debug >= DEBUG_MAX) fputs("Returning old data.\n", stderr);
    return data_set;
  }

  /* Data is ready or we have no existing data-set */

  new_set = getDataset(inputfp);
  if (new_set != NULL) {
    if (data_set != NULL) {
      freeDataset(data_set);
    }
    data_set = new_set;
  }

  if (cram) {
    
  }

  return data_set;
}

void do_update(void)
{
  static int first_time_thru = 1;
  static int count = 0;
  static char title[MAX_TEXT];

  int num_items;
  int num_values;

  Dataset *data_set = get_data(inputfp, halted);

  if (first_time_thru) {
    first_time_thru = 0;
    if (per_row == 0) {
      if (data_set->groupCount > 0) {
	per_row = data_set->groups[0]->itemCount;
	/*printf("%d\n", per_row);*/
      }
      if (per_row < PER_ROW) {
	per_row = PER_ROW;
      }
    }
  }


  ++count;

  if (skip && !halted) {
    while (is_data_ready(inputfp)) { /* Skip over data until we are up to date */
      ++count;
      if (debug >= DEBUG_MIN) {
	fprintf(stderr,
		"Can't render fast enough - skipping over data set %d.\n",
		count);
      }
      data_set = get_data(inputfp, halted);
    } 
  }

  if (data_set) {
    if (debug >= DEBUG_MAX) fprintf(stderr, "Displaying set %s.\n", data_set->id) ;
    glutSetWindowTitle(data_set->id);
    glutSetIconTitle(data_set->id);
  }

  if (isvisible && data_set) {
    if (debug >= DEBUG_MAX) fputs("Displaying.\n", stderr) ;
    display(data_set);    
    /*    glutPostRedisplay(); */
    if (debug >= DEBUG_MAX) fprintf(stderr, "Done (%d).\n", count) ;
  }
}

void do_idle(void)
{
				/* Only update if there's new data. */
  if (rotate || (!halted && is_data_ready(inputfp))) {	
    if (debug >= DEBUG_MAX) fputs("Update display during idle time.\n", stderr);
    do_update();
  }
  else {
    sleep(1);
  }
}

void key_help_to_stdout(void)
{
  int i;
  puts("\nKey Summary.\n");
  for (i = 0; *(key_help[i]); i++) {
    puts(key_help[i]);
  }
}

void
keyboard(unsigned char ch, int x, int y)
{
  int update = 1; /* Most commands need a display update */

  switch (ch) {

  case 27:             /* escape */
    exit(0);
    break;

  case 'L':
  case 'l':
    useLighting = !useLighting;
    useLighting ? glEnable(GL_LIGHTING) : glDisable(GL_LIGHTING);
    glutPostRedisplay();
    update = 0;
    break;
  case 'F':
  case 'f':
    useFog = !useFog;
    useFog ? glEnable(GL_FOG) : glDisable(GL_FOG);
    glutPostRedisplay();
    update = 0;
    break;

  case 'R':
  case 'r':
    rotate = !rotate;
    break;

  case 's':
    scale_value[value_index] += 0.2;
    break;
  case 'S':
    scale_value[value_index] -= 0.2;
    if (scale_value[value_index] < 0) scale_value[value_index] = 0.0;
    break;

  case 'n':
  case 'N':
    scale_value[value_index] = 1.0;
    break;

  case 'o':
  case 'O':
    hide_value[value_index] = !hide_value[value_index];
    break;

  case 'i':
    trans_x = 1 ;
    break;
  case 'I':
    trans_x = -1 ;
    break;
  case 'j':
    trans_y = 1;
    break;
  case 'J':
    trans_y = -1;
    break;
  case 'k':
    trans_z = 1;
    break;
  case 'K':
    trans_z = -1;
    break;

  case 'x':
    rotate_x = 1 ;
    break;
  case 'X':
    rotate_x = -1 ;
    break;
  case 'y':
    rotate_y = 1;
    break;
  case 'Y':
    rotate_y = -1;
    break;
  case 'z':
    rotate_z = 1;
    break;
  case 'Z':
    rotate_z = -1;
    break;


  case '0':
  case '1':
  case '2':
  case '3':
  case '4':
  case '5':
  case '6':
  case '7':
  case '8':
  case '9':
    value_index = ch - '0';
    update = 0;
    break;

  case 'p':
    per_row++;
    break;
  case 'P':
    if (per_row > 1) per_row--;
    break;

  case 'a':
    zoom_vert -= 0.1;
    zoom_horz -= 0.1;
    break;
  case 'A':
    zoom_vert += 0.1;
    zoom_horz += 0.1;
    break;

  case 'H':
  case 'h':
    halted = !halted;
    if (halted) update = 0;
    break;

  case ' ':
    update = 1;
    break;

  case '?':
    key_help_to_stdout();
    break;

  default:
    /*printf("key %d\n", ch); */
    break;
  }

  if (update) {
    if (debug >= DEBUG_MAX) fputs("Update display after keypress.\n", stderr);
    do_update();
  }

}

void strokePrintf(const char *format, ...)
{
  va_list ap; 

  static char buf[1000];

  int i;

  va_start (ap, format);
  
  vsprintf (buf, format, ap);

  glPushMatrix();

  for (i = 0; i < strlen(buf); i++) {
    glutStrokeCharacter(font, buf[i]);
  }

  glPopMatrix();
}


void
buildColormap(void)
{
  if (useRGB) {
    return;
  } else {
    int mapSize = 1 << glutGet(GLUT_WINDOW_BUFFER_SIZE);
    int rampSize = mapSize / 8;
    int entry;
    int i;

    for (entry = 0; entry < mapSize; ++entry) {
      int hue = entry / rampSize;
      GLfloat val = (entry % rampSize) * (1.0 / (rampSize - 1));
      GLfloat red, green, blue;

      red = (hue == 0 || hue == 1 || hue == 5 || hue == 6) ? val : 0;
      green = (hue == 0 || hue == 2 || hue == 4 || hue == 6) ? val : 0;
      blue = (hue == 0 || hue == 3 || hue == 4 || hue == 5) ? val : 0;

      glutSetColor(entry, red, green, blue);
    }

    for (i = 0; i < 8; ++i) {
      materialColor[i][0] = i * rampSize + 0.2 * (rampSize - 1);
      materialColor[i][1] = i * rampSize + 0.8 * (rampSize - 1);
      materialColor[i][2] = i * rampSize + 1.0 * (rampSize - 1);
      materialColor[i][3] = 0.0;
    }

    fogIndex[0] = -0.2 * (rampSize - 1);
  }
}

static void
setColor(int c)
{
  if (useLighting) {
    if (useRGB) {
      glMaterialfv(GL_FRONT_AND_BACK,
        GL_AMBIENT_AND_DIFFUSE, &materialColor[c][0]);
    } else {
      glMaterialfv(GL_FRONT_AND_BACK,
        GL_COLOR_INDEXES, &materialColor[c][0]);
    }
  } else {
    if (useRGB) {
      glColor4fv(&materialColor[c][0]);
    } else {
      glIndexf(materialColor[c][1]);
    }
  }
}

static void
drawBar(int color, float height)
{
  int i, j;
  /* printf("hieght=%f\n", height); */
  setColor(color);

  for (i = 0; i < 6; ++i) {
    glNormal3fv(&cube_normals[i][0]);
    glBegin(GL_POLYGON);
    for (j = 0; j < 4; j++) {
      if (cube_vertexes[i][j][1] >= 0) {
	glVertex4f(cube_vertexes[i][j][0],
		   height,
		   //		   height == 0 ? 0.00001 : height,
		   cube_vertexes[i][j][2],
		   cube_vertexes[i][j][3]
		   );
      }
      else {
	glVertex4fv(&cube_vertexes[i][j][0]);
      }      
    }
    glEnd();
  }
}

drawIDname( char *id )
{
  while (*(id) == ' ') ++id;	/* Skip spaces */
  if (debug > DEBUG_MAX) {
    fprintf(stderr, "draw %s\n", id);
  }
  setColor(BLACK);
  glPushMatrix();
  glTranslatef(-0.3, -0.65, 0.7);
  glScalef(FONT_SCALE,1.0,FONT_SCALE);
  glRotatef(-90, 1.0, 0.0, 0.0);
  strokePrintf("%s", id);
  glPopMatrix();
}

void 
drawOneItem(Item *item)
{

  int i;

  drawIDname(item->id);

  glPushMatrix();

  glTranslatef(0.0, -0.66, 0.0); 
  glScalef(0.3, 0.1, 0.3);

  for (i = 0; i <  item->dataCount; i++) {
    if (i >= MAX_CONTROLLABLE || !hide_value[i]) {
				/* draw bar */	
      if (i < MAX_CONTROLLABLE && scale_value[i] != 1.0) {
	drawBar(i + 1, item->datums[i]->value * scale_value[i]);
      }
      else {
	drawBar(i + 1, item->datums[i]->value);        /* draw bar */
      }
    }
    glTranslatef(0.0, 0.0, -3.33);
  }
  glPopMatrix();

}

drawValueName( char *bname, int num )
{
  setColor(BLACK);
  glPushMatrix();
  glTranslatef(-1.4, -0.65, -num);
  glScalef(FONT_SCALE*1.5,1.0,FONT_SCALE*1.5);
  glRotatef(-90, 1.0, 0.0, 0.0);
  strokePrintf("%s", bname);
  glPopMatrix();
}

drawGname( char *group, int length )
{
  setColor(RED);
  glPushMatrix();
  glTranslatef((length + 0.5)/2.0 - 2.0, -0.65, 1.3);
  glScalef(FONT_SCALE*2,1.0,FONT_SCALE*2);
  glRotatef(-90, 1.0, 0.0, 0.0);
  strokePrintf("%s", group);
  glPopMatrix();
}

void 
drawBase(char *gname, int length, Item *item)
{

  int i, j;
  int num_values = item->dataCount;
  drawGname(gname, length);
  for (i = 0; i < num_values; i++) {
    drawValueName(item->datums[i]->name, i);
  }

  glPushMatrix();

  /* printf("length=%d\n",length); */


  glTranslatef(-0.5, -1.0, -(num_values - 1));
  glScalef(1.0, 0.3, 1.0);

  setColor(GREY);

  for (i = 0; i < 6; ++i) {
    glNormal3fv(&cube_normals[i][0]);
    glBegin(GL_POLYGON);
    for (j = 0; j < 4; j++) {
      glVertex4f((cube_vertexes[i][j][0] >= 0) ? (length + 0.5) : cube_vertexes[i][j][0],
		 cube_vertexes[i][j][1],
		 (cube_vertexes[i][j][2] >= 0) ? (num_values + 0.5) : cube_vertexes[i][j][2],
		 cube_vertexes[i][j][3]
		 );
    }
    glEnd();
  }

  glPopMatrix();

}



void showData (Dataset *data_set)
{
  int num_groups = data_set->groupCount;
  int num_values = 0;
  int num_drawn;
  int step;
  char last_group[MAX_TEXT];
  float location;

  glPushMatrix();

  glTranslatef(-(per_row / 2 + 1), 0.0, 0.0);

  num_drawn = 0;
  step = 0;

  while (num_drawn < num_groups) {
    
    int i;
    
    int num_in_group = data_set->groups[num_drawn]->itemCount;
    drawBase(data_set->groups[num_drawn]->id, num_in_group, data_set->groups[num_drawn]->items[0]);
    
    for (i = 0; i < num_in_group; i++) {
      int new_num_values = data_set->groups[num_drawn]->items[i]->dataCount;
      if (new_num_values > num_values) {
	num_values = new_num_values;
      }
      drawOneItem(data_set->groups[num_drawn]->items[i]);
      glTranslatef(1, 0.0, 0.0);
      step++;
    }
    num_drawn++;
    
    if (num_drawn < num_groups) {

      if (step > per_row || step + data_set->groups[num_drawn]->itemCount > per_row
	  && (!cram 
	      || (step > (1 - cram_factor) * per_row)
	      || (step + data_set->groups[num_drawn]->itemCount > per_row * (1 + cram_factor)))
	  ) {
	glTranslatef(-step, 0.0, -(num_values + gap));
	step = 0;
      } else {
	glTranslatef(3, 0.0, 0.0);
	step += 3;
      }
    }
    
  }
  
  glPopMatrix();
}

void
display(Dataset *data_set)
{

  projection(zoom_horz, zoom_vert);

  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

  if (trans_x || trans_y || trans_z) {
    glTranslatef(trans_x, trans_y, trans_z); 
    trans_x = trans_y = trans_z = 0;
  }

  if (rotate || rotate_x || rotate_y || rotate_z) {
    glRotatef(rotate_angle, rotate_x, rotate_y, rotate_z);
    rotate_angle = ROTATION;
    rotate_x = rotate_y = rotate_z = 0;
    if (rotate) rotate_y = 1;
  }

  /* Draw bars */
  showData(data_set);

  glDepthMask(GL_TRUE);
  if (useRGB) {
    glDisable(GL_BLEND);
  } else {
    glDisable(GL_POLYGON_STIPPLE);
  }
  if (useFog) {
    glEnable(GL_FOG);
  }
  glutSwapBuffers();
}

void
visible(int state)
{
  isvisible = state == GLUT_VISIBLE ;
}

void
fog_select(int fog)
{
  glFogf(GL_FOG_MODE, fog);
  glutPostRedisplay();
}

void
menu_select(int mode)
{
  switch (mode) {
  case 1:
    halted = 0;
    break;
  case 2:
    halted = 1;
    break;
  case 3:
    useFog = !useFog;
    useFog ? glEnable(GL_FOG) : glDisable(GL_FOG);
    glutPostRedisplay();
    break;
  case 4:
    useLighting = !useLighting;
    useLighting ? glEnable(GL_LIGHTING) : glDisable(GL_LIGHTING);
    glutPostRedisplay();
    break;
  case 5:
    exit(0);
    break;
  }
}

void do_nothing(int choice)
{
				/* Dummy for key_help menu */
}

void projection(float zoom_horz, float zoom_vert)
{
  static float last_h = 0;
  static float last_v = 0;
  if (zoom_horz != last_h || zoom_vert != last_v) {
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glFrustum(CLIP_LEFT * zoom_horz, CLIP_RIGHT * zoom_horz, CLIP_BOTTOM * zoom_vert, CLIP_TOP * zoom_vert, CLIP_NEAR, CLIP_FAR);
    glMatrixMode(GL_MODELVIEW);
    last_h = zoom_horz;
    last_v = zoom_vert;
  }
}

void gl_setup(int argc, char **argv, int width, int height) {

  int fog_menu, i;
  char *name;

  glutInitWindowSize(width, height);
  glutInit(&argc, argv);
  /* choose visual */
  if (useRGB) {
    if (useDB) {
      glutInitDisplayMode(GLUT_DOUBLE | GLUT_RGB | GLUT_DEPTH);
      name = windowNameRGBDB;
    } else {
      glutInitDisplayMode(GLUT_SINGLE | GLUT_RGB | GLUT_DEPTH);
      name = windowNameRGB;
    }
  } else {
    if (useDB) {
      glutInitDisplayMode(GLUT_DOUBLE | GLUT_INDEX | GLUT_DEPTH);
      name = windowNameIndexDB;
    } else {
      glutInitDisplayMode(GLUT_SINGLE | GLUT_INDEX | GLUT_DEPTH);
      name = windowNameIndex;
    }
  }

  glutCreateWindow(name);

  buildColormap();

  glutKeyboardFunc(keyboard);
  glutDisplayFunc(do_update);
  glutVisibilityFunc(visible);

  key_menu = glutCreateMenu(do_nothing);

  for (i = 0; *(key_help[i]); i++) {
    glutAddMenuEntry(key_help[i], 0);
  }

  glutAttachMenu(GLUT_MIDDLE_BUTTON);
  

  fog_menu = glutCreateMenu(fog_select);
  glutAddMenuEntry("Linear fog", GL_LINEAR);
  glutAddMenuEntry("Exp fog", GL_EXP);
  glutAddMenuEntry("Exp^2 fog", GL_EXP2);

  glutCreateMenu(menu_select);
  glutAddSubMenu("Key summary", key_menu);
  glutAddMenuEntry("Unfreeze", 1);
  glutAddMenuEntry("Freeze", 2);
  glutAddMenuEntry("Toggle fog", 3);
  glutAddMenuEntry("Toggle lighting", 4);
  glutAddSubMenu("Fog type", fog_menu);
  glutAddMenuEntry("Quit", 5);
  glutAttachMenu(GLUT_RIGHT_BUTTON);

  /* setup context */
  projection(zoom_horz, zoom_vert);

  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  glTranslatef(0.0, 0.0, -3.0);

  glEnable(GL_DEPTH_TEST);

  if (useLighting) {
    glEnable(GL_LIGHTING);
  }
  glEnable(GL_LIGHT0);
  glLightfv(GL_LIGHT0, GL_POSITION, lightPos);
  glLightfv(GL_LIGHT0, GL_AMBIENT, lightAmb);
  glLightfv(GL_LIGHT0, GL_DIFFUSE, lightDiff);
  glLightfv(GL_LIGHT0, GL_SPECULAR, lightSpec);
  /* 
   * glLightfv(GL_LIGHT0, GL_SPOT_DIRECTION, lightDir);
   * glLightf(GL_LIGHT0, GL_SPOT_EXPONENT, 80);
   * glLightf(GL_LIGHT0, GL_SPOT_CUTOFF, 25);
   */

  glEnable(GL_NORMALIZE);

  if (useFog) {
    glEnable(GL_FOG);
  }
  glFogfv(GL_FOG_COLOR, fogColor);
  glFogfv(GL_FOG_INDEX, fogIndex);
  glFogf(GL_FOG_MODE, GL_EXP);
  glFogf(GL_FOG_DENSITY, 0.5);
  glFogf(GL_FOG_START, CLIP_NEAR);
  glFogf(GL_FOG_END, CLIP_FAR);

  glEnable(GL_CULL_FACE);
  glCullFace(GL_BACK);

  glShadeModel(GL_SMOOTH);

  glBlendFunc(GL_ONE_MINUS_SRC_ALPHA, GL_SRC_ALPHA);

#if 0
  if (useLogo) {
    glPolygonStipple((const GLubyte *) sgiPattern);
  } else {
    glPolygonStipple((const GLubyte *) shadowPattern);
  }
#endif

  glClearColor(0.0, 0.0, 0.0, 1);
  glClearIndex(0);
  glClearDepth(1);

}

void usage(void) 
{
  puts("");
  puts("gather | display [options]");
  puts(" ");
  puts("Options:");
  puts("    -t proc|free|rup|<cmd> type of data to gather or command to exec.");
  puts("    -host host rsh gather on host.");
  puts("    -witdh n   pixals.");
  puts("    -height n  pixals.");
  puts("    -perrow n  data sets per row.");
  puts("    -cram x    (x * perrow) <= in row or in row <= ((1 + x) * perrow)");
  puts("    -gap x     gap between rows in floating point.");
  puts("    -angle n   starting tilt toward or away from you.");
  puts("    -dolly n   starting viewing distance from origin.");
  puts("    -zoom x    zoom factor horizontal and vertical.");
  puts("    -zhorz x   horizontal zoom factor.");
  puts("    -zvert x   vertical zoom factor.");
  puts("    -geometry  window size and location");
  puts("    -color     take over color map");
  puts("    -nolight   disable lighting");
  puts("    -fog       enable fog effects");
  puts("    -onebuf    disable double buffering");
  puts("    -rotate    enable rotation");
  puts("    -stdin     take input from stdin");         
  puts("    -host h    rsh gather on host");         
  puts("    -sleep n   sleep seconds between updates.");         
  puts("    -noskip    don't skip data even if it comes in too fast.");        
  puts("    -verbose   all feedback/debugging output to stderr enabled."); 
  puts("    -v         report skipped data due to it arriving too fast.");     
  puts("");

}

int
main(int argc, char **argv)
{

  
  int width = WIN_WIDTH, height = WIN_HEIGHT;
  int i;

  char hostname[MAX_TEXT] = "";
  char *gather_type = GATHER_PROC;

  /* process commmand line args */
  for (i = 1; i < argc; ++i) {

    if (!strcmp("-c", argv[i])) {
      useRGB = !useRGB;

    } else if (!strcmp("-color", argv[i])) {
      useRGB = !useRGB;

    } else if (!strcmp("-nolight", argv[i])) {
      useLighting = !useLighting;

    } else if (!strcmp("-fog", argv[i])) {
      useFog = !useFog;

    } else if (!strcmp("-onebuf", argv[i])) {
      useDB = !useDB;

    } else if (!strcmp("-rotate", argv[i])) {
      rotate = 1;

    } else if (!strcmp("-width", argv[i])) {
      width = strtol(argv[++i],NULL,0);

    } else if (!strcmp("-height", argv[i])) {
      height = strtol(argv[++i],NULL,0);

    } else if (!strcmp("-perrow", argv[i])) {
      per_row = strtol(argv[++i],NULL,0);

    } else if (!strcmp("-cram", argv[i])) {
      cram = 1;
      cram_factor = strtod(argv[++i],NULL);

    } else if (!strcmp("-angle", argv[i])) {
      rotate_angle = strtol(argv[++i],NULL,0);

    } else if (!strcmp("-dolly", argv[i])) {
      trans_z = strtol(argv[++i],NULL,0);

    } else if (!strcmp("-gap", argv[i])) {
      gap = strtod(argv[++i],NULL);
      
    } else if (!strcmp("-stdin", argv[i])) {
      use_stdin = 1;

    } else if (!strcmp("-host", argv[i])) {
      strcpy(hostname, argv[++i]);

    } else if (!strcmp("-sleep", argv[i])) {
      sleep_seconds = strtod(argv[++i],NULL);
      if (sleep_seconds == 0) {
	halted = 1;
      }
    } else if (!strcmp("-noskip", argv[i])) {
      skip = 0;

    } else if (!strcmp("-zoom", argv[i])) {
      zoom_horz = zoom_vert = strtod(argv[++i],NULL);
    } else if (!strcmp("-zvert", argv[i])) {
      zoom_vert = zoom_vert = strtod(argv[++i],NULL);
    } else if (!strcmp("-zhorz", argv[i])) {
      zoom_horz = strtod(argv[++i],NULL);

    } else if (!strcmp("-verbose", argv[i])
	       || !strcmp("-debug", argv[i])) {
      debug = DEBUG_MAX;

    } else if (!strcmp("-v", argv[i])) {
      debug = DEBUG_MIN;

    } else if (!strcmp("-t", argv[i])) {
      gather_type = argv[++i];
      if (!strcmp("proc", gather_type)) {
	gather_type = GATHER_PROC;
      }
      else if (!strcmp("free", gather_type)) {
	gather_type = GATHER_FREE;
      }
      else if (!strcmp("rup", gather_type)) {
	gather_type = GATHER_RUP;
      }
    } else {
      usage();
    }
  }
  
  if (use_stdin) {
    inputfp = stdin;
  }
  else {
    char *cmd = getenv(GR_GATHER_COMMAND);
    char command[MAX_LINE];

    strcpy(command, cmd ? cmd : gather_type);

    if (sleep_seconds) {
      sprintf(command, "%s -sleep %d", command, sleep_seconds);
    }

    if (*hostname) {
      char rsh[MAX_LINE];
      sprintf(rsh, "rsh -n %s '%s'", hostname, command);
      inputfp = popen(rsh, "r");
    } else {
      inputfp = popen(command, "r");
    }
  }
	       
  puts(VERSION_STRING);
  puts("Press the middle mouse button for key summary.");
  puts("Starting up...");
  
  sleep(2);			/* Give the data gatherer a chance 
				 * to warm up.
				 */

  /* loop, collecting process info and sleeping */
  
  gl_setup(argc, argv, width, height);

  glutIdleFunc(do_idle);

  glutMainLoop();
}



/*
 * Normal end of execution.
 */
void
end(void)
{
    exit(0);
}


/*
 * SIGTSTP catcher.
 */
void
stop(void)
{
    exit(0);
}


