/*
 * This file is part of gshutdown.
 *
 * Copyright (c) 2006, GShutdown Team <gshutdown@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with This program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "action.h"
#include "misc.h"
#include "gui.h"
#include "values.h"
#include "gdm.h"

/* Run the action now ! */
static gboolean action_run_command(Action *action);

/*
 * Constructor.
 */
Action *action_new()
{
    Action *ptr = g_malloc(sizeof(Action));
    ptr->type = ACTION_TYPE_NONE;
    ptr->command = NULL;
    return ptr;
}

/* 
 * Destructor.
 */
void action_destroy(Action *action)
{
    if(action->command != NULL)
        g_free(action->command);

    g_free(action);
}

/*
 * Read the action type (see ActionType
 * in action.h)
 */
ActionType action_get_type(Action *action)
{
    return action->type;
}

/*
 * Modity the action type (see ActionType
 * in action.h)
 */
void action_set_type(Action *action, ActionType type)
{
    action->type = type;
}

/*
 * Change the custom command.
 *
 * NB: command are duplicated with g_strdup.
 */
void action_set_command(Action *action, const gchar *command)
{
    if(action->command != NULL)
        g_free(action->command);

    action->command = g_strdup(command);
}

/*
 * Return the custom command.
 *
 * Warning: The pointer returned are internal.
 *          DO NOT FREE IT!
 */
const gchar *action_get_command(Action *action)
{
    return (action->command) ? action->command : "";
}

/*
 * Run the actual action.
 *
 * Return TRUE on error.
 */
gboolean action_run(Action *action)
{
    Desktop current_desktop = values.detected_desktop;
    DisplayManager current_display_manager =  values.detected_display_manager;

    /* Command before action */
    if(values.command_before_action_enabled == TRUE)
        run_command(values.command_before_action);

    /* Method = Command */
    if(values.action_method == ACTION_METHOD_CUSTOM_COMMAND || 
       action_get_type(action) == ACTION_TYPE_COMMAND) {
        return action_run_command(action);
    }

    /* Method = Selected Manually */
    if(values.action_method == ACTION_METHOD_MANUAL) {
        if(values.selected_desktop != DESKTOP_AUTO)
            current_desktop = values.selected_desktop;

        if(values.selected_display_manager != DISPLAY_MANAGER_AUTO)
            current_display_manager = values.selected_display_manager;
    }

    /* WARNING: After this line, its important to use
     * current_display_manager and current_desktop */

    /* Not detected? */
    if(current_desktop== DESKTOP_NONE ||
            current_display_manager == DISPLAY_MANAGER_NONE) {
        return TRUE;
    }

    /* GDM */
    if(current_display_manager == DISPLAY_MANAGER_GDM) {
        GdmLogoutAction logout_action = GDM_LOGOUT_ACTION_NONE;

        if(action_get_type(action) == ACTION_TYPE_SHUTDOWN)
            logout_action = GDM_LOGOUT_ACTION_SHUTDOWN;
        else if(action_get_type(action) == ACTION_TYPE_RESTART)
            logout_action = GDM_LOGOUT_ACTION_REBOOT;

        gdm_set_logout_action(logout_action);
    } 
    /* KDM */
    else if(current_display_manager == DISPLAY_MANAGER_KDM) {
        if(action_get_type(action) == ACTION_TYPE_SHUTDOWN)
            run_command("kdmctl shutdown halt forcenow");
        else if(action_get_type(action) == ACTION_TYPE_RESTART)
            run_command("kdmctl shutdown reboot forcenow");
    }

    /* GNOME */
    if(current_desktop == DESKTOP_GNOME) 
        return run_command("gnome-session-save --kill --silent");
        /*TODO tester si le programme gnome-session-save existe 
         * au début du programme, et pourquoi pas le dire à
         * l'utilisateur */

    /* KDE */
    else if(current_desktop == DESKTOP_KDE)
        return run_command("kdeinit_shutdown");

    /* XFCE */
    /* TODO xfce support
    else if(current_desktop == DESKTOP_XFCE)
        return run_command("xfce4-session-logout");
    */

    return TRUE; /* Error */
}

/*
 * Run the custom command (action->command).
 *
 * Return TRUE on error.
 */
static gboolean action_run_command(Action *action)
{
    if(action->command == NULL) 
        return TRUE;

    return run_command(action->command);
}

/* vim:cindent:et:sw=4:ts=4:sts=4:tw=78:fenc=utf-8
 */
