/* GStreamer
 * Copyright (C) <1999> Erik Walthinsen <omega@cse.ogi.edu>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#include <sys/stat.h>
#include <unistd.h>

#include <libgnome/libgnome.h>

#include <config.h>

#include "gsteditor.h"
#include "gsteditoritem.h"
#include "stockicons.h"
#include <gst/common/gste-common.h>
#include <gst/common/gste-debug.h>
#include "../element-browser/browser.h"
#include "../../../pixmaps/pixmaps.h"

#define GST_CAT_DEFAULT gste_debug_cat

static void gst_editor_class_init (GstEditorClass * klass);
static void gst_editor_init (GstEditor * project_editor);
static void gst_editor_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);
static void gst_editor_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void gst_editor_dispose (GObject * object);
static void gst_editor_connect_func (const gchar * handler_name,
    GObject * object,
    const gchar * signal_name,
    const gchar * signal_data,
    GObject * connect_object, gboolean after, gpointer user_data);

static gint on_delete_event (GtkWidget * widget,
    GdkEvent * event, GstEditor * editor);
static void on_canvas_notify (GObject * object,
    GParamSpec * param, GstEditor * ui);
static void on_xml_loaded (GstXML * xml, GstObject * object, xmlNodePtr self,
    GData ** datalistp);


enum
{
  PROP_0,
  PROP_FILENAME
};

enum
{
  LAST_SIGNAL
};

typedef struct
{
  GstEditor *editor;
  GModule *symbols;
}
connect_struct;


static GObjectClass *parent_class;

static gint _num_editor_windows = 0;

/* GST_EDITOR_ERROR quark - FIXME: maybe move this to a more generic part */
#define GST_EDITOR_ERROR	(gst_editor_error_quark ())

GQuark
gst_editor_error_quark (void)
{
  static GQuark quark = 0;

  if (quark == 0)
    quark = g_quark_from_static_string ("gst-editor-error-quark");
  return quark;
}

/* error dialog response - FIXME: we can use positive values since the
   GTK_RESPONSE ones are negative ? */
#define GST_EDITOR_RESPONSE_DEBUG 1

GType
gst_editor_get_type (void)
{
  static GType project_editor_type = 0;

  if (!project_editor_type) {
    static const GTypeInfo project_editor_info = {
      sizeof (GstEditorClass),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,
      (GClassInitFunc) gst_editor_class_init,
      NULL,
      NULL,
      sizeof (GstEditor),
      0,
      (GInstanceInitFunc) gst_editor_init,
    };

    project_editor_type =
        g_type_register_static (G_TYPE_OBJECT, "GstEditor",
        &project_editor_info, 0);
  }
  return project_editor_type;
}

static void
gst_editor_class_init (GstEditorClass * klass)
{
  GList *list;
  GdkPixbuf *pixbuf;
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

  parent_class = g_type_class_ref (G_TYPE_OBJECT);

  gobject_class->get_property = gst_editor_get_property;
  gobject_class->set_property = gst_editor_set_property;
  gobject_class->dispose = gst_editor_dispose;

  g_object_class_install_property (gobject_class, PROP_FILENAME,
      g_param_spec_string ("filename", "Filename", "File name",
          NULL, G_PARAM_READWRITE | G_PARAM_CONSTRUCT));

  _gst_editor_stock_icons_init ();

  /* give all windows the same icon (in theory) */
  pixbuf = gdk_pixbuf_new_from_inline (-1, gst_editor_stock_image, FALSE, NULL);
  list = g_list_prepend (NULL, pixbuf);
  gtk_window_set_default_icon_list (list);

  g_list_free (list);
  g_object_unref (G_OBJECT (pixbuf));
}

static void
gst_editor_init (GstEditor * editor)
{
  connect_struct data;
  GModule *symbols;
  gchar *path;

  symbols = g_module_open (NULL, 0);

  data.editor = editor;
  data.symbols = symbols;

  path = gste_get_ui_file ("editor.glade2");
  if (!path)
    g_error ("GStreamer Editor user interface file 'editor.glade2' not found.");
  editor->xml = glade_xml_new (path, "main_project_window", NULL);

  if (!editor->xml) {
    g_error ("GStreamer Editor could not load main_project_window from %s",
        path);
  }
  g_free (path);

  glade_xml_signal_autoconnect_full (editor->xml, gst_editor_connect_func,
      &data);

  editor->window = glade_xml_get_widget (editor->xml, "main_project_window");
  gtk_widget_show (editor->window);

  editor->canvas =
      (GstEditorCanvas *) g_object_new (gst_editor_canvas_get_type (), "aa",
      FALSE, NULL);

  gtk_widget_show (GTK_WIDGET (editor->canvas));

  bonobo_dock_set_client_area (BONOBO_DOCK (glade_xml_get_widget (editor->xml,
              "bonobodock1")), GTK_WIDGET (editor->canvas));

  _num_editor_windows++;

  g_signal_connect (editor->window, "delete-event",
      G_CALLBACK (on_delete_event), editor);
  g_signal_connect (editor->canvas, "notify", G_CALLBACK (on_canvas_notify),
      editor);
}

static void
gst_editor_set_property (GObject * object, guint prop_id, const GValue * value,
    GParamSpec * pspec)
{
  gchar *title;
  GstEditor *editor = GST_EDITOR (object);
  const gchar *filename;
  static gint count = 1;

  switch (prop_id) {
    case PROP_FILENAME:
      if (editor->filename)
        g_free (editor->filename);
      filename = g_value_get_string (value);
      if (!filename) {
        editor->filename = g_strdup_printf ("untitled-%d.xml", count++);
        editor->need_name = TRUE;
      } else {
        editor->filename = g_strdup (filename);
        editor->need_name = FALSE;
      }

      title = g_strdup_printf ("%s%s - %s",
          editor->filename, editor->changed ? "*" : "",
          _("GStreamer Pipeline Editor"));
      gtk_window_set_title (GTK_WINDOW (editor->window), title);
      g_free (title);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_editor_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstEditor *editor = GST_EDITOR (object);

  switch (prop_id) {
    case PROP_FILENAME:
      g_value_set_string (value, g_strdup (editor->filename));
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_editor_dispose (GObject * object)
{
  GstEditor *editor = GST_EDITOR (object);

  gtk_widget_destroy (editor->window);

  _num_editor_windows--;

  g_message ("editor dispose: %d windows remaining", _num_editor_windows);

  if (_num_editor_windows == 0)
    gtk_main_quit ();
}

/*
 * helper functions
 */

/* show an error dialog with a gerror and debug string */
static void
gst_editor_dialog_gerror (GtkWindow * window, GError * error,
    const gchar * debug)
{
  GtkWidget *error_dialog;
  gint response;

  g_return_if_fail (error);
  error_dialog = gtk_message_dialog_new (window, GTK_DIALOG_MODAL,
      GTK_MESSAGE_ERROR, GTK_BUTTONS_NONE, error->message);
  /* add a debug button if there is debug info */
  if (debug) {
    gtk_dialog_add_button (GTK_DIALOG (error_dialog), GTK_STOCK_DIALOG_INFO,
        GST_EDITOR_RESPONSE_DEBUG);
  }
  /* make sure OK is on the right and default action to be HIG-compliant */
  gtk_dialog_add_button (GTK_DIALOG (error_dialog), GTK_STOCK_OK,
      GTK_RESPONSE_OK);
  gtk_dialog_set_default_response (GTK_DIALOG (error_dialog), GTK_RESPONSE_OK);
  response = gtk_dialog_run (GTK_DIALOG (error_dialog));

  if (response == GST_EDITOR_RESPONSE_DEBUG) {
    gtk_widget_destroy (error_dialog);
    error_dialog = gtk_message_dialog_new (GTK_WINDOW (window),
        GTK_DIALOG_MODAL,
        GTK_MESSAGE_INFO, GTK_BUTTONS_OK,
        "%s\n\nAdditional debug info: %s\n", error->message, debug);
    gtk_dialog_run (GTK_DIALOG (error_dialog));
  }
  gtk_widget_destroy (error_dialog);
}

#if 0
/* show an error dialog given just an error string (for internal use) */
static void
gst_editor_dialog_error (GtkWindow * window, const gchar * message)
{
  GError *error;

  error = g_error_new (GST_EDITOR_ERROR, 0, message);
  gst_editor_dialog_gerror (window, error, NULL);
  g_error_free (error);
}
#endif

/* GStreamer callbacks */
static void
gst_editor_pipeline_deep_notify (GstObject * object, GstObject * orig,
    GParamSpec * pspec, gchar ** excluded_props)
{
}

static void
gst_editor_pipeline_error (GObject * object, GstObject * source,
    GError * error, gchar * debug, GstEditor * editor)
{
  gchar *name = gst_object_get_path_string (source);
  GError *my_error;

  my_error = g_error_copy (error);
  g_free (my_error->message);
  my_error->message = g_strdup_printf ("%s: %s", name, error->message);

  gst_editor_dialog_gerror (GTK_WINDOW (editor->window), my_error, debug);
  g_error_free (my_error);
  g_free (name);
}

/* connect useful GStreamer signals to pipeline */
static void
gst_editor_element_connect (GstEditor * editor, GstElement * pipeline)
{
  g_signal_connect (pipeline, "deep_notify",
      G_CALLBACK (gst_editor_pipeline_deep_notify), editor);
  g_signal_connect (pipeline, "error",
      G_CALLBACK (gst_editor_pipeline_error), editor);
}

/* we need more control here so... */
static void
gst_editor_connect_func (const gchar * handler_name,
    GObject * object,
    const gchar * signal_name,
    const gchar * signal_data,
    GObject * connect_object, gboolean after, gpointer user_data)
{
  gpointer func_ptr;
  connect_struct *data = (connect_struct *) user_data;

  if (!g_module_symbol (data->symbols, handler_name, &func_ptr))
    g_warning ("GstEditor: could not find signal handler '%s'.", handler_name);
  else {
    GCallback func;

    func = *(GCallback) (func_ptr);
    if (after)
      g_signal_connect_after (object, signal_name, func,
          (gpointer) data->editor);
    else
      g_signal_connect (object, signal_name, func, (gpointer) data->editor);
  }
}

GtkWidget *
gst_editor_new (GstElement * element)
{
  GtkWidget *ret = g_object_new (gst_editor_get_type (), NULL);

  if (element) {
    g_object_set (GST_EDITOR (ret)->canvas, "bin", element, NULL);
    gst_editor_element_connect (GST_EDITOR (ret), element);
  }

  return ret;
}

typedef struct
{
  GtkWidget *selection;
  GstEditor *editor;
}
file_select;

static gboolean
do_save (GstEditor * editor)
{
  GstElement *element;
  gchar *message;
  FILE *file;

  if (!(file = fopen (editor->filename, "w"))) {
    g_warning ("%s could not be saved...", editor->filename);
    return FALSE;
  }

  g_object_get (editor->canvas, "bin", &element, NULL);

  if (gst_xml_write_file (element, file) < 0)
    g_warning ("error saving xml");
  fclose (file);

  message = g_strdup_printf ("Pipeline saved to %s.", editor->filename);
  gnome_appbar_set_status (GNOME_APPBAR (GNOME_APP (editor->window)->statusbar),
      message);
  g_free (message);

  return TRUE;
}

static void
on_save_as_file_selected (GtkWidget * button, file_select * data)
{
  GtkWidget *selector = data->selection;
  GstEditor *editor = data->editor;

  g_object_set (editor, "filename",
      gtk_file_selection_get_filename (GTK_FILE_SELECTION (selector)), NULL);

  do_save (editor);

  g_free (data);
}

void
gst_editor_on_save_as (GtkWidget * widget, GstEditor * editor)
{
  GtkWidget *file_selector;
  file_select *file_data = g_new0 (file_select, 1);

  file_selector = gtk_file_selection_new ("Please select a file for saving.");

  g_object_set (file_selector, "filename", editor->filename, NULL);

  file_data->selection = file_selector;
  file_data->editor = editor;

  g_signal_connect (G_OBJECT (GTK_FILE_SELECTION (file_selector)->
          ok_button), "clicked", GTK_SIGNAL_FUNC (on_save_as_file_selected),
      file_data);

  /* Ensure that the dialog box is destroyed when the user clicks a button. */
  g_signal_connect_swapped (G_OBJECT (GTK_FILE_SELECTION (file_selector)->
          ok_button), "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy),
      (gpointer) file_selector);
  g_signal_connect_swapped (G_OBJECT (GTK_FILE_SELECTION (file_selector)->
          cancel_button), "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy),
      (gpointer) file_selector);

  /* Display that dialog */
  gtk_widget_show (file_selector);
}

void
gst_editor_on_save (GtkWidget * widget, GstEditor * editor)
{
  if (editor->need_name) {
    gst_editor_on_save_as (NULL, editor);
    return;
  }

  do_save (editor);
}

static void
on_load_file_selected (GtkWidget * button, file_select * data)
{
  GtkWidget *selector = data->selection;
  const gchar *file_name;
  GstEditor *editor = data->editor;

  file_name = gtk_file_selection_get_filename (GTK_FILE_SELECTION (selector));

  gst_editor_load (editor, file_name);

  g_free (data);
}

void
gst_editor_load (GstEditor * editor, const gchar * file_name)
{
  gboolean err;
  GList *l;
  GstElement *pipeline;
  GstXML *xml;
  gchar *message;
  GData *datalist;

  xml = gst_xml_new ();

  /* create a datalist to retrieve extra information from xml */
  g_datalist_init (&datalist);

  g_signal_connect (G_OBJECT (xml), "object_loaded",
      G_CALLBACK (on_xml_loaded), &datalist);

  err = gst_xml_parse_file (xml, file_name, NULL);

  if (err != TRUE) {
    g_warning ("parse of xml file '%s' failed", file_name);
    return;
  }

  l = gst_xml_get_topelements (xml);
  if (!l) {
    g_warning ("no toplevel pipeline element in file '%s'", file_name);
    return;
  }

  if (l->next)
    g_warning ("only one toplevel element is supported at this time");

  pipeline = GST_ELEMENT (l->data);

  /* FIXME: through object properties ? */
  EDITOR_DEBUG ("loaded: attributes: %p", datalist);
  /* FIXME: attributes needs to come first ! */
  g_object_set (editor->canvas, "attributes", &datalist, "bin", pipeline, NULL);
  g_object_set (editor, "filename", file_name, NULL);

  message = g_strdup_printf ("Pipeline loaded from %s.", editor->filename);
  gnome_appbar_set_status (GNOME_APPBAR (GNOME_APP (editor->window)->statusbar),
      message);
  gst_editor_element_connect (editor, pipeline);
  g_free (message);
}

void
gst_editor_on_open (GtkWidget * widget, GstEditor * editor)
{
  GtkWidget *file_selector;
  file_select *file_data = g_new0 (file_select, 1);

  file_selector = gtk_file_selection_new ("Please select a file to load.");

  file_data->selection = file_selector;
  file_data->editor = editor;

  g_signal_connect (G_OBJECT (GTK_FILE_SELECTION (file_selector)->ok_button),
      "clicked", G_CALLBACK (on_load_file_selected), file_data);

  /* Ensure that the dialog box is destroyed when the user clicks a button. */
  g_signal_connect_swapped (G_OBJECT (GTK_FILE_SELECTION (file_selector)->
          ok_button), "clicked", G_CALLBACK (gtk_widget_destroy),
      (gpointer) file_selector);
  g_signal_connect_swapped (G_OBJECT (GTK_FILE_SELECTION (file_selector)->
          cancel_button), "clicked", G_CALLBACK (gtk_widget_destroy),
      (gpointer) file_selector);

  /* Display that dialog */
  gtk_widget_show (file_selector);
}

void
gst_editor_on_new_empty_pipeline (GtkWidget * widget, GstEditor * editor)
{
  gst_editor_new (gst_element_factory_make ("pipeline", NULL));
}

static void
have_pipeline_description (gchar * string, gpointer data)
{
  GstElement *pipeline;
  GError *error = NULL;
  GstEditor *editor = NULL;

  if (!string)
    return;

  pipeline = (GstElement *) gst_parse_launch (string, &error);

  if (!pipeline) {
    GtkWidget *dialog = gtk_message_dialog_new (GTK_WINDOW (data),
        GTK_DIALOG_DESTROY_WITH_PARENT,
        GTK_MESSAGE_WARNING,
        GTK_BUTTONS_CLOSE,
        "Pipeline failed to parse: %s", error->message);

    gtk_widget_show (dialog);
    g_signal_connect_swapped (GTK_OBJECT (dialog), "response",
        G_CALLBACK (gtk_widget_destroy), GTK_OBJECT (dialog));

    return;
  }

  editor = (GstEditor *) gst_editor_new (pipeline);

  gnome_appbar_set_status (GNOME_APPBAR (GNOME_APP (editor->window)->statusbar),
      "Pipeline loaded from launch description.");
}

void
gst_editor_on_new_from_pipeline_description (GtkWidget * widget,
    GstEditor * editor)
{
  static GtkWidget *request = NULL;

  if (!request) {
    request = gnome_request_dialog (FALSE,
        "Please enter in a pipeline description. "
        "See the gst-launch man page for help on the syntax.",
        "fakesrc ! fakesink",
        0, have_pipeline_description, editor, GTK_WINDOW (editor->window));
    gnome_dialog_close_hides (GNOME_DIALOG (request), TRUE);
  }

  gtk_widget_show (request);
}

void
gst_editor_on_close (GtkWidget * widget, GstEditor * editor)
{
  g_object_unref (G_OBJECT (editor));
}

void
gst_editor_on_quit (GtkWidget * widget, GstEditor * editor)
{
  gtk_main_quit ();
}

void
gst_editor_on_cut (GtkWidget * widget, GstEditor * editor)
{
  GstEditorElement *element = NULL;

  g_object_get (editor->canvas, "selection", &element, NULL);

  if (!element) {
    gnome_appbar_set_status (GNOME_APPBAR (GNOME_APP (editor->window)->
            statusbar),
        "Could not cut element: No element currently selected.");
    return;
  }

  gst_editor_element_cut (element);
}

void
gst_editor_on_copy (GtkWidget * widget, GstEditor * editor)
{
  GstEditorElement *element = NULL;

  g_object_get (editor->canvas, "selection", &element, NULL);

  if (!element) {
    gnome_appbar_set_status (GNOME_APPBAR (GNOME_APP (editor->window)->
            statusbar),
        "Could not copy element: No element currently selected.");
    return;
  }

  gst_editor_element_copy (element);
}

void
gst_editor_on_paste (GtkWidget * widget, GstEditor * editor)
{
  gst_editor_bin_paste (editor->canvas->bin);
}

void
gst_editor_on_add (GtkWidget * widget, GstEditor * editor)
{
  GstElementFactory *factory;
  GstElement *element;

  if ((factory = gst_element_browser_pick_modal ())) {
    if (!(element = gst_element_factory_create (factory, NULL))) {
      g_warning ("unable to create element of type '%s'",
          GST_OBJECT_NAME (factory));
      return;
    }
    /* the object_added signal takes care of drawing the gui */
    gst_bin_add (GST_BIN (GST_EDITOR_ITEM (editor->canvas->bin)->object),
        element);
  }
}

void
gst_editor_on_remove (GtkWidget * widget, GstEditor * editor)
{
  GstEditorElement *element = NULL;

  g_object_get (editor->canvas, "selection", &element, NULL);

  if (!element) {
    gnome_appbar_set_status (GNOME_APPBAR (GNOME_APP (editor->window)->
            statusbar),
        "Could not remove element: No element currently selected.");
    return;
  }

  gst_editor_element_remove (element);
}

void
gst_editor_show_element_inspector (GtkWidget * widget, GstEditor * editor)
{
  gboolean b;

  g_object_get (widget, "active", &b, NULL);
  g_object_set (editor->canvas, "properties-visible", b, NULL);
}

void
gst_editor_show_utility_palette (GtkWidget * widget, GstEditor * editor)
{
  gboolean b;

  g_object_get (widget, "active", &b, NULL);

  g_message ("show utility palette: %d", b);

  g_object_set (editor->canvas, "palette-visible", b, NULL);
}

void
gst_editor_on_help_contents (GtkWidget * widget, GstEditor * editor)
{
  gnome_help_display ("gst-editor-manual", NULL, NULL);
}

void
gst_editor_on_about (GtkWidget * widget, GstEditor * editor)
{
  GtkWidget *about;
  GdkPixbuf *pixbuf;
  const gchar *authors[] = { "Andy Wingo", "Erik Walthinsen",
    "Jan Schmidt", NULL
  };

  about = gnome_about_new ("GStreamer Pipeline Editor",
      VERSION,
      "(c) 2001-2004 GStreamer Team",
      "A graphical pipeline editor for GStreamer capable of loading "
      "and saving XML.\n\nhttp://gstreamer.net/", authors, NULL, NULL, NULL);

  pixbuf = gtk_widget_render_icon (about, GST_EDITOR_STOCK_LOGO,
      GTK_ICON_SIZE_DIALOG, NULL);
  if (!pixbuf)
    g_warning ("no pixbuf found");

  g_object_set (about, "logo", pixbuf, NULL);

  gtk_widget_show (about);
}

static gboolean
sort (GstEditor * editor)
{
  gchar *message;

  message = g_strdup_printf ("Change in bin pressure: %f",
      gst_editor_bin_sort (editor->canvas->bin, 0.1));
  gnome_appbar_set_status (GNOME_APPBAR (GNOME_APP (editor->window)->statusbar),
      message);
  g_free (message);
  return TRUE;
}

void
gst_editor_on_sort_toggled (GtkToggleButton * toggle, GstEditor * editor)
{
  gboolean active;

  active = gtk_toggle_button_get_active (toggle);

  if (active) {
    gnome_appbar_set_status (GNOME_APPBAR (GNOME_APP (editor->window)->
            statusbar), "Sorting bin...");

    g_timeout_add (200, (GSourceFunc) sort, editor);
  } else {
    gnome_appbar_set_status (GNOME_APPBAR (GNOME_APP (editor->window)->
            statusbar), "Finished sorting.");
    g_source_remove_by_user_data (editor);
  }
}

static gint
on_delete_event (GtkWidget * widget, GdkEvent * event, GstEditor * editor)
{
  g_object_unref (G_OBJECT (editor));

  return FALSE;
}

static void
on_canvas_notify (GObject * object, GParamSpec * param, GstEditor * editor)
{
  GValue v = { 0, };
  gchar *status;

  if (strcmp (param->name, "properties-visible") == 0) {
    g_value_init (&v, param->value_type);
    g_object_get_property (object, param->name, &v);
    g_object_set_property (G_OBJECT (glade_xml_get_widget (editor->xml,
                "view-element-inspector")), "active", &v);
  } else if (strcmp (param->name, "palette-visible") == 0) {
    g_message ("canvas property notify");
    g_value_init (&v, param->value_type);
    g_object_get_property (object, param->name, &v);
    g_object_set_property (G_OBJECT (glade_xml_get_widget (editor->xml,
                "view-utility-palette")), "active", &v);
  } else if (strcmp (param->name, "status") == 0) {
    g_object_get (object, "status", &status, NULL);
    gnome_appbar_set_status (GNOME_APPBAR (GNOME_APP (editor->window)->
            statusbar), status);
    g_free (status);
  }
}

static void
on_xml_loaded (GstXML * xml, GstObject * object, xmlNodePtr self,
    GData ** datalistp)
{
  xmlNodePtr children = self->xmlChildrenNode;
  GstEditorItemAttr *attr = NULL;       /* GUI attributes in editor canvas */

  attr = g_malloc (sizeof (GstEditorItemAttr));

  GST_DEBUG_OBJECT (object, "xml for object loaded, getting attrs");
  while (children) {
    if (!strcmp (children->name, "item")) {
      xmlNodePtr nodes = children->xmlChildrenNode;

      while (nodes) {
        if (!strcmp (nodes->name, "x")) {
          attr->x = g_ascii_strtod (xmlNodeGetContent (nodes), NULL);
        } else if (!strcmp (nodes->name, "y")) {
          attr->y = g_ascii_strtod (xmlNodeGetContent (nodes), NULL);
        } else if (!strcmp (nodes->name, "w")) {
          attr->w = g_ascii_strtod (xmlNodeGetContent (nodes), NULL);
        } else if (!strcmp (nodes->name, "h")) {
          attr->h = g_ascii_strtod (xmlNodeGetContent (nodes), NULL);
        }
        nodes = nodes->next;
      }
      GST_DEBUG_OBJECT (object, "loaded with x: %f, y: %f, w: %f, h: %f",
          attr->x, attr->y, attr->w, attr->h);
    }
    children = children->next;
  }

  /* save this in the datalist with the object's name as key */
  GST_DEBUG_OBJECT (object, "adding to datalistp %p", datalistp);
  g_datalist_set_data (datalistp, GST_OBJECT_NAME (object), attr);
}
