/* gt_data.c - Create additional data links dynamically
 *
 * Copyright (C) 1997, 1999 Free Software Foundation
 * Copyright (C) 1995, 1996 Eric M. Ludlam
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 * 
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 * 
 * Please send bug reports, etc. to zappo@gnu.org.
 * 
 * Description:
 *   These functions manage the creation and IO of data links.  A data
 * link is any additional TCP connection above and beyond those needed
 * for casual conversation.  Additional links to other people may also
 * be generated, and the accept routine can decide which type new
 * connections are.
 * 
 * $Log: gt_data.c,v $
 * Revision 1.14  1999/11/28 20:18:21  zappo
 * Fix some race conditions for incomplete connectsion.
 * Convert some 3s into the symbol for edit characters.
 *
 * Revision 1.13  1997/12/14 19:15:36  zappo
 * Renamed package to gtalk, renamed symbols and files apropriately
 * Fixed copyright and email address.
 *
 * Revision 1.12  1997/10/25 01:48:36  zappo
 * Added check to make sure that the remote is of a version that supports
 * receiving of files.
 *
 * Revision 1.11  1997/10/15 02:09:49  zappo
 * For sent data, only require a DISPLAY if not a subprocess.
 * (Assume emacs can display it.)
 *
 * Revision 1.10  1997/02/23 03:26:27  zappo
 * API change on GT_clean_dev changes.
 *
 * Revision 1.9  1997/02/01 14:27:14  zappo
 * Changed USER_send and USER_read to also take Ctxt
 *
 * Revision 1.8  1997/01/28  03:19:25  zappo
 * Removed unused local variable
 *
 * Revision 1.7  1996/07/27  21:27:40  zappo
 * Added in auxilliary port for emacs interface
 *
 * Revision 1.6  1996/05/18  15:33:34  zappo
 * Made AUX connections for emacs work
 *
 * Revision 1.5  1996/02/26  00:28:16  zappo
 * Fixed warnings, added links for datalink->user
 *
 * Revision 1.4  1996/02/01  01:14:24  zappo
 * Added link verification hooks (who a new link is from) and all the
 * shared app stuff
 *
 * Revision 1.3  1995/12/09  23:55:41  zappo
 * Added use of querynewtcp to query connection options
 *
 * Revision 1.2  1995/11/21  03:39:20  zappo
 * Added the additional support needed for file transfers, and data link
 * startup.
 *
 * Revision 1.1  1995/09/29  08:29:54  zappo
 * Initial revision
 *
 * History:
 * zappo   9/25/95    Created
 *
 * Tokens: ::Header:: gtalkc.h
 */
#include "gtalklib.h"
#include "gtalkc.h"
#include "talk.h"
#include "gtproc.h"


/*
 * Function: fail_connection
 *
 *   Locally defined function which sends the fail command back to
 * requesting client, with reason, then does housekeeping to clean up
 * the InputDevice.
 *
 * Returns:     static void  - 
 * Parameters:  uo     - Pointer to user object
 *              reason - Pointer toCharacter of reason
 * History:
 * zappo   11/6/95    Created
 */
static void fail_connection(Ctxt, uo, reason)
     struct TalkContext *Ctxt;
     struct UserObject  *uo;
     char               *reason;
{
  char *buffer;
  static char *msg = "Datalink failed: ";

  USER_send(Ctxt, uo, "\02", 1);
  USER_send(Ctxt, uo, reason, strlen(reason));
  USER_send(Ctxt, uo, "\n", 1);
  
  /* If attached to someone, just clean up the message */
  GT_clean_dev(uo->remote, Ctxt->cleanup);

  uo->state = USER_CLOSED;

  buffer = (char *)malloc(strlen(reason) + strlen(msg) + 1);
  sprintf(buffer, "%s%s", msg, reason);
  DISP_message(Ctxt, buffer);
}


/*
 * Function: DATA_link_accept
 *
 *   Accepts a new TCP connection of and set's it up as either an
 * unsolicited data connection, or a new unsolicited talk connection.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt - Context
 *              io   - Pointer to io
 * History:
 * zappo   9/25/95    Created
 */
void DATA_link_accept(Ctxt, io)
     struct TalkContext *Ctxt;
     struct InputDevice *io;
{
  struct InputDevice 	*newtcp;
  struct UserObject  	*newuo;
  char                	 ident[NUMEDITKEYS];
  int 			 numedit;
  int 			 n;

  newtcp = TCP_accept(io);

  if(!newtcp)
    {
      DISP_message(Ctxt, "Accepting socket has IO, but no new connection generated.");
      return;
    }
  
  /* Unsolicited connects will first send us data which we need to
     determine what to do with the new link */
  numedit = 0;
  do {
    n = GT_recv(newtcp, ident, NUMEDITKEYS - numedit);
    if (n == Fail)
      {
	DISP_message(Ctxt, "Failed to read data from new connection.");
	return;
      }
    numedit += n;
  } while (numedit != NUMEDITKEYS);
  
  /* Set up new tcp link to read data */
  newtcp->state = CONNECTED;
  newtcp->timeout = 0;
  newtcp->timefn = 0;

  /* First things first, create a new user for which we can soon classify
     as being a new user (which needs to have windows set up) _or_ a
     new data connection which has a different setup */
  newuo = USER_alloc();
  newuo->remote = newtcp;
  memcpy(newuo->editkeys, ident, NUMEDITKEYS);
  
  /* Lets classify the new connection */
  if(!ident[0] && !ident[1] && !ident[2])
    {
      char linktype;
      int doit = 0;
      int vcnt;
      char verifybuff[NAME_SIZE+1];
      struct UserObject *testuo;

      /* All NULLS means we have a DATA connection */
      newuo->type = DATALINK;
      newtcp->readme = STREAM_remote_read;

      /* Get the verifier string */
      for(vcnt = 0; vcnt < NAME_SIZE; vcnt++)
	{
	  /* Learn the type of new connection */
	  if(GT_recv(newtcp, verifybuff+vcnt, 1) == Fail)
	    {
	      fail_connection(Ctxt, newuo, "Error reading from your port");
	      return;
	    }
	}
      verifybuff[sizeof(verifybuff)] = 0;

      /* First part is the name, make sure we are talking to one of those */
      testuo = USER_namefind(verifybuff, newtcp->host->name);
      if(!testuo || testuo->state != USER_CONNECTED)
	{
	  fail_connection(Ctxt, newuo, "Authentication failed.");
	  return;
	}
      /* Store that association */
      newuo->datauser = testuo;
      
      /* Learn the type of new connection */
      if(GT_recv(newtcp, &linktype, 1) == Fail)
	{
	  fail_connection(Ctxt, newuo, "Error reading from your port");
	  return;
	}
      switch(linktype)
	{
	  /* DATA just means a transferred file of some sort */
	case NEWLINK_DATA:
	  {
	    int count;
	    char datachar;
	    char *env;
	    char dataname[200];
	    char buffer[200];

	    if(!(env = getenv("DISPLAY")) && (Ctxt->runstate != Socket))
	      {
		fail_connection(Ctxt, newuo, "Access to X not available.");
		return;
	      }

	    count = 0;
	    datachar = 0;
	    while((count < sizeof(dataname)) && datachar != '\n')
	      {
		if(GT_recv(newtcp, &datachar, 1) == Fail)
		  {
		    fail_connection(Ctxt, newuo, "Fail to read data name.");
		    return;
		  }
		dataname[count++] = datachar;
	      }
	    dataname[--count] = 0; /* terminate it */

	    sprintf(buffer, "%s/%s", Ctxt->downloadpath, dataname);

	    /* Now, after all that, make sure the user wants to do
	       this */
	    if(Ctxt->querynewtcp == AlwaysConnect) doit = 1;
	    if(Ctxt->querynewtcp == QueryConnect)
	      {
		char qbuff[100];
		sprintf(qbuff, "%s is sending you %s.  Accept and view?",
			verifybuff, buffer);
		if(DISP_yes_or_no(Ctxt, qbuff))
		  doit = 1;		
	      }

	    if(!doit)
	      {
		fail_connection(Ctxt, newuo, "File refused by user.");
		return;
	      }
	    newuo->name = strdup(dataname);
	    newuo->local = GT_open_output_file(buffer);

	    if(!newuo->local)
	      {
		fail_connection(Ctxt, newuo, "Unable to create requested file!");
		return;
	      }

	    /* Send the success signal */
	    USER_send(Ctxt, newuo, "\01", 1);
	  }
	  break;
	  /* Auxiliarry link initiated by superior process in Socket mode */
	case NEWLINK_AUX:
	  if(Ctxt->runstate != Socket)
	    {
	      fail_connection(Ctxt, newuo,
			      "AUXILIARY cannot be created: Not running as inferior process");
	      return;
	    }
	  /*
	   * All we are going to get is the master-process function we want
	   * to run, which we promptly report up to our parent after
	   * we get them to  connect down to us
	   */
	  {
	    char buffer[100];
	    char datachar = 0;
	    int  bcnt = 0;
	    int  doit = 0;
	    /* Scan in the function we want */
	    while ((bcnt < (sizeof(buffer)-1)) && (datachar != '\n'))
	      {
		if(GT_recv(newtcp, &datachar, 1) == Fail)
		  {
		    fail_connection(Ctxt, newuo, "Fail to read data name.");
		    return;
		  }
		buffer[bcnt++] = datachar;
	      }
	    buffer[--bcnt] = '\0'; /* null out CR */

	    printf("Read in request for aux [%s]\n", buffer);

	    /* Ask for permission before doing anything */
	    if(Ctxt->querynewtcp == AlwaysConnect) doit = 1;
	    if(Ctxt->querynewtcp == QueryConnect)
	      {
		char qbuff[100];
		sprintf(qbuff, "User %s wants to use %s. Do it?",
			testuo->name, buffer);
		if(DISP_yes_or_no(Ctxt, qbuff))
		  doit = 1;
	      }
	      
	    if(doit)
	      {
		char buffer2[100];

		sprintf(buffer2, "%s with %s", buffer, testuo->name);
		newtcp->name = strdup(buffer2);
		if(!newtcp->name)
		  gtalk_shutdown("strdup failed in DATA_link_accept()=aux");

		/* Report this to parent process, then wait for response */
		printf("%c%c%d %s\n", ETALK_ESCAPE, TTY_NEW_DATA,
		       testuo->id, buffer);
		
		/* Place this new socket directly into our new user struct,
		   and set it up as a local port.  Since we are only
		   acting as a proxy we behave exactly as if we were
		   containing a talk process */
		newuo->local          = TCP_accept(Ctxt->emacs_connect);
		newuo->local->state   = CONNECTED;
		newuo->local->readme  = STREAM_local_read;
		newuo->local->timeout = 0;
		newuo->local->timefn  = NULL;

		if(newuo->local == NULL)
		  {
		    /* Ooops */
		    fail_connection(Ctxt, newuo, "Connectivity error!");
		  }
		else
		  {
		    /* Once this is successful, report back YES */
		    USER_send(Ctxt, newuo, "yes\n", 4);
		  }
	      }
	    else
	      {
		/* Once this is successful, report back YES */
		fail_connection(Ctxt, newuo, "Request refused");
	      }
	  }
	  /* continue on our way... */
	  return;
	  /* Here we want to fork off a new process. */
	case NEWLINK_APP:
	  {
	    char proxy, dispreq, datachar;
	    int count;
	    char appbuff[100];
	    
	    /* Load and check the display requirements */
	    if(GT_recv(newtcp, &dispreq, 1) == Fail)
	      {
		fail_connection(Ctxt, newuo, "Fail to read display requirements.");
		return;
	      }
	    if(dispreq > DISPREQ_MAX)
	      {
		fail_connection(Ctxt, newuo, "Unknown display requirement value recieved!");
		return;
	      }
	    /* Load and check the proxy */
	    if(GT_recv(newtcp, &proxy, 1) == Fail)
	      {
		fail_connection(Ctxt, newuo, "Fail to read proxy specifier.");
		return;
	      }
	    if(proxy > PROXY_MAX)
	      {
		fail_connection(Ctxt, newuo, "Unknown proxy value recieved!");
		return;
	      }
	    /* Load in the application symbol */
	    count = 0;
	    datachar = 0;
	    while((count < sizeof(appbuff)) && datachar != '\n')
	      {
		if(GT_recv(newtcp, &datachar, 1) == Fail)
		  {
		    fail_connection(Ctxt, newuo, "Fail to read application symbol name.");
		    return;
		  }
		appbuff[count++] = datachar;
	      }
	    appbuff[--count] = 0; /* terminate it */	    

	    /* Lookup the app to make sure we hve a match */
	    if(!SHAPP_verify(appbuff, proxy, dispreq))
	      {
		fail_connection(Ctxt, newuo, "Unknown app, or bad requirements match");
		return;
	      }
	    
	    /* Ask the user if they want to do that */
	    if(Ctxt->querynewtcp == AlwaysConnect) doit = 1;
	    if(Ctxt->querynewtcp == QueryConnect)
	      {
		char qbuff[100];
		sprintf(qbuff, "Start %s with %s?", 
			appbuff, verifybuff);
		if(DISP_yes_or_no(Ctxt, qbuff))
		  doit = 1;		
	      }

	    if(!doit)
	      {
		fail_connection(Ctxt, newuo, "File refused by user.");
		return;
	    
	      }

	    if(SHAPP_fork_shared_unsolicited(Ctxt, appbuff, newuo, newtcp)
	       == Fail)
	      {
		fail_connection(Ctxt, newuo, "System failure during launch");
		return;
	      }
	    
	    /* The RETURN SUCCESS part will be sent by the above
	     * procedure.
	     */
	  }
	  return;
	default:
	  {
	    char buffer[100];
	    sprintf(buffer, "Connections of type %d not supported",  linktype);
	    fail_connection(Ctxt, newuo, buffer);
	    break;
	  }
	}
    }
  else
    {
      char *unamebuff, *t;
      int   buffsize, i, ret;
      char  lastchar;
      int   doit = 0;

      /* We must immediatly send our editkeys */
      GT_send(newtcp, Ctxt->editkeys, NUMEDITKEYS);

      /* Next, for both modes, read their ident string, this way we
	 know their user name, as well as thier host name.  This is
	 required or the window names are all messed up. */
      buffsize = 80;
      unamebuff = (char *)malloc(buffsize);
      lastchar = 0;
      i = 0;
      while(lastchar != '\n')
	{
	  ret = GT_recv(newtcp, &lastchar, 1);
	  if(ret)
	    {
	      unamebuff[i] = lastchar;
	    }
	  else
	    {
	      fail_connection(Ctxt, newuo, "Error reading ident string");
	      return;
	    }
	  i += ret;
	  /* boost it's size if it keeps going. */
	  if(i == buffsize)
	    {
	      buffsize += 80;
	      t = (char *)malloc(buffsize);
	      strncpy(t, unamebuff, i);
	      free(unamebuff);
	      unamebuff = t;
	    }
	}
      unamebuff[i+1] = 0;	/* make sure it's terminated */

      /* Associate this scan buffer with this TCP socket */
      newuo->scanbuff = unamebuff;

      /* Quickly scan the buffer for the username... */
      /* <03><010>name 1111 TALKVER v.v<DELLINE><\n> */
      t = strchr(unamebuff, ' ');
      newuo->name = (char *)malloc(t - newuo->scanbuff);
      strncpy(newuo->name, newuo->scanbuff + 2, t - 2 - newuo->scanbuff);
      newuo->name[t - 2 - newuo->scanbuff] = 0;

      /* Now we know something about this character.  Make sure the
	 user wants to talk to them. */
      if(Ctxt->querynewtcp == AlwaysConnect) doit = 1;
      if(Ctxt->querynewtcp == QueryConnect)
	{
	  char qbuff[100];
	  sprintf(qbuff, "User %s has attached.  Talk them?", newuo->name);
	  if(DISP_yes_or_no(Ctxt, qbuff))
	    doit = 1;
	}
      
      if(doit)
	{
	  /* We can keep going, so alert whatever that we can accept IO
	     from this user */
	  newuo->state = USER_CONNECTED;
	  
	  /* This means we have edit characters.  Set up for a new user */
	  if(Ctxt->runstate == Socket)
	    {
	      /* Tell them our found name */
	      printf("%c%c%d %s@%s\n", ETALK_ESCAPE, TTY_NEW_USER,
		     newuo->id, newuo->name, newtcp->host->name);
	      /* Place this new socket directly into our new user struct,
		 and set it up as a local port */
	      newuo->local          = TCP_accept(Ctxt->emacs_connect);
	      newuo->local->state   = WAITING;
	      newuo->local->readme  = STREAM_local_read;
	      newuo->local->timeout = 0;
	      newuo->local->timefn  = NULL;
	      
	      /* We are connected, so tell them our code and editchars too */
	      GT_send(newuo->local, newuo->editkeys, NUMEDITKEYS);
	      /* Now send them the code we just read... */
	      GT_send(newuo->local, newuo->scanbuff, i);
	      /* And finally, let's interpret that code ourselves! */
	      SCAN_evaluate(Ctxt, newuo);
	    }
#ifndef NO_IFACE
	  else
	    {
	      /* Now send our state data */
	      SCAN_send_version(Ctxt, newuo);
	      /* and update our user a little */
	      DISP_new_user(Ctxt, newuo);
	      /* And scan that string for goodies */
	      SCAN_evaluate(Ctxt, newuo);
	    }
#endif
	  /* Once all that work is done, setup to be read */
	  newtcp->readme = STREAM_remote_read;
	}
      else
	{
	  /* We don't want to talk to them.. Well, lets just clean this
	     mess up.  There are no windows yet, so there isn't much to
	     do */
	  GT_clean_dev(newuo->remote, Ctxt->cleanup);
	  newuo->state = USER_CLOSED;
	}
    }
}


/*
 * Function: DATA_open_new_connection
 *
 *   Connects to to user specified in UO with a new datalink.  All
 * data links start the same with with some minor validation, and if i
 * change it in the future, here is the one spot to do it in.
 *
 * Returns:     struct InputDevice * - 
 * Parameters:  Ctxt     - Context
 *              uo       - Pointer to uo
 *              typecode - Type of
 * History:
 * zappo   1/25/96    Created
 */
struct InputDevice *DATA_open_new_connection(Ctxt, uo, typecode)
     struct TalkContext *Ctxt;
     struct UserObject  *uo;
     char typecode;
{
  struct InputDevice *newtcp;
  struct sockaddr_in addr;
  char namebuff[NAME_SIZE];

  if(!uo || !uo->remote || !uo->remote->host) return NULL;

  /* generate the address struct */
  addr = uo->remote->host->addr;
  addr.sin_port = htons(uo->connect);

  newtcp = TCP_connect((struct sockaddr*)&addr);

  if(!newtcp)
    {
      return NULL;
    }

  memset(namebuff, 0, sizeof(namebuff));
  strcpy(namebuff, Ctxt->myname);

  GT_send(newtcp, "\0\0\0", 3); /* editchars (none) and DATA code */
  GT_send(newtcp, namebuff, NAME_SIZE); /* send the name */
  GT_send(newtcp, &typecode, 1); /* send the type of new connection */

  return newtcp;
}


/*
 * Function: DATA_aux_connection
 *
 *   Connects to UO and binds new linke into AUXLINK.  This becomes
 * the AUX connection request by our parent process.
 *
 * Returns:     int  - 
 * Parameters:  Ctxt     - Context
 *              uo       - Pointer to user object we connect to
 *              auxlink  - Pointer to auxiliary user Object
 *              function - the function we want the remote to run
 * History:
 * zappo   4/6/96     Created
 */
int DATA_aux_connection(Ctxt, uo, auxlink, function)
     struct TalkContext *Ctxt;
     struct UserObject  *uo;
     struct UserObject  *auxlink;
     char *function;
{
  struct InputDevice *newtcp;
  char buffer[100];

  /* Get a link to the remote process */
  newtcp = DATA_open_new_connection(Ctxt, uo, NEWLINK_AUX);

  if(!newtcp) 
    {
      DISP_message(Ctxt, "\03Could not create auxiliary link.");
      return Fail;
    }

  newtcp->state = CONNECTED;
  newtcp->readme = STREAM_remote_read;
  newtcp->timeout = 0;
  newtcp->timefn = 0;

  /* Update the newly created user object with all the new data we have */
  sprintf(buffer, "%s with %s", function, uo->name);
  auxlink->name = strdup(buffer);
  if(!auxlink->name)
    gtalk_shutdown("strdup failed in DATA_aux_connection()");
  auxlink->datauser = auxlink;
  auxlink->remote = newtcp;
  auxlink->editkeys[0] = 0;
  auxlink->editkeys[1] = 0;
  auxlink->editkeys[2] = 0;

  /* Send the name of the function we want to use */
  printf("Sending string [%s] through new aux\n", function);
  GT_send(newtcp, function, strlen(function));
  GT_send(newtcp, "\n", 1);
 
  return Success;
}

/*
 * Function: DATA_send_file
 *
 *   Sends a file to USER over an unsolicited data channel.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt - Context
 *              user - Pointer to user
 *              name - Name of the file to send
 * History:
 * zappo   11/7/95    Created
 */
int DATA_send_file(Ctxt, user, name)
     struct TalkContext *Ctxt;
     struct UserObject  *user;
     char *name;
{
  FILE *f;
  char block[1000];
  char dbuff[100];
  int ret, count;
  struct InputDevice *newtcp;
  char answercode;

  /* First, make sure we can contact them */
  if(!((user->type == GNUTALK) || 
       ((user->type == ETALK) && (((user->ver == 0) && (user->num >= 11)) ||
				  (user->ver > 0)))))
    {
      DISP_message(Ctxt, "Selected user client can not receieve a file.");
      return Fail;
    }

  /* first, see if the file is openable */
  f = fopen(name, "r");

  if(!f)
    {
      DISP_message(Ctxt, "\03Could not open file.");
      return Fail;
    }

  /* Next, create the data connection */
  newtcp = DATA_open_new_connection(Ctxt, user, NEWLINK_DATA);

  if(!newtcp) 
    {
      fclose(f);
      DISP_message(Ctxt, "\03Could not create datalink.");
      return Fail;
    }

  GT_send(newtcp, name, strlen(name)); /* send the name */
  GT_send(newtcp, "\n", 1);	/* terminate name */

  /* wait for response */
  if(GT_recv(newtcp, &answercode, 1) == Fail)
    {
      fclose(f);
      DISP_message(Ctxt, "\03Failed to read answer code.");
      return Fail;
    }

  /* check the code */
  if(answercode == 2)
    {
      int msgsize;
      fclose(f);
      msgsize = GT_recv(newtcp, block+1, sizeof(block));
      block[msgsize] = 0;
      block[0] = ETALK_ESCAPE;
      DISP_message(Ctxt, block); /* report the error */
      GT_clean_dev(newtcp, Ctxt->cleanup);
      return Fail;
    }
  
  /* All sustems go: transmit the data.
     TODO: make this asynchronous        */
  count = 0;
  do {
    sprintf(dbuff, "%cTransmitting %s (%d bytes) ...",
	    ETALK_ESCAPE, name, count);
    DISP_message(Ctxt, dbuff);
    ret = fread(block, 1, sizeof(block), f);
    if(ret) 
      if(GT_send(newtcp, block, ret) == Fail)
	{
	  DISP_message(Ctxt,"\03Unexpected disconnect during transmit.");
	  /* Cleanup our mess */
	  fclose(f);
	  GT_clean_dev(newtcp, Ctxt->cleanup);
	  return Fail;
	}
    count += ret;
  } while(ret);

  sprintf(dbuff, "%cTransmitting %s (%d bytes) ... done!", 
	  ETALK_ESCAPE, name, count);
  DISP_message(Ctxt, dbuff);

  /* Cleanup our mess */
  fclose(f);
  GT_clean_dev(newtcp, Ctxt->cleanup);

  return Success;
}
