/*
 * Copyright (C) 1994,95,96,97,98,99 Free Software Foundation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 *
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 *
 * Please send bug reports, etc. to zappo@gnu.org.
 *
 * Purpose:
 *  This file implements the talk protocol.  It is dependant on the
 * recursive nature of the select_all fn which is similar to RPC
 * calls.  This will allow us to detect timeouts and all that kind of
 * good stuff, and view it logically in a linear manor.
 * 
 * $Log: gt_proto.c,v $
 * Revision 1.41  1999/11/29 17:03:26  zappo
 * Fixed a bunch of references to the old etalk program.  (Now gtalk)
 *
 * Revision 1.40  1999/11/28 20:24:16  zappo
 * Fix some race conditions for incomplete connectsion.
 * Convert some 3s into the symbol for edit characters.
 *
 * Revision 1.39  1998/05/01 16:39:24  zappo
 * Updated the test so that a failed announce delents the announce.
 *
 * Revision 1.38  1998/01/09 22:31:57  zappo
 * You can now test a machine other than local_host.
 *
 * Revision 1.37  1998/01/04 13:31:59  zappo
 * Fixed warnings
 *
 * Revision 1.36  1997/12/14 19:16:33  zappo
 * Renamed package to gtalk, renamed symbols and files apropriately
 * Fixed copyright and email address.
 *
 * Revision 1.35  1997/11/01 13:25:10  zappo
 * Fixed test so that if we have a V1 daemon, it correctly regresses from V2
 * during brief daemon check.
 *
 * Revision 1.34  1997/10/18 02:59:50  zappo
 * Added PROTOCOL_lookup_verify to do a regressive lookup command.
 * Removed that part from PROTOCOL_attach, and and also called it on
 * the local socket just before making a local announcement.
 *
 * Revision 1.33  1997/03/23 15:43:13  zappo
 * _abort now calls USER_hangup for management stuff, and sets state to
 * UNREACHABLE to prevent recursion.
 *
 * Revision 1.32  1997/03/12 00:26:31  zappo
 * PROTOCOL_abort now returns TRUE of FALSE.  TRUE if there was a call to abort.
 *
 * Revision 1.31  1997/02/25 02:47:43  zappo
 * Fixed DISP_message calls w/ /ns at the end.
 *
 * Revision 1.30  1997/02/23 03:26:19  zappo
 * API change on GT_clean_dev changes.
 *
 * Revision 1.29  1997/02/21 04:03:28  zappo
 * Added new test for control-characters
 *
 * Revision 1.28  1997/01/16 00:07:32  zappo
 * Fixed horrible bug in GET_MSG and response_wait_stuff
 * Made test procedure regress talk version for more robust testing
 *
 * Revision 1.27  1996/03/02  03:17:51  zappo
 * Fixed erroneous return value in response_wait_stuff and some warnings
 *
 * Revision 1.26  1996/01/06  02:15:38  zappo
 * Fixed references to ringer (requires username now), and simplified the
 * ugly GET_MSG macros
 *
 * Revision 1.25  1995/11/26  23:56:41  zappo
 * Fixed reply in socket mode to use same codes as unsolicited connect
 *
 * Revision 1.24  1995/11/22  03:39:23  zappo
 * When connecting, some errors were not resulting in correct deletion of
 * emacs links, and disposal of user structs.
 *
 * Revision 1.23  1995/09/29  09:10:30  zappo
 * Removed traces of ringer response message
 *
 * Revision 1.22  1995/09/29  08:37:04  zappo
 * Upgraded reading of edit chars to be more reliable on delayed TCP
 * links.  Added SCAN_send_version after every new connection.  Modified
 * to user DATA_link_accept whenever remote_connect was modified to wait
 * for a connection.  Re-ordered some opporations to protect agains
 * DATA_link_accept when it would prove unpleasant.
 *
 * Revision 1.21  1995/09/22  13:49:47  zappo
 * Added codes needed to optionally have the X toolkit and curses
 * interfaces removed from the binary.  Fixed abort command which was
 * NOT deinitializing the talk daemon
 *
 * Revision 1.20  1995/09/20  23:27:31  zappo
 * Made updates needed for use with new structure names
 *
 * Revision 1.19  1995/09/14  11:21:34  zappo
 * Added parts needed for _attach and _reply to handle curses interface
 *
 * Revision 1.18  1995/07/23  18:00:08  zappo
 * When in test mode, variable "myname" is now filled in by querying with
 * the "getpw" functions.  Used to get the environment $USER which
 * doesn't exist on all systems.
 *
 * Revision 1.17  1995/07/16  16:34:23  zappo
 * Code involving degrading talk protocols is now much more robust.  In
 * fact, it works now.  Uses USER_hangup command to clean up from failed
 * attempts instead of poking information in by hand.  GET_MSG_RETRY now
 * works the way it was indended.  PROTOCOL_test now tests sun-forwarding
 * method correctly after doing a GNU daemon check.
 *
 * Revision 1.16  1995/07/15  19:48:27  zappo
 * Modified section dealing with "LOOK_HERE".  no longer terminate user
 * struct, but recycle when the userid is re-used by parent process.
 * Also added more error-checking when looking up name from address
 *
 * Revision 1.15  1995/05/25  23:12:52  zappo
 * the static/local "io" variable was not reliably cleared: fixed.
 *
 * Revision 1.14  1995/05/09  01:52:32  zappo
 * reply function did not always return FAIL to the controlling process
 * when errors occured, thus locking emacs up until a C-g was pressed.
 *
 * Revision 1.13  1995/04/08  20:05:19  zappo
 * Replaced some malloc/strcpy with strdups
 *
 * Revision 1.12  1995/04/01  17:04:06  zappo
 * Fixed a rouge slash, and added an appname before the announcement to
 * test the ability of gtalk to read this in correctly.
 *
 * Revision 1.11  1995/03/25  03:35:21  zappo
 * Update copyright.
 *
 * Revision 1.10  1995/03/23  02:58:45  zappo
 * Greatly improved the testing facility.
 *
 * Revision 1.9  1995/01/30  02:27:06  zappo
 * Fixed non-apparent host problem to prevent a core dump
 *
 * Revision 1.8  1995/01/29  14:24:24  zappo
 * Fixed some -Wall warnings
 *
 * Revision 1.7  1994/12/12  23:51:48  zappo
 * changed headers and fixed up test procedure a wee bit
 *
 * Revision 1.6  1994/12/07  23:18:55  zappo
 * Added checks to make sure that a daemon has a type before sending
 * various messages.
 *
 * Revision 1.5  1994/11/24  02:46:50  zappo
 * Include sitecnfg.h which contains site configurable parameters.
 *
 * Revision 1.4  1994/11/17  03:07:28  zappo
 * Added extra checks for announcement and reply-query check.
 *
 * Revision 1.3  1994/10/11  00:30:07  zappo
 * Improved testing for both etalk vs BSD/Sun and gtalkd testing.
 *
 * Revision 1.2  1994/09/15  02:06:26  zappo
 * Added cast to printing of sockaddress in test function
 *
 * Revision 1.1  1994/08/29  23:28:21  zappo
 * Initial revision
 *
 * History:
 * interran@uluru.Stanford.EDU (John Interrante)
 *   Spelling error in PROTOCOL_test.
 * eml
 *   Increased the number of things tested against for etalk.
 *
 * ::Header:: gtalkc.h 
 */
#include "gtalklib.h"
#include "gtalkc.h"
#include "gtalk.h"		/* this contains retry limits and such */
#include "talk.h"
#include "otalk.h"
#include "gtl_union.h"

#include "sitecnfg.h"

#if HAVE_PWD_H == 1
#include <pwd.h>
#endif

/* The current objects used during user aquisition
 */
static struct UserObject  *uobj = NULL;
static struct InputDevice *io = NULL;
static int                 announce_vers;
static int                 version_check = FALSE;
static char               *idlestr = "IDLE";
static char               *status = "IDLE";
static int                 testringerflag = 0;

static void no_action();

/*
 * This macro will simply encompass a loop to try to get a UDP message.
 * for the following attach function.  I should probably replace this
 * with something kinder in the future.
 */
#define GET_MSG(fn, io) \
{ int cnt = 0; \
  do { int rws = response_wait_stuff(Ctxt, io, &cnt); \
       if(rws == 1) return; if(rws == 2) { USER_hangup(Ctxt,uobj->id); \
					   DISP_update_user(Ctxt,uobj); \
					   PROTOCOL_delete_all(Ctxt, TRUE); \
					   return; }\
       if(!(fn)) { fn_badcall_cleanup(Ctxt); return; } \
       (io)->timeout = (UDP_LIFE); (io)->readme = no_action; cnt++; } \
    while(GT_select_all(Ctxt, io) == Fail); \
    (io)->timeout = 0; (io)->readme = NULL; }

#define GET_MSG_RETRY(fn, io) \
{ int cnt = 0; \
  do { int rws = response_wait_stuff(Ctxt, io, &cnt); \
       if(rws == 1) return NULL; if(rws == 2) break; \
       if(!(fn)) { fn_badcall_cleanup(Ctxt); return NULL; } \
       (io)->timeout = (UDP_LIFE); (io)->readme = no_action; cnt++; } \
    while(GT_select_all(Ctxt, io) == Fail); \
    (io)->timeout = 0; (io)->readme = NULL; }


/*
 * Function: response_wait_stuff, fn_badcall_cleanup
 *
 *   Locally defined function which waits for waits patiently for a
 * response from a remote talk daemon.  Called from MSG_* macro's.
 *
 * Returns:     static int  - 
 * Parameters:  Ctxt    - Context
 *              io      - Pointer to io
 *              counter - Number of itterations so far
 * History:
 * zappo   1/4/96     Created
 */
static int response_wait_stuff(Ctxt, io, counter)
     struct TalkContext *Ctxt;
     struct InputDevice *io;
     int                *counter;
{
  /* This relies upon some static-global variables!!! */
  if(!uobj || !(io) || !io)
    { 
      status = idlestr;
      DISP_message(Ctxt, "Bad GET_MSG environment");
      return 1; 
    } 
  if(uobj->state == USER_CLOSED)
    {
      PROTOCOL_delete_all(Ctxt, TRUE); 
      DISP_message(Ctxt, "Bad GET_MSG environment");
      status = idlestr; 
      return 1;
    }
  if(*counter >= UDP_RETRY)
    {
      uobj->state = USER_UNREACHABLE;
      DISP_message(Ctxt, "\03Retry limit reached.");
      io->readme = NULL; 
      status = idlestr;
      version_check = TRUE;
      return 2;			/* exit code */
    }
  io->timeout = UDP_LIFE; 
  io->readme = no_action;
  (*counter)++;
  return 0;
}
static void fn_badcall_cleanup(Ctxt)
     struct TalkContext *Ctxt;
{
  DISP_message(Ctxt, "\03Error running function, no more actions taken.");
  PROTOCOL_delete_all(Ctxt, TRUE);
  io->readme = NULL;
  status = idlestr; 
}

/*
 * Function: no_action
 *
 * Do nothing so that the readme flag gets set...
 * 
 * Parameters: Ctxt - talk context
 *             io   - the input device
 * History:
 * eml 4/6/94
 */
static void no_action(Ctxt, io)
     struct TalkContext *Ctxt;
     struct InputDevice *io;
{ /* Do nothing so README fn can be set while waiting about... */ }

/*
 * Function: display_timeout
 *
 * display a message discussing timeout till next announcement.
 * 
 * Parameters: Ctxt - context
 *             io - the io device in question
 * History:
 * eml 4/15/94
 */
void display_timeout(Ctxt, io)
     struct TalkContext *Ctxt;
     struct InputDevice *io;
{
  char buffer[200];

  /* if the calling user object is blank, ignore. */
  if(!uobj) 
    {
      DISP_message(Ctxt, "protocol display_timeout called with no active user obect.");
      return;
    }
  
  sprintf(buffer, "\03%cCalling %s: %d Announces, %d seconds till next ring.", 
	 TTY_NOLOG,
	 uobj->name,
	 announce_vers,
	 io->timeout);
  DISP_message(Ctxt, buffer);
}

/*
 * Function: PROTOCOL_connect
 *
 * If we mysteriously get a socket number, use that to connect directly
 * to some remote.
 * 
 * Parameters: Ctxt   - talk context
 *             userid - the user id of the emacs socket available
 *             sock   - the socket id on remote's machine
 *             user   - the user name
 *             node   - the remote node name
 *             tty    - the tty of remote
 * History:
 * eml 4/18/94
 */
void PROTOCOL_connect(Ctxt, userid, sock, user, node, tty)
     struct TalkContext *Ctxt;
     int userid;
     int sock;
     char *user;
     char *node;
     char *tty;
{
  struct HostObject *host;
  struct InputDevice *newtcp;
  struct sockaddr_in addr;

  if(verbose)
    printf("Attempt to connect to user %s@%s (%d), on socket %d\n",
	   user, node, userid, sock);

  /* now get the user struct from our list...
   */
  uobj = USER_find(userid);

  if(!uobj) {
    DISP_message(Ctxt, "Bad user id structure...");
    uobj = NULL;
    io = NULL;
    return;
  }

  if(uobj->state != USER_NEW)
    {
      DISP_message(Ctxt, "Attempt to call out on a used user struct!");
      uobj = NULL;
      io = NULL;
      return;
    }
  announce_vers = 1;

  uobj->name = strdup(user);

  if(uobj->local)
    {
      uobj->local->name = (char *)malloc(strlen(user) + 8);
      sprintf(uobj->local->name, "TCP to %s", uobj->name);
    }

  /* Ok, now just try to connect...
   */
  host = HOST_gen_host(node);

  if(!host)
    {
      char buffer[100];
      sprintf(buffer, "Cannot connect to %s because host does not exist.",
	      node);
      DISP_message(Ctxt, buffer);
      uobj->state = USER_CLOSED;
      GT_clean_dev(uobj->local, Ctxt->cleanup);
      DISP_update_user(Ctxt, uobj);
      return;
    }

  addr = host->addr;
  addr.sin_port = htons(sock);

  newtcp = TCP_connect((struct sockaddr *)&addr);
  
  if(newtcp)
    {
      int numedit;
      int n;

      newtcp->state = CONNECTED;
      newtcp->readme = STREAM_remote_read;
      newtcp->timeout = 0;
      newtcp->timefn = 0;
	  
      if(uobj->local)
	uobj->local->state = CONNECTED;
      uobj->remote = newtcp;
      uobj->state = USER_CONNECTED;
	  
      uobj->remote->name = (char *)malloc(strlen(user) + 10);
      sprintf(uobj->remote->name, "TCP from %s", uobj->name);
	  
      GT_send(newtcp, Ctxt->editkeys, NUMEDITKEYS);
      numedit = 0;
      do
	{
	  n = GT_recv(newtcp, uobj->editkeys, NUMEDITKEYS-numedit);
	  if(n == Fail)
	    {
	      DISP_message(Ctxt, "Failed to read data from new connection.");
	      uobj->state = USER_CLOSED;
	      GT_clean_dev(uobj->local, Ctxt->cleanup);
	      DISP_update_user(Ctxt, uobj);
	      uobj = NULL;
	      return;
	    }
	  numedit += n;
	} while(numedit != NUMEDITKEYS);
	      
      SCAN_send_version(Ctxt, uobj);

      if(uobj->local)
	{
	  GT_send(uobj->local, uobj->editkeys, NUMEDITKEYS);
	}
      if(Ctxt->runstate != Socket)
	{
	  DISP_message(Ctxt, "Connected to remote.");
	  DISP_update_user(Ctxt, uobj);
	}
    }
  else
    {
      char buffer[100];
      sprintf(buffer, "\03Unable to connect to %s.", node);
      DISP_message(Ctxt, buffer);
      uobj->state = USER_CLOSED;
      GT_clean_dev(uobj->local, Ctxt->cleanup);
      DISP_update_user(Ctxt, uobj);
    }
  uobj = NULL;
}

/*
 * Function: PROTOCOL_wait
 *
 * Wait for a connection to a given user id.
 * 
 * Parameters: Ctxt - talk context
 *             userid - the user id of the emacs socket available
 *             user - the user name
 *             node - the remote node name
 *             tty - the tty of remote
 * History:
 * eml 4/18/94
 */
void PROTOCOL_wait(Ctxt, userid, user, node, tty)
     struct TalkContext *Ctxt;
     int                 userid;
     char               *user;
     char               *node;
     char               *tty;
{
  struct InputDevice *newtcp;

  if(io || uobj)
    {
      DISP_message(Ctxt, "\03You may only make one outgoing call at a time!");
      USER_hangup(Ctxt, userid);
      return;
    }
  uobj = USER_find(userid);

  if(!uobj) {
    DISP_message(Ctxt, "Bad user id structure...");
    uobj = NULL;
    io = NULL;
    return;
  }

  if(uobj->state != USER_NEW)
    {
      DISP_message(Ctxt, "Attempt to call out on a used user struct!");
      uobj = NULL;
      io = NULL;
      return;
    }
  
  status = "Waiting for call with known address.";

  /*
   * Wait for connection from the remote...
   */
  Ctxt->remote_connect->readme  = no_action;
  Ctxt->remote_connect->timeout = RING_WAIT;
  Ctxt->remote_connect->timefn  = NULL;

  while(GT_select_all(Ctxt, Ctxt->remote_connect) == Fail)
    {
      /* first, check to see if we are even checking anymore...
       */
      if(!uobj) 
	{
	  status = idlestr;
	  return;
	}
      Ctxt->remote_connect->timeout = RING_WAIT;
    }
  
  Ctxt->remote_connect->readme  = DATA_link_accept;
  Ctxt->remote_connect->timeout = 0;
  Ctxt->remote_connect->timemsg = NULL;
  /*
   * now run accept on the socket to get the new user and do the final
   * setup
   */
  newtcp = TCP_accept(Ctxt->remote_connect);
  
  newtcp->state = CONNECTED;
  newtcp->readme = STREAM_remote_read;
  newtcp->timeout = 0;
  newtcp->timefn = 0;

  if(uobj->local)
    uobj->local->state = CONNECTED;
  uobj->remote = newtcp;
  uobj->state = USER_CONNECTED;

  uobj->remote->name = (char *)malloc(strlen(user) + 10);
  sprintf(uobj->remote->name, "TCP from %s", uobj->name);

  GT_send(newtcp, Ctxt->editkeys, NUMEDITKEYS); 
  {
    int numedit;
    int n;
    numedit = 0;
    do
      {
	n = GT_recv(newtcp, uobj->editkeys, NUMEDITKEYS-numedit);
	if (n == Fail)
	  {
	    DISP_message(Ctxt, "Failed to read data from new connection!");
	    uobj = NULL;
	    io = NULL;
	    return;
	  }
	numedit += n;
      } while(numedit != NUMEDITKEYS);
  }

  SCAN_send_version(Ctxt, uobj);

  if(uobj->local)
    {
      GT_send(uobj->local, uobj->editkeys, NUMEDITKEYS);
    }
  DISP_update_user(Ctxt, uobj);

  /* cleanup static varialbes used herin
   */
  uobj = NULL;
  io = NULL;

  status = idlestr;
}

/*
 * Function: PROTOCOL_reply
 *
 *   Ask local server who to the last person to send us an
 * announcement was, and print out that information.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt   - Context
 *
 * History:
 * zappo   11/18/94   Created
 */
void PROTOCOL_reply(Ctxt)
     struct TalkContext *Ctxt;
{
  struct sockaddr_in *addr;	/* address used for TCP creation */
  int cnt = 0; 

  if(io || uobj)
    {
      DISP_message(Ctxt, "\03You may only make one outgoing call request at a time!");
      return;
    }

  /* First, make sure we have an IO device for talking to remote daemon.
   */
  io = Ctxt->local_daemon;	/* use local daemon for remote daemon */
  
  if(io->host->type <= TALK_VERSION)
    {
      DISP_message(Ctxt, "\03You may only use reply if YOUR machine runs the GNU talk daemon.");
      io = NULL;
      return;
    }

  status = "Recieving REPLY information.";

  do
    { 
      if(cnt >= (UDP_RETRY))
	{
	  DISP_message(Ctxt, "\03Retry limit reached. Abandoning Call.");
	  io->readme = NULL; 
	  io = NULL;
	  status = idlestr; 
	  return; 
	}
      if(!DMN_Reply_Query(Ctxt, io))
	{
	  DISP_message(Ctxt, "\03Error running function, no more actions taken.");
	  io->readme = NULL;
	  io = NULL;
	  status = idlestr;
	  return; 
	}
      io->timeout = (UDP_LIFE); 
      io->readme = no_action; cnt++; 
    }
  while(GT_select_all(Ctxt, io) == Fail);

  io->timeout = 0; 
  io->readme = NULL;

  addr = DMN_get_reply_query_response(Ctxt, Ctxt->local_daemon);

  if(addr == (struct sockaddr_in *)NULL)  
    {
      DISP_message(Ctxt, "\03No one is calling you right now.");
    }
  else
    {
      char *user;
      struct HostObject *host;
      struct UserObject *new;

      user = DMN_last_response_user(io);
      host = HOST_gen_host_by_addr((struct sockaddr *)addr, sizeof(addr));

      /* If someone said "reply", immediatly go and create a connection */

      /* We successfully got some information, now create the talk
       * windows, and all that other good stuff
       */
      new = USER_alloc();
	  
      if(!new)
	{
	  gtalk_shutdown("Error creating new user for reply!\n");
	}

      new->name = strdup(user);
      if(!new->name)
	{
	  gtalk_shutdown("Error strduping name in PROTOCOL_reply");
	}
      if(Ctxt->runstate == Socket)
	{
	  /* Send unsolicited new tcp connection request */
	  printf("%c%c%d %s@%s\n", ETALK_ESCAPE, TTY_NEW_USER,
		 new->id, new->name, host->name);
	  /* Place this new socket directly into our new user struct,
	     and set it up as a local port */
	  new->local          = TCP_accept(Ctxt->emacs_connect);
	  new->local->state   = WAITING;
	  new->local->readme  = STREAM_local_read;
	  new->local->timeout = 0;
	  new->local->timefn  = NULL;
	  /* And fall into PROTOCOL_attach with new connection up! */
	}
#ifndef NO_IFACE
      else
	{
	  /* Create the window needed to hold this user's text */
	  DISP_new_user(Ctxt, new);
	}
#endif
      /* Reset status variables so we can call out! */
      io = NULL;
      status = idlestr;
      PROTOCOL_attach(Ctxt, new->id, new->name, host->name, NULL);
    }
  io = NULL;
  status = idlestr;
}


/*
 * Function: PROTOCOL_lookup_verify
 *
 *   Execute a lookup command that also verifies the version of the
 * talk deamon.  Used to identify the type of deamon on a given host.
 * If USER is NULL, then this is a general check, and not a real check
 * so we will do the minimum work necessary, including skipping the
 * test if applicable.
 *
 * Returns:     Address pointer from last server response, or NULL, or -1
 * Parameters:  Ctxt - Context
 *              io   - Input device & associated host.
 *              user - Name of the user to look for.
 * History:
 * zappo   10/17/97   Created
 */
static
struct sockaddr_in *PROTOCOL_lookup_verify(Ctxt, io_test, user_in, tty_in)
     struct TalkContext *Ctxt;
     struct InputDevice *io_test;
     char               *user_in;
     char               *tty_in;
{
  char buffer[150];
  struct sockaddr_in *addr;
  char *user = user_in?user_in:"test";
  char *tty = tty_in?tty_in:"";

  /* Do nothing if this is a boring test. */
  if(io_test->host->verified && (user == NULL)) return NULL;

  do
    {
      version_check = FALSE;
      /* Now, lets find out if we have been invited or not by asking the
       * local machine...
       */
      if(user_in != NULL) {
	sprintf(buffer, "\03Sending lookup message [V %d].",
		io_test->host->type);
	DISP_message(Ctxt, buffer);
      }

      GET_MSG_RETRY(DMN_Lookup(Ctxt, io_test, user, tty), io_test);
      if(!version_check)
	addr = DMN_get_lookup_response(Ctxt, io_test);
      else
	addr = NULL;

      if(addr == (struct sockaddr_in *)-1) version_check = TRUE;
      
      if(addr && (addr != (struct sockaddr_in *)-1))
	{
	  addr->sin_family = ntohs(addr->sin_family);

	  if(verbose)
	    {
	      printf("Address found:");
	      print_sockaddr((struct sockaddr *)addr);
	      printf("\n");
	    }

	  /* If we get this far, then that deamon really exists. */
	  io_test->host->verified = 1;
	  return addr;
	}

      /* check for potential errors, and we may need to downgrade the
       * supposed talk daemon we are connecting to.
       */
      if(version_check)
	{
	  /* The daemon may or may not respond to this query,
	   * so slowly drop to nothing...
	   */
	  if(io_test->host->type == 0)
	    {
	      sprintf(buffer, "\03Host %s has no apparent talk daemon.",
		      io_test->host->name);
	      DISP_message(Ctxt, buffer);
	      uobj->state = USER_UNREACHABLE;
	      USER_hangup(Ctxt, uobj->id);
	      DISP_update_user(Ctxt, uobj);

	      io = NULL;
	      uobj = NULL;
	      status = idlestr;
	      return (struct sockaddr_in *)-1;
	    }
	  io_test->host->type--;
	  UDP_daemon_change(io_test);
	}
      /*
       * If we do get a response, and no addr, then
       * check version error.
       */
      else if(DMN_last_response_numeric(io_test) == BADVERSION)
	{
	  if(io_test->host->type == OTALKD)
	    {
	      /* This can't happen as OTALKD doesn't define this symbol! */
	      sprintf(buffer, "\03Host %s has no apparent talk daemon!!",
		      io_test->host->name);
	      DISP_message(Ctxt, buffer);
	      uobj->state = USER_UNREACHABLE;
	      USER_hangup(Ctxt, uobj->id);
	      DISP_update_user(Ctxt, uobj);

	      io = NULL;
	      uobj = NULL;
	      status = idlestr;
	      return (struct sockaddr_in *)-1;
	    }
	  /* In this case, we must degrade down to the level descibed.
	   */
	  io_test->host->type = DMN_last_response_version(io_test);
	  UDP_daemon_change(io_test);
	}
      /* and if that is all ok, then just call out!!
       */
    } while((DMN_last_response_numeric(io_test) == BADVERSION) ||
	    version_check);

  /* If we get this far, then that deamon really exists. */
  io_test->host->verified = 1;
  return NULL;
}

/*
 * Function: PROTOCOL_attach
 *
 * Simply do the "talk thing" to connect to a remote user.
 * 
 * Parameters: Ctxt - talk context
 *             uid  - userobject id number.
 *             user - the user
 *             node - the node to attach to
 *             tty  - the users tty.
 * History:
 * eml 4/6/94
 */
void PROTOCOL_attach(Ctxt, userid, user, node, tty)
     struct TalkContext *Ctxt;
     int                 userid;
     char               *user;
     char               *node;
     char               *tty;
{
  char buffer[150];
  struct sockaddr_in *addr;	/* address used for TCP creation */
  struct InputDevice *newtcp;	/* new TCP socket                */

  if(io || uobj)
    {
      DISP_message(Ctxt,"\03You may only make one outgoing call at a time!");
      USER_hangup(Ctxt, userid);
      return;
    }

  /* First, make sure we have an IO device for talking to remote daemon.
   */
  if(!node) {
    io = Ctxt->local_daemon;	/* use local daemon for remote daemon */
    node = "";
  } else {
    io = UDP_host(node);
  }
  if(!tty) tty = "";

  if(!io)
    {
      /* Send the user a message */
      DISP_message(Ctxt, "\03Error connecting to host.");
      USER_hangup(Ctxt, userid);
      return;
    }
  
  /* Do nothing if there is not daemon. */
  if(io->host && (io->host->type == -1))
    {
      DISP_message(Ctxt, "\03That host has no daemon.");
      io = NULL;
      USER_hangup(Ctxt, userid);
      return;
    }

  /* now get the user struct from our list...
   */
  uobj = USER_find(userid);

  if(!uobj) {
    DISP_message(Ctxt, "Bad user id...");
    uobj = NULL;
    io = NULL;
    return;
  }

  if(uobj->state != USER_NEW)
    {
      DISP_message(Ctxt, "Attempt to call out on a used user struct!");
      uobj = NULL;
      io = NULL;
      return;
    }
  announce_vers = 1;

  uobj->name = strdup(user);
  if(!uobj->name)
    {
      gtalk_shutdown("Error strduping name in protocol attach");
    }

  uobj->state = USER_CALLING;
  DISP_update_user(Ctxt, uobj);

  if(uobj->local)
    {
      /* Only need to name this socket if we are talking to emacs */
      uobj->local->name = (char *)malloc(strlen(user) + 8);
      sprintf(uobj->local->name, "TCP to %s", uobj->name);
    }
  status = "Looking for invite.";

  addr = PROTOCOL_lookup_verify(Ctxt, io, user, tty);

  if(addr == (struct sockaddr_in *)-1)
    {
      /* We have encountered an error such as no deamon at all.  Quit now. */
      /* All the regular work has already been done. */
      return;
    }

  if(addr)
    {
      /* The lookup was successful! */
      newtcp = TCP_connect((struct sockaddr *)addr);
	  
      if(newtcp)
	{
	  int numedit;
	  int n;

	  newtcp->state = CONNECTED;
	  newtcp->readme = STREAM_remote_read;
	  newtcp->timeout = 0;
	  newtcp->timefn = 0;
	      
	  if(uobj->local)
	    {
	      uobj->local->state = CONNECTED;
	    }
	  uobj->remote = newtcp;
	  uobj->state = USER_CONNECTED;
	      
	  uobj->remote->name = (char *)malloc(strlen(user) + 10);
	  sprintf(uobj->remote->name, "TCP from %s", uobj->name);
	      
	  /* Transfer edit keys. */
	  GT_send(newtcp, Ctxt->editkeys, NUMEDITKEYS);
	  numedit = 0;
	  do		/* wait for all 3 editkeys */
	    {
	      n = GT_recv(newtcp, uobj->editkeys, NUMEDITKEYS-numedit);
	      if (n == Fail)
		{
                 DISP_message(Ctxt, "Failed to read data from new connection.");
                 return;
               }
	      numedit += n;
	    } while(numedit != NUMEDITKEYS);

	  /* Forward edit keys to emacs if applicable */
	  if(uobj->local)
	    {
	      GT_send(uobj->local, uobj->editkeys, NUMEDITKEYS);
	    }

	  /* Send version information */
	  SCAN_send_version(Ctxt, uobj);

	  DISP_update_user(Ctxt, uobj);

	  /* make sure we don't delete non-existing thingies
	   */
	  uobj = NULL;
	  io = NULL;
	      
	  status = idlestr;

	  DISP_message(Ctxt, "\03Connection Established!");

	  return;
	}
      else
	{
	  /*
	   * If we get crap from the daemon, just call them!!
	   */
	  DISP_message(Ctxt, "\03Invite information failed.  Calling out...");
	}
    }

  /* if we have not yet connected, then we must:
   * 1) prepare TCP listener for a connection.
   * do {
   *   2) leave an invitation
   *   3) announce to terminal
   * } while we don't hear from him.
   *
   *************
   *
   * Loop on the RING-WAIT thing after doing the invite/announce thing.
   */
  status = "Making outgoing call.";

  if(verbose)
    printf("Sending invitation message.\n");

  /* We do this to verify our local deamon, otherwise we can really 
   * mess ourselves up.  This is usually a no-op.
   */
  if(PROTOCOL_lookup_verify(Ctxt, Ctxt->local_daemon, NULL, NULL) ==
     ((struct sockaddr_in *)-1))
    {
      /* We have no local deamon, so tallyho and bugger outa here, */
      return;
    }
  
  GET_MSG(DMN_LeaveInvite(Ctxt, user, tty), Ctxt->local_daemon);
  if(DMN_get_invite_response(Ctxt, Ctxt->local_daemon) == Fail) 
    {
      sprintf(buffer, "\03Error leaving invitation for %s is [%s]", 
	      user, DMN_last_response(Ctxt->local_daemon));
      DISP_message(Ctxt, buffer);
      PROTOCOL_delete_all(Ctxt, TRUE);
      status = idlestr;
      return;
    }
  DISP_message(Ctxt, "\03Announcing...");
  GET_MSG(DMN_Announce(Ctxt, io, user, tty), io);
  if(DMN_get_announce_response(Ctxt, io) == Fail) 
    {
      /* In this case, just tell parent where they should be looking, */
      /* and let adjust their address/pointers, and they can then     */
      /* re-assign the current user-object.                           */
      if(DMN_last_response_numeric(io) == TRY_HERE)
	{
	  struct HostObject *host;

	  host = HOST_gen_host_by_addr((struct sockaddr *)DMN_last_addr(io),
				       sizeof(struct sockaddr));

	  if(!host)
	    {
	      if(verbose)
		{
		  printf("\03Address: ");
		  print_sockaddr((struct sockaddr
				  *)DMN_last_addr(Ctxt->local_daemon));
		  printf("\n");
		}
	      DISP_message(Ctxt,"\03Talk Daemon responded LOOK_HERE but address is invalid.");
		     
	    }
	  else
	    {
	      if(Ctxt->runstate == Socket)
		{
		  printf("\03%c%d %s\n", TTY_LOOK_HERE, uobj->id,
			 host->name);
		}
	      else
		{
		  /* Do curses init in this case... */
		  DISP_message(Ctxt,"Got a look here message...Adjusting.");
		  /* Reset the user object to be re-initialized */
		  uobj->state = USER_NEW;
		}
	    }

	  /* reguardless of look_here, the daemon won't let us connect
	   * so shutdown now.  DO NOT TERMINATE USER OBJECT.  This
	   * will be (maybe) recycled by the controlling process
	   */
	  PROTOCOL_delete_all(Ctxt, FALSE);
	  uobj = NULL;		/* reset user object  */
	  io   = NULL;		/* cancel io object   */
	  GT_reset_ids();	/* reset callout ids  */
	  status = idlestr;

	  if((Ctxt->runstate != Socket) && host)
	    {
	      /* If we have an interface, and we have a valid host,
		 lets try this again with the adjusted host name */
	      PROTOCOL_attach(Ctxt, userid, user, host->name, tty);
	    }
	  else
	    {
	      GT_end_recursion(); /* now cancel any lazy calls */
	    }
	  return;		/* all done */
	}
      else
	{
	  char buffer[100];
	  sprintf(buffer, "\03Response to announcement to %s is [%s]",
		  user, DMN_last_response(io));
	  PROTOCOL_delete_all(Ctxt, TRUE);
	  status = idlestr;
	  DISP_message(Ctxt, buffer);
	  return;
	}
    }
  
  /*
   * now wait for the connect.  If connect occurs during this time,
   * reads will be buffered... I hope anyway... so prepare rc for cnct
   */
  Ctxt->remote_connect->readme  = no_action;
  Ctxt->remote_connect->timeout = RING_WAIT;
  Ctxt->remote_connect->timefn  = NULL;
  Ctxt->remote_connect->timemsg = display_timeout;

  while(GT_select_all(Ctxt, Ctxt->remote_connect) == Fail)
    {
      /*
       * Check for a behind the back cancel of this opperation.
       */
      if(!uobj || !io) 
	{
	  status = idlestr;
	  return;
	}
      /* Delay buffer connects until we finish doing our UDP thing 
       * by simply removing it from the Q of Input devices to read from.
       */
      Ctxt->remote_connect->readme  = NULL;
      Ctxt->remote_connect->timeout = 0;
      /*
       * There was no connect, so delete the old ones, and put in some
       * new ones...
       */ 
      PROTOCOL_delete_all(Ctxt, 0);
      /*
       * setup for receiving again.
       */
      Ctxt->remote_connect->readme  = no_action;
      Ctxt->remote_connect->timeout = RING_WAIT;
      /*
       * Now, reannounce, and leave a new copy of the invitation...
       */
      if(verbose)
	printf("Sending invite message.\n");

      GET_MSG(DMN_LeaveInvite(Ctxt, user, tty), Ctxt->local_daemon);
      if(DMN_get_invite_response(Ctxt, Ctxt->local_daemon) == Fail)
	{
	  PROTOCOL_delete_all(Ctxt, TRUE);
	  sprintf(buffer, "\03Error leaving invitation for %s is  [%s]", 
		 user, DMN_last_response(Ctxt->local_daemon));
	  DISP_message(Ctxt, buffer);
	  status = idlestr;
	  return;
	}

      sprintf(buffer, "\03Announcing again [%d]...", announce_vers++);
      DISP_message(Ctxt, buffer);

      GET_MSG(DMN_Announce(Ctxt, io, user, tty), io);
      if(DMN_get_announce_response(Ctxt, io) == Fail)
	{
	  PROTOCOL_delete_all(Ctxt, TRUE);
	  sprintf(buffer, "\03Response to announcement to %s is [%s]",
		  user, DMN_last_response(io));
	  DISP_message(Ctxt, buffer);
	  status = idlestr;
	  return;
	}
  
    }
  /* Turn off connects while we manage our accepting */
  Ctxt->remote_connect->readme  = NULL;;
  Ctxt->remote_connect->timeout = 0;
  Ctxt->remote_connect->timemsg = NULL;
  /*
   * now run accept on the socket to get the new user and do the final
   * setup
   */
  newtcp = TCP_accept(Ctxt->remote_connect);
  
  newtcp->state = CONNECTED;
  newtcp->readme = STREAM_remote_read;
  newtcp->timeout = 0;
  newtcp->timefn = 0;

  GT_send(newtcp, Ctxt->editkeys, NUMEDITKEYS); 
  {
    int numedit;
    int n;
    numedit = 0;
    do
      {
	n= GT_recv(newtcp, uobj->editkeys, NUMEDITKEYS-numedit);
	if (n == Fail)
	  {
	    DISP_message(Ctxt, "Failed to read data from new connection.");
	    io = NULL;
	    uobj = NULL;
	  }
	numedit += n;
      } while(numedit != NUMEDITKEYS);
  }

  /* Fix up the user structure */
  if(uobj->local)
    uobj->local->state = CONNECTED;
  uobj->remote = newtcp;
  uobj->state = USER_CONNECTED;

  /*
   * we have a connection, so delete the announces and invites, and
   * put in some new ones...
   */ 
  PROTOCOL_delete_all(Ctxt, 0);

  uobj->remote->name = (char *)malloc(strlen(user) + 10);
  sprintf(uobj->remote->name, "TCP from %s", uobj->name);

  SCAN_send_version(Ctxt, uobj);

  if(uobj->local)
    {
      GT_send(uobj->local, uobj->editkeys, NUMEDITKEYS);
    }
  DISP_update_user(Ctxt, uobj);

  DISP_message(Ctxt, "\03Connection Established.");

  /* Restore the connector to it's old value */
  Ctxt->remote_connect->readme  = DATA_link_accept;

  /* cleanup static varialbes used herin
   */
  GT_reset_ids();
  io = NULL;
  uobj = NULL;
}

/*
 * Function: PROTOCOL_delete_all
 *
 * Delete the last announcement, and the last invitation.
 * 
 * Parameters: uobj      - user object of current pass
 *             terminate - if non-0, stop currently outbound call
 *
 * History:
 * eml 4/15/94
 */
void PROTOCOL_delete_all(Ctxt, terminate)
     struct TalkContext *Ctxt;
     int                 terminate;
{
  char buffer[150];
  if(!io && !uobj) return;

  /* don't try this unless we have a daemon type. */
  if(io->host && (io->host->type == -1))
    return;

  status = "Deleting Announcements.";
  
  /*
   * We cannot use the nifty macro or we get a nasty little recursive 
   * problem.
   */
  if(verbose)
    printf("Sending delete announce message.\n");

  {
    int cnt = 0; 
    do {
      if(cnt >= (UDP_RETRY))
	{
	  DISP_message(Ctxt, "\03Retry limit reached. Can't delete announce."); 
	  GT_clean_dev(uobj->local, Ctxt->cleanup);
	  uobj->state = USER_CLOSED;
	  DISP_update_user(Ctxt, uobj);
	  io->readme = NULL;
	  break;
	}
      else
	{
	  if(!DMN_Delete(Ctxt, io, DMN_announce))
	    { 
	      DISP_message(Ctxt, "\03Error running delete invite function.");
	      if(uobj->local)
		{
		  close(uobj->local->fd);
		  GT_clean_dev(uobj->local, Ctxt->cleanup);
		}
	      uobj->state = USER_CLOSED;
	      DISP_update_user(Ctxt, uobj);

	      io->readme = NULL;
	      break;
	    }
	  io->timeout = (UDP_LIFE);
	  io->readme = no_action;
	  cnt++; 
	}
    }
    while(GT_select_all(Ctxt, io) == Fail);
    io->timeout = 0; 
    io->readme = NULL;
  }
  if(DMN_get_delete_response(Ctxt, io) == Fail)
    {
      sprintf(buffer, "\03Delete announce error  %s", DMN_last_response(io));
      DISP_message(Ctxt, buffer);
    }
  if(verbose)
    printf("Sending delete invite message.\n");

  {
    int cnt = 0; 
    do {
      if(cnt >= (UDP_RETRY))
	{
	  DISP_message(Ctxt, "\03Retry limit reached. Can't delete invite."); 
	  uobj->state = USER_CLOSED;
	  if(uobj->local)
	    {
	      GT_clean_dev(uobj->local, Ctxt->cleanup);
	      Ctxt->local_daemon->readme = NULL;
	    }
	  DISP_update_user(Ctxt, uobj);
	  break;
	}
      else
	{
	  if(!DMN_Delete(Ctxt, Ctxt->local_daemon, DMN_invite))
	    { 
	      DISP_message(Ctxt, "\03Error running delete invite function.");
	      uobj->state = USER_CLOSED;
	      if(uobj->local)
		{
		  GT_clean_dev(uobj->local, Ctxt->cleanup);
		  Ctxt->local_daemon->readme = NULL;
		}
	      DISP_update_user(Ctxt, uobj);
	      break;
	    }
	  Ctxt->local_daemon->timeout = (UDP_LIFE);
	  Ctxt->local_daemon->readme = no_action; 
	  cnt++; 
	}
    }
    while(GT_select_all(Ctxt, Ctxt->local_daemon) == Fail);
    Ctxt->local_daemon->timeout = 0; 
    Ctxt->local_daemon->readme = NULL;
  }
  if(DMN_get_delete_response(Ctxt, Ctxt->local_daemon) == Fail)
    {
      sprintf(buffer, "\03Delete invite error  %s", 
	      DMN_last_response(Ctxt->local_daemon));
      DISP_message(Ctxt, buffer);
    }

  Ctxt->remote_connect->readme = DATA_link_accept;
  Ctxt->remote_connect->timeout = 0;

  if(terminate)
    {
      if(verbose)
	printf("Executing terminate call information...\n");

      io = NULL;
      if(uobj)
	uobj->state = USER_CLOSED;
      if(uobj && uobj->local)
	{
	  GT_clean_dev(uobj->local, Ctxt->cleanup);
	}
      DISP_update_user(Ctxt, uobj);

      uobj = NULL;
      GT_reset_ids();
      GT_end_recursion();	/* now cancel any lazy calls */
    }

  status = idlestr;
}


/*
 * Function: PROTOCOL_status
 *
 *   Return a string point to a description of the current status of
 * any given operation which may be happening.
 *
 * Returns:     char * - 
 * Parameters:  Ctxt - Context
 *
 * History:
 * zappo   11/19/94   Created
 */
char *PROTOCOL_status(Ctxt)
     struct TalkContext *Ctxt;
{
  return status;
}

/*
 * Function: PROTOCOL_abort
 *
 *   Aborts the currently active call
 *
 * Returns:     int - 0 if no active calls, 1 if we canceled something.
 * Parameters:  Ctxt - Context
 *
 * History:
 * zappo   4/19/94    Created
 * zappo   9/21/95    Added delete_all call to remove current outstanding
 *                    invitations, and announcements
 */
int PROTOCOL_abort(Ctxt)
     struct TalkContext *Ctxt;
{
  if(!io && !uobj)
    {
      DISP_message(Ctxt, "No active calls waiting...");
      return 0;
    }

  status = "Aborting ongoing call.";

  /* first, remove any pointers from the deamon */
  PROTOCOL_delete_all(Ctxt, 0);

  /* Basically, null out any active call structures.
   */
  if(io)
    {
      io->readme = NULL;
      io = NULL;
    }
  if(uobj)
    {
      if(verbose)
	printf("Call to %s UID %d deleted.\n",
	       uobj->name?uobj->name:"Unknown", uobj->id);

      /* Set as unreachable, then hang up on it for cleanup */
      uobj->state = USER_UNREACHABLE;
      USER_hangup(Ctxt, uobj->id);
      uobj = NULL;
    }
  if(Ctxt->remote_connect->timemsg)
    {
      Ctxt->remote_connect->timemsg = NULL;
      Ctxt->remote_connect->timeout = 1; /* to free a waitfor */
      Ctxt->remote_connect->readme = DATA_link_accept;
    }

  status = idlestr;

  return 1;
}


/*
 * Function: PROTOCOL_ringerread, PROTOCOL_test
 *
 * Test the local talk daemon by looking up USER.  Don't bother
 * reading the message in since we'll just quit in a minute anyway.
 *
 * The ringread function is used solely for reading/printing the
 * contents of a ringer message passed in during a -t test.
 *
 * Parameters:  Ctxt    - talk context
 *              machine - The machine to test against. NULL or BLANK
 *                        means the local machine
 *
 * History:
 * eml	May 27, 1994  Created
 * eml  Aug 10, 1994  Spelling, OTALKD and NTALKD not OTALK and NTALK
 * eml  Aug 19, 1994  Added much more comprehensive test of the
 *                    whole protocol exchange between daemon and server.
 * zappo   9/14/94    Added cast to sockaddr printing function.
 * zappo   9/24/94    Improved error checking for no-daemon conditions.
 * zappo   10/1/94    Added check when daemon forgets our invite
 * zappo   11/16/94   Added Announcement and reply-query check
 * zappo   12/12/94   Fixed up some printing.
 * zappo   3/15/95    Added PROTOCOL_ringread for use with ringer
 *                    testing functions, and added numbers before each
 *                    testing seequence.
 * zappo   1/7/98     Added machine parameters
 */
static void PROTOCOL_ringerread(Ctxt, dev)
     struct TalkContext *Ctxt;
     struct InputDevice *dev;
{
  GNU_RING_CONTROL    rc;
  struct InputDevice *newudp;

  printf("3.2) ringerread: Recieved a ringer message.\n");
  if(GT_recv(dev, &rc, sizeof(rc)) == Fail)
    {
      printf("3.2) Failed to read in ringer message from daemon!\n");
      return;
    }

  /* Fix byte swapping in the family so we can use this address. */
  ((struct sockaddr_in *)&rc.addr)->sin_family = 
    ntohs(((struct sockaddr_in *)&rc.addr)->sin_family);

  /* We could also use dev->lraddr, but this is more stable. In     */
  /* addition, this particular address will always be recycled too. */
  newudp = UDP_byaddr(dev, (struct sockaddr_in *)&rc.addr);
  newudp->name = "daemon_ringport";
 
  printf("3.2.2) ringerread: Daemon ringport at: ");
  print_sockaddr((struct sockaddr *)&rc.addr);
  printf("\n");
  
  /* Always print this: This will let us see the full message in the log. */
  if(verbose) RING_print_control(&rc);

  printf("3.2.2) ringerread: Person calling is %s\n", rc.name);

  testringerflag = 1;

  return;  
}

int PROTOCOL_test(Ctxt, machine)
     struct TalkContext *Ctxt;
     char *machine;
{
  int myflag = 1;
  struct InputDevice *udp_host = NULL;

  Ctxt->myname = "testing";

  if(Ctxt->runstate != Socket)
    {
      DISP_message(Ctxt, "To run the test from command line type: gtalk -t");
      return 1;
    }

  Ctxt->tty->readme = NULL;	/* turn off command line from intrusion */

  if(machine && *machine)
    udp_host = UDP_host(machine);

  if(udp_host == NULL)
    udp_host = Ctxt->local_daemon;

  printf("\n** Performing tests on host %s\n\n", udp_host->host->name);

  do
    {
      printf("1) Checking for daemon: Sending test lookup message [V %d].\n", 
	     udp_host->host->type);
      DMN_Lookup(Ctxt, udp_host, "test", "");

      udp_host->timeout = (UDP_LIFE); 
      udp_host->readme = no_action;

      if(GT_select_all(Ctxt, udp_host))
	{
	  /* Read in the text sent back, and make sure it's ok. */
	  DMN_get_lookup_response(Ctxt, udp_host);
	  if(DMN_last_response_numeric(udp_host) != SUCCESS)
	    myflag = 1;
	  else
	    {
	      printf("1) Incorrect answer from daemon...\n");
	      myflag = 0;
	    }
	}
      else
	{
	  printf("1) Nothing recieved or error selecting inputs...\n");
	  myflag = 0;
	}

      if((myflag == 0) && (udp_host->host->type == 0))
	myflag = 0;
      else if((DMN_last_response_numeric(udp_host) == BADVERSION) ||
	      (myflag != 1))
	{
	  udp_host->host->type --;
	  /* 2 means retry */
	  myflag = 2;
	}

      /* A 2 means retry */
    } while (myflag == 2);
  

  if(myflag)
    {
      printf("1) Brief daemon check successful!\n");
    }
  else
    {
      printf("1) Daemon check failed!\n\n");

      printf("1) You probably don't have a talk daemon running or your\n");
      printf("1) existing talk daemon crashes when recieving V2 packets.\n");
      
      printf("1) If you intend to use gtalkd, you can get the makefile\n");
      printf("1) to pass by running gtalkd in terminal mode as root like this:\n1)\n");

      printf("1) gtalkd -ft\n1)\n");
      printf("1) which will permit you to have all this test completed.\n");

      printf("1)\n1) Read the gtalk info file under `Setup Problems' for more help.\n\n");
      return Fail;
    }

  {
    uid_t          my_id;
    struct passwd *pw;

    my_id = getuid();
    pw = getpwuid(my_id);
    
    if(!pw)
      {
	printf("You don't exist! Can't fill in user name.\n");
      }

    /* Some systems it's static, others it's malloced.  Since gtalk is
     * about to exit anyway, ignore this problem.
     */
    Ctxt->myname = pw->pw_name;
  }

  printf("2) Initiating protocol test.\n");

  printf("2.1) Sending announcement message...\n");
  /* Make it bold to test bomb removal */
  Ctxt->myappname = "\033[1mgtalk_running_a_test_procedure\033[m";
  DMN_Announce(Ctxt, udp_host, Ctxt->myname, "");

  udp_host->timeout = (UDP_LIFE); 
  udp_host->readme = no_action;

  if(GT_select_all(Ctxt, udp_host))
    {
      int r = DMN_get_announce_response(Ctxt, udp_host);

      printf("2.1) Received response %s to announcement!\n",
	     DMN_last_response(udp_host));
      if(r == Fail)
	{
	  printf("2.1) Announcement in protocol test failed.  See info file.\n");
	  printf("2.1.1) Sending deletion message for previous announce...\n");
	  DMN_Delete(Ctxt, udp_host, DMN_announce);
	  exit(1);
	}
    }
  else
    {
      printf("2.1) Announcement in protocol test failed.  See info file.\n");
      printf("2.1.1) Sending deletion message for previous announce...\n");
      DMN_Delete(Ctxt, udp_host, DMN_announce);
      exit(1);
    }

  printf("2.2) Sending invitation...\n");
  DMN_LeaveInvite(Ctxt, Ctxt->myname, "");

  udp_host->timeout = (UDP_LIFE); 
  udp_host->readme = no_action;

  if(GT_select_all(Ctxt, udp_host))
    {
      int r = DMN_get_invite_response(Ctxt, udp_host);

      printf("2.2) Received response %s to invitation!\n",
	     DMN_last_response(udp_host));
      if(r == Fail)
	{
	  printf("2.2) Invitation in protocol test failed.  See info file.\n");
	  exit(1);
	}
    }
  else
    {
      printf("2.2) Invitation in protocol test failed.  See info file.\n");
      exit(1);
    }

  if(udp_host->host->type >= TALK_VERSION_GNU)
    {
      printf("2.2.1) Sending reply query message for previous announce...\n");
      DMN_Reply_Query(Ctxt, udp_host);

      udp_host->timeout = (UDP_LIFE); 
      udp_host->readme = no_action;

      if(GT_select_all(Ctxt, udp_host))
	{
	  struct sockaddr_in *r;

	  r = DMN_get_reply_query_response(Ctxt, udp_host);

	  if(!r || r == (struct sockaddr_in *)-1)
	    {
	      printf("2.2.1) Talk daemon forgot about our announcement! Daemon error!\n");
	      exit(1);
	    }
	  else
	    {
	      /* convert address out of network order. */
	      r->sin_family = ntohs(r->sin_family);

	      printf("2.2.1) Received response %s to reply request!\n",
		     DMN_last_response(udp_host));
	      if(DMN_last_response_numeric(udp_host) != SUCCESS)
		{
		  printf("2.2.1) Reply lookup in  protocol test failed.  See info file.\n");
		  exit(1);
		}
	      printf("2.2.1) Daemon has sent us   :");
	      print_sockaddr((struct sockaddr *)r);
	      printf("\n");
	    }
	}
      else
	{
	  printf("Reply lookup in protocol test failed.  See info file.\n");
	  exit(1);
	}
    }

  printf("2.3) Sending lookup query message for previous invite...\n");

  DMN_Lookup(Ctxt, udp_host, Ctxt->myname, "");

  udp_host->timeout = (UDP_LIFE); 
  udp_host->readme = no_action;

  if(GT_select_all(Ctxt, udp_host))
    {
      struct sockaddr_in *r;

      r = DMN_get_lookup_response(Ctxt, udp_host);

      if(!r || r == (struct sockaddr_in *)-1)
	{
	  printf("2.3) Talk daemon forgot about our invitation! Daemon error!\n");
	  exit(1);
	}
      else
	{
	  /* convert address out of network order. */
	  r->sin_family = ntohs(r->sin_family);

	  printf("2.3) Received response %s to lookup request!\n",
		 DMN_last_response(udp_host));
	  if(DMN_last_response_numeric(udp_host) != SUCCESS)
	    {
	      printf("2.3) Lookup in  protocol test failed.  See info file.\n");
	      exit(1);
	    }
	  printf("2.3) Our remote connect socket is:");
	  print_sockaddr((struct sockaddr *)&Ctxt->remote_connect->raddr);
	  printf("\n     Daemon thinks it is         :");
	  print_sockaddr((struct sockaddr *)r);
	  printf("\n");

	  if((r->sin_family != Ctxt->remote_connect->raddr.sin_family) ||
	     (r->sin_port != Ctxt->remote_connect->raddr.sin_port) ||
	     (r->sin_addr.s_addr != Ctxt->remote_connect->raddr.sin_addr.s_addr))
	    {
	      printf("2.3) Response from daemon garbled our address!\n");
	      exit(0);
	    }
	  else
	    {
	      printf("2.3) Address to our socket transferred correctly.\n");
	    }
	}
    }
  else
    {
      printf("2.3) Lookup in protocol test failed.  See info file.\n");
      exit(1);
    }

  printf("2.4) Sending deletion message for previous invite...\n");
  DMN_Delete(Ctxt, udp_host, DMN_invite);

  udp_host->timeout = (UDP_LIFE); 
  udp_host->readme = no_action;

  if(GT_select_all(Ctxt, udp_host))
    {
      int r = DMN_get_delete_response(Ctxt, udp_host);

      printf("2.4) Received response %s to delete request!\n",
	     DMN_last_response(udp_host));
      if(r == Fail)
	{
	  printf("2.4) Deletion (invite) in protocol test failed.  See info file.\n");
	  exit(1);
	}
    }
  else
    {
      printf("2.4) Deletion (announce) in protocol test failed.  See info file.\n");
      exit(1);
    }

  printf("2.5) Sending deletion message for previous announce...\n");
  DMN_Delete(Ctxt, udp_host, DMN_announce);

  udp_host->timeout = (UDP_LIFE); 
  udp_host->readme = no_action;

  if(GT_select_all(Ctxt, udp_host))
    {
      int r = DMN_get_delete_response(Ctxt, udp_host);

      printf("2.5) Received response %s to delete request!\n",
	     DMN_last_response(udp_host));
      if(r == Fail)
	{
	  printf("2.5) Deletion (announce) in protocol test failed.  See info file.\n");
	  exit(1);
	}
    }
  else
    {
      printf("2.5) Deletion (announce) in protocol test failed.  See info file.\n");
      exit(1);
    }
  printf("2) Protocol test successful.\n");

  /* Now lets test the ringer facility (if available) */
  if(udp_host->host->type >= TALK_VERSION_GNU)
    {
      char buff[80];

      printf("3) Ringer test.\n");

      /* First, just make sure we can get called back... */
      if(RING_activate(Ctxt->myname, &testringerflag, Ctxt->udp_ring, PROTOCOL_ringerread)
	 == Fail)
	{
	  printf("3.1) Ringer file activation failed.\n");
	  return Fail;
	}
      printf("3.1) Ringer file successfully engaged.\n");

      printf("3.2) Sending announcement message to activate ringer.\n");

      DMN_Announce(Ctxt, udp_host, Ctxt->myname, "");

      udp_host->timeout = (UDP_LIFE); 
      udp_host->readme = no_action;

      if(GT_select_all(Ctxt, udp_host))
	{
	  int r = DMN_get_announce_response(Ctxt, udp_host);
	  
	  printf("3.2) Received response %s to announcement!\n",
		 DMN_last_response(udp_host));
	  if(r == Fail)
	    {
	      printf("3.2) Announcement in protocol test failed.  See info file.\n");
	      exit(1);
	    }
	}
      else
	{
	  printf("3.2) Announcement in protocol test failed.  See info file.\n");
	  exit(1);
	}

      printf("3.2) Response to message 0k.  Testflag = %d\n", testringerflag);

      printf("3.3) Deleting that pesky announce we just did.\n");
      DMN_Delete(Ctxt, udp_host, DMN_announce);

      udp_host->timeout = (UDP_LIFE); 
      udp_host->readme = no_action;

      if(GT_select_all(Ctxt, udp_host))
	{
	  int r = DMN_get_delete_response(Ctxt, udp_host);
	  
	  printf("3.3) Received response %s to delete request!\n",
		 DMN_last_response(udp_host));
	  if(r == Fail)
	    {
	      printf("3.3) Deletion (announce) in protocol test failed.  See info file.\n");
	      exit(1);
	    }
	}
      else
	{
	  printf("3.3) Deletion (announce) in protocol test failed.  See info file.\n");
	  exit(1);
	}

      printf("3.4) Testing ringer file forwarding via daemon.\n");

      /* first, we must create a recognizable bogus address */
      {
	char *env;
	FILE *f;			/* ringer file. */

	/* Open for replacing. 
	 */
	env = getenv("HOME");
	
	if(env == NULL)
	  {
	    printf("You do not have the HOME environment variable set!\n");
	    return Fail;
	  }

	strcpy(buff, env);
	strcat(buff, "/");
	strcat(buff, RINGER_RC);

	f = fopen(buff, "w");

	if(f == NULL)
	  {
	    printf("Failure to open Ringer File [%s].\n", buff);
	    return Fail;
	  }

	/* Write an identifiable address here! */
	fprintf(f, "%d %d %ld", 1, htons(2), htonl(420000));
	 

	printf("3.4) Stowing address: 1 2 420000 into ringer file.\n");
	  
	fclose(f);
      }

      /* Now send an announcement! */
      printf("3.5) Sending announcement message to activate ringer.\n");

      DMN_Announce(Ctxt, udp_host, Ctxt->myname, "");

      udp_host->timeout = (UDP_LIFE); 
      udp_host->readme = no_action;

      if(GT_select_all(Ctxt, udp_host))
	{
	  int r = DMN_get_announce_response(Ctxt, udp_host);
	  
	  printf("3.5) Received response %s to announcement!\n",
		 DMN_last_response(udp_host));
	  if(r == Fail)
	    {
	      if(DMN_last_response_numeric(udp_host) == TRY_HERE)
		{
		  printf("3.5) Recieved TRY_HERE: Note, only last number matters.\n");
		  printf("3.5) Daemon responded with: ");
		  print_sockaddr((struct sockaddr *)DMN_last_addr(udp_host));
		  printf("\n");
		  if(DMN_last_addr(udp_host)->sin_addr.s_addr !=
		     420000)
		    {
		      printf("3.5) Transfer of address was not successful!\n");
		      exit(1);
		    }
		}
	      else
		{
		  printf("3.5) Failed to get TRY_HERE failure response, got %d instead.\n",
			 DMN_last_response_numeric(udp_host));
		  printf("3.6) Deleting the ringer file.\n");
		  if(RING_deactivate(Ctxt->myname) == Fail)
		    {
		      printf("3.6) Deactivation failed!\n");
		      exit(1);
		    }
		  else
		    printf("3.6) Deactivation (deletion of file) successful.\n");

		  exit(1);
		}
	    }
	  else 
	    {
	      printf("3.5) Announcement in protocol test was unsespectedly Successful\n");
	      printf("3.6) Deleting the ringer file.\n");
	      if(RING_deactivate(Ctxt->myname) == Fail)
		{
		  printf("3.6) Deactivation failed!\n");
		  exit(1);
		}
	      else
		printf("3.6) Deactivation (deletion of file) successful.\n");

	      exit(1);
	    }
	}
      else
	{
	  printf("3.5) Announcement in protocol test failed.  See info file.\n");
	  printf("3.6) Deleting the ringer file.\n");
	  if(RING_deactivate(Ctxt->myname) == Fail)
	    {
	      printf("3.6) Deactivation failed!\n");
	      exit(1);
	    }
	  else
	    printf("3.6) Deactivation (deletion of file) successful.\n");

	  exit(1);
	}

      printf("3.6) Deleting the ringer file.\n");
      if(RING_deactivate(Ctxt->myname) == Fail)
	{
	  printf("3.6) Deactivation failed!\n");
	  exit(1);
	}
      else
	printf("3.6) Deactivation (deletion of file) successful.\n");

      printf("3.7) Deleting that pesky announce we just did.\n");
      DMN_Delete(Ctxt, udp_host, DMN_announce);

      udp_host->timeout = (UDP_LIFE); 
      udp_host->readme = no_action;

      if(GT_select_all(Ctxt, udp_host))
	{
	  int r = DMN_get_delete_response(Ctxt, udp_host);
	  
	  printf("3.7) Received response %s to delete request!\n",
		 DMN_last_response(udp_host));
	  if(r == Fail)
	    {
	      printf("3.7) Deletion (announce) in protocol test failed.  See info file.\n");
	      exit(1);
	    }
	}
      else
	{
	  printf("3.7) Deletion (announce) in protocol test failed.  See info file.\n");
	  exit(1);
	}

      printf("3) Ringer test successful!\n");

      printf("4) Reverting back to 1, and using OTALK protocol to test forwarding.\n");

      udp_host->host->type = OTALKD;

      UDP_daemon_change(udp_host);

      PROTOCOL_test(Ctxt, machine);

    }

  return Success;
}
