/* gtl_list.c - Generic List Management
 *
 * Copyright (C) 1997, 1998 Free Software Foundation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 *
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 *
 * Please send bug reports, etc. to zappo@gnu.org
 * 
 * Purpose:
 *   Generic List management.
 *
 * $Log: gtl_list.c,v $
 * Revision 1.6  1998/05/01 16:44:51  zappo
 * Very verbose behavior now only happens on 3.
 *
 * Revision 1.5  1997/12/14 19:21:24  zappo
 * Renamed package to gtalk, renamed symbols and files apropriately
 * Fixed copyright and email address.
 *
 * Revision 1.4  1997/10/18 03:07:23  zappo
 * Fixed magic number assignments to be cast-happy.
 *
 * Revision 1.3  1997/07/27 20:40:59  zappo
 * Fixed bugs in the verifier.
 *
 * Revision 1.2  1997/07/23 01:22:44  zappo
 * Added debugging and verification of list structures.
 *
 * Revision 1.1  1997/07/11 21:27:46  zappo
 * Initial revision
 *
 *
 * ::Header:: gtalklib.h
 */

#include "gtalklib.h"

#define LIST_MAGIC_NUMBER 0xF0F0F0F0

static struct GenericList *listlist = ((void *)LIST_MAGIC_NUMBER);


/*
 * Function: LIST_alloc
 *
 *   Allocates and initializes a new Generic List node.  This node
 * can be used to alloc other structures that inherit from the
 * GenericListNode structure.
 *
 * Returns:     GenericList * - 
 * Parameters:  list - Pointer to a new list element
 *              size - Size of the structure to allocate
 * History:
 * zappo   7/11/97    Created
 */
struct GenericListNode *LIST_alloc(list, size)
     struct GenericList *list;
     size_t size;
{
  struct GenericListNode *new;

  if(list->next == NULL) {
    /* For list list maintenance... */
    list->next = listlist;
    listlist = list;
    list->verify = LIST_MAGIC_NUMBER;
    if(verbose)
      printf("New list %p in front of %p\n", listlist, listlist->next);
  }

  new = (struct GenericListNode *)malloc(size);

  if(!new)
    {
      perror("malloc failure!! : ");
      return NULL;
    }

  memset((char *)new, 0, size);

  new->verify = LIST_MAGIC_NUMBER;
  
  if(list->last) {
    list->last->next = new;
  } else {
    list->first = new;
  }
  new->prev = list->last;
  new->next = NULL;
  list->last = new;

  if(verbose >= 3) {
    char buffer[100];
    sprintf(buffer, "Alloced a node [ %p %p %p ]", new, new->next, new->prev);
    DISP_message(NULL, buffer);
    sprintf(buffer, "Allocated node in list [ %p %p ]",
	    list->first, list->last);
    DISP_message(NULL, buffer);
  }

  LIST_verify();

  return new;  
}


/*
 * Function: LIST_find, LIST_find_next
 *
 *   Finds some node in List based on predicate.  Predicate must be a
 * function pointer that takes two arguments, the Node and a generic
 * pointer representing to match against.  It does not find multiple
 * matching occurances.
 *
 * Returns:     struct GenericListNode * - 
 * Parameters:  list      - Generic List object
 *              start     - Starting point of search (_next only)
 *              predicate - Matching function
 *              criteria  - Criteria to match against.
 * History:
 * zappo   7/11/97    Created
 */
struct GenericListNode *LIST_find(list, predicate, criteria)
     struct GenericList *list;
     unsigned char     (*predicate)();
     void               *criteria;
{
  struct GenericListNode *loop;

  for(loop = list->first; 
      loop && !predicate(loop, criteria);
      loop = loop->next);

  return loop;
}
struct GenericListNode *LIST_find_next(start, predicate, criteria)
     struct GenericListNode *start;
     unsigned char     (*predicate)();
     void               *criteria;
{
  struct GenericListNode *loop;

  for(loop = start->next;
      loop && !predicate(loop, criteria);
      loop = loop->next);

  return loop;
}


/*
 * Function: LIST_map
 *
 *   Finds the length of the list, and can also run a predicate
 * function on each element of the node.  The predicate function must
 * take only one parameter, the node being mapped.
 *
 * Returns:     int - Length of the list
 * Parameters:  list      - Pointer the list
 *              predicate - Function to run on each node.
 *              data      - User data to be passed to predicate
 * History:
 * zappo   7/11/97    Created
 */
int LIST_map(list, predicate, data)
     struct GenericList *list;
     void (*predicate)();
     void *data;
{
  struct GenericListNode *loop;
  int i;

  for(i = 0, loop = list->first;
      loop; i++, loop = loop->next) 
    {
      if(predicate != NULL)
	{
	  predicate(loop, data);
	}
    }

  return i;
}


/*
 * Function: LIST_deallocate
 *
 *   Frees all memory associated w/ the GenericListNode node which is
 * found in GenericList.  Does not account for other forms of data.
 * The reference to node is also unlinked from the list object.
 *
 * Returns:     Nothing
 * Parameters:  list - GenericList in which node is saved
 *              node - The node to free.
 * History:
 * zappo   7/11/97    Created
 */
void LIST_deallocate(list, node)
     struct GenericList *list;
     void *node;
{
  struct GenericListNode *n = node;

  LIST_verify();

  if(verbose >= 3) {
    char buffer[100];
    sprintf(buffer, "Deallocing node [ %p %p %p ]", n, n->next, n->prev);
    DISP_message(NULL, buffer);
  }

  if(n == list->first)
    list->first = n->next;
  if(n == list->last)
    list->last = n->prev;
  if(n->next)
    n->next->prev = n->prev;
  if(n->prev)
    n->prev->next = n->next;

  if(verbose >= 3) {
    char buffer[100];
    sprintf(buffer, "Deallocated node from list [ %p %p ]",
	    list->first, list->last);
    DISP_message(NULL, buffer);
  }

  free(n);
}


/*
 * Function: LIST_verify
 *
 *   Verify that all active lists have valid pointers and such.
 * Verification is only in effect when verbosity is on. 
 *
 * Returns:     Nothing
 * Parameters:  None
 *
 * History:
 * zappo   7/17/97    Created
 */
static void PrintPtr(msg, ptr)
     char *msg;
     void *ptr;
{
  char buffer[100];
  sprintf(buffer, msg, ptr);
  DISP_message(NULL, buffer, LOG_CRIT);
}

void LIST_verify()
{
  struct GenericList *loop;

  if(verbose)
    for(loop=listlist; loop != ((void *)LIST_MAGIC_NUMBER); loop=loop->next)
      {
	struct GenericListNode *node;

	/* Verify this lists magic number */
	if(loop->verify != LIST_MAGIC_NUMBER) {
	  PrintPtr("List at %p failed Magic Number check. Abort\n", loop);
	  abort();
	}
	if(verbose >= 3)
	  printf("List @ %p (%p %p) -> %p\n", loop, loop->first,
		 loop->last, loop->next);

	for(node = loop->first; node; node = node->next)
	  {
	    if(node->verify != LIST_MAGIC_NUMBER)
	      {
		PrintPtr("  Node at %p Failed Magic Number check. Abort\n", node);
		abort();
	      }
	    if(verbose >= 3)
	      printf("  Node @ %p (%p %p)\n", node, node->next, node->prev);
	  }
      }
}
