/*
 *  Copyright (C) 1999-2002 Bruno Pires Marinho
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#include <applet-widget.h>
#include <liboaf/liboaf.h>
#include <libgnomeui/gnome-window-icon.h>
#include <gdk/gdkx.h>
#include <gdk/gdkprivate.h>
#include <X11/extensions/shape.h>
#include <sys/utsname.h>
#include <sys/socket.h>
#include <net/route.h>
#include <net/if.h>
#include <sys/ioctl.h> 
#include "Gtm.h"
#include "dialogs.h"
#include "gtm_applet.h"
#include "tray_empty.xpm"
#include "tray_full.xpm"
#ifdef HAVE_PANEL_PIXEL_SIZE
#include "tray_empty_huge.xpm"
#include "tray_full_huge.xpm"
#include "tray_empty_large.xpm"
#include "tray_full_large.xpm"
#include "tray_empty_small.xpm"
#include "tray_full_small.xpm"
#include "tray_empty_tiny.xpm"
#include "tray_full_tiny.xpm"
#endif

/* Pixmaps used on the application */
static GdkPixmap *i_open;
static GdkPixmap *i_open_mask;
static GdkPixmap *i_closed;
static GdkPixmap *i_closed_mask;

static GtkWidget *eventbox;
static GtkWidget *applet;
static GtkWidget *pixmap;

#ifdef HAVE_PANEL_PIXEL_SIZE
static int panel_size;
#endif
static PanelOrientType panel_orient;

/* Used to keep current drag state */
static gboolean have_drag;

/* Used to keep if we already start GTM on interface on */
static gboolean already_open_gtm;

CORBA_Environment ev;

/* The types of stuff we accept on dnd */
enum {
    TARGET_URL,
    TARGET_NETSCAPE_URL
};

/* The applet properties */
Properties gtm_applet_prop;

/* Variable holding the timeout id of the check network function */
static gint check_network_timeout_id = -1;


/* CORBA functions */
static void
Exception (CORBA_Environment *ev)
{
    switch (ev->_major) {
	case CORBA_SYSTEM_EXCEPTION:
	    g_log ("GNOME Transfer Manager", G_LOG_LEVEL_DEBUG, 
		   "CORBA system exception %s.\n", CORBA_exception_id (ev));
	    exit (1);
	case CORBA_USER_EXCEPTION:
	    g_log ("GNOME Transfer Manager", G_LOG_LEVEL_DEBUG, 
		   "CORBA user exception: %s.\n",
		   CORBA_exception_id (ev));
	    exit (1);
	default:
	    break;
    }
}


/* Open gtm */
gboolean
gtm_open (void)
{
    gboolean res;
    res = 
        (!CORBA_Object_is_nil (oaf_activate_from_id ("OAFIID:GNOME_GTM", 
                                                     0, NULL, NULL), NULL));
    return res;
}


/* Add a new URL to gtm */
GTMAppletError
gtm_add_url (gchar *url, gchar *dir, gboolean use_proxy, gboolean use_auto_dl)
{
    GTM_Download gtm_download_client;
    GTMAppletError result;

    gtm_download_client = 
    	    oaf_activate_from_id ("OAFIID:GNOME_GTM", 
    				  0, NULL, &ev);
    Exception(&ev);
    if (gtm_download_client != NULL) {
        if (GTM_Download_add_url (gtm_download_client, url, dir,
                                   !use_proxy, !use_auto_dl, &ev))
            result = GTM_ADDURL_NO_ERROR;
        else
            result = GTM_ADDURL_REPEAT_ERROR;
        if (ev._major != CORBA_NO_EXCEPTION) {
            g_print ("Exception thrown: %s\n",
                     CORBA_exception_id (&ev));
            CORBA_exception_free (&ev);
            result = GTM_ADDURL_CORBA_ERROR;
        }
    } else
        result = GTM_ADDURL_GTM_ERROR;

    return result;
}


/* Properties manipulation functions */
static void
property_load (gchar *path)
{
    gboolean def;

    /* Check to see if we have a saved session, if not used a default saved 
     * config (if any to use) */
    if (gnome_config_has_section (path))
        gnome_config_push_prefix (path);
    else
        gnome_config_push_prefix ("gtm_applet/Default/");
    gtm_applet_prop.confirm_dnd_url =
        gnome_config_get_bool ("confirm_dnd_url=FALSE");
    gtm_applet_prop.binding_button1_click =
        gnome_config_get_int ("binding_button1_click=1");
    gtm_applet_prop.presume_protocol = 
	gnome_config_get_int_with_default ("presume_protocol", &def);
    if (def)
	gtm_applet_prop.presume_protocol = PROTOCOL_HTTP;
    gtm_applet_prop.open_gtm_on_network =
        gnome_config_get_bool ("open_gtm_on_network=FALSE");
    gtm_applet_prop.interface =
        gnome_config_get_string ("interface=ppp0");
    gtm_applet_prop.chk_interf_time =
	gnome_config_get_int ("chk_interf_time=1000");
    gnome_config_pop_prefix ();
}

static void
property_save (gchar *path)
{
    gnome_config_push_prefix (path);
    gnome_config_set_bool ("gtm_applet/confirm_dnd_url",
                           gtm_applet_prop.confirm_dnd_url);
    gnome_config_set_int ("gtm_applet/binding_button1_click",
                          gtm_applet_prop.binding_button1_click);
    gnome_config_set_int ("gtm_applet/presume_protocol",
                          gtm_applet_prop.presume_protocol);
    gnome_config_set_bool ("gtm_applet/open_gtm_on_network",
                           gtm_applet_prop.open_gtm_on_network);
    gnome_config_set_string ("gtm_applet/interface",
                             gtm_applet_prop.interface);
    gnome_config_set_int ("gtm_applet/check_interf_time",
                          gtm_applet_prop.chk_interf_time);
    gnome_config_pop_prefix ();
    gnome_config_sync ();
}


/* Save applet session */
static gint 
applet_save_session (GtkWidget *widget, char *privcfgpath, char *globcfgpath)
{
    property_save (privcfgpath);

    return FALSE;
}

/* Function to check network and automaticaly start downloads if possible */
#if defined (__NetBSD__) || defined (linux)
static gint
check_network (gpointer data)
{
    struct ifreq ifr;
    int fd;
	
    fd = socket (PF_INET, SOCK_DGRAM, 0);
    if (fd == -1) {
        perror ("Couldn't open a socket");
        exit (1);
    }
    
    memset (&ifr, 0, sizeof (ifr)); 
    strcpy (ifr.ifr_name, gtm_applet_prop.interface);
    if (ioctl (fd, SIOCGIFFLAGS, &ifr) < 0) {
       close (fd);
       return FALSE;
    }
    close (fd);

    if (ifr.ifr_flags & IFF_UP) {
        /* We are connected to the network so we need to check if user want
         * us to open GTM */
        if (gtm_applet_prop.open_gtm_on_network && !already_open_gtm) {
            already_open_gtm = TRUE;
            gtm_open ();
        }
    } else
        already_open_gtm = FALSE;

    return TRUE;
}

#else
static gboolean
check_network (gpointer data)
{
    /* We don't know if we are connected on the interface so we don't start
     * anything automatically */
    return TRUE;
}

#endif

/* Command open gtm */
static void
cmd_gtm_open (AppletWidget *widget, gpointer data)
{
    GtkWidget *error;

    if (!gtm_open ()) {
	error = gnome_message_box_new (
	    _ ("Couldn't start GTM"), GNOME_MESSAGE_BOX_ERROR, 
	    GNOME_STOCK_BUTTON_OK, NULL);
	gnome_dialog_run_and_close (GNOME_DIALOG (error));        
    }
}


/* Command new URL */
static void
cmd_new_url (AppletWidget *widget, gpointer data)
{
    dialog_new ();
}


/* Command properties */
static void
cmd_properties (AppletWidget *widget, gpointer data)
{
    dialog_prop ();
}


/* Command about */
static void
cmd_about (AppletWidget *widget, gpointer data)
{
    dialog_about ();
}

/* redraw the applet pixmap because either the drag state or size changed */
static void
applet_redraw_pixmap ()
{
    GdkWindowPrivate *sock_win =
        (GdkWindowPrivate *) GTK_PLUG(applet)->socket_window;
    Window root_return;
    Window parent_return;
    Window *children_return;
    unsigned int nchildren_return;

    /* find the evil parent window */
    XQueryTree (GDK_DISPLAY (), GDK_WINDOW_XWINDOW (sock_win),
                &root_return, &parent_return, &children_return,
                &nchildren_return);
    XFree(children_return);

    if (have_drag) {
        gtk_pixmap_set (GTK_PIXMAP (pixmap), i_open, i_open_mask);
        gtk_widget_shape_combine_mask (applet, i_open_mask, 0, 0);

        /* Set the mask for our parent, too */
        XShapeCombineMask (GDK_DISPLAY (), parent_return,
                           ShapeBounding, 0, 0,
                           ((GdkWindowPrivate *)i_open_mask)->xwindow,
                           ShapeSet);
    } else {
        gtk_pixmap_set (GTK_PIXMAP (pixmap), i_closed, i_closed_mask);
        gtk_widget_shape_combine_mask (applet, i_closed_mask, 0, 0);

        /* Set the mask for our parent, too */
        XShapeCombineMask (GDK_DISPLAY (), parent_return,
                           ShapeBounding, 0, 0,
                           ((GdkWindowPrivate *)i_closed_mask)->xwindow,
                           ShapeSet);
    }
}

/* update the pixmaps for the current panel size */
static void
applet_update_pixmaps ()
{
    gchar **full_xpm;
    gchar **empty_xpm;

    if (i_open)
        gdk_pixmap_unref (i_open);
    if (i_closed)
        gdk_pixmap_unref (i_closed);

#ifdef HAVE_PANEL_PIXEL_SIZE
    switch (panel_size) {
        case PIXEL_SIZE_HUGE:
            full_xpm = tray_full_huge_xpm;
            empty_xpm = tray_empty_huge_xpm;
            break;
        case PIXEL_SIZE_LARGE:
            full_xpm = tray_full_large_xpm;
            empty_xpm = tray_empty_large_xpm;
            break;
        case PIXEL_SIZE_SMALL:
            full_xpm = tray_full_small_xpm;
            empty_xpm = tray_empty_small_xpm;
            break;
        case PIXEL_SIZE_TINY:
            full_xpm = tray_full_tiny_xpm;
            empty_xpm = tray_empty_tiny_xpm;
            break;
        case PIXEL_SIZE_STANDARD:
        default:
            full_xpm = tray_full_xpm;
            empty_xpm = tray_empty_xpm;
    }
#else
    full_xpm = tray_full_xpm;
    empty_xpm = tray_empty_xpm;
#endif
    
    i_open = gdk_pixmap_colormap_create_from_xpm_d (
        NULL,
        gtk_widget_get_colormap (applet),
        &i_open_mask, NULL, full_xpm);
    i_closed = gdk_pixmap_colormap_create_from_xpm_d (
        NULL,
        gtk_widget_get_colormap (applet),
        &i_closed_mask, NULL, empty_xpm);
}

/* Drag 'n drop stuff */
static void  
drag_leave_cb (GtkWidget *widget, GdkDragContext *context, guint time)
{
    have_drag = FALSE;
    applet_redraw_pixmap();
}

static gboolean
drag_motion_cb (GtkWidget *widget, GdkDragContext *context, gint x, gint y,
                guint time)
{
    if (!have_drag) {
        have_drag = TRUE;
        applet_redraw_pixmap();
    }

    gdk_drag_status (context, context->suggested_action, time);
    return TRUE;
}

static void
drag_data_received_cb (GtkWidget *widget, GdkDragContext *context,
                       gint x, gint y, GtkSelectionData *selection_data,
                       guint info, guint time)
{
    if ((selection_data->length >= 0) && (selection_data->format == 8)) {
        switch (info) {
            case TARGET_URL:
            case TARGET_NETSCAPE_URL:
                dialog_add (selection_data->data);
                break;
        }
        gtk_drag_finish (context, TRUE, FALSE, time);
        return;
    }  
    gtk_drag_finish (context, FALSE, FALSE, time);
}

static void
applet_cmd_by_type (GTMAppletCmd cmd)
{
    switch (cmd) {
        case GTM_ACTION_NEW_URL:
            dialog_new ();
            break;

        case GTM_ACTION_OPEN_GTM:
            cmd_gtm_open (NULL, NULL);
            break;

        case GTM_ACTION_NONE:
            break;

        default:
            break;
    }
}


/* Applet click event */
static gint
applet_clicked_cb (GtkWidget *widget, GdkEventButton *event, gpointer data)
{
    if (event != NULL && event->button == 1
        && event->type == GDK_BUTTON_PRESS) {
        applet_cmd_by_type (gtm_applet_prop.binding_button1_click);
        return TRUE;
    }

    return FALSE;
}

/* Match the panel's orientation */
static void
applet_change_orient (GtkWidget *w, PanelOrientType o, gpointer data)
{
    panel_orient = o;
}

/* Match the panel's background */
static void
applet_back_change (GtkWidget *w,
                    PanelBackType type,
                    gchar *pix,
                    GdkColor *color,
                    gpointer data)
{
    GtkStyle *ns;

    switch (type) {
        case PANEL_BACK_NONE:
            ns = gtk_style_new ();
            gtk_style_ref (ns);
            gtk_widget_set_style (applet, ns);
            gtk_style_unref (ns);
            break;

        case PANEL_BACK_COLOR:
            ns = gtk_style_copy (GTK_WIDGET (applet)->style);
            gtk_style_ref (ns);
            ns->bg[GTK_STATE_NORMAL] = *color;
            gtk_widget_set_style (GTK_WIDGET (applet), ns);
            gtk_style_unref (ns);
            break;

        case PANEL_BACK_PIXMAP:
            if (pix != NULL && g_file_exists (pix)) {
                GdkImlibImage *im = gdk_imlib_load_image (pix);
                if (im) {
                    GdkPixmap *pmap;

                    gdk_imlib_render (im, im->rgb_width, im->rgb_height);
                    pmap = gdk_imlib_copy_image (im);
                    gdk_imlib_destroy_image (im);
                    gdk_window_set_back_pixmap (applet->window, pmap,
                                                FALSE);
                    gdk_window_clear (applet->window);
                    gdk_pixmap_unref (pmap);
                }
            }
            break;

        case PANEL_BACK_TRANSLUCENT:
            break;
    }
}

#ifdef HAVE_PANEL_PIXEL_SIZE
/* Match the panel's size */
static void
applet_change_pixel_size (GtkWidget *w, int size, gpointer data)
{
    panel_size = size;
    applet_update_pixmaps();
    applet_redraw_pixmap();
}
#endif

/* Update the check interface timeout */
void
update_check_interface_timeout (guint timeout)
{
    /* Set the new value and set the already open GTM to false */
    gtm_applet_prop.chk_interf_time = timeout;
    already_open_gtm = FALSE;

    /* Update the timeout */
    if (check_network_timeout_id != -1)
        gtk_timeout_remove (check_network_timeout_id);
    check_network_timeout_id =
        gtk_timeout_add (gtm_applet_prop.chk_interf_time, check_network, NULL);
}


/* Main program */
int 
main (int argc, char *argv[])
{
    static GtkTargetEntry target_table[] = {
        { "text/uri-list", 0, TARGET_URL },
        { "text/plain", 0, TARGET_URL },
        { "x-url/http", 0, TARGET_NETSCAPE_URL },
#ifdef ENABLE_SSL
	{ "x-url/https", 0, TARGET_NETSCAPE_URL },
#endif
        { "x-url/ftp", 0, TARGET_NETSCAPE_URL },
        { "_NETSCAPE_URL", 0, TARGET_NETSCAPE_URL }
    };

    /* Initialize the i18n stuff */
    bindtextdomain (PACKAGE, GNOMELOCALEDIR);
    textdomain (PACKAGE);

    /* Initialize CORBA stuff */
    CORBA_exception_init (&ev);
    oaf_init(argc, argv);

   
    /* Initialize the applet */
    applet_widget_init ("gtm_applet", VERSION, argc, argv, NULL, 0, NULL);
    gnome_window_icon_set_default_from_file (GNOME_ICONDIR "/gtm_applet.png");
    applet = applet_widget_new ("gtm_applet");
    if (!applet)
	g_error ("Can't create applet!\n");
    gtk_widget_set_events (applet, GDK_BUTTON_PRESS_MASK);
    gtk_widget_realize (applet);
    property_load (APPLET_WIDGET (applet)->privcfgpath);

    /* Load pixmaps */
    i_open = 0;
    i_closed = 0;
    applet_update_pixmaps();
                                               
    /* Create widget eventbox with a pixmap for to new URL */
    eventbox = gtk_event_box_new ();
    pixmap = gtk_pixmap_new (i_closed, i_closed_mask);
    gtk_container_add (GTK_CONTAINER (eventbox), pixmap);
    gtk_widget_show (pixmap);
    gtk_signal_connect (GTK_OBJECT (eventbox), "button_press_event", 
			GTK_SIGNAL_FUNC (applet_clicked_cb), NULL);
    applet_widget_set_widget_tooltip (
        APPLET_WIDGET(applet), eventbox, 
        _ ("Drop URL from netscape, gmc, gftp..."));
    gtk_widget_show (eventbox);

    /* Set the drag 'n drop stuff */
    gtk_drag_dest_set (pixmap, GTK_DEST_DEFAULT_MOTION 
		       | GTK_DEST_DEFAULT_DROP,
                       target_table, 
                       sizeof (target_table) / sizeof (target_table[0]),
                       GDK_ACTION_COPY);
    gtk_signal_connect (GTK_OBJECT (pixmap), "drag_leave",
                        GTK_SIGNAL_FUNC (drag_leave_cb), NULL);
    gtk_signal_connect (GTK_OBJECT (pixmap), "drag_motion",
                        GTK_SIGNAL_FUNC (drag_motion_cb), NULL);
    gtk_signal_connect (GTK_OBJECT (pixmap), "drag_data_received",
                        GTK_SIGNAL_FUNC(drag_data_received_cb), NULL);

    /* Handle panel background change */
    gtk_signal_connect (GTK_OBJECT (applet),"back_change",
                        GTK_SIGNAL_FUNC (applet_back_change), NULL);

    /* Handle panel orientation change */
    gtk_signal_connect (GTK_OBJECT (applet),"change_orient",
                        GTK_SIGNAL_FUNC (applet_change_orient), NULL);

#ifdef HAVE_PANEL_PIXEL_SIZE
    /* Set panel resize handler */
    gtk_signal_connect (GTK_OBJECT (applet), "change_pixel_size",
                        GTK_SIGNAL_FUNC (applet_change_pixel_size),
                        NULL);
#endif

    /* Add the eventbox to the applet and show the applet */
    applet_widget_add (APPLET_WIDGET (applet), eventbox);
    gtk_widget_show (applet);

    /* Session management stuff */
    gtk_signal_connect (GTK_OBJECT (applet),"save_session",
			GTK_SIGNAL_FUNC (applet_save_session), NULL);

    /* Add the menu items */
    applet_widget_register_stock_callback (APPLET_WIDGET (applet),
					   "properties",
					   GNOME_STOCK_MENU_PROP,
					   _ ("Properties..."),
					   cmd_properties, 
					   NULL);
    applet_widget_register_stock_callback (APPLET_WIDGET (applet),
					   "about",
					   GNOME_STOCK_MENU_ABOUT,
					   _ ("About..."),
					   cmd_about,
					   NULL);
    applet_widget_register_stock_callback (APPLET_WIDGET (applet),
					   "new",
					   GNOME_STOCK_MENU_NEW,
					   _ ("New URL"),
					   cmd_new_url, 
					   NULL);
    applet_widget_register_stock_callback (APPLET_WIDGET (applet),
					   "gtm",
					   GNOME_STOCK_PIXMAP_EXEC,
					   _ ("Open GTM"),
					   cmd_gtm_open, 
					   NULL);

    /* Install the timer */
    already_open_gtm = FALSE;
    check_network_timeout_id = 
        gtk_timeout_add (gtm_applet_prop.chk_interf_time, check_network, NULL);

    applet_widget_gtk_main();

    /* Destroy the timer */
    gtk_timeout_remove (check_network_timeout_id);
    
    return 0;
}
