/* piping functions for data filters and cd writing */

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <gtk/gtk.h>

#include "piping.h"
#include "varman.h"
#include "preferences.h"
#include "rterm.h"
#include "calc.h"
#include "stdfiletrack.h"
#include "dependencies.h"

#define FDREAD 0
#define FDWRITE 1

/* uncomment for debugging */
// #define DEBUG
//
int piping_pidwatch_callback(gpointer data)
{
   piping_watch *w;
   int callagain;

   w=(piping_watch*)data;
   if (waitpid(w->pid,&w->status,WNOHANG)!=0)
     {
	callagain=w->finishcb(w->status,w->data);
     }
   else
     callagain=1;
   if (callagain==0) piping_pidwatch_remove(w);

   return callagain;
}
;

piping_watch *piping_pidwatch_add(int pid,
				  piping_finishcb callback,
				  gpointer data)
{
   piping_watch *w;

   w=(piping_watch*)malloc(sizeof(piping_watch));
   w->pid=pid;
   w->status=0;
   w->finishcb=callback;
   w->data=data;
   w->handle=gtk_timeout_add(PIPING_PIDWATCH_TIMEOUT,
			     piping_pidwatch_callback,
			     (gpointer)w);

   return w;
}
;

void piping_pidwatch_remove(piping_watch *w)
{
   gtk_timeout_remove(w->handle);
   free(w);
}
;

typedef struct
{
   char *command;
   int waitfordata;
}
piping_create_info;

/* callback called by piping_create_function() when invoked with
 * piping_create()
 * if waitfordata is true,this function will wait for data to arrive
 * at stdin before control is handed down to the client program */
int piping_create_callback(int inpipe,
			   int outpipe,
			   int errpipe,
			   gpointer data)
{
   char *command;
   char *args[PIPING_MAX_ARGS+1];
   char *argstring;
   int  currentpos;
   int  x;
   int  specialchar;
   int  quoted;
   int  sensitive;
   int  waitfordata;
   fd_set rfds;

   command=((piping_create_info*)data)->command;
   waitfordata=((piping_create_info*)data)->waitfordata;
   argstring=(char*)malloc(strlen(command)+1);
   strcpy(argstring,command);  // make a copy first,then modify it appropriately

#ifdef DEBUG
   printf ("commandstring: %s\n",argstring);
#endif

   for (x=0;x<PIPING_MAX_ARGS+1;x++) /* initialize parameters */
     args[x]=NULL;
   x=0;currentpos=0;
   quoted=0;sensitive=1;specialchar=0;
   args[x]=(char*)malloc(256);
   args[x][0]=0;
   do
     {
	specialchar=0;
	if (sensitive)
	  {
	     switch (argstring[currentpos])
	       {
		case '\\':
		  specialchar=1;
		  sensitive=0;
		  break;
		case '"':
		  specialchar=1;
		  quoted=1-quoted;
		  break;
		case ' ':
		  if ((!quoted)&&(x<PIPING_MAX_ARGS-1))
		    {
		       specialchar=1;
		       x++;
		       args[x]=(char*)malloc(256);
		       args[x][0]=0;
		    }
		  ;
		  break;
	       }
	     ;
	  }
	else
	  sensitive=1;
	if (!specialchar)
	  strncat(args[x],&argstring[currentpos],1);
	currentpos++;
     }
   while (currentpos<strlen(argstring));
	/* last entry has to be a NULL pointer */
	/* due to preinitialization this is generally the case */

#ifdef DEBUG
   printf ("calling with args:\n");
   for (x=0;args[x]!=NULL;x++)
     {
	printf ("%s\n",args[x]);
     }
   ;
#endif

   if ((inpipe!=-1)&&(inpipe!=0))
     {
	close (0);
	dup(inpipe);
	close(inpipe);
     }
   ;

   /* see errpipe for a comment about why we don't do anything in case
    * outpipe has the value 1 */
   if ((outpipe!=-1)&&(outpipe!=1))
     {
	close (1);
	dup(outpipe);
	close(outpipe);
     }
   ;

   /* the !=2 check is necessary to catch the event that someone would
    * want the stderr output of our client to be sent to the stderr of our
    * parent's process (which we inherit automatically just by doing
    * nothing) */
   if ((errpipe!=-1)&&(errpipe!=2))
     {
	close (2);
	dup(errpipe);
	close(errpipe);
     }
   ;

   if (waitfordata)
     {
	FD_ZERO(&rfds);
	FD_SET(0,&rfds);
	select(1,&rfds,NULL,NULL,NULL);
     };

   execvp(args[0],args);
   perror ("couldnt run client");
   _exit(-1);
}
;

/* The new "light" version of piping_create(). it utilizes piping_create_function,
 * effectively contracting all fork and piping calls to a single function
 * piping_create_function. Thus future big in the piping routines have
 * to be fixed only once
 *    command: program to start and parameters separated by spaces
 *    *in:     thats where the filedescriptor for sending data to the
 *             client will be written to
 *             if *in is set to something other than -1 it is assumed that
 *             that filedescriptor represented by *in is already open and
 *             should be used to read data by the client
 *    *out:    the filedescriptor with which you can obtain data from the client
 *             will be written here
 *    *err:    the same for the error channel of the client
 * any of the pointers in,out and err set to NULL means that youre not
 * interested in communicating with the client over that pipe.
 * the corresponding pipe of the client will be connected to /dev/zero in that
 * case */
int piping_create(char *command,int *in,int *out,int *err)
{
   piping_create_info info;
   int pid=-1;

   info.command=command;
   info.waitfordata=0;

   pid=piping_create_function(GTK_SIGNAL_FUNC(piping_create_callback),
			      (gpointer)&info,
			      in,out,err);
   return pid;
}
;

/* same as above,just wait for data to arrive at client's stdin
 * before actually running it */
int piping_create_wait(char *command,int *in,int *out,int *err)
{
   piping_create_info info;

   info.command=command;
   info.waitfordata=1;

   return piping_create_function(GTK_SIGNAL_FUNC(piping_create_callback),
				 (gpointer)&info,
				 in,out,err);
}
;

/* create a pipe to part of the main program (essentially a fork with
 * a few pipes for communication with the main process)
 * if *in,*out,*err is -1 a new pipe will be created
 * if in,out,err is NULL the corresponding pipe is connected to /dev/zero
 * if *in,*out,*err !=-1 we take that pipe as input,output or stderror
 * if *in,*out,*err !=-1 we return -1 as there is no way of controlling
 * a direct link to a client.
 * WARNING! the filedescriptors in *in,*out,*err
 * are closed on the callers side. */
int piping_create_function(GtkSignalFunc func,gpointer data,int *in,int *out,int *err)
{
   int inpipe[2];
   int outpipe[2];
   int errpipe[2];

   int childpid;

   if (in!=NULL)
     {
	/* create new input pipe only if value of in is -1 */
	if (*in==-1)
	  {
	     if (pipe(inpipe)==-1)
	       {
		  perror("piping_create_func couldnt create inpipe");
		  inpipe[FDREAD]=-1;
		  inpipe[FDWRITE]=-1;
	       }
	     ;
	  }
	else
	  {
	     inpipe[FDREAD]=*in;
	     /* if inpipe was open theres no way of
	      * controlling the data flow to the child */
	     inpipe[FDWRITE]=-1;
	  }
	;
     }
   else
     {
	inpipe[FDREAD]=open("/dev/zero",O_RDONLY);
	inpipe[FDWRITE]=-1;
     }
   ;
   if (out!=NULL)
     {
	/* create new output pipe only if value of out is -1 */
	if (*out==-1)
	  {
	     if (pipe(outpipe)==-1)
	       {
		  perror("piping_create_func couldnt create outpipe");
		  outpipe[FDREAD]=-1;
		  outpipe[FDWRITE]=-1;
	       }
	     ;
	  }
	else
	  {
	     outpipe[FDREAD]=-1;
	     outpipe[FDWRITE]=*out;
	  }
	;

     }
   else
     {
	outpipe[FDREAD]=-1;
	outpipe[FDWRITE]=open("/dev/zero",O_WRONLY);
     }
   ;
   if (err!=NULL)
     {
	/* create new error pipe only if value of out is -1 */
	if (*err==-1)
	  {
	     if (pipe(errpipe)==-1)
	       {
		  perror("piping_create_func couldnt create errpipe");
		  errpipe[FDREAD]=-1;
		  errpipe[FDWRITE]=-1;
	       }
	     ;
	  }
	else
	  {
	     errpipe[FDREAD]=-1;
	     errpipe[FDWRITE]=*err;
	  }
	;
     }
   else
     {
	/* when in debugging mode, set stderror of child process to
	 * stderror of parent process */
	errpipe[FDREAD]=-1;
	errpipe[FDWRITE]=open("/dev/zero",O_WRONLY);
     }
   ;
   childpid=fork();
   switch (childpid)
     {
      case 0:
	/* this section is a bit different to the implementation
	 * of the normal pipe function. It doesnt redirect stdin
	 * stdout and stderr but merely gives the descriptors used
	 * for communication with the same names and meanings
	 * as arguments to the function.
	 * Thus,the client function may still use the stdin,stdout
	 * and stderr of the main program while at the same time
	 * having access to the pipes created for communication
	 * with the main process. */

	if (inpipe[FDWRITE]!=-1)      /* only client side pipes are needed here */
	  close (inpipe[FDWRITE]);
	if (outpipe[FDREAD]!=-1)
	  close (outpipe[FDREAD]);
	if (errpipe[FDREAD]!=-1)
	  close(errpipe[FDREAD]);

	func(inpipe[FDREAD],
	     outpipe[FDWRITE],
	     errpipe[FDWRITE],
	     data);

	_exit(0);
      case -1:
	perror ("forking error");
	exit(-1);
      default:
	
	if (inpipe[FDREAD]!=-1)
	  close(inpipe[FDREAD]);
	if (outpipe[FDWRITE]!=-1)
	  close(outpipe[FDWRITE]);
	if (errpipe[FDWRITE]!=-1)
	  close(errpipe[FDWRITE]);
	if (in!=NULL)
	  *in=inpipe[FDWRITE];
	if (out!=NULL)
	  *out=outpipe[FDREAD];
	if (err!=NULL)
	  *err=errpipe[FDREAD];

	return childpid;
     }
   ;
}
;

/* write the output of a command into *buf */
void piping_create_getoutput(char *command,char *buf,int maxoutput,char watchmask)
{
   int outp=-1,errp=-1;
   int *outpp,*errpp;
   int pid;
   int count;
   int status;
   int readresult_err;
   int readresult_out;

   if (watchmask&PIPING_WATCHSTDOUT)
     outpp=&outp;
   else
     outpp=NULL;

   if (watchmask&PIPING_WATCHSTDERR)
     errpp=&errp;
   else
     errpp=NULL;

#ifdef DEBUG
   printf ("outpp:%p errpp:%p\n",outpp,errpp);
#endif
   pid=piping_create(command,NULL,outpp,errpp);
#ifdef DEBUG
   printf ("outpp:%p errpp:%p\n",outpp,errpp);
   printf ("outp:%d errp:%d\n",outp,errp);
#endif
   count=0;

   if (watchmask&PIPING_WATCHSTDOUT)
     fcntl(outp,F_SETFL,O_NONBLOCK);
   if (watchmask&PIPING_WATCHSTDERR)
     fcntl(errp,F_SETFL,O_NONBLOCK);

   do
     {
	readresult_out=0;
	if  (watchmask&PIPING_WATCHSTDOUT)
	  {
	     if ((readresult_out=read(outp,&buf[count],1))>0)
	       {
		  if (count<maxoutput-1)
		    {
#ifdef DEBUG
		       printf ("%c",buf[count]);
#endif
		       count++;
		    }
		  ;
	       }
	     ;
	  }
	;
#ifdef DEBUG
	if ((readresult_out==-1) && (errno!=EAGAIN))
	  perror ("piping_create_getoutput: error while reading from stdoutpipe");
#endif
	readresult_err=0;
	if   (watchmask&PIPING_WATCHSTDERR)
	  {
	     if ((readresult_err=read(errp,&buf[count],1))>0)
	       {
		  if (count<maxoutput-1)
		    {
#ifdef DEBUG
		       printf ("%c",buf[count]);
#endif
		       count++;
		    }
		  ;
	       }
	     ;
	  }
	;
#ifdef DEBUG
	if ((readresult_err==-1) && (errno!=EAGAIN))
	  perror ("piping_create_getoutput: error while reading from stderrpipe");
#endif
     }
   while ((readresult_out!=0)||(readresult_err!=0));

   if (watchmask&PIPING_WATCHSTDOUT)
     close(outp);
   if (watchmask&PIPING_WATCHSTDERR)
     close(errp);

   waitpid(pid,&status,0); /* wait for the client to finish */

   buf[count]=0; /* terminate the output with 0 to create a valid string */
}
;

#define SCANTEXTBUFFERSIZE 16384

int piping_create_scantextoutput(char *command,
				 char *function,char watchmask)
{
   int result=0;
   int outp=-1,errp=-1;
   int *outpp=NULL,*errpp=NULL;
   FILE *fd=NULL;
   int cmdpid=-1;
   int pidstatus=0;

   OUTBUF_LOCK;

   /* there can be only one */
   if ((watchmask&PIPING_WATCHSTDOUT)&&(watchmask&PIPING_WATCHSTDERR))
     return 0;
   if (watchmask&PIPING_WATCHSTDOUT)
     outpp=&outp;
   if (watchmask&PIPING_WATCHSTDERR)
     errpp=&errp;
   cmdpid=piping_create(command,NULL,outpp,errpp);

   fd=fdopen((watchmask&PIPING_WATCHSTDOUT)?outp:errp,
	     "r");
   if (fd)
     {
	while ((!result)&&(!feof(fd)))
	  {
	     outbuf[0]=0;
	     fgets(outbuf,SCANTEXTBUFFERSIZE,fd);
	     result=calc_function(function,0);
#ifdef DEBUG
	     printf("piping_create_scantextoutput: scanning '%s',result=%i\n",
		    outbuf,
		    result);
#endif
	  };
	fclose(fd);
     };
   kill(cmdpid,SIGTERM);
   if (cmdpid!=-1)
     waitpid(cmdpid,&pidstatus,0);

   OUTBUF_UNLOCK;
   return result;
};

/* is the absolute pointer to exec aiming at a valid executable ?
 * returns true if valid */
int piping_isvalid_exec_absolute(char *exec)
{
   struct stat buf;
   int result=0;

#ifdef DEBUG
   printf("piping_isvalid_exec_absolute: looking for executable '%s'\n",exec);
#endif
   if (!stat(exec,&buf))
     {
	/* FIXME: This only checks if *anybody* has exec permission on
	 * this.
	 * This is not correct. We'd have to check if we're the owner,
	 * we're in a group with the owner or if the file is generally
	 * executable */
	result=((buf.st_mode&(S_IXUSR|S_IXGRP|S_IXOTH))>0);
     };
   return result;
};

int piping_isvalid_exec(char *exec)
{
   int result=0;
   if (!strlen(exec))
     /* No filename is invalid */
     return 0;
   else
     {
	if (strchr(exec,'/'))
	  result=piping_isvalid_exec_absolute(exec);
	else
	  {
	/* get current executable path */
	     char *orig_env=getenv("PATH");
	     if (orig_env)
	       {
		  char *env;
		  char *current;
	     /* make local copy */
		  env=strdup(orig_env);
		  current=env;
		  do
		    {
		       char *absolute;
		       char *next=strchr(current,':');
		       if (!next)
			 next=(char*)(strlen(current)+(unsigned int)current);
		       *next=0;
		       next++;

		       absolute=malloc(strlen(current)+strlen(exec)+2);
		       strcpy(absolute,current);
		       if (absolute[strlen(absolute)-1]!='/')
			 strcat(absolute,"/");
		       strcat(absolute,exec);
		       result|=piping_isvalid_exec_absolute(absolute);
		       free(absolute);

		       current=next;
		    }
		  while (strlen(current));
		  free(env);
	       };
	  };
     };
   return result;
};

int piping_isvalid_command(char *command)
{
   int result=0;
   char *exec=strdup(command);
   if (strchr(exec,' '))
     *strchr(exec,' ')=0;
#ifdef DEBUG
   printf ("piping_isvalid_command: checking for executable '%s'\n",exec);
#endif
   result=piping_isvalid_exec(exec);
   free(exec);
   return result;
};

int piping_isvalid_commandchain(char *chain,const char *message)
{
   char *local=strdup(chain);
   char *current=local;
   char *next=NULL;
   int  result=1;
   while (current)
     {
	int myresult=0;
	next=strchr(current,'|');
	if (next)
	  {
	     *next=0;
	     ++next;
	  };
	myresult=(piping_isvalid_command(current));
	if ((!myresult)&&(message))
	  dependencies_showdep(current,message);

	result&=myresult;
	current=next;
     };
   free(local);
   return result;
};

int piping_finishchainitemcb(int status,void *data)
{
   return 0; // stop pipewatch
};

int piping_createcommandchain(char *commandchain,int *in,int *out)
{
   char *local=strdup(commandchain);
   char *current=local;
   char *next=NULL;
   int  lastpid=-1;
   int  *inp=in;
   int  myin=-1;
   int  myout=-1;
   int  *outp=&myout;
   do
     {
	next=strchr(current,'|');
	if (next)
	  {
	     *next=0;
	     ++next;
	  }
	else
	  outp=out; // connect last element
#ifdef DEBUG	
	printf("Passing fds %i,%i\n",inp?*inp:-1,outp?*outp:-1);
#endif	
	lastpid=piping_create(current,inp,outp,NULL);
#ifdef DEBUG	
	printf("Got fds %i,%i\n",inp?*inp:-1,outp?*outp:-1);
#endif	

	if ((lastpid!=-1)&&next)
	  {
	     piping_pidwatch_add(lastpid,
				 piping_finishchainitemcb,
				 NULL);
	  };
	myin=myout;myout=-1;
	inp=&myin; // ok, connect next command's inp to current command's outp
	current=next;
     }
   while (current&&(lastpid!=-1));
   free(local);
#ifdef DEBUG   
   printf("At the end, fds are %i,%i\n",in?*in:-1,out?*out:-1);
#endif   
   return lastpid;
};

