/* This file implements a generic definition of a track list 
 * it also provides some handler functions to manipulate that list */

#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>

#include "tracklist.h"
#include "selectlist.h"
#include "menusys.h"
#include "tracks.h"
#include "streams.h"
#include "helpings.h"
#include "updatehandlers.h"
#include "main.h"

#include "ttaudio.xpm"
GdkPixmap *ttaudioxpm;
GdkBitmap *ttaudiomask;

#include "ttdata.xpm"
GdkPixmap *ttdataxpm;
GdkBitmap *ttdatamask;

GdkPixmap *tticon;
GdkBitmap *tticonmask;

/* uncomment for debugging */
/* #define DEBUG */

static int tracklist_resize(GtkWidget *column,
			    GtkAllocation *alloc,
			    gpointer data)
{
   #ifdef DEBUG
      printf ("tracklist_resize: got resize event for clist\n");
   #endif
      gtk_clist_set_column_width(GTK_CLIST(column),0,alloc->width-(96+100));
      gtk_clist_set_column_width(GTK_CLIST(column),1,96);
   
      return TRUE;
};

/* this function gets called by any of the tracks stored in the tracklist 
 * whenever their properties change */
void tracklist_updatehandler(tracklist_info*tracklist)
{
   char sizebuf[40];
   char *trackent[2];
   tracks_trackinfo *track;

   int i;

   #ifdef DEBUG
     printf ("tracklist_updatehandler: updating display due to a track updateevent\n");
   #endif
   /* update display */
   for (i=0;i<tracklist->entries;i++)
     {	   
	track=streams_getstreambyid((char*)gtk_clist_get_row_data(tracklist->selectlistinfo->self,i));
	trackent[0]=track->name;  
	helpings_formatnumber(sizebuf,20,tracks_audiotracksize(track));
	strcat(sizebuf," (");
	strcat(sizebuf,helpings_secs2hms(tracks_audiotracksize(track)/(44100*4),0));
	strcat(sizebuf,")");   
	trackent[1]=sizebuf;	
	tticon=ttdataxpm;
	tticonmask=ttdatamask;
	/* give audio tracks the right look */
	if (!strcasecmp(track->tracktype,"audio"))
	  {
	     tticon=ttaudioxpm;
	     tticonmask=ttaudiomask;
	  };       
	gtk_clist_set_pixtext(GTK_CLIST(tracklist->selectlistinfo->self),
			      i,0,
			      track->name,
			      5,
			      tticon,
			      tticonmask);
        gtk_clist_set_text(GTK_CLIST(tracklist->selectlistinfo->self),
			   i,1,
	                   trackent[1]);
     };
   
   /* call the updatehandlers of interested functions like the
    * fillstate display */
   updatehandlers_call(tracklist->updatehandlers);
};

/* call this function at the beginning of each dropcallback-handler
 * coping with tracklists. its return value is 1 if the drop was
 * tracklist internal and 0 if the drop data came from outside the
 * tracklist. So essentially the dropcallback-handler should execute
 * its main task only if tracklist_handleinternaldrop returned 0.
 * p is the position within the selectlist,where the drop item was
 * placed by the user 
 * FIXME: this function currently works with one selected item only
 * the problem inherent in processing multiple tracks is that
 * the clist changes whenever one item is processed so that the locations
 * of all the following items change as well */
int tracklist_handleinternaldrop(tracklist_info*i,int p)
{
   int linetomove,destination;
   
   if ((i->selectlistinfo->is_drag_source)&&(i->selectlistinfo->selected_lines!=NULL))
     {
	#ifdef DEBUG
	  printf ("tracklist_handleinternaldrop: moving row %i to position %i\n",
		  (int)i->selectlistinfo->selected_lines->data,
		  p);
	#endif

       	linetomove=(int)i->selectlistinfo->selected_lines->data;
	
	/* if item was dropped outside the tracklist,simply append it */
	if (p==-1)
	  destination=i->entries+1;
	else
	  destination=p;	  
	
	/* we got to do that as a row_move messes all our informations
	 * about selections. They all dont resemble reality any longer */
	gtk_clist_unselect_all(GTK_CLIST(i->selectlistinfo->self));
	
	gtk_clist_row_move(GTK_CLIST(i->selectlistinfo->self),
			   linetomove,
			   destination);
	/* call the updatehandlers of interested functions like the
	 * fillstate display */
	updatehandlers_call(i->updatehandlers);
     }
   ;
   return i->selectlistinfo->is_drag_source;
}
;

const char *titledisplay_layout_at=N_("%s by %s");
const char *titledisplay_layout_a =N_("Medium by %s");
const char *titledisplay_layout_t =N_("%s");

void tracklist_updatetitledisplay(tracklist_info *info)
{
   char *constrdisplay = NULL;
   const char *newdisplay = info->titledefaultheader;
   if (info->author&&info->title)
     {
	constrdisplay = (char*)malloc(strlen(info->author)+strlen(info->title)+strlen(titledisplay_layout_at));
	sprintf(constrdisplay,_(titledisplay_layout_at),info->title,info->author);
     }
   else
     {
	if (info->author)
	  {
	     constrdisplay = (char*)malloc(strlen(info->author)+strlen(titledisplay_layout_a));
	     sprintf(constrdisplay,_(titledisplay_layout_a),info->author);
	  };
	if (info->title)
	  {
	     constrdisplay = (char*)malloc(strlen(info->title)+strlen(titledisplay_layout_t));
	     sprintf(constrdisplay,_(titledisplay_layout_t),info->title);
	  };
     };
   if (constrdisplay)
     newdisplay = constrdisplay;
   if (!newdisplay)
     newdisplay = _("Trackname");
   gtk_clist_set_column_title(info->selectlistinfo->self, 
			      0, newdisplay);
   if (constrdisplay)
     free(constrdisplay);
};

void tracklist_setauthortitleinfo(tracklist_info *info,
				  const char *newauthor,
				  const char *newtitle)
{
   if (info->author)
     free(info->author);
   if (info->title)
     free(info->title);
   info->author=((newauthor&&strlen(newauthor))?strdup(newauthor):NULL);
   info->title =((newtitle&&strlen(newtitle))?strdup(newtitle) :NULL);
   tracklist_updatetitledisplay(info);
};				  

/* if you want to allow track rearranging,both dropcallback and send have
 * to be enabled (dropcallback has to point to a valid handler) */
tracklist_info *tracklist_info_create(selectlist_dropcallback dropcallback,
		       	  	      int send,
				      menusys_menu *popup,
				      GtkSignalFunc doubleclick,
				      const char *titledefaultheader)
{
   tracklist_info *info;
   GtkWidget *content;
   
   const char *headings[3];
   
   info=(tracklist_info*)malloc(sizeof(tracklist_info));
   headings[0]=((titledefaultheader)?titledefaultheader:"");
   headings[1]=_("Tracksize");
   headings[2]=NULL;
   
   /* -1 is the data field of the clist */
   info->selectlistinfo=selectlist_info_create("stream:",-1,
					       dropcallback,
					       send,popup,doubleclick,
					       NULL,	
					       headings);
   /* No author/title infos yet */
   info->author = NULL;
   info->title  = NULL;
   info->titledefaultheader=((titledefaultheader)?strdup(titledefaultheader):NULL);   
   
   content=selectlist_create(info->selectlistinfo,2);

   gtk_signal_connect(GTK_OBJECT(content),"size_allocate",
		      GTK_SIGNAL_FUNC(tracklist_resize),NULL);
      
   gtk_clist_set_selection_mode(GTK_CLIST(content),GTK_SELECTION_EXTENDED);
   gtk_clist_set_column_justification(GTK_CLIST(content),0,GTK_JUSTIFY_LEFT);
   gtk_clist_set_column_justification(GTK_CLIST(content),1,GTK_JUSTIFY_RIGHT);
   
   info->widget=gtk_scrolled_window_new(NULL,NULL);
   gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(info->widget),
				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_AUTOMATIC);
   gtk_widget_show(content);
   /* Warning! Its not a bug but a feature that we do a CONTAINER_ADD instead
    * of a SCROLLED_WINDOW_ADD_WITH_VIEWPORT(). This way,the title section
    * of the clist does not get scrolled as the Viewport management is done
    * by the CList itself */
   gtk_container_add(GTK_CONTAINER(info->widget),
		     content);

   /* no entries to the tracklist yet */
   info->entries=0;
   /* no handlers installed yet */
   info->updatehandlers=NULL;

   tracklist_updatetitledisplay(info);
   
   return info;
};


void tracklist_inserttrack(tracklist_info *tracklist,int n,
         		   tracks_trackinfo *track)
{
   char *trackent[2];
   char sizebuf[40];
   char *streamid;
     
   #ifdef DEBUG
     printf ("tracklist_addtrack: claiming track \n");
   #endif
     
   /* get a streamid and claim() the track for our list at the same time */
   streamid=streams_getstreamid(track);

     
   /* build the clist entry for this track */
   trackent[0]=track->name;
   helpings_formatnumber(sizebuf,20,tracks_audiotracksize(track));
   strcat(sizebuf," (");
   strcat(sizebuf,helpings_secs2hms(tracks_audiotracksize(track)/(44100*4),0));
   strcat(sizebuf,")");   
   trackent[1]=sizebuf;
   
   #ifdef DEBUG
     printf ("tracklist_addtrack: adding stream %s size %s\n",
	     track->name,
	     sizebuf);
   #endif

   if (n!=-1)
     {
        selectlist_insert(tracklist->selectlistinfo,n,
			  trackent);
     }
   else
     {
	n=tracklist->entries;
	gtk_clist_append(GTK_CLIST(tracklist->selectlistinfo->self),trackent);
     };   
   gtk_clist_set_row_data(GTK_CLIST(tracklist->selectlistinfo->self),
			  n,
			  streamid);
   
   tticon=ttdataxpm;
   tticonmask=ttdatamask;
   /* give audio tracks the right look */
   if (!strcasecmp(track->tracktype,"audio"))
     {
	tticon=ttaudioxpm;
	tticonmask=ttaudiomask;
     };       

   gtk_clist_set_pixtext(GTK_CLIST(tracklist->selectlistinfo->self),
			 n,0,
			 track->name,
			 5,
			 tticon,
			 tticonmask);
   
   /* install the update handler responsible for both keeping the display
    * up to date and serving its update clients */
   updatehandlers_register(&track->updatehandlers,
			   (updatehandlers_handler)tracklist_updatehandler,
			   (gpointer)tracklist);
   tracklist->entries++;   
   /* call the updatehandlers of interested functions like the
    * fillstate display */
   updatehandlers_call(tracklist->updatehandlers);
};

void tracklist_addtrack(tracklist_info *tracklist,
			tracks_trackinfo *track)
{
   tracklist_inserttrack(tracklist,-1,track);
};

void tracklist_removetrack(tracklist_info *tracklist,int n)
{
   tracks_trackinfo *track;
   char *streamid=(char*)gtk_clist_get_row_data(tracklist->selectlistinfo->self,n);
   
   track=streams_getstreambyid(streamid);
   free(streamid);
			       
   #ifdef DEBUG
     printf ("tracklist_removetrack: removing stream %s\n",
	     (char*)track->name);
   #endif
     
   updatehandlers_unregister(&track->updatehandlers,
			     (updatehandlers_handler)tracklist_updatehandler,
			     (gpointer)tracklist);
   tracks_unclaim(track);
   
   selectlist_remove(tracklist->selectlistinfo,
	             n);
   /* decrease number of tracklist entries by one */
   tracklist->entries--;
   /* call the updatehandlers of interested functions like the
    * fillstate display */
   updatehandlers_call(tracklist->updatehandlers);
};

/* replace a track within the tracklist by a new one.
 * this function will be used extensively by the precaching mechanism */
void tracklist_replacetrack(tracklist_info *tracklist,int n,
			    tracks_trackinfo *track)
{
   tracklist_removetrack(tracklist,n);
   tracklist_inserttrack(tracklist,n,track);
   /* call the updatehandlers of interested functions like the
    * fillstate display */
   updatehandlers_call(tracklist->updatehandlers);
};

/* clear a whole tracklist */
void tracklist_clear(tracklist_info *tracklist)
{
   tracks_trackinfo *track;
   int i;

   /* delete invalid selection informations */
   gtk_clist_unselect_all(GTK_CLIST(tracklist->selectlistinfo->self));
			  
   for (i=0;i<tracklist->entries;i++)
     {
	char *streamid=(char*)gtk_clist_get_row_data(tracklist->selectlistinfo->self,i);
	track=streams_getstreambyid(streamid);
	free(streamid);
			       
	updatehandlers_unregister(&track->updatehandlers,	
				  (updatehandlers_handler)tracklist_updatehandler,
				  (gpointer)tracklist);	
	tracks_unclaim(track);
     };
   gtk_clist_clear(tracklist->selectlistinfo->self);
   /* reset tracklist entry count */
   tracklist->entries=0;
   /* call the updatehandlers of interested functions like the
    * fillstate display */
   updatehandlers_call(tracklist->updatehandlers);
};
   
/* get track n of the track list */
tracks_trackinfo *tracklist_gettrack(tracklist_info *tracklist,int n)
{
   tracks_trackinfo *track;
   
   if (n<tracklist->entries)     
     track=streams_getstreambyid((char*)gtk_clist_get_row_data(tracklist->selectlistinfo->self,n));
   else
     track=NULL;
   
   #ifdef DEBUG
     if (track!=NULL)
       printf ("tracklist_gettrack: processed track request for %s (track %i)\n",
	       track->name,n);
   #endif
     return track;
};

int tracklist_hastrack(tracklist_info *tracklist,tracks_trackinfo *track)
{
   int i;
   int result=0;
   for (i=0;(i<tracklist->entries)&&(!result);++i)
     if (track==streams_getstreambyid((char*)gtk_clist_get_row_data(tracklist->selectlistinfo->self,i)))
       result=1;
   return result;
};

/* destroy a tracklist */
void tracklist_info_destroy(tracklist_info*tracklist)
{
   if (tracklist->title)
     free(tracklist->title);
   if (tracklist->author)
     free(tracklist->author);
   
   #ifdef DEBUG
     printf("tracklist_info_destroy: clearing tracklist\n");
   #endif
   /* unclaim() all the tracks first */
   tracklist_clear(tracklist);
   /* destroy widgets */
   #ifdef DEBUG
     printf("tracklist_info_destroy: destroying widget\n");
   #endif
   gtk_widget_destroy(tracklist->widget);
   /* free selectlist info structure*/
   #ifdef DEBUG
     printf("tracklist_info_destroy: deleting selectlistinfo\n");
   #endif
   selectlist_info_destroy(tracklist->selectlistinfo);
   /* free the list of update handlers */
   g_list_free(tracklist->updatehandlers);
   /* free the tracklist info structure itself */
   free(tracklist);
};

void tracklist_init()
{
   GtkStyle *style;
   
   style=gtk_widget_get_style(window);
   ttaudioxpm=gdk_pixmap_create_from_xpm_d(window->window,
					   &ttaudiomask,
					   &style->bg[GTK_STATE_NORMAL],
					   ttaudio_xpm);
   ttdataxpm=gdk_pixmap_create_from_xpm_d(window->window,					   
					  &ttdatamask,
					  &style->bg[GTK_STATE_NORMAL],
					  ttdata_xpm);
};

   

   
   
   
   
	
   

