; NSIS Script For Guifications Plugin
; Author Daniel A. Atallah
; Based on the Gaim installer by Herman Bloggs and the Gaim-Encryption installer by Bill Tompkins
; Uses NSIS v2.0

;--------------------------------
;Include Modern UI
  !include "MUI.nsh"

;--------------------------------
;General
  Name "Gaim Guifications ${GUIFICATIONS_VERSION}"

  ;Do A CRC Check
  CRCCheck On

  ;Output File Name
  OutFile "gaim-guifications-${GUIFICATIONS_VERSION}.exe"

  ;The Default Installation Directory
  InstallDir "$PROGRAMFILES\gaim"
  InstallDirRegKey HKLM SOFTWARE\gaim ""

  ShowInstDetails show
  ShowUnInstDetails show
  SetCompressor /SOLID lzma

  !define GUIFICATIONS_UNINST_EXE     "gaim-guifications-uninst.exe"
  !define GUIFICATIONS_DLL            "guifications.dll"
  !define GUIFICATIONS_UNINSTALL_LNK  "Guifications Uninstall.lnk"


;--------------------------------
; Registry keys:
  !define GUIFICATIONS_REG_KEY        "SOFTWARE\gaim-guifications"
  !define GUIFICATIONS_UNINSTALL_KEY  "SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall\gaim-guifications"

;-------------------------------
; Gaim Plugin installer helper stuff
  !addincludedir "../../win32/nsis"
  !include "gaim-plugin.nsh"

;--------------------------------
; Modern UI Configuration
  !define MUI_ICON .\nsis\install.ico
  !define MUI_UNICON .\nsis\install.ico
  !define MUI_HEADERIMAGE
  !define MUI_HEADERIMAGE_BITMAP "nsis\header.bmp"
  !define MUI_CUSTOMFUNCTION_GUIINIT gf_checkGaimVersion
  !define MUI_ABORTWARNING  


;--------------------------------
; Pages
  ;Welcome Page
  !define MUI_WELCOMEPAGE_TITLE $(WELCOME_TITLE)
  !define MUI_WELCOMEPAGE_TEXT $(WELCOME_TEXT)
  !insertmacro MUI_PAGE_WELCOME

  ;License Page
  !define MUI_LICENSEPAGE_RADIOBUTTONS
  !insertmacro MUI_PAGE_LICENSE  ".\COPYING"

  ;Directory Page
  !define MUI_DIRECTORYPAGE_TEXT_TOP $(DIR_SUBTITLE)
  !define MUI_DIRECTORYPAGE_TEXT_DESTINATION $(DIR_INNERTEXT)
  !insertmacro MUI_PAGE_DIRECTORY

  ;Installation Page
  !insertmacro MUI_PAGE_INSTFILES

  ;Finish Page
  !define MUI_FINISHPAGE_TITLE $(FINISH_TITLE)
  !define MUI_FINISHPAGE_TEXT $(FINISH_TEXT)
  !insertmacro MUI_PAGE_FINISH


;--------------------------------
; Languages
  !insertmacro MUI_LANGUAGE "English"
  !insertmacro MUI_LANGUAGE "Italian"
  !insertmacro MUI_LANGUAGE "Spanish"

  ;Translations
  !include "nsis\translations\english.nsh"
  !include "nsis\translations\italian.nsh"
  !include "nsis\translations\spanish.nsh"

; Uninstall the previous version if it exists
Section -SecUninstallOldPlugin
  ; Check install rights..
  Call CheckUserInstallRights
  Pop $R0

  StrCmp $R0 "HKLM" rights_hklm
  StrCmp $R0 "HKCU" rights_hkcu done

  rights_hkcu:
    ReadRegStr $R1 HKCU "${GUIFICATIONS_REG_KEY}" ""
    ReadRegStr $R2 HKCU "${GUIFICATIONS_REG_KEY}" "Version"
    ReadRegStr $R3 HKCU "${GUIFICATIONS_UNINSTALL_KEY}" "UninstallString"
    Goto try_uninstall

  rights_hklm:
    ReadRegStr $R1 HKLM "${GUIFICATIONS_REG_KEY}" ""
    ReadRegStr $R2 HKLM "${GUIFICATIONS_REG_KEY}" "Version"
    ReadRegStr $R3 HKLM "${GUIFICATIONS_UNINSTALL_KEY}" "UninstallString"

  ; If previous version exists .. remove
  try_uninstall:
    StrCmp $R1 "" done
      StrCmp $R2 "" uninstall_problem
        IfFileExists $R3 0 uninstall_problem
          ; Have uninstall string.. go ahead and uninstall.
          SetOverwrite on
          ; Need to copy uninstaller outside of the install dir
          ClearErrors
          CopyFiles /SILENT $R3 "$TEMP\${GUIFICATIONS_UNINST_EXE}"
          SetOverwrite off
          IfErrors uninstall_problem
            ; Ready to uninstall..
            ClearErrors
            ExecWait '"$TEMP\${GUIFICATIONS_UNINST_EXE}" /S _?=$R1'
            IfErrors exec_error
              Delete "$TEMP\${GUIFICATIONS_UNINST_EXE}"
              Goto done

            exec_error:
              Delete "$TEMP\${GUIFICATIONS_UNINST_EXE}"
              Goto uninstall_problem

        uninstall_problem:
          ; Just delete the plugin and uninstaller, and remove Registry key
          MessageBox MB_YESNO $(GUIFICATIONS_PROMPT_WIPEOUT) IDYES do_wipeout IDNO cancel_install
        cancel_install:
          Quit

        do_wipeout:
          StrCmp $R0 "HKLM" del_lm_reg del_cu_reg
          del_cu_reg:
            DeleteRegKey HKCU ${GUIFICATIONS_REG_KEY}
            Goto uninstall_prob_cont
          del_lm_reg:
            DeleteRegKey HKLM ${GUIFICATIONS_REG_KEY}

        uninstall_prob_cont:
          ; plugin DLL
          Delete "$R1\plugins\${GUIFICATIONS_DLL}"
          ; pixmaps
          RMDir /r "$R1\pixmaps\gaim\guifications"
          Delete "$R3"

  done:
SectionEnd


Section "Install"
  Call CheckUserInstallRights
  Pop $R0

  StrCmp $R0 "NONE" instrights_none
  StrCmp $R0 "HKLM" instrights_hklm instrights_hkcu

  instrights_hklm:
    ; Write the version registry keys:
    WriteRegStr HKLM ${GUIFICATIONS_REG_KEY} "" "$INSTDIR"
    WriteRegStr HKLM ${GUIFICATIONS_REG_KEY} "Version" "${GUIFICATIONS_VERSION}"

    ; Write the uninstall keys for Windows
    WriteRegStr HKLM ${GUIFICATIONS_UNINSTALL_KEY} "DisplayName" "$(GUIFICATIONS_UNINSTALL_DESC)"
    WriteRegStr HKLM ${GUIFICATIONS_UNINSTALL_KEY} "UninstallString" "$INSTDIR\${GUIFICATIONS_UNINST_EXE}"
    SetShellVarContext "all"
    Goto install_files

  instrights_hkcu:
    ; Write the version registry keys:
    WriteRegStr HKCU ${GUIFICATIONS_REG_KEY} "" "$INSTDIR"
    WriteRegStr HKCU ${GUIFICATIONS_REG_KEY} "Version" "${GUIFICATIONS_VERSION}"

    ; Write the uninstall keys for Windows
    WriteRegStr HKCU ${GUIFICATIONS_UNINSTALL_KEY} "DisplayName" "$(GUIFICATIONS_UNINSTALL_DESC)"
    WriteRegStr HKCU ${GUIFICATIONS_UNINSTALL_KEY} "UninstallString" "$INSTDIR\${GUIFICATIONS_UNINST_EXE}"
    Goto install_files

  instrights_none:
    ; No registry keys for us...

  install_files:
    SetOutPath "$INSTDIR\plugins"
    SetCompress Auto
    SetOverwrite on
    File "src\${GUIFICATIONS_DLL}"

    SetOutPath "$INSTDIR\pixmaps\gaim\guifications\conf"
    File "pixmaps\*.png"

    SetOutPath "$INSTDIR\pixmaps\gaim\guifications\themes\default"
    File "themes\default\theme.xml"
    File "themes\default\background.png"
    SetOutPath "$INSTDIR\pixmaps\gaim\guifications\themes\mini"
    File "themes\mini\theme.xml"
    File "themes\mini\background.png"
    SetOutPath "$INSTDIR\pixmaps\gaim\guifications\themes\Penguins"
    File "themes\Penguins\theme.xml"
    File "themes\Penguins\penguin.png"

    ; translations - if there is a way to automate this, i can't find it
    SetOutPath "$INSTDIR\locale\de\LC_MESSAGES"
    File /oname=guifications.mo po\de.gmo
    SetOutPath "$INSTDIR\locale\en_AU\LC_MESSAGES"
    File /oname=guifications.mo po\en_AU.gmo
    SetOutPath "$INSTDIR\locale\en_GB\LC_MESSAGES"
    File /oname=guifications.mo po\en_GB.gmo
    SetOutPath "$INSTDIR\locale\es\LC_MESSAGES"
    File /oname=guifications.mo po\es.gmo
    SetOutPath "$INSTDIR\locale\fr\LC_MESSAGES"
    File /oname=guifications.mo po\fr.gmo
    SetOutPath "$INSTDIR\locale\he\LC_MESSAGES"
    File /oname=guifications.mo po\he.gmo
    SetOutPath "$INSTDIR\locale\it\LC_MESSAGES"
    File /oname=guifications.mo po\it.gmo
    SetOutPath "$INSTDIR\locale\ja\LC_MESSAGES"
    File /oname=guifications.mo po\ja.gmo
    SetOutPath "$INSTDIR\locale\nl\LC_MESSAGES"
    File /oname=guifications.mo po\nl.gmo
    SetOutPath "$INSTDIR\locale\no\LC_MESSAGES"
    File /oname=guifications.mo po\no.gmo
    SetOutPath "$INSTDIR\locale\pt\LC_MESSAGES"
    File /oname=guifications.mo po\pt.gmo
	SetOutPath "$INSTDIR\locale\sk\LC_MESSAGES"
	file /oname=guifications.mo po\sk.gmo
    SetOutPath "$INSTDIR\locale\sv\LC_MESSAGES"
    File /oname=guifications.mo po\sv.gmo
    SetOutPath "$INSTDIR\locale\zh_CN\LC_MESSAGES"
    File /oname=guifications.mo po\zh_CN.gmo
    SetOutPath "$INSTDIR\locale\zh_TW\LC_MESSAGES"
    File /oname=guifications.mo po\zh_TW.gmo

    StrCmp $R0 "NONE" done
    CreateShortCut "$SMPROGRAMS\Gaim\${GUIFICATIONS_UNINSTALL_LNK}" "$INSTDIR\${GUIFICATIONS_UNINST_EXE}"
    WriteUninstaller "$INSTDIR\${GUIFICATIONS_UNINST_EXE}"
    SetOverWrite off

  done:
SectionEnd

Section Uninstall
  Call un.CheckUserInstallRights
  Pop $R0
  StrCmp $R0 "NONE" no_rights
  StrCmp $R0 "HKCU" try_hkcu try_hklm

  try_hkcu:
    ReadRegStr $R0 HKCU "${GUIFICATIONS_REG_KEY}" ""
    StrCmp $R0 $INSTDIR 0 cant_uninstall
      ; HKCU install path matches our INSTDIR.. so uninstall
      DeleteRegKey HKCU "${GUIFICATIONS_REG_KEY}"
      DeleteRegKey HKCU "${GUIFICATIONS_UNINSTALL_KEY}"
      Goto cont_uninstall

  try_hklm:
    ReadRegStr $R0 HKLM "${GUIFICATIONS_REG_KEY}" ""
    StrCmp $R0 $INSTDIR 0 try_hkcu
      ; HKLM install path matches our INSTDIR.. so uninstall
      DeleteRegKey HKLM "${GUIFICATIONS_REG_KEY}"
      DeleteRegKey HKLM "${GUIFICATIONS_UNINSTALL_KEY}"
      ; Sets start menu and desktop scope to all users..
      SetShellVarContext "all"

  cont_uninstall:
    ; plugin
    Delete "$INSTDIR\plugins\${GUIFICATIONS_DLL}"
    ; pixmaps
    RMDir /r "$INSTDIR\pixmaps\gaim\guifications"

    ; translations
    ; loop through locale dirs and try to delete any guifications translations
    ClearErrors
    FindFirst $R1 $R2 "$INSTDIR\locale\*"
    IfErrors doneFindingTranslations

    processCurrentTranslationDir:
      ;Ignore "." and ".."
      StrCmp $R2 "." readNextTranslationDir
      StrCmp $R2 ".." readNextTranslationDir
      IfFileExists "$INSTDIR\locale\$R2\LC_MESSAGES\guifications.mo" +1 readNextTranslationDir
      Delete "$INSTDIR\locale\$R2\LC_MESSAGES\guifications.mo"
      RMDir  "$INSTDIR\locale\$R2\LC_MESSAGES"
      RMDir  "$INSTDIR\locale\$R2"
      ClearErrors
    readNextTranslationDir:
      FindNext $R1 $R2
    IfErrors doneFindingTranslations processCurrentTranslationDir

    doneFindingTranslations:
    FindClose $R1
    RMDir  "$INSTDIR\locale"

    ; uninstaller
    Delete "$INSTDIR\${GUIFICATIONS_UNINST_EXE}"
    ; uninstaller shortcut
    Delete "$SMPROGRAMS\Gaim\${GUIFICATIONS_UNINSTALL_LNK}"

    ; try to delete the Gaim directories, in case it has already uninstalled
    RMDir "$INSTDIR\plugins"
    RMDir "$INSTDIR"
    RMDir "$SMPROGRAMS\Gaim"

    Goto done

  cant_uninstall:
    MessageBox MB_OK $(un.GUIFICATIONS_UNINSTALL_ERROR_1) IDOK
    Quit

  no_rights:
    MessageBox MB_OK $(un.GUIFICATIONS_UNINSTALL_ERROR_2) IDOK
    Quit

  done:
SectionEnd

; Check that the selected installation dir contains gaim.exe
Function .onVerifyInstDir
  IfFileExists $INSTDIR\gaim.exe +2
    Abort
FunctionEnd

; Check that the currently installed gaim version is compatible with the version of guifications we are installing
Function gf_checkGaimVersion
  Push $R0

  Push ${GAIM_VERSION}
  Call CheckGaimVersion
  Pop $R0

  StrCmp $R0 ${GAIM_VERSION_OK} gf_checkGaimVersion_OK
  StrCmp $R0 ${GAIM_VERSION_INCOMPATIBLE} +1 +6
    Call GetGaimVersion
    IfErrors +3
    Pop $R0
    MessageBox MB_OK|MB_ICONSTOP "$(BAD_GAIM_VERSION_1) $R0 $(BAD_GAIM_VERSION_2)"
    goto +2
    MessageBox MB_OK|MB_ICONSTOP "$(NO_GAIM_VERSION)"
    Quit

  gf_checkGaimVersion_OK:
  Pop $R0
FunctionEnd

Function CheckUserInstallRights
  ClearErrors
  UserInfo::GetName
  IfErrors Win9x
  Pop $0
  UserInfo::GetAccountType
  Pop $1

  StrCmp $1 "Admin" 0 +3
    StrCpy $1 "HKLM"
    Goto done
  StrCmp $1 "Power" 0 +3
    StrCpy $1 "HKLM"
    Goto done
  StrCmp $1 "User" 0 +3
    StrCpy $1 "HKCU"
    Goto done
  StrCmp $1 "Guest" 0 +3
    StrCpy $1 "NONE"
    Goto done

  ; Unknown error
  StrCpy $1 "NONE"
  Goto done

  Win9x:
    StrCpy $1 "HKLM"

  done:
  Push $1
FunctionEnd

; This is necessary because the uninstaller doesn't have access to installer functions
; (it is identical to CheckUserInstallRights)
Function un.CheckUserInstallRights
  ClearErrors
  UserInfo::GetName
  IfErrors Win9x
  Pop $0
  UserInfo::GetAccountType
  Pop $1

  StrCmp $1 "Admin" 0 +3
    StrCpy $1 "HKLM"
    Goto done
  StrCmp $1 "Power" 0 +3
    StrCpy $1 "HKLM"
    Goto done
  StrCmp $1 "User" 0 +3
    StrCpy $1 "HKCU"
    Goto done
  StrCmp $1 "Guest" 0 +3
    StrCpy $1 "NONE"
    Goto done

  ; Unknown error
  StrCpy $1 "NONE"
  Goto done

  Win9x:
    StrCpy $1 "HKLM"

  done:
  Push $1
FunctionEnd
