/*
 * Guifications - The end all, be all, toaster popup plugin
 * Copyright (C) 2003-2005 Gary Kramlich
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <glib.h>
#include <string.h>

struct _GfThemeInfo {
	gchar *name;
	gchar *version;
	gchar *summary;
	gchar *description;
	gchar *author;
	gchar *website;
};

#include "gf_theme_info.h"

#include <debug.h>
#include <xmlnode.h>

GfThemeInfo *
gf_theme_info_new() {
	GfThemeInfo *info;

	info = g_new0(GfThemeInfo, 1);

	return info;
}

GfThemeInfo *
gf_theme_info_new_from_xmlnode(xmlnode *node) {
	GfThemeInfo *info;
	xmlnode *child;
	gchar *data;

	g_return_val_if_fail(node, NULL);

	info = gf_theme_info_new();

	child = xmlnode_get_child(node, "name");
	if(child && (data = xmlnode_get_data(child))) {
		info->name = g_strdup(data);
		g_free(data);
	}

	child = xmlnode_get_child(node, "version");
	if(child && (data = xmlnode_get_data(child))) {
		info->version = g_strdup(data);
		g_free(data);
	}

	child = xmlnode_get_child(node, "summary");
	if(child && (data = xmlnode_get_data(child))) {
		info->summary = g_strdup(data);
		g_free(data);
	}

	child = xmlnode_get_child(node, "description");
	if(child && (data = xmlnode_get_data(child))) {
		info->description = g_strdup(data);
		g_free(data);
	}

	child = xmlnode_get_child(node, "author");
	if(child && (data = xmlnode_get_data(child))) {
		info->author = g_strdup(data);
		g_free(data);
	}

	child = xmlnode_get_child(node, "website");
	if(child && (data = xmlnode_get_data(child))) {
		info->website = g_strdup(data);
		g_free(data);
	}

	return info;
}

xmlnode *
gf_theme_info_to_xmlnode(GfThemeInfo *info) {
	xmlnode *parent, *child;

	parent = xmlnode_new("info");

	if(info->name && strlen(info->name)) {
		child = xmlnode_new_child(parent, "name");
		xmlnode_insert_data(child, info->name, strlen(info->name));
	}

	if(info->version && strlen(info->version)) {
		child = xmlnode_new_child(parent, "version");
		xmlnode_insert_data(child, info->version, strlen(info->version));
	}

	if(info->summary && strlen(info->summary)) {
		child = xmlnode_new_child(parent, "summary");
		xmlnode_insert_data(child, info->summary, strlen(info->summary));
	}

	if(info->description && strlen(info->description)) {
		child = xmlnode_new_child(parent, "description");
		xmlnode_insert_data(child, info->description, strlen(info->description));
	}

	if(info->author && strlen(info->author)) {
		child = xmlnode_new_child(parent, "author");
		xmlnode_insert_data(child, info->author, strlen(info->author));
	}

	if(info->website && strlen(info->website)) {
		child = xmlnode_new_child(parent, "website");
		xmlnode_insert_data(child, info->website, strlen(info->website));
	}

	return parent;
}

void
gf_theme_info_destroy(GfThemeInfo *info) {
	g_return_if_fail(info);

	if(info->name)
		g_free(info->name);

	if(info->version)
		g_free(info->version);

	if(info->summary)
		g_free(info->summary);

	if(info->description)
		g_free(info->description);

	if(info->author)
		g_free(info->author);

	if(info->website)
		g_free(info->website);

	g_free(info);
	info = NULL;
}

gchar *
gf_theme_info_strip_name(GfThemeInfo *info) {
	GString *str;
	const gchar *iter;
	gchar *dir;

	g_return_val_if_fail(info, NULL);

	if(!info->name)
		return g_strdup("untitled");

	str = g_string_new("");

	iter = info->name;

	if(iter[0] == '.' && strlen(iter) > 1)
		iter++;

	for(; *iter != '\0'; iter++)
	{
		switch(iter[0]) {
			case '\\':
			case '/':
			case ':':
			case '*':
			case '?':
			case '"':
			case '<':
			case '>':
			case '|':
			case '[':
			case ']':
			case '{':
			case '}':
				break;
			case ' ':
				str = g_string_append_c(str, '_');
				break;
			default:
				str = g_string_append_c(str, iter[0]);
		}
	}

	dir = str->str;
	g_string_free(str, FALSE);

	if(!dir)
		dir = g_strdup("untitled");

	return dir;
}

void
gf_theme_info_set_name(GfThemeInfo *info, const gchar *name) {
	g_return_if_fail(info);
	g_return_if_fail(name);

	if(info->name)
		g_free(info->name);

	info->name = g_strdup(name);
}

const gchar *
gf_theme_info_get_name(GfThemeInfo *info) {
	g_return_val_if_fail(info, NULL);

	return info->name;
}

void
gf_theme_info_set_version(GfThemeInfo *info, const gchar *version) {
	g_return_if_fail(info);
	g_return_if_fail(version);

	if(info->version)
		g_free(info->version);

	info->version = g_strdup(version);
}

const gchar *
gf_theme_info_get_version(GfThemeInfo *info) {
	g_return_val_if_fail(info, NULL);

	return info->version;
}

void
gf_theme_info_set_summary(GfThemeInfo *info, const gchar *summary) {
	g_return_if_fail(info);
	g_return_if_fail(summary);

	if(info->summary)
		g_free(info->summary);

	info->summary = g_strdup(summary);
}

const gchar *
gf_theme_info_get_summary(GfThemeInfo *info) {
	g_return_val_if_fail(info, NULL);

	return info->summary;
}

void
gf_theme_info_set_description(GfThemeInfo *info, const gchar *description) {
	g_return_if_fail(info);
	g_return_if_fail(description);

	if(info->description)
		g_free(info->description);

	info->description = g_strdup(description);
}

const gchar *
gf_theme_info_get_description(GfThemeInfo *info) {
	g_return_val_if_fail(info, NULL);

	return info->description;
}

void
gf_theme_info_set_author(GfThemeInfo *info, const gchar *author) {
	g_return_if_fail(info);
	g_return_if_fail(author);

	if(info->author)
		g_free(info->author);

	info->author = g_strdup(author);
}

const gchar *
gf_theme_info_get_author(GfThemeInfo *info) {
	g_return_val_if_fail(info, NULL);

	return info->author;
}

void
gf_theme_info_set_website(GfThemeInfo *info, const gchar *website) {
	g_return_if_fail(info);
	g_return_if_fail(website);

	if(info->website)
		g_free(info->website);

	info->website = g_strdup(website);
}

const gchar *
gf_theme_info_get_website(GfThemeInfo *info) {
	g_return_val_if_fail(info, NULL);

	return info->website;
}
