/* 
 * File: gmedialib.c
 *
 * Medialib search and playlist editor
 * Copyright (c) 2005-2006 Johannes Heimansberg
 *
 * requires GTK+ 2.6 or better and XMMS2
 *
 * Released under the GNU General Public License v2
 */

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <gtk/gtk.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkdialog.h>
#include "gmedialib.h"
#include "gtrackinfo/trackinfo.h"
#include "gtrackinfo/gtrackinfo.h"
#include "playlists.xpm"
#include "browser.xpm"

static void gmedialib_class_init(GMedialibClass *klass);
static void gmedialib_init(GMedialib *gml);

extern xmmsc_connection_t *connection; /* available from gxmms2.c */

GType gmedialib_get_type(void)
{
	static GType gml_type = 0;

	if (!gml_type) {
		static const GTypeInfo gml_info =
		{
			sizeof(GMedialibClass),
			NULL, /* base_init */
			NULL, /* base_finalize */
			(GClassInitFunc) gmedialib_class_init,
			NULL, /* class_finalize */
			NULL, /* class_data */
        	sizeof(GMedialib),
			0,
			(GInstanceInitFunc) gmedialib_init,
		};

		gml_type = g_type_register_static(GTK_TYPE_DIALOG, 
		                                  "GMedialib", &gml_info, 0);
	}
	return gml_type;
}

static void gmedialib_class_init(GMedialibClass *klass)
{
}

void gml_set_statusbar_text(GMedialib *gml, gchar *text)
{
	gtk_statusbar_pop(GTK_STATUSBAR(gml->statusbar), gml->statusbar_id);
	gtk_statusbar_push(GTK_STATUSBAR(gml->statusbar), gml->statusbar_id, text);
}

static void cb_close_medialib_dialog(GtkWidget *widget, gpointer pt)
{
	GtkWidget *gml = (GtkWidget *)pt;
	gtk_widget_hide(gml);
}

/* callback: close window (return FALSE = close window) */
static gint cb_delete_event(GtkWidget *widget, GdkEvent *event, gpointer pt)
{
	GtkWidget *gml = (GtkWidget *)pt;
	gtk_widget_hide(gml);
	return TRUE;
}

void gml_set_album_tracknr_digits(GMedialib *gml, gint digits)
{
	if (digits >= 0 && digits <= 3)
		gml->album_tracknr_digits = digits;
}

gint gml_window_get_pos_x(GMedialib *gml)
{
	return gml->window_pos_x;
}

gint gml_window_get_pos_y(GMedialib *gml)
{
	return gml->window_pos_y;
}

gint gml_window_get_width(GMedialib *gml)
{
	return gml->window_width;
}

gint gml_window_get_height(GMedialib *gml)
{
	return gml->window_height;
}

void gml_set_font_size(GMedialib *gml, gint size)
{
	if (size > 0 && size <= 100) {
		gml->font_size = size;
		g_object_set(gml->renderer_text, "size", gml->font_size * PANGO_SCALE,
		             "size-set", TRUE, NULL);
		g_object_set(gml->renderer_text_ellipsize, "size", gml->font_size * PANGO_SCALE,
		             "size-set", TRUE, NULL);
	}
}

void gml_cell_data_function(GtkTreeViewColumn *col,
                            GtkCellRenderer   *renderer,
                            GtkTreeModel      *model,
                            GtkTreeIter       *iter,
                            gpointer           user_data)
{
	GMedialib *gml = GMEDIALIB(user_data);
	g_object_set(gml->renderer_text, "weight-set", FALSE, NULL);
	g_object_set(gml->renderer_text_ellipsize, "weight-set", FALSE, NULL);
}

static gint cb_configure_event(GtkWidget *widget, GdkEvent *event, gpointer pt)
{
	gint               x, y;
	GMedialib         *gml = (GMedialib *)pt;
	GdkEventConfigure *ce = (GdkEventConfigure *)event;

	gtk_window_get_position(GTK_WINDOW(gml), &x, &y);
	gml->window_width  = ce->width;
	gml->window_height = ce->height;
	gml->window_pos_x  = x;
	gml->window_pos_y  = y;
	return FALSE;
}

static void bc_handle_mediainfo_reader_status(xmmsc_result_t *res, void *userdata)
{
	GMedialib *gml = (GMedialib *)userdata;
	gint32     status;

	xmmsc_result_get_int(res, &status);
	if (status == 0)
		gml_set_statusbar_text(gml, "Done.");
	else if (status == 1)
		gml_set_statusbar_text(gml, "Reading media info of newly added files...");
}

static void cb_handle_mediainfo_reader_unindexed(xmmsc_result_t *res, void *userdata)
{
	guint32         remaining_files;
	gchar           buf[256];
	GMedialib      *gml = (GMedialib *)userdata;
	xmmsc_result_t *res2;

	xmmsc_result_get_uint(res, &remaining_files);
	snprintf(buf, sizeof(buf), 
	         "Reading media info of newly added files: %d files remaining",
	         remaining_files);
	gml_set_statusbar_text(gml, buf);
	res2 = xmmsc_result_restart(res);
	xmmsc_result_unref(res);
	xmmsc_result_unref(res2);
}

static gboolean cb_key_press(GtkWidget   *widget,
                             GdkEventKey *event,
                             gpointer     userdata)
{
	GMedialib *gml = (GMedialib *)userdata;
	/* 0 = search tab */
	if (gtk_notebook_get_current_page(GTK_NOTEBOOK(gml->notebook)) == 0) {
		gchar text[6] = "";
		g_unichar_to_utf8(gdk_keyval_to_unicode(event->keyval), text);
		if (text[0] >= 32) { /* valid text character */
			gtk_entry_set_text(GTK_ENTRY(gml->search.entry_searchstring), text);
			gtk_widget_grab_focus(gml->search.entry_searchstring);
			gtk_editable_set_position(GTK_EDITABLE(gml->search.entry_searchstring), -1);
			gtk_editable_select_region(GTK_EDITABLE(gml->search.entry_searchstring), 0, 0);
			gtk_editable_set_position(GTK_EDITABLE(gml->search.entry_searchstring), -1);
		}
	}
	return FALSE;
}

void gml_setup_xmms_callbacks(GMedialib *gml)
{
	XMMS_CALLBACK_SET(connection, xmmsc_broadcast_mediainfo_reader_status,
 	                  bc_handle_mediainfo_reader_status, gml);
	XMMS_CALLBACK_SET(connection, xmmsc_signal_mediainfo_reader_unindexed,
	                  cb_handle_mediainfo_reader_unindexed, gml);
}

gboolean gml_notebook_is_mlib_search_visible(GMedialib *gml)
{
	return gml->notebook_mlib_search;
}
gboolean gml_notebook_is_mlib_browser_visible(GMedialib *gml)
{
	return gml->notebook_mlib_browser;
}
gboolean gml_notebook_is_mlib_add_files_visible(GMedialib *gml)
{
	return gml->notebook_mlib_add_files;
}
gboolean gml_notebook_is_playlists_visible(GMedialib *gml)
{
	return gml->notebook_playlists;
}
gboolean gml_notebook_is_playlist_visible(GMedialib *gml)
{
	return gml->notebook_playlist;
}

static void gml_create_notebook(GMedialib *gml)
{
	GdkPixbuf *icon;

	if (gml->notebook_mlib_search) {
		gml_create_mlib_search(gml);
		gml->image_mlib_search = gtk_image_new_from_stock(GTK_STOCK_FIND, 
		                                                  GTK_ICON_SIZE_MENU);
		gml->label_mlib_search = gtk_label_new_with_mnemonic("MLib _Search");
		gml->hbox_mlib_search = gtk_hbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(gml->hbox_mlib_search), 
		                   gml->image_mlib_search, FALSE, FALSE, 3);
		gtk_box_pack_start(GTK_BOX(gml->hbox_mlib_search), 
		                   gml->label_mlib_search, FALSE, FALSE, 3);
		gtk_widget_show_all(gml->hbox_mlib_search);
		gtk_notebook_append_page(GTK_NOTEBOOK(gml->notebook), 
		                         gml->search.frame, gml->hbox_mlib_search);
	}

	if (gml->notebook_mlib_browser) {
		gml_create_mlib_browser(gml);
		icon = gdk_pixbuf_new_from_xpm_data((const char **)browser_xpm);
		gml->image_mlib_browse = gtk_image_new_from_pixbuf(icon);
		gml->label_mlib_browse = gtk_label_new_with_mnemonic("MLib _Browser");
		gml->hbox_mlib_browse = gtk_hbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(gml->hbox_mlib_browse), 
		                   gml->image_mlib_browse, FALSE, FALSE, 3);
		gtk_box_pack_start(GTK_BOX(gml->hbox_mlib_browse), 
		                   gml->label_mlib_browse, FALSE, FALSE, 3);
		gtk_widget_show_all(gml->hbox_mlib_browse);
		gtk_notebook_append_page(GTK_NOTEBOOK(gml->notebook), 
		                         gml->browser.frame, gml->hbox_mlib_browse);
	}

	if (gml->notebook_mlib_add_files) {
		gml_create_mlib_add_files(gml);
		gml->image_mlib_add_files = gtk_image_new_from_stock(GTK_STOCK_OPEN,
		                                                     GTK_ICON_SIZE_MENU);
		gml->label_mlib_add_files = gtk_label_new_with_mnemonic("MLib _Add files");
		gml->hbox_mlib_add_files = gtk_hbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(gml->hbox_mlib_add_files), 
		                   gml->image_mlib_add_files, FALSE, FALSE, 3);
		gtk_box_pack_start(GTK_BOX(gml->hbox_mlib_add_files), 
		                   gml->label_mlib_add_files, FALSE, FALSE, 3);
		gtk_widget_show_all(gml->hbox_mlib_add_files);
		gtk_notebook_append_page(GTK_NOTEBOOK(gml->notebook), 
		                         gml->addfiles.frame, gml->hbox_mlib_add_files);
	}
	
	if (gml->notebook_playlists) {
		gml_create_playlists(gml);
		icon = gdk_pixbuf_new_from_xpm_data((const char **)playlists_xpm);
		gml->image_playlists = gtk_image_new_from_pixbuf(icon);
		gml->label_playlists = gtk_label_new_with_mnemonic("Pla_ylists");
		gml->hbox_playlists = gtk_hbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(gml->hbox_playlists), 
		                   gml->image_playlists, FALSE, FALSE, 3);
		gtk_box_pack_start(GTK_BOX(gml->hbox_playlists), 
		                   gml->label_playlists, FALSE, FALSE, 3);
		gtk_widget_show_all(gml->hbox_playlists);
		gtk_notebook_append_page(GTK_NOTEBOOK(gml->notebook), 
		                         gml->playlists.frame, gml->hbox_playlists);
	}

	if (gml->notebook_playlist) {
		gml_create_playlist(gml);
		gml->image_playlist = gtk_image_new_from_stock(GTK_STOCK_MEDIA_PLAY,
		                                               GTK_ICON_SIZE_MENU);
		gml->label_playlist = gtk_label_new_with_mnemonic("_Playlist");
		gml->hbox_playlist = gtk_hbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(gml->hbox_playlist), 
		                   gml->image_playlist, FALSE, FALSE, 3);
		gtk_box_pack_start(GTK_BOX(gml->hbox_playlist), 
		                   gml->label_playlist, FALSE, FALSE, 3);
		gtk_widget_show_all(gml->hbox_playlist);
		gtk_notebook_append_page(GTK_NOTEBOOK(gml->notebook), 
		                         gml->playlist.frame, gml->hbox_playlist);
	}
	gtk_widget_show_all(GTK_WIDGET(gml));
}

static void gmedialib_init(GMedialib *gml)
{
	gml->renderer_text = gtk_cell_renderer_text_new();
	gml->renderer_text_ellipsize = gtk_cell_renderer_text_new();
	g_object_set(gml->renderer_text_ellipsize, "ellipsize", PANGO_ELLIPSIZE_END, 
	             "ellipsize-set", TRUE, NULL);

	gtk_window_set_title(GTK_WINDOW(gml), "Playlist Editor");
	gtk_window_set_modal(GTK_WINDOW(gml), FALSE);

	gml->notebook = gtk_notebook_new();
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(gml)->vbox),
	                   gml->notebook, TRUE, TRUE, 0);

	/* statusbar */
	gml->statusbar = gtk_statusbar_new();
	gtk_statusbar_set_has_resize_grip(GTK_STATUSBAR(gml->statusbar), FALSE);
	gml->statusbar_id = gtk_statusbar_get_context_id(
		GTK_STATUSBAR(gml->statusbar),
		"default");
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(gml)->vbox),
                       gml->statusbar, FALSE, FALSE, 0);
	gtk_statusbar_push(GTK_STATUSBAR(gml->statusbar), 
	                   gml->statusbar_id, "Ready");

	/* close button */
	gml->button_close = gtk_button_new_from_stock(GTK_STOCK_CLOSE);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(gml)->action_area),
                       gml->button_close, TRUE, TRUE, 0);

	/* signals */
    g_signal_connect(G_OBJECT(gml->button_close),       "clicked",
	                 G_CALLBACK(cb_close_medialib_dialog),
	                 (gpointer)gml);
	g_signal_connect(G_OBJECT(gml),                     "delete_event",
	                 G_CALLBACK(cb_delete_event),
	                 (gpointer)gml);
	g_signal_connect(G_OBJECT(gml),                     "configure_event",
	                 G_CALLBACK(cb_configure_event),
	                 (gpointer)gml);
	g_signal_connect(G_OBJECT(gml->notebook),           "key-press-event",
                     G_CALLBACK(cb_key_press),
                     (gpointer)gml);

	gtk_widget_set_size_request(GTK_WIDGET(gml), 540, 360);

	gtk_widget_show_all(GTK_WIDGET(gml));
	
	gml_setup_xmms_callbacks(gml);
	
	gml->font_size = -1; /* default */
	gml->album_tracknr_digits = 2;
}

GtkWidget* gmedialib_new(gboolean search,
                         gboolean browser,
                         gboolean add_files,
                         gboolean playlists,
                         gboolean playlist)
{
	GMedialib *gml = g_object_new(gmedialib_get_type(), NULL);
	gml->notebook_mlib_search    = search;
	gml->notebook_mlib_browser   = browser;
	gml->notebook_mlib_add_files = add_files;
	gml->notebook_playlists      = playlists;
	gml->notebook_playlist       = playlist;
	gml_create_notebook(gml);
	if (gml->notebook_playlist)     gml_pl_refresh_playlist(gml);
	if (gml->notebook_mlib_browser) gml_mlib_browse_update_artists(gml);
	if (gml->notebook_playlists)    gml_playlists_update_playlists(gml);
	return GTK_WIDGET(gml);
}
