/* 
 * File: gmlbrowser.c
 *
 * Medialib search and playlist editor
 * Copyright (c) 2005 Johannes Heimansberg
 *
 * requires GTK+ 2.6 or better and XMMS2 DR2 or better
 *
 * Released under the GNU General Public License v2
 */

#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkdialog.h>
#include "gmedialib.h"
#include "gtrackinfo/trackinfo.h"
#include "gtrackinfo/gtrackinfo.h"

extern xmmsc_connection_t *connection; /* available from gxmms2.c */

static void cb_browser_artist_activated(GtkTreeView       *view,
                                        GtkTreePath       *path,
                                        GtkTreeViewColumn *col,
                                        gpointer           userdata)
{
	GtkTreeIter   iter;
	GtkTreeModel *model;
	GMedialib    *gml = (GMedialib *)userdata;

	model = gtk_tree_view_get_model(view);

	if (gtk_tree_model_get_iter(model, &iter, path)) {
		gchar *search_str = "Unknown";
		gtk_tree_model_get(model, &iter, 0, &search_str, -1);

		if (gml_notebook_is_mlib_search_visible(GMEDIALIB(gml))) {
			if (gtk_tree_model_iter_has_child(model, &iter)) { /* Artist */
				gml_search_do_search(gml, SEARCH_ARTIST, search_str);
			} else {
				gml_search_do_search(gml, SEARCH_ALBUM,  search_str);
			}
			gtk_notebook_set_current_page(GTK_NOTEBOOK(gml->notebook), 0);
		}
		g_free(search_str);
	}
}

static void cb_browser_button_search_pressed(GtkWidget *widget, gpointer pt)
{
	GMedialib        *gml = (GMedialib *)pt;
	GtkTreeSelection *sel;
	GtkTreeModel     *model;
	GtkTreeIter       selected_row;

	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->browser.list));

	g_assert(gtk_tree_selection_get_mode(sel) == GTK_SELECTION_SINGLE);

	if (gtk_tree_selection_get_selected(sel, &model, &selected_row)) {
		gchar *search_str;

		gtk_tree_model_get(model, &selected_row, 0, &search_str, -1);

		if (gml_notebook_is_mlib_search_visible(GMEDIALIB(gml))) {
			if (gtk_tree_model_iter_has_child(model, &selected_row)) { /* Artist */
				gml_search_do_search(gml, SEARCH_ARTIST, search_str);
			} else {
				gml_search_do_search(gml, SEARCH_ALBUM,  search_str);
			}
			gtk_notebook_set_current_page(GTK_NOTEBOOK(gml->notebook), 0);
		}
		g_free(search_str);
	}
}

static void cb_browser_popup_refresh_pressed(GtkWidget *widget, gpointer pt)
{
	GMedialib *gml = (GMedialib *)pt;
	gml_mlib_browse_update_artists(gml);
}

static void browser_view_popup_menu(GtkWidget      *treeview,
                                    GdkEventButton *event,
                                    gpointer        userdata)
{
	GtkWidget        *menu, *menuitem;
	GtkTreeSelection *selection;
	GMedialib        *gml = (GMedialib *)userdata;
	gboolean          multiple = FALSE, none = FALSE;

	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->browser.list));
	if (gtk_tree_selection_count_selected_rows(selection) > 1)
		multiple = TRUE;
	if (gtk_tree_selection_count_selected_rows(selection) == 0)
		none = TRUE;

	menu = gtk_menu_new();

	menuitem = gtk_image_menu_item_new_from_stock(GTK_STOCK_FIND, NULL);
	if (multiple || none || !gml_notebook_is_mlib_search_visible(GMEDIALIB(gml)))
		gtk_widget_set_sensitive(menuitem, FALSE);

	g_signal_connect(menuitem, "activate",
	                 G_CALLBACK(cb_browser_button_search_pressed), userdata);

	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);


	menuitem = gtk_separator_menu_item_new();
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	menuitem = gtk_image_menu_item_new_from_stock(GTK_STOCK_REFRESH, NULL);

	g_signal_connect(menuitem, "activate",
	                 G_CALLBACK(cb_browser_popup_refresh_pressed), userdata);

	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	gtk_widget_show_all(menu);

	gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL,
	               (event != NULL) ? event->button : 0,
	               gdk_event_get_time((GdkEvent*)event));
}

static gboolean cb_browser_selection_button_press(GtkWidget      *treeview,
                                                  GdkEventButton *event,
                                                  gpointer        userdata)
{
	gboolean result = FALSE;
	if (event->type == GDK_BUTTON_PRESS  &&  event->button == 3) {
		browser_view_popup_menu(treeview, event, userdata);
		result = TRUE;
	}
	return result;
}

static gboolean cb_browser_selection_popup_menu(GtkWidget *treeview, 
                                                gpointer   userdata)
{
	browser_view_popup_menu(treeview, NULL, userdata);
	return TRUE;
}

static void n_update_artists(xmmsc_result_t *res, void *userdata)
{
	GMedialib    *gml = (GMedialib *)userdata;
	gchar        *artist, *album;
	gint          id;
	gchar         artist_tmp[192] = "", album_tmp[192] = "";
	GtkTreeIter   iter;
	GtkTreeModel *model;

	model = gtk_tree_view_get_model(GTK_TREE_VIEW(gml->browser.list));
	/* Make sure the model stays with us after the tree view unrefs it */
	g_object_ref(model);
	/* Detach model from view */
	gtk_tree_view_set_model(GTK_TREE_VIEW(gml->browser.list), NULL);

	for (; xmmsc_result_list_valid(res); xmmsc_result_list_next(res)) {
		xmmsc_result_get_dict_entry_str(res, "artist", &artist);
		xmmsc_result_get_dict_entry_str(res, "album",  &album);
		xmmsc_result_get_dict_entry_int32(res, "id", &id);
		if (!artist)
			artist = "Unknown";
		if (g_ascii_strcasecmp(artist, artist_tmp) != 0) {
			g_strlcpy(artist_tmp, artist, sizeof(artist_tmp));
			g_strlcpy(album_tmp, album, sizeof(artist_tmp));
			if (gml->browser.ts != NULL) {
				GtkTreeIter child_iter;

				gtk_tree_store_append(gml->browser.ts, &iter, NULL);
				gtk_tree_store_set(gml->browser.ts, &iter,
				                   0, artist, -1);

				/* add child */
				gtk_tree_store_append(gml->browser.ts, &child_iter, &iter);
				gtk_tree_store_set(gml->browser.ts, &child_iter, 0, album, -1);
			}
		} else if (artist[0] != '\0' && 
		           g_ascii_strcasecmp(album, album_tmp) != 0) {
			GtkTreeIter child_iter;
			g_strlcpy(album_tmp, album, sizeof(artist_tmp));
			/* add child */
			gtk_tree_store_append(gml->browser.ts, &child_iter, &iter);
			gtk_tree_store_set(gml->browser.ts, &child_iter, 0, album, -1);
		}
	}

	/* Re-attach model to view */
	gtk_tree_view_set_model(GTK_TREE_VIEW(gml->browser.list), model);
	g_object_unref(model);

	xmmsc_result_unref(res);
	gml_set_statusbar_text(gml, "Ready.");
}

void gml_mlib_browse_update_artists(GMedialib *gml)
{
	xmmsc_result_t *res;
	gchar          *query = "SELECT DISTINCT m1.value AS artist, " \
	                        "IFNULL(m2.value, '[Unknown Album]') AS album " \
	                        "FROM Media m1 LEFT JOIN Media m2 " \
	                        "ON m1.id = m2.id AND m2.key='album' " \
	                        "AND m1.key='artist' " \
	                        "ORDER BY UPPER(artist),UPPER(album);";

	gml_set_statusbar_text(gml, "Loading artists...");

	if (gml->browser.ts != NULL) {
		gtk_tree_store_clear(gml->browser.ts);
	}

	res = xmmsc_medialib_select(connection, query);
	xmmsc_result_notifier_set(res, n_update_artists, gml);
	xmmsc_result_unref(res);
}

void gml_create_mlib_browser(GMedialib *gml)
{
	GtkTreeViewColumn *column_artist;

	gml->browser.frame = gtk_frame_new("Media Lib");
	gtk_container_set_border_width(GTK_CONTAINER(gml->browser.frame), 2);

	/* table mlib browser */
	gml->browser.table = gtk_table_new(2, 1, FALSE);
	gtk_container_set_border_width(GTK_CONTAINER(gml->browser.table), 3);
	gtk_table_set_row_spacings(GTK_TABLE(gml->browser.table), 4);
	gtk_table_set_col_spacings(GTK_TABLE(gml->browser.table), 4);
	gtk_container_add(GTK_CONTAINER(gml->browser.frame), 
	                  gml->browser.table);

	/* list artist browser */
	gml->browser.ts = gtk_tree_store_new(1, G_TYPE_STRING);

	gml->browser.list = 
		gtk_tree_view_new_with_model(GTK_TREE_MODEL(gml->browser.ts));
	gtk_tree_view_set_enable_search(GTK_TREE_VIEW(gml->browser.list), TRUE);

	gml->browser.scroll_widget = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
		GTK_SCROLLED_WINDOW(gml->browser.scroll_widget),
		GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);

	gtk_container_add(GTK_CONTAINER(gml->browser.scroll_widget),
	                  gml->browser.list);

	gtk_table_attach(GTK_TABLE(gml->browser.table),
		             gml->browser.scroll_widget,
		             0, 1, 0, 1, 
		             GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 0, 0);

	column_artist = gtk_tree_view_column_new_with_attributes("Artist", 
	                                                         gml->renderer_text,
	                                                         "text", 0, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(gml->browser.list),
	                            column_artist);
	gtk_tree_view_column_set_resizable(column_artist, TRUE);
	gtk_tree_view_column_set_cell_data_func(column_artist, gml->renderer_text,
	                                        gml_cell_data_function, gml, NULL);

	/* search button */
	gml->browser.button_search = gtk_button_new_from_stock(GTK_STOCK_FIND);
	gtk_table_attach(GTK_TABLE(gml->browser.table),
		             gml->browser.button_search,
		             0, 1, 1, 2,
		             GTK_SHRINK, GTK_SHRINK, 0, 0);

	g_signal_connect(G_OBJECT(gml->browser.list),     "row-activated",
	                 G_CALLBACK(cb_browser_artist_activated),
	                 (gpointer)gml);
	g_signal_connect(G_OBJECT(gml->browser.button_search),  "clicked",
	                 G_CALLBACK(cb_browser_button_search_pressed),
	                 (gpointer)gml);
	g_signal_connect(G_OBJECT(gml->browser.list), "button-press-event",
	                 G_CALLBACK(cb_browser_selection_button_press),
	                 (gpointer)gml);
	g_signal_connect(G_OBJECT(gml->browser.list),         "popup-menu",
	                 G_CALLBACK(cb_browser_selection_popup_menu),
	                 (gpointer)gml);
}
