/* ***** BEGIN LICENSE BLOCK *****
 * Source last modified: $Id: animattr.cpp,v 1.1.22.1 2004/07/09 01:58:01 hubbe Exp $
 * 
 * Portions Copyright (c) 1995-2004 RealNetworks, Inc. All Rights Reserved.
 * 
 * The contents of this file, and the files included with this file,
 * are subject to the current version of the RealNetworks Public
 * Source License (the "RPSL") available at
 * http://www.helixcommunity.org/content/rpsl unless you have licensed
 * the file under the current version of the RealNetworks Community
 * Source License (the "RCSL") available at
 * http://www.helixcommunity.org/content/rcsl, in which case the RCSL
 * will apply. You may also obtain the license terms directly from
 * RealNetworks.  You may not use this file except in compliance with
 * the RPSL or, if you have a valid RCSL with RealNetworks applicable
 * to this file, the RCSL.  Please see the applicable RPSL or RCSL for
 * the rights, obligations and limitations governing use of the
 * contents of the file.
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL") in which case the provisions of the GPL are applicable
 * instead of those above. If you wish to allow use of your version of
 * this file only under the terms of the GPL, and not to allow others
 * to use your version of this file under the terms of either the RPSL
 * or RCSL, indicate your decision by deleting the provisions above
 * and replace them with the notice and other provisions required by
 * the GPL. If you do not delete the provisions above, a recipient may
 * use your version of this file under the terms of any one of the
 * RPSL, the RCSL or the GPL.
 * 
 * This file is part of the Helix DNA Technology. RealNetworks is the
 * developer of the Original Code and owns the copyrights in the
 * portions it created.
 * 
 * This file, and the files included with this file, is distributed
 * and made available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS
 * ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET
 * ENJOYMENT OR NON-INFRINGEMENT.
 * 
 * Technology Compatibility Kit Test Suite(s) Location:
 *    http://www.helixcommunity.org/content/tck
 * 
 * Contributor(s):
 * 
 * ***** END LICENSE BLOCK ***** */

// system
#include <math.h>
#include <time.h>
// include
#include "hxtypes.h"
#include "hxwintyp.h"
#include "hxcom.h"
#include "hxxml.h"
#include "smiltype.h"
// pnmisc
#include "hxwinver.h"
#include "hxparse.h"
// pncont
#include "hxslist.h"
// rnxmllib
#include "hxxmlprs.h"
// rmasmil
#include "smlparse.h"
#include "animattr.h"

CAttr::CAttr(UINT32 ulAttrName, const char* pszStr)
{
    m_lLastError = HXR_OK;
    m_ulAttrName = ulAttrName;
    m_ulAttrType = kAttrTypeString;
    m_dValue[0]  = 0.0;
    m_dValue[1]  = 0.0;
    m_dValue[2]  = 0.0;
    m_dValue[3]  = 0.0;
    m_eType[0]   = CSS2TypeAuto;
    m_eType[1]   = CSS2TypeAuto;
    m_eType[2]   = CSS2TypeAuto;
    m_eType[3]   = CSS2TypeAuto;
    m_pszValue   = NULL;

    if (m_ulAttrName == kAttrNameLeft   ||
        m_ulAttrName == kAttrNameTop    ||
        m_ulAttrName == kAttrNameRight  ||
        m_ulAttrName == kAttrNameBottom ||
        m_ulAttrName == kAttrNameWidth  ||
        m_ulAttrName == kAttrNameHeight)
    {
        m_ulAttrType = kAttrTypeRealScalar;
        if (pszStr)
        {
            UINT32 ulIndex = (m_ulAttrName == kAttrNameTop ? 1 : 0);
            BOOL   bRel    = FALSE;
            m_lLastError   = ParsePosLenValue(pszStr, m_dValue[ulIndex], bRel);
            if (SUCCEEDED(m_lLastError))
            {
                m_eType[ulIndex] = (bRel ? CSS2TypePercentage : CSS2TypeLength);
            }
        }
    }
    else if (m_ulAttrName == kAttrNameBackgroundColor ||
             m_ulAttrName == kAttrNameColor)
    {
        m_ulAttrType = kAttrTypeRealVector;
        if (pszStr)
        {
            UINT32   ulColor = 0;
            CSS2Type eType   = CSS2TypeTransparent;
            m_lLastError     = CSmilParser::parseColor(pszStr, ulColor, eType);
            if (SUCCEEDED(m_lLastError))
            {
                m_dValue[0] = (double) ((ulColor & 0x00FF0000) >> 16);
                m_dValue[1] = (double) ((ulColor & 0x0000FF00) >>  8);
                m_dValue[2] = (double)  (ulColor & 0x000000FF);
                m_dValue[3] = (double) ((ulColor & 0xFF000000) >> 24);
            }
        }
    }
    else if (m_ulAttrName == kAttrNameZIndex)
    {
        m_ulAttrType = kAttrTypeRealScalar;
        if (pszStr)
        {
            m_dValue[0]  = atof(pszStr);
        }
    }
    else if (m_ulAttrName == kAttrNameSoundLevel)
    {
        m_ulAttrType = kAttrTypeRealScalar;
        if (pszStr)
        {
            char* pEndPtr = NULL;
            m_dValue[0]   = strtod(pszStr, &pEndPtr);
            if (!pEndPtr || (pEndPtr && *pEndPtr != '%'))
            {
                m_lLastError = HXR_FAIL;
            }
        }
    }
    else if (m_ulAttrName == kAttrNameMediaOpacity ||
             m_ulAttrName == kAttrNameBackgroundOpacity)
    {
        m_ulAttrType = kAttrTypeRealScalar;
        if (pszStr)
        {
            UINT32 ulOpacity = 0;
            m_lLastError     = HXParseOpacity(pszStr, ulOpacity);
            if (SUCCEEDED(m_lLastError))
            {
                m_dValue[0] = (double) ulOpacity;
            }
        }
    }
    else if (m_ulAttrName == kAttrNameCoords ||
             m_ulAttrName == kAttrNameValue)
    {
        m_ulAttrType = kAttrTypeString;
        if (pszStr)
        {
            m_pszValue = new char [strlen(pszStr) + 1];
            if (m_pszValue)
            {
                strcpy(m_pszValue, pszStr); /* Flawfinder: ignore */
            }
            else
            {
                m_lLastError = HXR_OUTOFMEMORY;
            }
        }
    }
    else if (m_ulAttrName == kAttrNameLeftTop)
    {
        m_ulAttrType = kAttrTypeRealVector;
        if (pszStr)
        {
            // SPEC: coordinate-pair ::= ( coordinate comma-wsp coordinate )
            //       coordinate      ::= num
            //       num             ::= Number
            //       comma-wsp       ::= S (spacechar|",") S
            //       S               ::= spacechar*
            //       spacechar       ::= (#x20 | #x9 | #xD | #xA)
            //
            // Here we parse coordinate-pair's
            //
            // Allocate a buffer to hold a copy
            char* pszTmp = new char [strlen(pszStr) + 1];
            if (pszTmp)
            {
                // Copy the string
                strcpy(pszTmp, pszStr); /* Flawfinder: ignore */
                // Parse the string
                UINT32 ulNumFound = 0;
                char*  pszToken   = strtok(pszTmp, " \t\r\n,");
                while (pszToken && SUCCEEDED(m_lLastError) && ulNumFound < 2)
                {
                    // Parse the value
                    BOOL bRel    = FALSE;
                    m_lLastError = ParsePosLenValue(pszToken, m_dValue[ulNumFound], bRel);
                    if (SUCCEEDED(m_lLastError))
                    {
                        // Set the type
                        m_eType[ulNumFound] = (bRel ? CSS2TypePercentage : CSS2TypeLength);
                        // Increment the number parsed
                        ulNumFound++;
                        // Get the next token
                        pszToken = strtok(NULL, " \t\r\n,");
                    }
                }
                if (SUCCEEDED(m_lLastError))
                {
                    // Make sure we parsed exactly two
                    if (ulNumFound != 2)
                    {
                        // We found less than 2
                        m_lLastError = HXR_FAIL;
                    }
                }
            }
            else
            {
                m_lLastError = HXR_OUTOFMEMORY;
            }
        }
    }
}

CAttr::CAttr(UINT32 ulAttrName,
             double dVal0, CSS2Type eType0,
             double dVal1, CSS2Type eType1,
             double dVal2, CSS2Type eType2,
             double dVal3, CSS2Type eType3)
{
    m_lLastError = HXR_OK;
    m_ulAttrName = ulAttrName;
    m_ulAttrType = kAttrTypeString;
    m_dValue[0]  = dVal0;
    m_dValue[1]  = dVal1;
    m_dValue[2]  = dVal2;
    m_dValue[3]  = dVal3;
    m_eType[0]   = eType0;
    m_eType[1]   = eType1;
    m_eType[2]   = eType2;
    m_eType[3]   = eType3;
    m_pszValue   = NULL;

    if (m_ulAttrName == kAttrNameLeft         ||
        m_ulAttrName == kAttrNameTop          ||
        m_ulAttrName == kAttrNameRight        ||
        m_ulAttrName == kAttrNameBottom       ||
        m_ulAttrName == kAttrNameWidth        ||
        m_ulAttrName == kAttrNameHeight       ||
        m_ulAttrName == kAttrNameZIndex       ||
        m_ulAttrName == kAttrNameSoundLevel   ||
        m_ulAttrName == kAttrNameMediaOpacity ||
        m_ulAttrName == kAttrNameBackgroundOpacity)
    {
        m_ulAttrType   = kAttrTypeRealScalar;
    }
    else if (m_ulAttrName == kAttrNameBackgroundColor ||
             m_ulAttrName == kAttrNameColor           ||
             m_ulAttrName == kAttrNameLeftTop)
    {
        m_ulAttrType   = kAttrTypeRealVector;
    }
}

CAttr::CAttr(const CAttr& rAttr)
{
    m_lLastError = rAttr.m_lLastError;
    m_ulAttrName = rAttr.m_ulAttrName;
    m_ulAttrType = rAttr.m_ulAttrType;
    m_dValue[0]  = rAttr.m_dValue[0];
    m_dValue[1]  = rAttr.m_dValue[1];
    m_dValue[2]  = rAttr.m_dValue[2];
    m_dValue[3]  = rAttr.m_dValue[3];
    m_eType[0]   = rAttr.m_eType[0];
    m_eType[1]   = rAttr.m_eType[1];
    m_eType[2]   = rAttr.m_eType[2];
    m_eType[3]   = rAttr.m_eType[3];
    m_pszValue   = NULL;
    if (rAttr.m_pszValue)
    {
        m_pszValue = new char [strlen(rAttr.m_pszValue) + 1];
        if (m_pszValue)
        {
            strcpy(m_pszValue, rAttr.m_pszValue); /* Flawfinder: ignore */
        }
        else
        {
            m_lLastError = HXR_OUTOFMEMORY;
        }
    }
}

CAttr::~CAttr()
{
    HX_VECTOR_DELETE(m_pszValue);
}

double CAttr::Dist(CAttr* pAttr1, CAttr* pAttr2, CAttr* pDepend)
{
    double dRet = 0.0;

    if (Compatible(pAttr1, pAttr2))
    {
        if (pAttr1->m_ulAttrType == kAttrTypeRealScalar ||
            pAttr1->m_ulAttrType == kAttrTypeRealVector)
        {
            double dSum = 0.0;
            for (UINT32 i = 0; i < kVectorSize; i++)
            {
                double dDiff = GetAbsolute(pAttr2, i, pDepend) -
                               GetAbsolute(pAttr1, i, pDepend);
                dSum += dDiff * dDiff;
            }
            dRet = sqrt(dSum);
        }
    }

    return dRet;
}

double CAttr::GetAbsolute(CAttr* pAttr, UINT32 i, CAttr* pDepend)
{
    double dRet = 0.0;

    if (i < kVectorSize)
    {
        dRet = pAttr->m_dValue[i];
        if (pAttr->IsRelative(i) && pDepend && !pDepend->IsRelative(i))
        {
            dRet = pAttr->m_dValue[i] * pDepend->m_dValue[i] / 100.0;
        }
    }

    return dRet;
}

void CAttr::Interp(CAttr* pAttr1, double t1,
                   CAttr* pAttr2, double t2,
                   double t, CAttr* pDepend)
{
    if (Compatible(pAttr1, pAttr2) &&
        t1 < t2 && t1 <= t && t <= t2)
    {
        // Copy the first attribute
        *this = *pAttr1;
        // Perform the interpolation
        if (pAttr1->m_ulAttrType == kAttrTypeRealScalar ||
            pAttr1->m_ulAttrType == kAttrTypeRealVector)
        {
            double dFract = (t - t1) / (t2 - t1);
            for (UINT32 i = 0; i < kVectorSize; i++)
            {
                double d1 = pAttr1->m_dValue[i];
                double d2 = pAttr2->m_dValue[i];
                if (pAttr1->IsRelative(i) != pAttr2->IsRelative(i) &&
                    pDepend && !pDepend->IsRelative(i))
                {
                    // If we are trying to interpolate between
                    // an absolute and relative value, we will first
                    // convert the relative value to absolute
                    if (pAttr1->IsRelative(i) && !pAttr2->IsRelative(i))
                    {
                        // Convert attribute 1 to absolute
                        d1 = GetAbsolute(pAttr1, i, pDepend);
                    }
                    else
                    {
                        // Convert attribute 2 to absolute
                        d2 = GetAbsolute(pAttr2, i, pDepend);
                    }
                    // Since we made sure that both attributes were
                    // absolute, then the result must be absolute as well.
                    m_eType[i] = CSS2TypeLength;
                }
                m_dValue[i] = d1 + (d2 - d1) * dFract;
            }
        }
    }
}

void CAttr::Add(CAttr* pAttr, CAttr* pDepend)
{
    if (Compatible(this, pAttr))
    {
        if (m_ulAttrType == kAttrTypeRealScalar ||
            m_ulAttrType == kAttrTypeRealVector)
        {
            for (UINT32 i = 0; i < kVectorSize; i++)
            {
                double d1 = m_dValue[i];
                double d2 = pAttr->m_dValue[i];
                if (IsRelative(i) != pAttr->IsRelative(i) &&
                    pDepend && !pDepend->IsRelative(i))
                {
                    // If we are trying to add an absolute value
                    // to a relative value, we will first
                    // convert the relative value to absolute
                    if (IsRelative(i) && !pAttr->IsRelative(i))
                    {
                        // Convert attribute 1 to absolute
                        d1 = GetAbsolute(this, i, pDepend);
                    }
                    else
                    {
                        // Convert attribute 2 to absolute
                        d2 = GetAbsolute(pAttr, i, pDepend);
                    }
                    // Since we made sure that both attributes were
                    // absolute, then the result must be absolute as well.
                    m_eType[i] = CSS2TypeLength;
                }
                // Add the values
                m_dValue[i] = d1 + d2;
            }
        }
    }
}

void CAttr::Mult(double dMult)
{
    if (m_ulAttrType == kAttrTypeRealScalar ||
        m_ulAttrType == kAttrTypeRealVector)
    {
        for (UINT32 i = 0; i < kVectorSize; i++)
        {
            m_dValue[i] *= dMult;
        }
    }
}

void CAttr::Clamp()
{
    if (m_ulAttrName == kAttrNameBackgroundColor ||
        m_ulAttrName == kAttrNameColor)
    {
        for (UINT32 i = 0; i < kVectorSize; i++)
        {
            // Round to nearest integer
            m_dValue[i] = floor(m_dValue[i] + 0.5);
            // Clip bottom
            if (m_dValue[i] < 0.0)
            {
                m_dValue[i] = 0.0;
            }
            // Clip top
            if (m_dValue[i] > 255.0)
            {
                m_dValue[i] = 255.0;
            }
        }
    }
    else if (m_ulAttrName == kAttrNameZIndex)
    {
        // Round to nearest integer
        m_dValue[0] = floor(m_dValue[0] + 0.5);
        // Clip bottom
        if (m_dValue[0] < 0.0)
        {
            m_dValue[0] = 0.0;
        }
    }
    else if (m_ulAttrName == kAttrNameSoundLevel)
    {
        // Clip bottom
        if (m_dValue[0] < 0.0)
        {
            m_dValue[0] = 0.0;
        }
    }
    else if (m_ulAttrName == kAttrNameMediaOpacity ||
             m_ulAttrName == kAttrNameBackgroundOpacity)
    {
        // Clip bottom
        if (m_dValue[0] < 0.0)
        {
            m_dValue[0] = 0.0;
        }
        // Clip top
        if (m_dValue[0] > 255.0)
        {
            m_dValue[0] = 255.0;
        }
    }
}

double CAttr::GetValueDouble(UINT32 i) const
{
    double dRet = 0.0;

    if (i < kVectorSize)
    {
        dRet = m_dValue[i];
    }

    return dRet;
}

CSS2Type CAttr::GetCSS2Type(UINT32 i) const
{
    CSS2Type eRet = CSS2TypeAuto;

    if (i < kVectorSize)
    {
        eRet = m_eType[i]; 
    }

    return eRet;
}

BOOL CAttr::IsRelative(UINT32 i) const
{
    BOOL bRet = FALSE;

    if (i < kVectorSize)
    {
        bRet = (m_eType[i] == CSS2TypePercentage ? TRUE : FALSE);
    }

    return bRet;
}

const char* CAttr::GetValueString(UINT32 i) const
{
    return m_pszValue;
}

CAttr& CAttr::operator  = (const CAttr& rAttr)
{
    m_lLastError = rAttr.m_lLastError;
    m_ulAttrName = rAttr.m_ulAttrName;
    m_ulAttrType = rAttr.m_ulAttrType;
    m_dValue[0]  = rAttr.m_dValue[0];
    m_dValue[1]  = rAttr.m_dValue[1];
    m_dValue[2]  = rAttr.m_dValue[2];
    m_dValue[3]  = rAttr.m_dValue[3];
    m_eType[0]   = rAttr.m_eType[0];
    m_eType[1]   = rAttr.m_eType[1];
    m_eType[2]   = rAttr.m_eType[2];
    m_eType[3]   = rAttr.m_eType[3];
    HX_VECTOR_DELETE(m_pszValue);
    if (rAttr.m_pszValue)
    {
        m_pszValue = new char [strlen(rAttr.m_pszValue) + 1];
        if (m_pszValue)
        {
            strcpy(m_pszValue, rAttr.m_pszValue); /* Flawfinder: ignore */
        }
        else
        {
            m_lLastError = HXR_OUTOFMEMORY;
        }
    }
    return *this;
}

BOOL CAttr::Compatible(CAttr* pAttr1, CAttr* pAttr2)
{
    BOOL bRet = FALSE;

    if (pAttr1 && pAttr2)
    {
        if (pAttr1->m_ulAttrName == pAttr2->m_ulAttrName &&
            pAttr1->m_ulAttrType == pAttr2->m_ulAttrType)
        {
            bRet = TRUE;
        }
        else if (pAttr1->m_ulAttrName == kAttrNameLeftTop)
        {
            if (pAttr2->m_ulAttrName == kAttrNameLeft ||
                pAttr2->m_ulAttrName == kAttrNameTop)
            {
                bRet = TRUE;
            }
        }
        else if (pAttr2->m_ulAttrName == kAttrNameLeftTop)
        {
            if (pAttr1->m_ulAttrName == kAttrNameLeft ||
                pAttr1->m_ulAttrName == kAttrNameTop)
            {
                bRet = TRUE;
            }
        }
    }

    return bRet;
}

HX_RESULT CAttr::ParsePosLenValue(const char* pszStr,
                                  double&     rdValue,
                                  BOOL&       rbIsPercent)
{
    HX_RESULT retVal = HXR_OK;

    if (pszStr)
    {
        // Parse the numeric value
        char* pEndPtr = NULL;
        rdValue       = strtod(pszStr, &pEndPtr);
        // Now decide if it was a percent or not
        if (pEndPtr && *pEndPtr == '%')
        {
            rbIsPercent = TRUE;
        }
        else
        {
            rbIsPercent = FALSE;
        }
    }
    else
    {
        retVal = HXR_FAIL;
    }

    return retVal;
}

