/*
 *  JLib - Jacob's Library.
 *  Copyright (C) 2003, 2004  Juan Carlos Seijo Prez
 * 
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 * 
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 * 
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 *  Juan Carlos Seijo Prez
 *  jacob@mainreactor.net
 */

/** Clase para gestin de texturas. El objeto JGLTexture consta de las 
 * coordenadas de textura dentro de la superficie de datos y de un ndice 
 * de textura OpenGL. El llamador es responsable de que la textura sea vlida 
 * (por ejemplo de ancho y alto como potencia de 2, donde se aplique o 
 * dimensiones mnimas y/o mximas dependientes de la implementacin OpenGL).
 * @file    JGLTexture.cpp
 * @author  Juan Carlos Seijo Prez
 * @date    01/04/2003
 * @version 0.0.1 - 01/04/2003 - Primera Versin.
 */

#include <JLib/Graphics/JGLTexture.h>

// Constructor
JGLTexture::JGLTexture() : id(0), hasMipmaps(false), width(0), height(0), bpp(0)
{
}

// Destructor
JGLTexture::~JGLTexture()
{
  Destroy();
}

// Libera la memoria asociada
void JGLTexture::Destroy()
{
  if (id)
  {
    glDeleteTextures(1, (GLuint *)&id);
    id = 0;
  }
}

u32 JGLTexture::Next2Power(u32 value)
{
	u32 newVal = 1;
	while (value > newVal)
	{
		newVal <<= 1;
	}

	return newVal;
}

// Crea la textura a partir de un buffer.
u32 JGLTexture::Create(JImage *image, bool withMipmaps /*= false*/, s32 internalFormat /*= 4*/, s32 border /*= 0*/, s32 format /*= GL_RGBA*/, s32 type /*= GL_UNSIGNED_BYTE*/)
{
	// Si no hay datos
  if(!image)
		return 0;

  width = image->Width();
  height = image->Height();
  bpp = image->BitsPP();

	hasMipmaps = withMipmaps;

	// Genera la textura con el ID asociado en el array
	glGenTextures(1, (GLuint *)&id);

	if (0 == id)
	{
		fprintf(stderr, "JGLTexture: No se pudo generar la textura.\n");

		return 0;
	}

	// Enlaza el ID a la textura
	glBindTexture(GL_TEXTURE_2D, id);

	// Crea los mipmaps, si es necesario
	if (hasMipmaps)
	{
		fprintf(stderr, "Creando con mipmaps a %dx%dx%d\n", image->Width(), image->Height(), image->BitsPP());
    if (0 != gluBuild2DMipmaps(GL_TEXTURE_2D,
															 internalFormat,
															 image->Width(),
															 image->Height(),
															 format,
															 type,
															 image->Pixels()))
		{
			fprintf(stderr, "No se pudieron construir los mipmaps: %s\n", gluErrorString(glGetError()));
			Destroy();
			
			return 0;
		}
	}
  else
	{
    glTexImage2D(GL_TEXTURE_2D, 
								 0, 
								 internalFormat,
								 image->Width(), 
								 image->Height(), 
								 border, 
								 format,
								 type,
								 image->Pixels());
	
		if (glGetError() != 0)
		{
			fprintf(stderr, "Fallo al invocar glTexImage2D(). Error de OpenGL: %s\n", gluErrorString(glGetError()));
			Destroy();
			
			return 0;
		}
	}

  // Calidad de la textura
  if (hasMipmaps)
  {
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_NEAREST);
	  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR_MIPMAP_LINEAR);
  }
  else
  {
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  }

  return id;
}

// Crea la textura desde un fichero bmp, tga o jpg.
// Devuelve el id de textura.
u32 JGLTexture::Load(const char *fileName, bool withMipmaps /*= false*/, s32 internalFormat /*= 4*/, s32 border /*= 0*/, s32 format /*= GL_RGBA*/, s32 type /*= GL_UNSIGNED_BYTE*/)
{
	if (fileName == 0)
		return 0;

	JImage image;
	if (!image.Load(fileName, 0))
	{
		return 0;
	}

  return Create(&image, withMipmaps, internalFormat, border, format, type);
}
