/****************************************************************************
 *
 * Copyright (c) 2005 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#ifndef _LIBICAL2_H
#define _LIBICAL2_H

#include <config.h>
#include <xpl.h>

#define ICAL2_TEXT_CHAR (1 << 0)
#define ICAL2_PARAM_TEXT_CHAR (1 << 1)
#define ICAL2_X_NAME_CHAR (1 << 2)
#define ICAL2_PROPERTY_DELIMITER (1 << 3)
#define ICAL2_VALUE_CHAR (1 << 4)

#define ICal2IsText(c) (((ICal2CharDefinitions[(int)(unsigned char)(c)]) & ICAL2_TEXT_CHAR) != 0)
#define ICal2IsParamText(c) (((ICal2CharDefinitions[(int)(unsigned char)(c)]) & ICAL2_PARAM_TEXT_CHAR) != 0)
#define ICal2IsXName(c) (((ICal2CharDefinitions[(int)(unsigned char)(c)]) & ICAL2_X_NAME_CHAR) != 0)
#define ICal2IsPropertyDelim(c) (((ICal2CharDefinitions[(int)(unsigned char)(c)]) & ICAL2_PROPERTY_DELIMITER) != 0)
#define ICal2IsValue(c) (((ICal2CharDefinitions[(int)(unsigned char)(c)]) & ICAL2_VALUE_CHAR) != 0)

#define ICAL2_KEYWORD_ALTREP "ALTREP"
#define ICAL2_KEYWORD_CN "CN"
#define ICAL2_KEYWORD_CUTYPE "CUTYPE"
#define ICAL2_KEYWORD_DELEGATED_FROM "DELEGATED-FROM"
#define ICAL2_KEYWORD_DELEGATED_TO "DELEGATED-TO"
#define ICAL2_KEYWORD_DIR "DIR"
#define ICAL2_KEYWORD_ENCODING "ENCODING"
#define ICAL2_KEYWORD_FMTTYPE "FMTTYPE"
#define ICAL2_KEYWORD_FBTYPE "FBTYPE"
#define ICAL2_KEYWORD_LANGUAGE "LANGUAGE"
#define ICAL2_KEYWORD_MEMBER "MEMBER"
#define ICAL2_KEYWORD_PARTSTAT "PARTSTAT"
#define ICAL2_KEYWORD_RANGE "RANGE"
#define ICAL2_KEYWORD_RELATED "RELATED"
#define ICAL2_KEYWORD_RELTYPE "RELTYPE"
#define ICAL2_KEYWORD_ROLE "ROLE"
#define ICAL2_KEYWORD_RSVP "RSVP"
#define ICAL2_KEYWORD_SENT_BY "SENT-BY"
#define ICAL2_KEYWORD_TZID "TZID"
#define ICAL2_KEYWORD_VALUE "VALUE"

#define ICAL2_KEYWORD_BEGIN "BEGIN"
#define ICAL2_KEYWORD_END "END"
#define ICAL2_KEYWORD_CALSCALE "CALSCALE"
#define ICAL2_KEYWORD_METHOD "METHOD"
#define ICAL2_KEYWORD_PRODID "PRODID"
#define ICAL2_KEYWORD_VERSION "VERSION"
#define ICAL2_KEYWORD_ATTACH "ATTACH"
#define ICAL2_KEYWORD_CATEGORIES "CATEGORIES"
#define ICAL2_KEYWORD_CLASS "CLASS"
#define ICAL2_KEYWORD_COMMENT "COMMENT"
#define ICAL2_KEYWORD_DESCRIPTION "DESCRIPTION"
#define ICAL2_KEYWORD_GEO "GEO"
#define ICAL2_KEYWORD_LOCATION "LOCATION"
#define ICAL2_KEYWORD_PERCENT_COMPLETE "PERCENT-COMPLETE"
#define ICAL2_KEYWORD_PRIORITY "PRIORITY"
#define ICAL2_KEYWORD_RESOURCES "RESOURCES"
#define ICAL2_KEYWORD_STATUS "STATUS"
#define ICAL2_KEYWORD_SUMMARY "SUMMARY"
#define ICAL2_KEYWORD_COMPLETED "COMPLETED"
#define ICAL2_KEYWORD_DTEND "DTEND"
#define ICAL2_KEYWORD_DUE "DUE"
#define ICAL2_KEYWORD_DTSTART "DTSTART"
#define ICAL2_KEYWORD_DURATION "DURATION"
#define ICAL2_KEYWORD_FREEBUSY "FREEBUSY"
#define ICAL2_KEYWORD_TRANSP "TRANSP"
#define ICAL2_KEYWORD_TZID "TZID"
#define ICAL2_KEYWORD_TZNAME "TZNAME"
#define ICAL2_KEYWORD_TZOFFSETFROM "TZOFFSETFROM"
#define ICAL2_KEYWORD_TZOFFSETTO "TZOFFSETTO"
#define ICAL2_KEYWORD_TZURL "TZURL"
#define ICAL2_KEYWORD_ATTENDEE "ATTENDEE"
#define ICAL2_KEYWORD_CONTACT "CONTACT"
#define ICAL2_KEYWORD_ORGANIZER "ORGANIZER"
#define ICAL2_KEYWORD_RECURRENCE_ID "RECURRENCE-ID"
#define ICAL2_KEYWORD_RELATED_TO "RELATED-TO"
#define ICAL2_KEYWORD_URL "URL"
#define ICAL2_KEYWORD_UID "UID"
#define ICAL2_KEYWORD_EXDATE "EXDATE"
#define ICAL2_KEYWORD_EXRULE "EXRULE"
#define ICAL2_KEYWORD_RDATE "RDATE"
#define ICAL2_KEYWORD_RRULE "RRULE"
#define ICAL2_KEYWORD_ACTION "ACTION"
#define ICAL2_KEYWORD_REPEAT "REPEAT"
#define ICAL2_KEYWORD_TRIGGER "TRIGGER"
#define ICAL2_KEYWORD_CREATED "CREATED"
#define ICAL2_KEYWORD_DTSTAMP "DTSTAMP"
#define ICAL2_KEYWORD_LAST_MODIFIED "LAST-MODIFIED"
#define ICAL2_KEYWORD_SEQUENCE "SEQUENCE"
#define ICAL2_KEYWORD_X "X-"
#define ICAL2_KEYWORD_REQUEST_STATUS "REQUEST-STATUS"

#define ICAL2_KEYWORD_DAYLIGHT "DAYLIGHT"
#define ICAL2_KEYWORD_STANDARD "STANDARD"
#define ICAL2_KEYWORD_VALARM "VALARM"
#define ICAL2_KEYWORD_VEVENT "VEVENT"
#define ICAL2_KEYWORD_VCALENDAR "VCALENDAR"
#define ICAL2_KEYWORD_VJOURNAL "VJOURNAL"
#define ICAL2_KEYWORD_VFREEBUSY "VFREEBUSY"
#define ICAL2_KEYWORD_VTIMEZONE "VTIMEZONE"
#define ICAL2_KEYWORD_VTODO "VTODO"
#define ICAL2_KEYWORD_IANA "IANA"

#define ICAL2_KEYWORD_RECUR_FREQ "FREQ"
#define ICAL2_KEYWORD_RECUR_UNTIL "UNTIL"
#define ICAL2_KEYWORD_RECUR_COUNT "COUNT"
#define ICAL2_KEYWORD_RECUR_INTERVAL "INTERVAL"
#define ICAL2_KEYWORD_RECUR_BYSECOND "BYSECOND"
#define ICAL2_KEYWORD_RECUR_BYMINUTE "BYMINUTE"
#define ICAL2_KEYWORD_RECUR_BYHOUR "BYHOUR"
#define ICAL2_KEYWORD_RECUR_BYDAY "BYDAY"
#define ICAL2_KEYWORD_RECUR_BYMONTHDAY "BYMONTHDAY"
#define ICAL2_KEYWORD_RECUR_BYYEARDAY "BYYEARDAY"
#define ICAL2_KEYWORD_RECUR_BYWEEKNO "BYWEEKNO"
#define ICAL2_KEYWORD_RECUR_BYMONTH "BYMONTH"
#define ICAL2_KEYWORD_RECUR_BYSETPOS "BYSETPOS"
#define ICAL2_KEYWORD_RECUR_WKST "WKST"

#define ICAL2_PULL_PARAMETER_DETAIL(prop, param, detail) \
            { \
                register ICal2Parameter *r = (param); \
                if ((r->items.count == 0) || ((r->items.count + 1) < r->items.allocated)) { \
                    (detail) = &r->items.detail[r->items.count++]; \
                } else { \
                    r->items.allocated += 4; \
                    r = (ICal2Parameter *)MemRealloc(r, sizeof(ICal2Parameter) + (r->items.allocated * sizeof(ICal2ParameterDetail))); \
                    if (r) { \
                        register ICal2Property *p = (prop); \
                        if (r->next) { \
                            r->next->prev = r; \
                        } else { \
                            p->parameters.tail = r; \
                        } \
                        if (r->prev) { \
                            r->prev->next = r; \
                        } else { \
                            p->parameters.head = r; \
                        } \
                        (param) = r; \
                        (detail) = &r->items.detail[r->items.count++]; \
                    } else { \
                        (detail) = NULL; \
                    } \
                } \
            }

#define ICAL2_ADD_PROPERTY_PARAMETER(prop, param) \
            { \
                register ICal2Property *p = (prop); \
                register ICal2Parameter *r = (param); \
                r->next = NULL; \
                if ((r->prev = p->parameters.tail) != NULL) { \
                    r->prev->next = r; \
                } else { \
                    p->parameters.head = r; \
                } \
                p->parameters.tail = r; \
            }

#define ICAL2_PULL_VALUE_DETAIL(prop, detail) \
            { \
                register ICal2Value *v = (prop)->value; \
                if ((v->items.count == 0) || ((v->items.count + 1) < v->items.allocated)) { \
                    (detail) = &v->items.detail[v->items.count++]; \
                    memset((detail), 0, sizeof(ICal2ValueDetail)); \
                } else { \
                    v->items.allocated += 4; \
                    v = (ICal2Value *)MemRealloc(v, sizeof(ICal2Value) + (v->items.allocated * sizeof(ICal2ValueDetail))); \
                    if (v) { \
                        (prop)->value = v; \
                        (detail) = &v->items.detail[v->items.count++]; \
                        memset((detail), 0, sizeof(ICal2ValueDetail)); \
                    } else { \
                        (detail) = NULL; \
                    } \
                } \
            }

#define ICAL2_ADD_COMPONENT_PROPERTY(comp, prop) \
            { \
                register ICal2Property *p = (prop); \
                register ICal2Component *c = (comp); \
                p->next = NULL; \
                if ((p->prev = c->properties.tail) != NULL) { \
                    p->prev->next = p; \
                } else { \
                    c->properties.head = p; \
                } \
                c->properties.tail = p; \
            }

#define ICAL2_UNFOLD_AND_COPY_LINE(s, d) \
            { \
                register int i; \
                register unsigned char *n; \
                do { \
                    n = strchr((s), '\r'); \
                    if (n) { \
                        i = n - (s); \
                        memcpy((d), (s), i); \
                        (d) += i; \
                        if (n[1] == '\n') { \
                            (s) = n + 2; \
                        } else { \
                            (s) = n + 1; \
                        } \
                        if ((*(s) != ' ') && (*(s) != '\t')) { \
                            break; \
                        } \
                        (s)++; \
                        continue; \
                    } \
                    i = strlen((s)); \
                    memcpy((d), (s), i); \
                    (d) += i; \
                    (s) += i; \
                    break; \
                } while (*(s)); \
                *(d)++ = '\0'; \
                (d)++; \
            }

/*
    This enumeration should be ordered in conjunction
    with the ICal2ParameterKeywords array defined in ical2.c
*/
enum ICal2ParamaterTypes {
    ICAL2_PARAMETER_ALTREP = 1, 
    ICAL2_PARAMETER_CN, 
    ICAL2_PARAMETER_CUTYPE, 
    ICAL2_PARAMETER_DELEGATED_FROM, 
    ICAL2_PARAMETER_DELEGATED_TO, 
    ICAL2_PARAMETER_DIR, 
    ICAL2_PARAMETER_ENCODING, 
    ICAL2_PARAMETER_FMTTYPE, 
    ICAL2_PARAMETER_FBTYPE, 
    ICAL2_PARAMETER_LANGUAGE, 
    ICAL2_PARAMETER_MEMBER, 
    ICAL2_PARAMETER_PARTSTAT, 
    ICAL2_PARAMETER_RANGE, 
    ICAL2_PARAMETER_RELATED, 
    ICAL2_PARAMETER_RELTYPE, 
    ICAL2_PARAMETER_ROLE, 
    ICAL2_PARAMETER_RSVP, 
    ICAL2_PARAMETER_SENT_BY, 
    ICAL2_PARAMETER_TZID, 
    ICAL2_PARAMETER_VALUE, 
    ICAL2_PARAMETER_X, 
    ICAL2_PARAMETER_IANA
};

/*
    This enumeration should be ordered in conjunction
    with the ICal2ValueKeywords array defined in ical2.c
*/
enum ICal2ValueTypes {
    ICAL2_VALUE_TYPE_NONE = 0, 
    ICAL2_VALUE_TYPE_BINARY, 
    ICAL2_VALUE_TYPE_BOOLEAN, 
    ICAL2_VALUE_TYPE_CAL_ADDRESS, 
    ICAL2_VALUE_TYPE_DATE, 
    ICAL2_VALUE_TYPE_DATE_TIME, 
    ICAL2_VALUE_TYPE_DURATION, 
    ICAL2_VALUE_TYPE_FLOAT, 
    ICAL2_VALUE_TYPE_INTEGER, 
    ICAL2_VALUE_TYPE_PERIOD, 
    ICAL2_VALUE_TYPE_RECUR, 
    ICAL2_VALUE_TYPE_REQUEST_STATUS, 
    ICAL2_VALUE_TYPE_STATUS, 
    ICAL2_VALUE_TYPE_TEXT, 
    ICAL2_VALUE_TYPE_TIME, 
    ICAL2_VALUE_TYPE_URI, 
    ICAL2_VALUE_TYPE_UTC_OFFSET, 
    ICAL2_VALUE_TYPE_X, 
    ICAL2_VALUE_TYPE_IANA
};

enum ICal2PropertyTypes {
    ICAL2_PROPERTY_BEGIN = 1, 
    ICAL2_PROPERTY_END, 
    ICAL2_PROPERTY_CALSCALE, 
    ICAL2_PROPERTY_METHOD, 
    ICAL2_PROPERTY_PRODID, 
    ICAL2_PROPERTY_VERSION, 
    ICAL2_PROPERTY_ATTACH, 
    ICAL2_PROPERTY_CATEGORIES, 
    ICAL2_PROPERTY_CLASS, 
    ICAL2_PROPERTY_COMMENT, 
    ICAL2_PROPERTY_DESCRIPTION, 
    ICAL2_PROPERTY_GEO, 
    ICAL2_PROPERTY_LOCATION, 
    ICAL2_PROPERTY_PERCENT_COMPLETE, 
    ICAL2_PROPERTY_PRIORITY, 
    ICAL2_PROPERTY_RESOURCES, 
    ICAL2_PROPERTY_STATUS, 
    ICAL2_PROPERTY_SUMMARY, 
    ICAL2_PROPERTY_COMPLETED, 
    ICAL2_PROPERTY_DTEND, 
    ICAL2_PROPERTY_DUE, 
    ICAL2_PROPERTY_DTSTART, 
    ICAL2_PROPERTY_DURATION, 
    ICAL2_PROPERTY_FREEBUSY, 
    ICAL2_PROPERTY_TRANSP, 
    ICAL2_PROPERTY_TZID, 
    ICAL2_PROPERTY_TZNAME, 
    ICAL2_PROPERTY_TZOFFSETFROM, 
    ICAL2_PROPERTY_TZOFFSETTO, 
    ICAL2_PROPERTY_TZURL, 
    ICAL2_PROPERTY_ATTENDEE, 
    ICAL2_PROPERTY_CONTACT, 
    ICAL2_PROPERTY_ORGANIZER, 
    ICAL2_PROPERTY_RECURRENCE_ID, 
    ICAL2_PROPERTY_RELATED_TO, 
    ICAL2_PROPERTY_URL, 
    ICAL2_PROPERTY_UID, 
    ICAL2_PROPERTY_EXDATE, 
    ICAL2_PROPERTY_EXRULE, 
    ICAL2_PROPERTY_RDATE, 
    ICAL2_PROPERTY_RRULE, 
    ICAL2_PROPERTY_ACTION, 
    ICAL2_PROPERTY_REPEAT, 
    ICAL2_PROPERTY_TRIGGER, 
    ICAL2_PROPERTY_CREATED, 
    ICAL2_PROPERTY_DTSTAMP, 
    ICAL2_PROPERTY_LAST_MODIFIED, 
    ICAL2_PROPERTY_SEQUENCE, 
    ICAL2_PROPERTY_REQUEST_STATUS, 
    ICAL2_PROPERTY_X, 
    ICAL2_PROPERTY_IANA
};

enum ICal2AlarmType {
    ICAL2_ALARM_AUDIO = 1, 
    ICAL2_ALARM_DISPLAY, 
    ICAL2_ALARM_EMAIL, 
    ICAL2_ALARM_IANA, 
    ICAL2_ALARM_PROCEDURE, 
    ICAL2_ALARM_X, 
};

/*
    This enumeration is order sensitive.
*/
enum ICal2ComponentType {
    ICAL2_COMPONENT_VCALENDAR = 1, 
    ICAL2_COMPONENT_VALARM, 
    ICAL2_COMPONENT_VEVENT, 
    ICAL2_COMPONENT_VTODO, 
    ICAL2_COMPONENT_VJOURNAL, 
    ICAL2_COMPONENT_VFREEBUSY, 
    ICAL2_COMPONENT_VTIMEZONE, 
    ICAL2_COMPONENT_X, 
    ICAL2_COMPONENT_IANA, 
    ICAL2_COMPONENT_DAYLIGHT, 
    ICAL2_COMPONENT_STANDARD
};

enum ICal2ObjectType {
    ICAL2_OBJECT_CALENDAR = 1, 
    ICAL2_OBJECT_ICAL
};

enum ICal2PropertyFlags {
    ICAL2_PROPERTY_PRESENT = 1,
};

enum ICal2ComponentFlags {
    ICAL2_COMPONENT_DELETED = 1, 
};

enum ICal2ObjectFlags {
    ICAL2_OBJECT_FLAG_MEMORY = (1 << 0), 
    ICAL2_OBJECT_FLAG_FILE_HANDLE = (1 << 1), 
    ICAL2_OBJECT_FLAG_SOCKET = (1 << 2)
};

enum ICal2CalendarUserTypes {
    ICAL2_CALENDAR_USER_TYPE_INDIVIDUAL = 1, 
    ICAL2_CALENDAR_USER_TYPE_GROUP, 
    ICAL2_CALENDAR_USER_TYPE_RESOURCE, 
    ICAL2_CALENDAR_USER_TYPE_ROOM, 
    ICAL2_CALENDAR_USER_TYPE_UNKNOWN, 
    ICAL2_CALENDAR_USER_TYPE_X, 
    ICAL2_CALENDAR_USER_TYPE_IANA
};

enum ICal2EncodingTypes {
    ICAL2_ENCODING_TYPE_8BIT = 1, 
    ICAL2_ENCODING_TYPE_BASE64, 
    ICAL2_ENCODING_TYPE_X, 
    ICAL2_ENCODING_TYPE_IANA
};

enum ICal2FormatTypes {
    ICAL2_FORMAT_TYPE_X = 1, 
    ICAL2_FORMAT_TYPE_IANA
};

enum ICal2FreeBusyTypes {
    ICAL2_FREE_BUSY_TYPE_FREE = 1, 
    ICAL2_FREE_BUSY_TYPE_BUSY, 
    ICAL2_FREE_BUSY_TYPE_BUSY_UNAVAILABLE, 
    ICAL2_FREE_BUSY_TYPE_BUSY_TENTATIVE, 
    ICAL2_FREE_BUSY_TYPE_X, 
    ICAL2_FREE_BUSY_TYPE_IANA
};

enum ICal2ParticipantStatusTypes {
    ICAL2_PART_STATUS_TYPE_NEEDS_ACTION = 1, 
    ICAL2_PART_STATUS_TYPE_ACCEPTED, 
    ICAL2_PART_STATUS_TYPE_DECLINED, 
    ICAL2_PART_STATUS_TYPE_TENTATIVE, 
    ICAL2_PART_STATUS_TYPE_DELEGATED, 
    ICAL2_PART_STATUS_TYPE_COMPLETED, 
    ICAL2_PART_STATUS_TYPE_IN_PROCESS, 
    ICAL2_PART_STATUS_TYPE_X, 
    ICAL2_PART_STATUS_TYPE_IANA
};

enum ICal2RangeTypes {
    ICAL2_RANGE_TYPE_THISANDPRIOR = 1, 
    ICAL2_RANGE_TYPE_THISANDFUTURE
};

enum ICal2RelatedTypes {
    ICAL2_RELATED_TYPE_START = 1, 
    ICAL2_RELATED_TYPE_END
};

enum ICal2RelationshipTypes {
    ICAL2_RELTYPE_TYPE_PARENT = 1, 
    ICAL2_RELTYPE_TYPE_CHILD, 
    ICAL2_RELTYPE_TYPE_SIBLING, 
    ICAL2_RELTYPE_TYPE_IANA, 
    ICAL2_RELTYPE_TYPE_X
};

enum ICal2RoleTypes {
    ICAL2_ROLE_TYPE_CHAIR = 1, 
    ICAL2_ROLE_TYPE_REQ_PARTICIPANT, 
    ICAL2_ROLE_TYPE_OPT_PARTICIPANT, 
    ICAL2_ROLE_TYPE_NON_PARTICIPANT, 
    ICAL2_ROLE_TYPE_IANA, 
    ICAL2_ROLE_TYPE_X
};

enum ICal2RecurKeywords {
    ICAL2_RECUR_FREQ = 1, 
    ICAL2_RECUR_UNTIL, 
    ICAL2_RECUR_COUNT, 
    ICAL2_RECUR_INTERVAL, 
    ICAL2_RECUR_BYSECOND, 
    ICAL2_RECUR_BYMINUTE, 
    ICAL2_RECUR_BYHOUR, 
    ICAL2_RECUR_BYDAY, 
    ICAL2_RECUR_BYMONTHDAY, 
    ICAL2_RECUR_BYYEARDAY, 
    ICAL2_RECUR_BYWEEKNO, 
    ICAL2_RECUR_BYMONTH, 
    ICAL2_RECUR_BYSETPOS, 
    ICAL2_RECUR_WKST
};

enum ICal2RecurFreqTypes {
    ICAL2_RECUR_SECONDLY = 1, 
    ICAL2_RECUR_MINUTELY, 
    ICAL2_RECUR_HOURLY, 
    ICAL2_RECUR_DAILY, 
    ICAL2_RECUR_MONTHLY, 
    ICAL2_RECUR_WEEKLY, 
    ICAL2_RECUR_YEARLY
};

enum ICal2RecurWeekdays {
    ICAL2_RECUR_WEEKDAY_NONE = 0, 
    ICAL2_RECUR_WEEKDAY_SUNDAY, 
    ICAL2_RECUR_WEEKDAY_MONDAY, 
    ICAL2_RECUR_WEEKDAY_TUESDAY, 
    ICAL2_RECUR_WEEKDAY_WEDNESDAY, 
    ICAL2_RECUR_WEEKDAY_THURSDAY, 
    ICAL2_RECUR_WEEKDAY_FRIDAY, 
    ICAL2_RECUR_WEEKDAY_SATURDAY
};

enum ICal2StatusTypes {
    ICAL2_STATUS_TENTATIVE = 1, 
    ICAL2_STATUS_CONFIRMED, 
    ICAL2_STATUS_CANCELLED, 
    ICAL2_STATUS_NEEDS_ACTION, 
    ICAL2_STATUS_COMPLETED, 
    ICAL2_STATUS_IN_PROCESS, 
    ICAL2_STATUS_DRAFT, 
    ICAL2_STATUS_FINAL 
};

typedef struct _ICal2AltRepParameter {
    unsigned char *uri;
} ICal2AltRepParameter;

typedef struct _ICal2CnParameter {
    unsigned char *value;
} ICal2CnParameter;

typedef struct _ICal2CuTypeParameter {
    enum ICal2CalendarUserTypes type;
    unsigned char *value;
} ICal2CuTypeParameter;

typedef struct _ICal2DelFromParameter {
    unsigned char *calAddress;
} ICal2DelFromParameter;

typedef struct _ICal2DelToParameter {
    unsigned char *calAddress;
} ICal2DelToParameter;

typedef struct _ICal2DirParameter {
    unsigned char *uri;
} ICal2DirParameter;

typedef struct _ICal2EncodingParameter {
    enum ICal2EncodingTypes type;
    unsigned char *value;
} ICal2EncodingParameter;

typedef struct _ICal2FormatParameter {
    enum ICal2FormatTypes type;
    unsigned char *value;
} ICal2FormatParameter;

typedef struct _ICal2FreeBusyParameter {
    enum ICal2FreeBusyTypes type;
    unsigned char *value;
} ICal2FreeBusyParameter;

typedef struct _ICal2LanguageParameter {
    unsigned char *value;
} ICal2LanguageParameter;

typedef struct _ICal2MemberParameter {
    unsigned char *calAddress;
} ICal2MemberParameter;

typedef struct _ICal2PartStatParameter {
    enum ICal2ParticipantStatusTypes type;
    unsigned char *status;
} ICal2PartStatParameter;

typedef struct _ICal2RangeParameter {
    enum ICal2RangeTypes type;
    unsigned char *value;
} ICal2RangeParameter;

typedef struct _ICal2RelatedParameter {
    enum ICal2RelatedTypes type;
    unsigned char *value;
} ICal2RelatedParameter;

typedef struct _ICal2RelTypeParameter {
    enum ICal2RelationshipTypes type;
    unsigned char *value;
} ICal2RelTypeParameter;

typedef struct _ICal2RoleParameter {
    enum ICal2RoleTypes type;
    unsigned char *value;
} ICal2RoleParameter;

typedef struct _ICal2RsvpParameter {
    BOOL yes;
    unsigned char *value;
} ICal2RsvpParameter;

typedef struct _ICal2SentByParameter {
    unsigned char *calAddress;
} ICal2SentByParameter;

typedef struct _ICal2TzIDParameter {
    unsigned char *value;
} ICal2TzIDParameter;

typedef struct _ICal2ValueParameter {
    enum ICal2ValueTypes type;
    unsigned char *value;
} ICal2ValueParameter;

typedef struct _ICal2XParameter {
    unsigned char *value;
} ICal2XParameter;

typedef struct _ICal2IanaParameter {
    unsigned char *value;
} ICal2IanaParameter;

typedef union {
    ICal2AltRepParameter altRep;
    ICal2CnParameter cn;
    ICal2CuTypeParameter cuType;
    ICal2DelFromParameter delFrom;
    ICal2DelToParameter delTo;
    ICal2DirParameter dir;
    ICal2EncodingParameter encoding;
    ICal2FormatParameter format;
    ICal2FreeBusyParameter freeBusy;
    ICal2LanguageParameter language;
    ICal2MemberParameter member;
    ICal2PartStatParameter partStat;
    ICal2RangeParameter range;
    ICal2RelatedParameter related;
    ICal2RelTypeParameter relType;
    ICal2RoleParameter role;
    ICal2RsvpParameter rsvp;
    ICal2SentByParameter sentBy;
    ICal2TzIDParameter tzID;
    ICal2ValueParameter value;
    ICal2XParameter x;
    ICal2IanaParameter iana;
} ICal2ParameterDetail;

typedef struct _ICal2Parameter {
    struct _ICal2Parameter *prev;
    struct _ICal2Parameter *next;

    enum ICal2ParamaterTypes type;

    unsigned char *name;

    struct {
        unsigned long allocated;
        unsigned long count;

        ICal2ParameterDetail detail[1];
    } items;
} ICal2Parameter;

typedef struct _ICal2BinaryValue {
    unsigned char *encoded;
} ICal2BinaryValue;

typedef struct _ICal2BooleanValue {
    BOOL yes;
    unsigned char *value;
} ICal2BooleanValue;

typedef struct _ICal2CalAddressValue {
    unsigned char *uri;
} ICal2CalAddressValue;

typedef struct _ICal2DateValue {
    int year;
    int month;
    int day;

    unsigned char *value;
} ICal2DateValue;

typedef struct _ICal2DurationValue {
    int sign;
    int weeks;
    int days;
    int hours;
    int minutes;
    int seconds;
    int total;

    unsigned char *value;
} ICal2DurationValue;

typedef struct _ICal2FloatValue {
    double f;

    unsigned char *value;
} ICal2FloatValue;

typedef struct _ICal2IntegerValue {
    int i;

    unsigned char *value;
} ICal2IntegerValue;

typedef struct _ICal2TextValue {
    unsigned char *value;
} ICal2TextValue;

typedef struct _ICal2TimeValue {
    int hour;
    int minute;
    int second;

    BOOL utc;

    unsigned char *value;
} ICal2TimeValue;

typedef struct _ICal2DateTimeValue {
    ICal2DateValue date;
    ICal2TimeValue time;

    unsigned char *value;
} ICal2DateTimeValue;

typedef struct _ICal2RecurByListValue {
    enum ICal2RecurWeekdays weekday;

    int number;
    unsigned char *string;
} ICal2RecurByListValue;

typedef struct _ICal2RecurByList {
    struct _ICal2RecurByList *next;
    struct _ICal2RecurByList *prev;

    unsigned char *name;

    unsigned int count;
    unsigned int allocated;

    ICal2RecurByListValue list[];
} ICal2RecurByList;

typedef struct _ICal2RecurValue {
    enum ICal2RecurFreqTypes freq;
    enum ICal2ValueTypes untilType;
    enum ICal2RecurWeekdays weekday;

    unsigned int interval;

    ICal2RecurByList *bySeconds;
    ICal2RecurByList *byMinutes;
    ICal2RecurByList *byHours;
    ICal2RecurByList *byDays;
    ICal2RecurByList *byMonthDays;
    ICal2RecurByList *byYearDays;
    ICal2RecurByList *byWeekNumbers;
    ICal2RecurByList *byMonths;
    ICal2RecurByList *bySets;
    ICal2RecurByList *xNames;

    union {
        unsigned int count;
        ICal2DateValue date;
        ICal2DateTimeValue end;
    } until;

    unsigned char *value;
} ICal2RecurValue;

typedef struct _ICal2PeriodValue {
    BOOL duration;

    ICal2DateTimeValue start;

    union {
        ICal2DateTimeValue stop;
        ICal2DurationValue duration;
    } u;

    unsigned char *value;
} ICal2PeriodValue;

typedef struct _ICal2UriValue {
    unsigned char *value;
} ICal2UriValue;

typedef struct _ICal2UtcOffsetValue {
    int hours;
    int minutes;
    int seconds;
    int offset;

    unsigned char *value;
} ICal2UtcOffsetValue;

typedef struct _ICal2StatusValue {
    enum ICal2StatusTypes type;

    unsigned char *value;
} ICal2StatusValue;

typedef struct _ICal2RequestStatusValue {
    unsigned char *code;
    unsigned char *description;
    unsigned char *exception;
} ICal2RequestStatusValue;

typedef union {
    ICal2BinaryValue binary;
    ICal2BooleanValue boolean;
    ICal2CalAddressValue calAddress;
    ICal2DateValue date;
    ICal2DateTimeValue dateTime;
    ICal2DurationValue duration;
    ICal2FloatValue f;
    ICal2IntegerValue i;
    ICal2PeriodValue period;
    ICal2RecurValue recur;
    ICal2TextValue text;
    ICal2TimeValue time;
    ICal2UriValue uri;
    ICal2UtcOffsetValue utcOffset;
    ICal2StatusValue status;
    ICal2RequestStatusValue requestStatus;
} ICal2ValueDetail;

typedef struct _ICal2Value {
    enum ICal2ValueTypes type;

    struct {
        unsigned long allocated;
        unsigned long count;

        ICal2ValueDetail detail[1];
    } items;
} ICal2Value;

typedef struct _ICal2Property {
    struct _ICal2Property *next;
    struct _ICal2Property *prev;

    enum ICal2PropertyTypes type;
    enum ICal2PropertyFlags flags;

    unsigned char *name;

    struct {
        struct _ICal2Property *next;
        struct _ICal2Property *prev;
    } list;

    struct {
        ICal2Parameter *head;
        ICal2Parameter *tail;
    } parameters;

    ICal2Value *value;
} ICal2Property;

typedef struct _ICal2Properties {
    ICal2Property *head;
    ICal2Property *tail;
} ICal2Properties;

typedef struct _ICal2VCalendarComponent {
    struct _ICal2Component *next;
    struct _ICal2Component *prev;

    ICal2Property *prodID;
    ICal2Property *version;

    ICal2Property *calScale;
    ICal2Property *method;

    ICal2Properties xProperties;
    ICal2Properties iana;
} ICal2VCalendarComponent;

typedef struct _ICal2VAlarmComponent {
    struct _ICal2Component *next;
    struct _ICal2Component *prev;

    enum ICal2AlarmType type;

    ICal2Property *action;
    ICal2Property *trigger;
    ICal2Property *duration;
    ICal2Property *repeat;
    ICal2Property *description;
    ICal2Property *summary;

    ICal2Properties attach;
    ICal2Properties attendee;
    ICal2Properties xProperties;
    ICal2Properties iana;
} ICal2VAlarmComponent;

typedef struct _ICal2VEventComponent {
    struct _ICal2Component *next;
    struct _ICal2Component *prev;

    ICal2Property *course;
    ICal2Property *created;
    ICal2Property *description;
    ICal2Property *dtStart;
    ICal2Property *geo;
    ICal2Property *lastModified;
    ICal2Property *location;
    ICal2Property *organizer;
    ICal2Property *priority;
    ICal2Property *dtStamp;
    ICal2Property *seq;
    ICal2Property *status;
    ICal2Property *summary;
    ICal2Property *transp;
    ICal2Property *uid;
    ICal2Property *url;
    ICal2Property *recurID;

    ICal2Property *dtEnd;
    ICal2Property *duration;

    ICal2Properties attach;
    ICal2Properties attendee;
    ICal2Properties categories;
    ICal2Properties comment;
    ICal2Properties contact;
    ICal2Properties exDate;
    ICal2Properties exRule;
    ICal2Properties rstatus;
    ICal2Properties related;
    ICal2Properties resources;
    ICal2Properties rDate;
    ICal2Properties rRule;
    ICal2Properties xProperties;
    ICal2Properties iana;
} ICal2VEventComponent;

typedef struct _ICal2VToDoComponent {
    struct _ICal2Component *next;
    struct _ICal2Component *prev;

    ICal2Property *course;
    ICal2Property *completed;
    ICal2Property *created;
    ICal2Property *description;
    ICal2Property *dtStamp;
    ICal2Property *dtStart;
    ICal2Property *geo;
    ICal2Property *lastModified;
    ICal2Property *location;
    ICal2Property *organizer;
    ICal2Property *percent;
    ICal2Property *priority;
    ICal2Property *recurID;
    ICal2Property *seq;
    ICal2Property *status;
    ICal2Property *summary;
    ICal2Property *uid;
    ICal2Property *url;

    ICal2Property *due;
    ICal2Property *duration;

    ICal2Properties attach;
    ICal2Properties attendee;
    ICal2Properties categories;
    ICal2Properties comment;
    ICal2Properties contact;
    ICal2Properties exDate;
    ICal2Properties exRule;
    ICal2Properties rStatus;
    ICal2Properties related;
    ICal2Properties resources;
    ICal2Properties rDate;
    ICal2Properties rRule;
    ICal2Properties xProperties;
    ICal2Properties iana;
} ICal2VToDoComponent;

typedef struct _ICal2VJournalComponent {
    struct _ICal2Component *next;
    struct _ICal2Component *prev;

    ICal2Property *course;
    ICal2Property *created;
    ICal2Property *description;
    ICal2Property *dtStart;
    ICal2Property *dtStamp;
    ICal2Property *lastModified;
    ICal2Property *organizer;
    ICal2Property *recurID;
    ICal2Property *seq;
    ICal2Property *status;
    ICal2Property *summary;
    ICal2Property *uid;
    ICal2Property *url;

    ICal2Properties attach;
    ICal2Properties attendee;
    ICal2Properties categories;
    ICal2Properties comment;
    ICal2Properties contact;
    ICal2Properties exDate;
    ICal2Properties exRule;
    ICal2Properties related;
    ICal2Properties rDate;
    ICal2Properties rRule;
    ICal2Properties rStatus;
    ICal2Properties xProperties;
    ICal2Properties iana;
} ICal2VJournalComponent;

typedef struct _ICal2VFreeBusyComponent {
    struct _ICal2Component *next;
    struct _ICal2Component *prev;

    ICal2Property *contact;
    ICal2Property *dtStart;
    ICal2Property *dtEnd;
    ICal2Property *duration;
    ICal2Property *dtStamp;
    ICal2Property *organizer;
    ICal2Property *uid;
    ICal2Property *url;

    ICal2Properties attendee;
    ICal2Properties comment;
    ICal2Properties freeBusy;
    ICal2Properties rStatus;
    ICal2Properties xProperties;
    ICal2Properties iana;
} ICal2VFreeBusyComponent;

typedef struct _ICal2VTimeZoneComponent {
    struct _ICal2Component *next;
    struct _ICal2Component *prev;

    ICal2Property *tzID;
    ICal2Property *tzUrl;
    ICal2Property *lastModified;

    ICal2Properties xProperties;
    ICal2Properties iana;
} ICal2VTimeZoneComponent;

typedef struct _ICal2XComponent {
    struct _ICal2Component *next;
    struct _ICal2Component *prev;

    ICal2Properties xProperties;
    ICal2Properties iana;
} ICal2XComponent;

typedef struct _ICal2IanaComponent {
    struct _ICal2Component *next;
    struct _ICal2Component *prev;

    ICal2Properties xProperties;
    ICal2Properties iana;
} ICal2IanaComponent;

typedef struct _ICal2DaylightComponent {
    struct _ICal2Component *next;
    struct _ICal2Component *prev;

    ICal2Property *dtStart;
    ICal2Property *tzOffsetTo;
    ICal2Property *tzOffsetFrom;

    ICal2Properties comment;
    ICal2Properties rDate;
    ICal2Properties rRule;
    ICal2Properties tzName;
    ICal2Properties xProperties;
    ICal2Properties iana;
} ICal2DaylightComponent;

typedef struct _ICal2StandardComponent {
    struct _ICal2Component *next;
    struct _ICal2Component *prev;

    ICal2Property *dtStart;
    ICal2Property *tzOffsetTo;
    ICal2Property *tzOffsetFrom;

    ICal2Properties comment;
    ICal2Properties rDate;
    ICal2Properties rRule;
    ICal2Properties tzName;
    ICal2Properties xProperties;
    ICal2Properties iana;
} ICal2StandardComponent;

typedef union {
    ICal2VCalendarComponent vCalendar;
    ICal2VAlarmComponent vAlarm;
    ICal2VEventComponent vEvent;
    ICal2VToDoComponent vToDo;
    ICal2VJournalComponent vJournal;
    ICal2VFreeBusyComponent vFreeBusy;
    ICal2VTimeZoneComponent vTimeZone;
    ICal2XComponent x;
    ICal2IanaComponent iana;
    ICal2DaylightComponent daylight;
    ICal2StandardComponent standard;
} ICal2ComponentDetail;

typedef struct _ICal2Component {
    struct _ICal2Component *parent;
    struct _ICal2Component *next;
    struct _ICal2Component *prev;
    struct _ICal2Component *head;
    struct _ICal2Component *tail;

    enum ICal2ComponentType type;
    enum ICal2ComponentFlags flags;

    unsigned char *name;

    struct {
        int length;
        int comp;
        int prop;
    } counts;

    ICal2Properties properties;

    ICal2ComponentDetail detail;
} ICal2Component;

typedef struct _ICal2ComponentList {
    ICal2Component *head;
    ICal2Component *tail;
} ICal2ComponentList;

typedef struct _ICal2ObjectParsing {
    unsigned char *cur;
    unsigned char *dest;
    unsigned char *name;
    unsigned char *param;
    unsigned char *value;

    ICal2Component *component;
    ICal2Property *property;
    void *detail;

    struct _ICal2Token *token;
} ICal2ObjectParsing;

typedef struct _ICal2Object {
    enum ICal2ObjectFlags flags;

    unsigned long length;

    ICal2ObjectParsing parse;

    struct {
        struct _ICal2Component *head;
        struct _ICal2Component *tail;

        int count;

        ICal2ComponentList vCalendar;
        ICal2ComponentList vAlarm;
        ICal2ComponentList vEvent;
        ICal2ComponentList vToDo;
        ICal2ComponentList vJournal;
        ICal2ComponentList vFreeBusy;
        ICal2ComponentList vTimeZone;
        ICal2ComponentList x;
        ICal2ComponentList iana;
        ICal2ComponentList daylight;
        ICal2ComponentList standard;
   } components;

    unsigned char data[];
} ICal2Object;

typedef BOOL (* ICal2Parser)(ICal2Object *ical);

typedef struct _ICal2Token {
    unsigned char *name;
    unsigned long nameLen;
    unsigned long type;

    struct {
        ICal2Parser property;
        ICal2Parser value;
    } parser;
} ICal2Token;

/* ical2.c */
extern HulaKeywordIndex *ICal2ParameterIndex;
extern HulaKeywordIndex *ICal2RecurIndex;
extern HulaKeywordIndex *ICal2PropertyIndex;
extern HulaKeywordIndex *ICal2ComponentIndex;

extern ICal2Token ICal2ParameterTokens[];
extern ICal2Token ICal2RecurTokens[];
extern ICal2Token ICal2PropertyTokens[];
extern ICal2Token ICal2ComponentTokens[];

extern const unsigned char ICal2CharDefinitions[];
extern const unsigned char *ICal2ParameterKeywords[];
extern const unsigned char *ICal2ValueKeywords[];

EXPORT BOOL OpenICal2Parser(void);
EXPORT void CloseICal2Parser(void);
EXPORT ICal2Object *ICal2ObjectParse(void *object, unsigned long flags, unsigned long length);
EXPORT void ICal2ObjectFree(ICal2Object *ical);

/* param.c */
BOOL ICal2AltRepParameterParser(ICal2Object *ical);
BOOL ICal2CnParameterParser(ICal2Object *ical);
BOOL ICal2CuTypeParameterParser(ICal2Object *ical);
BOOL ICal2DelegatedFromParameterParser(ICal2Object *ical);
BOOL ICal2DelegatedToParameterParser(ICal2Object *ical);
BOOL ICal2DirParameterParser(ICal2Object *ical);
BOOL ICal2EncodingParameterParser(ICal2Object *ical);
BOOL ICal2FmtTypeParameterParser(ICal2Object *ical);
BOOL ICal2FbTypeParameterParser(ICal2Object *ical);
BOOL ICal2LanguageParameterParser(ICal2Object *ical);
BOOL ICal2MemberParameterParser(ICal2Object *ical);
BOOL ICal2PartStatParameterParser(ICal2Object *ical);
BOOL ICal2RangeParameterParser(ICal2Object *ical);
BOOL ICal2RelatedParameterParser(ICal2Object *ical);
BOOL ICal2RelTypeParameterParser(ICal2Object *ical);
BOOL ICal2RoleParameterParser(ICal2Object *ical);
BOOL ICal2RsvpParameterParser(ICal2Object *ical);
BOOL ICal2SentByParameterParser(ICal2Object *ical);
BOOL ICal2TzIDParameterParser(ICal2Object *ical);
BOOL ICal2ValueParameterParser(ICal2Object *ical);
BOOL ICal2XParameterParser(ICal2Object *ical);
BOOL ICal2IanaParameterParser(ICal2Object *ical);

/* value.c */
BOOL ICal2BinaryValueParser(ICal2Object *ical);
BOOL ICal2BooleanValueParser(ICal2Object *ical);
BOOL ICal2CalAddressValueParser(ICal2Object *ical);
BOOL ICal2DateValueParser(ICal2Object *ical);
BOOL ICal2DateTimeValueParser(ICal2Object *ical);
BOOL ICal2DurationValueParser(ICal2Object *ical);
BOOL ICal2FloatValueParser(ICal2Object *ical);
BOOL ICal2IntegerValueParser(ICal2Object *ical);
BOOL ICal2PeriodValueParser(ICal2Object *ical);
BOOL ICal2RecurFreqParser(ICal2Object *ical);
BOOL ICal2RecurUntilParser(ICal2Object *ical);
BOOL ICal2RecurCountParser(ICal2Object *ical);
BOOL ICal2RecurIntervalParser(ICal2Object *ical);
BOOL ICal2RecurBySecondParser(ICal2Object *ical);
BOOL ICal2RecurByMinuteParser(ICal2Object *ical);
BOOL ICal2RecurByHourParser(ICal2Object *ical);
BOOL ICal2RecurByDayParser(ICal2Object *ical);
BOOL ICal2RecurByMonthDayParser(ICal2Object *ical);
BOOL ICal2RecurByYearDayParser(ICal2Object *ical);
BOOL ICal2RecurByWeekNoParser(ICal2Object *ical);
BOOL ICal2RecurByMonthParser(ICal2Object *ical);
BOOL ICal2RecurBySetPosParser(ICal2Object *ical);
BOOL ICal2RecurWkstParser(ICal2Object *ical);
BOOL ICal2RecurValueParser(ICal2Object *ical);
BOOL ICal2TextValueParser(ICal2Object *ical);
BOOL ICal2TimeValueParser(ICal2Object *ical);
BOOL ICal2UriValueParser(ICal2Object *ical);
BOOL ICal2UtcOffsetValueParser(ICal2Object *ical);
BOOL ICal2ValueParser(ICal2Object *ical);
BOOL ICal2AttachValueParser(ICal2Object *ical);
BOOL ICal2GeoValueParser(ICal2Object *ical);
BOOL ICal2DateDateTimeValueParser(ICal2Object *ical);
BOOL ICal2TriggerValueParser(ICal2Object *ical);
BOOL ICal2StatusValueParser(ICal2Object *ical);
BOOL ICal2RequestStatusValueParser(ICal2Object *ical);

/* prop.c */
BOOL ICal2PropertyParser(ICal2Object *ical);
BOOL ICal2BeginPropertyParser(ICal2Object *ical);
BOOL ICal2EndPropertyParser(ICal2Object *ical);
BOOL ICal2XPropertyParser(ICal2Object *ical);
BOOL ICal2IanaPropertyParser(ICal2Object *ical);

/* comp.c */
BOOL ICal2DaylightComponentParser(ICal2Object *ical);
BOOL ICal2StandardComponentParser(ICal2Object *ical);
BOOL ICal2AlarmComponentParser(ICal2Object *ical);
BOOL ICal2CalendarComponentParser(ICal2Object *ical);
BOOL ICal2EventComponentParser(ICal2Object *ical);
BOOL ICal2FreeBusyComponentParser(ICal2Object *ical);
BOOL ICal2JournalComponentParser(ICal2Object *ical);
BOOL ICal2TimeZoneComponentParser(ICal2Object *ical);
BOOL ICal2ToDoComponentParser(ICal2Object *ical);
BOOL ICal2XComponentParser(ICal2Object *ical);
BOOL ICal2IanaComponentParser(ICal2Object *ical);

#endif /* _LIBICAL2_H */
