/****************************************************************************
 *
 * Copyright (c) 2001-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>

#include <xpl.h>
#include <memmgr.h>
#include <logger.h>
#include <hulautil.h>
#include <mdb.h>
#include <nmap.h>
#include <nmlib.h>
#include <msgapi.h>
#include <streamio.h>
#include <openssl/md5.h>
#include <openssl/ssl.h>
#include <openssl/err.h>
#include <connio.h>

#include "calagent.h"

#if defined(RELEASE_BUILD)
#define CalClientAlloc() MemPrivatePoolGetEntry(Cal.nmap.pool)
#else
#define CalClientAlloc() MemPrivatePoolGetEntryDebug(Cal.nmap.pool, __FILE__, __LINE__)
#endif

static void SignalHandler(int sigtype);

#define QUEUE_WORK_TO_DO(c, id, r) \
        { \
            XplWaitOnLocalSemaphore(Cal.nmap.semaphore); \
            if (XplSafeRead(Cal.nmap.worker.idle)) { \
                (c)->queue.previous = NULL; \
                if (((c)->queue.next = Cal.nmap.worker.head) != NULL) { \
                    (c)->queue.next->queue.previous = (c); \
                } else { \
                    Cal.nmap.worker.tail = (c); \
                } \
                Cal.nmap.worker.head = (c); \
                (r) = 0; \
            } else { \
                XplSafeIncrement(Cal.nmap.worker.active); \
                XplSignalBlock(); \
                XplBeginThread(&(id), HandleConnection, 24 * 1024, XplSafeRead(Cal.nmap.worker.active), (r)); \
                XplSignalHandler(SignalHandler); \
                if (!(r)) { \
                    (c)->queue.previous = NULL; \
                    if (((c)->queue.next = Cal.nmap.worker.head) != NULL) { \
                        (c)->queue.next->queue.previous = (c); \
                    } else { \
                        Cal.nmap.worker.tail = (c); \
                    } \
                    Cal.nmap.worker.head = (c); \
                } else { \
                    XplSafeDecrement(Cal.nmap.worker.active); \
                    (r) = -1; \
                } \
            } \
            XplSignalLocalSemaphore(Cal.nmap.semaphore); \
        }


CalGlobals Cal;

static BOOL 
CalClientAllocCB(void *buffer, void *data)
{
    register CalClient *c = (CalClient *)buffer;

    memset(c, 0, sizeof(CalClient));

    return(TRUE);
}

static void 
CalClientFree(CalClient *client)
{
    register CalClient *c = client;

    if (c->conn) {
        ConnClose(c->conn, 1);
        ConnFree(c->conn);
        c->conn = NULL;
    }

    MemPrivatePoolReturnEntry(c);

    return;
}

static void 
FreeClientData(CalClient *client)
{
    if (client && !(client->flags & CAL_CLIENT_FLAG_EXITING)) {
        client->flags |= CAL_CLIENT_FLAG_EXITING;

        if (client->conn) {
            ConnClose(client->conn, 1);
            ConnFree(client->conn);
            client->conn = NULL;
        }

        if (client->iCal) {
            ICalFreeObjects(client->iCal);
            client->iCal = NULL;
        }

        if (client->vs) {
            MDBDestroyValueStruct(client->vs);
            client->vs = NULL;
        }

        if (client->envelope) {
            MemFree(client->envelope);
            client->envelope = NULL;
        }

        if (client->from) {
            MemFree(client->from);
            client->from = NULL;
        }
    }

    return;
}

/* The allocation is made in one chunk, so add the strings to the end of attendee */
static ICalVAttendee *
CreateNewAttendee(char status, char role, char rsvp, char type, char *address, char *delegatedTo, char *delegatedFrom, char *userName)
{
    int userNameOffset = sizeof(ICalVAttendee);
    int addressOffset = userNameOffset + strlen(userName) + 1;
    int delegatedToOffset = addressOffset + strlen(address) + 1;
    int delegatedFromOffset = delegatedToOffset + strlen(delegatedTo) + 1;
    int totalLen = delegatedFromOffset + strlen(delegatedFrom) + 1;
    ICalVAttendee *attendee = MemMalloc(totalLen);

    if (attendee) {
        strcpy((char *)attendee + userNameOffset, userName);
        ((char *)attendee)[addressOffset - 1] = '\0';
        strcpy((char *)attendee + addressOffset,address);
        ((char *)attendee)[delegatedToOffset - 1] = '\0';
        ((char *)attendee)[delegatedToOffset] = '\0'; /* remove beginning quote */
        strcpy((char *)attendee + delegatedToOffset, delegatedTo);
        ((char *)attendee)[delegatedFromOffset - 2] = '\0'; /* remove ending quote */
        ((char *)attendee)[delegatedFromOffset - 1] = '\0';
        ((char *)attendee)[delegatedFromOffset] = '\0'; /* remove beginning quote */
        strcpy((char *)attendee + delegatedFromOffset, delegatedFrom);
        ((char *)attendee)[totalLen - 2] = '\0'; /* remove ending quote */
        ((char *)attendee)[totalLen - 1] = '\0';

        attendee->Name = (char *)attendee + userNameOffset;
        attendee->Address = (char *)attendee + addressOffset;

        if (((char *)attendee)[delegatedToOffset + 1]) {
            attendee->Delegated = ((char *)attendee) + delegatedToOffset + 1;
        } else {
            attendee->Delegated = NULL;
        }

        if (((char *)attendee)[delegatedFromOffset + 1]) {
            attendee->DelegatedFrom = ((char *)attendee) + delegatedFromOffset + 1;
        } else {
            attendee->DelegatedFrom = NULL;
        }

        if (rsvp == 'Y') {
            attendee->RSVP = 1;
        } else {
            attendee->RSVP = 0;
        }

        switch (type) { 
            case NMAP_CAL_TYPE_INDIVIDUAL: attendee->Type = ICAL_CUTYPE_INDIVIDUAL; break;
            case NMAP_CAL_TYPE_GROUP: attendee->Type = ICAL_CUTYPE_GROUP; break;
            case NMAP_CAL_TYPE_RESOURCE: attendee->Type = ICAL_CUTYPE_RESOURCE; break;
            case NMAP_CAL_TYPE_ROOM: attendee->Type = ICAL_CUTYPE_ROOM; break;
            case NMAP_CAL_TYPE_UNKNOWN: attendee->Type = ICAL_CUTYPE_UNKNOWN; break;
        }
        
        switch (role) { 
            case NMAP_CAL_ROLE_CHAIR: attendee->Role = ICAL_ROLE_CHAIR; break;
            case NMAP_CAL_ROLE_REQUIRED: attendee->Role = ICAL_ROLE_REQUIRED_PARTICIPANT; break;
            case NMAP_CAL_ROLE_OPTIONAL: attendee->Role = ICAL_ROLE_OPTIONAL_PARTICIPANT; break;
            case NMAP_CAL_ROLE_NOT: attendee->Role = ICAL_ROLE_NON_PARTICIPANT; break;
        }
        switch (status) {
            default:
            case NMAP_CAL_STATE_NEED_ACT:  attendee->State = ICAL_PARTSTAT_NEEDS_ACTION; break;
            case NMAP_CAL_STATE_ACCEPTED:  attendee->State = ICAL_PARTSTAT_ACCEPTED; break;
            case NMAP_CAL_STATE_DECLINED:  attendee->State = ICAL_PARTSTAT_DECLINED; break;
            case NMAP_CAL_STATE_TENTATIVE: attendee->State = ICAL_PARTSTAT_TENTATIVE; break;
            case NMAP_CAL_STATE_COMPLETED: attendee->State = ICAL_PARTSTAT_COMPLETED; break;
            case NMAP_CAL_STATE_INPROCESS: attendee->State = ICAL_PARTSTAT_IN_PROCESS; break;
            case NMAP_CAL_STATE_DELEGATED: attendee->State = ICAL_PARTSTAT_DELEGATED; break; 
        }

        attendee->Next = NULL;
    }

    return(attendee);
}

static int 
GrabOrganizersICalObject(CalClient *client, ICalObject **lppICal, long unsigned int entryID) 
{
    int ccode;
    char status;
    char role;
    char rsvp;
    char type;
    char address[MDB_MAX_OBJECT_CHARS + 1];
    char delegatedTo[MDB_MAX_OBJECT_CHARS + 1];
    char delegatedFrom[MDB_MAX_OBJECT_CHARS + 1];
    char userName[MDB_MAX_OBJECT_CHARS + 1];
    ICalVAttendee *attendee;
    ICalObject *iCal = MemMalloc(sizeof(ICalObject));

    if (iCal) {
        memset(iCal, 0, sizeof(ICalObject));

        if (((ccode = NMAPSendCommandF(client->conn, "CSATND %lu\r\n", entryID)) != -1) 
                && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE)) != -1) 
                && (ccode != 1000)) {
            while (((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE)) != -1) 
                    && (ccode != 1000)) {
                if (sscanf(client->line,"%c %c %c %c %s %s %s %s\n", 
                              &status, &role, &rsvp, &type, &address[0], &delegatedTo[0], &delegatedFrom[0], &userName[0]) == 8) {
                    for (attendee = iCal->Attendee; attendee; attendee = attendee->Next) {
                        /* Compare to the address we are trying to add */
                        if (XplStrCaseCmp(address, attendee->Address) == 0) {
                            /* Update the attendee */
                            switch(status) {
                                case  NMAP_CAL_STATE_NEED_ACT: attendee->State = ICAL_PARTSTAT_NEEDS_ACTION; break;
                                case  NMAP_CAL_STATE_ACCEPTED: attendee->State = ICAL_PARTSTAT_ACCEPTED; break;
                                case  NMAP_CAL_STATE_DECLINED: attendee->State = ICAL_PARTSTAT_DECLINED; break;
                                case  NMAP_CAL_STATE_TENTATIVE: attendee->State = ICAL_PARTSTAT_TENTATIVE; break;
                                case  NMAP_CAL_STATE_COMPLETED: attendee->State = ICAL_PARTSTAT_COMPLETED; break;
                                case  NMAP_CAL_STATE_INPROCESS: attendee->State = ICAL_PARTSTAT_IN_PROCESS; break;
                                case  ICAL_PARTSTAT_DELEGATED: attendee->State = ICAL_PARTSTAT_DELEGATED; break; 
                                /* This is tricky since we may not know who we are delegating to or from
                                   if we are the organizer. 
                                   When we copy the message we don't change the ICAL object in this case.
                                   We just update the attendee status to delegated.  */
                            }

                            break;
                        } else if (!attendee->Next) {
                            /* If next pointer is NULL and we still haven't found the attendee, create him */
                            if ((attendee->Next = CreateNewAttendee(status, role, rsvp, type, address, delegatedTo, delegatedFrom, userName))){
                                continue;
                            } else {
                                ICalFreeObjects(iCal);
                                return(0);
                            }
                        }
                    }

                    if (!iCal->Attendee) {
                        /* no attendees yet */
                        if ((iCal->Attendee = CreateNewAttendee(status, role, rsvp, type, address, delegatedTo, delegatedFrom, userName))){
                            continue;
                        } else {
                            ICalFreeObjects(iCal);
                            return(-1);
                        }
                    }
                } else { 
                    /* sscanf failed */
                    continue; 
                }
            }
        } else {
            ccode = -1;
        }

        if (ccode != -1) {
            *lppICal = iCal;
            return(ccode);
        }

        ICalFreeObjects(iCal);
    }

    return(-1);
}

static int 
PrintICalAttendee(char *buffer, ICalVAttendee *attendee)
{
    char *ptr = buffer + sprintf(buffer, "ATTENDEE;CN=%s;", attendee->Name);

    /* calendar user type */
    switch(attendee->Type) {
        case ICAL_CUTYPE_INDIVIDUAL: {
            break;
        }

        case ICAL_CUTYPE_GROUP: {
            strcpy(ptr, "CUTYPE=GROUP;");
            ptr += 13;
            break;
        }

        case ICAL_CUTYPE_RESOURCE: {
            strcpy(ptr, "CUTYPE=RESOURCE;");
            ptr += 16;
            break;
        }

        case ICAL_CUTYPE_ROOM: {
            strcpy(ptr, "CUTYPE=ROOM;");
            ptr += 12;
            break; 
        }

        case ICAL_CUTYPE_UNKNOWN: {
            strcpy(ptr, "CUTYPE=UNKNOWN;");
            ptr += 15;
            break; 
        }

        default: {
            break;
        }
    }

    /* participation status */
    switch(attendee->State){
        case ICAL_PARTSTAT_NEEDS_ACTION: {
            break;
        }

        case ICAL_PARTSTAT_ACCEPTED: {
            strcpy(ptr, "PARTSTAT=ACCEPTED;");
            ptr += 18;
            break;
        }

        case ICAL_PARTSTAT_DECLINED: {
            strcpy(ptr, "PARTSTAT=DECLINED;");
            ptr += 18;
            break;
        }

        case ICAL_PARTSTAT_TENTATIVE: {
            strcpy(ptr, "PARTSTAT=TENTATIVE;");
            ptr += 19;
            break;
        }

        case ICAL_PARTSTAT_DELEGATED: {
            strcpy(ptr, "PARTSTAT=DELEGATED;");
            ptr += 19;
            break;
        }

        case ICAL_PARTSTAT_COMPLETED: {
            strcpy(ptr, "PARTSTAT=COMPLETED;");
            ptr += 19;
            break;
        }

        case ICAL_PARTSTAT_IN_PROCESS: {
            strcpy(ptr, "PARTSTAT=IN-PROCESS;");
            ptr += 20;
            break;
        }

        default: {
            break;
        }
    }

    /* role */
    switch(attendee->Role){
        case ICAL_ROLE_CHAIR: {
            strcpy(ptr, "ROLE=CHAIR;");
            ptr += 11;
            break;
        }

        case ICAL_ROLE_REQUIRED_PARTICIPANT: {
            strcpy(ptr, "ROLE=REQ-PARTICIPANT;");
            ptr += 21;
            break;
        }

        case ICAL_ROLE_OPTIONAL_PARTICIPANT: {
            strcpy(ptr, "ROLE=OPT-PARTICIPANT;");
            ptr += 21;
            break;
        }

        case ICAL_ROLE_NON_PARTICIPANT: {
            strcpy(ptr, "ROLE=NON-PARTICIPANT;");
            ptr += 21;
            break;
        }

        default: {
            break;
        }
    }

    /* check for RSVP */
    if (attendee->RSVP) {
        strcpy(ptr, "RSVP=TRUE;");
        ptr += 10;
    }

    /* check for delegated from and delegated to */
    if (attendee->Delegated) {
        ptr += sprintf(ptr, "DELEGATED-TO=\"Mailto:%s\"",attendee->Delegated);
    }

    if (attendee->DelegatedFrom) {
        ptr += sprintf(ptr, "DELEGATED-FROM=\"Mailto:%s\"",attendee->DelegatedFrom);
    }
        
    /* finally add the attendees address */
    sprintf(ptr, ":MAILTO:%s\r\n", attendee->Address);

    return(ptr - buffer);
}

static int 
SendAttendeeList(CalClient *client, char *Buffer, ICalObject *iCal) 
{
    int ccode = 0;
    ICalVAttendee *attendee;

    for (attendee = iCal->Attendee; (ccode != -1) && attendee; attendee = attendee->Next) {
        ccode = ConnWrite(client->conn, client->line, PrintICalAttendee(client->line, attendee));
    }

    return(ccode);
}

/* Create a new calendar object using updated ICal memory image. */
/* Then delete the original object. The UID is used to find new object */
static int 
UpdateCalendarObject(CalClient *client, ICalObject *iCal, long unsigned int entryID, ICalVAttendee *delegator) 
{
    int ccode;
    unsigned long size;
    unsigned char *original;
    unsigned char *begin;
    unsigned char *end;
    BOOL sent = FALSE;
    BOOL inAttendeeLine = FALSE;

    /*
      First, we read the object into memory.  Then send it back line by line, and
      as soon as we find an attendee line send the new attendee list line be line. 

      We keep track of the entryID for the new object, and then once it is there
      remove the original object, and update entryID to point to the new object,
      so that the rest of this function will work.
    */
    if (((ccode = NMAPSendCommandF(client->conn, "CSLIST %lu\r\n", entryID)) != -1) 
            && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE)) != -1)) {
        size = atol(client->line);
        original = MemMalloc(size + 1);

        if ((ccode = NMAPRead(client->conn, original, size)) == 1000) {
            original[size] = '\0';

            if (((ccode = NMAPSendCommand(client->conn, "CSSTOR MAIN\r\n", 13)) != -1) 
                    && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE)) == 2002)) {
                begin = original;
                while ((ccode != -1) && begin) {
                    end = strchr(begin, '\r');
                    if (!end) {
                        end = strchr(begin, '\n');
                    }
            
                    if (end) {
                        *end = '\0';
                    }
            
                    if (XplStrNCaseCmp(begin, "ATTENDEE", 8) == 0) {
                        inAttendeeLine = TRUE;

                        if (!sent) {
                            ccode = SendAttendeeList(client, client->line, iCal );
                            sent = TRUE;
                        }
                    } else {
                        if (!XplStrNCaseCmp(begin, "END:VCALENDAR", 13)) {
                            /* We are done with it.  We don't want to send the NIMS flags, or they will get duplicated */
                            end = NULL;
                        }

                        /* skip if we are a folded attendee line */
                        if (!inAttendeeLine || begin[0] != ' ') {
                            inAttendeeLine = FALSE;
                            if ((ccode = ConnWrite(client->conn, begin, strlen(begin))) != -1) {
                                ccode = ConnWrite(client->conn, "\r\n", 2);
                            }
                        }
                    }

                    if (end) {
                        end++;
                
                        if (*end == '\n') {
                            end++;
                        }
                
                        begin = end;
                    } else {
                        begin = NULL;
                    }
                }
            }

            /* Close it off */
            if ((ccode != -1) 
                    && ((ccode = NMAPSendCommand(client->conn, "\r\n.\r\n", 5)) != -1)) {
                ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE);
            }
        }

        MemFree(original);

        /* Delete the original */
        if ((ccode != -1) 
                && ((ccode = NMAPSendCommand(client->conn, "CSUPDA\r\n", 8)) != -1) 
                && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE)) != -1) 
                && ((ccode = NMAPSendCommandF(client->conn, "CSDELE %lu\r\n", entryID)) != -1) 
                && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE)) != -1) 
                && ((ccode = NMAPSendCommandF(client->conn, "CSPURG %lu\r\n", entryID)) != -1)) {
            ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE);
        }
    }

    return(ccode);
}



/* This function handles updating a calendar for attendee delegating to new attendee. */
/* It requires cal object to be rewritten since a new address is required for the attendee. */
/* The delegator requires an update as well. */
/* We first update the attendee list with the NIMS attendee status for the ICal Object */
/* We then update it with the given delegatee and delegator information. */
/* We then write out the new ICal Object. */
/* Note: that state at this point may be removed from the NIMS attendees into the actual ICAL object. */
/* Watch out! the attendee can delegate his appointment to more that one recipient. */
/* This is handled by adding a new recipient line for the old delegatee. And putting the new */
/* Delegatee in the object. */
static int 
SetAttendeeDelegationStatus(CalClient *client, ICalObject *iCal, unsigned long int entryID, ICalVAttendee *delegator, struct sockaddr_in *saddr, unsigned char *user)
{
    int ccode;
    BOOL delegateeFound = FALSE;
    BOOL delegatorFound = FALSE;
    ICalVAttendee *attendee = NULL;
    ICalVAttendee *prevAttendee;
    ICalVAttendee *newAttendee;
    ICalObject *orgICal =  NULL;

    /* grab the organizers ICal Object */
    ccode = GrabOrganizersICalObject(client, &orgICal, entryID);
    if (ccode != -1) {
        attendee = orgICal->Attendee;
    } else {
        return(ccode);
    }

    /* find the delegator attendee structure */
    if (QuickCmp(delegator->Address, attendee->Address)) {
        delegatorFound = TRUE;

        /* Create new entry */
        /* inherit properties from delegator where possible, use emailaddress as display name*/
        sprintf(client->line, "\"%s\"", delegator->Delegated);
        if ((newAttendee = CreateNewAttendee(NMAP_CAL_STATE_DELEGATED, delegator->Role, delegator->RSVP, delegator->Type, delegator->Address, client->line, "\"\"", delegator->Name ))) {
            /* Ok - fall through */
        } else {
            /* problem creating new attendee */
            ICalFreeObjects(orgICal);
            return(-1);
        }

        /* Fix up links */
        newAttendee->Next = attendee->Next;
        orgICal->Attendee = newAttendee;

        /* Free old entry */
        MemFree(attendee);
        attendee = newAttendee;
    } else {
        /* now update the attendee list with the new delegator, and delegatee information. */
        for (prevAttendee = attendee, attendee = attendee->Next;
            attendee != NULL;
            prevAttendee = attendee, attendee = attendee->Next) {

            /* find the delegator attendee structure */
            if (QuickCmp(delegator->Address, attendee->Address)) {
                delegatorFound = TRUE;

                /* update his delegate address with the new address */
                /* the old address should already be a separate attendee */
                if (!attendee->Delegated || XplStrCaseCmp(attendee->Delegated, delegator->Delegated)) {
                    /* create new updated entry for delegator */
                  sprintf(client->line, "\"%s\"", delegator->Delegated);
                    if ((newAttendee = CreateNewAttendee(NMAP_CAL_STATE_DELEGATED, delegator->Role, delegator->RSVP, delegator->Type, delegator->Address, client->line, "\"\"", delegator->Name ))) {
                        /* Ok - fall through */
                    } else {
                        /* problem creating new attendee */
                        ICalFreeObjects(orgICal);
                        return(FALSE);
                    }

                    /* Fix up links */
                    newAttendee->Next = attendee->Next;
                    prevAttendee->Next = newAttendee;

                    /* Free old entry */
                    MemFree(attendee);
                    attendee = newAttendee;
                }

                break;
            }
        }
    }

    if (!delegatorFound) {
        /* The delegator must be in attendee list to be legal */
        ICalFreeObjects(orgICal);
        return(0);
    }

    attendee = orgICal->Attendee;

    /* find the delegatees attendee structure or add it */
    if (QuickCmp(delegator->Delegated, attendee->Address)) {
        ICalVAttendee *newAttendee;

        delegateeFound = TRUE;

        /* see if the DelegatedFrom address needs to be updated on the existing delegated entry*/
        if (!attendee->DelegatedFrom || XplStrCaseCmp(attendee->DelegatedFrom, delegator->Address)) {
            /* Create new entry for delegatee*/
            /* inherit properties from delegator where possible, use emailaddress as display name*/
            /* update his delegatefrom address with the delegators address */
            sprintf(client->line,"\"%s\"", delegator->Address);
            if ((newAttendee = CreateNewAttendee(NMAP_CAL_STATE_NEED_ACT, NMAP_CAL_ROLE_REQUIRED, delegator->RSVP, delegator->Type, delegator->Delegated, "\"\"", client->line, delegator->Delegated ))) {
                /* Ok - fall through */
            } else {
                /* problem createing new attendee */
                ICalFreeObjects(orgICal);
                return(-1);
            }

            /* Fix up links */
            newAttendee->Next = attendee->Next;
            orgICal->Attendee = newAttendee;

            /* Free old entry */
            MemFree(attendee);
            attendee = newAttendee;
        }
    } else {
        ICalVAttendee *newAttendee;

        /* now update the attendee list with the new delegator, and delegatee information. */
        for (prevAttendee = attendee, attendee = attendee->Next;
                attendee != NULL;
                prevAttendee = attendee, attendee = attendee->Next) {

            /* find the delegator attendee structure */
            if (QuickCmp(delegator->Delegated, attendee->Address)) {
                delegateeFound = TRUE;

                /* see if the DelegatedFrom address needs to be updated on the existing delegated entry*/
                if (!attendee->DelegatedFrom || XplStrCaseCmp(attendee->DelegatedFrom, delegator->Address)) {
                    /* create new updated entry */
                    /* inherit properties from delegator where possible, use emailaddress as display name*/
                    sprintf(client->line,"\"%s\"", delegator->Address);
                    if ((newAttendee = CreateNewAttendee(NMAP_CAL_STATE_NEED_ACT, NMAP_CAL_ROLE_REQUIRED, delegator->RSVP, delegator->Type, delegator->Delegated, "\"\"", client->line, delegator->Delegated ))) {
                        /* Ok - fall through */
                    } else {
                        /* problem creating new attendee */
                        ICalFreeObjects(orgICal);
                        return(-1);
                    }

                    /* Fix up links */
                    newAttendee->Next = attendee->Next;
                    prevAttendee->Next = newAttendee;
                    
                    /* Free old entry */
                    MemFree(attendee);
                    attendee = newAttendee;
                }

                break;
            }
        }
    }

    /* If we didn't find the delegatee then create and insert it into the attendee list */
    if (!delegateeFound) {
        sprintf(client->line, "\"%s\"", delegator->Address);
        if ((newAttendee = CreateNewAttendee(NMAP_CAL_STATE_NEED_ACT, NMAP_CAL_ROLE_REQUIRED, delegator->RSVP, delegator->Type, delegator->Delegated, "\"\"", client->line, delegator->Delegated ))) {
            newAttendee->Next = orgICal->Attendee;
            orgICal->Attendee = newAttendee;
        } else {
            /* problem creating new attendee */
            ICalFreeObjects(orgICal);
            return(-1);
        }
    }

    /* At this point our list should be updated. We would have
        returned earlier if there were no updates to be made. 
        Since NMAP doesn't allow us to modify an ICal object
        we will have to create a new object, and delete the original.
    */
    ccode = UpdateCalendarObject(client, orgICal, entryID, delegator);
    ICalFreeObjects(orgICal);

    return(ccode);

}

static int 
SetAttendeeStatus(CalClient *client, ICalObject *iCal, ICalVAttendee *attendee, struct sockaddr_in *saddr, unsigned char *user)
{
    int ccode;
    unsigned long entryID;
    unsigned char state;
    Connection *conn = client->conn;

    /*
        a) Build a connection to the right NMAP
        b) Find the matching entry
        c) Search for the specified attendee
        d) Update the status of the attendee
    */

    client->conn = NMAPConnect(NULL, saddr);
    if (client->conn) {
        if (NMAPAuthenticate(client->conn, client->line, CONN_BUFSIZE)) {
            ccode = NMAPSendCommandF(client->conn, "USER %s\r\n", user);
        } else {
            NMAPQuit(client->conn);
            ConnFree(client->conn);
            client->conn = conn;
            return(0);
        }
    } else {
        client->conn = conn;
        return(0);
    }

    if ((ccode != -1) 
            && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE)) == 1000) 
            && ((ccode = NMAPSendCommand(client->conn, "CSOPEN MAIN\r\n", 13)) != -1) 
            && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE)) == 1000) 
            && ((ccode = NMAPSendCommand(client->conn, "CSUPDA\r\n", 8)) != -1)) {
        do {
            if ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE)) == 1000) {
                break;
            }
        } while (ccode != -1);

        if ((ccode == 1000) 
                && ((ccode = NMAPSendCommandF(client->conn, "CSFIND %lu %s\r\n", iCal->Sequence, iCal->UID)) != -1) 
                && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE)) == 1000)) {
            entryID = atol(client->line);

            /* Update the attendee */
            switch(attendee->State) {
                default:
                case ICAL_PARTSTAT_NEEDS_ACTION:{
                    state = NMAP_CAL_STATE_NEED_ACT;        
                    if ((ccode = NMAPSendCommandF(client->conn, "CSATND %lu %c %s\r\n", entryID, state, attendee->Address)) != -1) {
                        ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE);
                    }

                    break;
                }

                case ICAL_PARTSTAT_ACCEPTED:{
                    state = NMAP_CAL_STATE_ACCEPTED;        
                    if ((ccode = NMAPSendCommandF(client->conn, "CSATND %lu %c %s\r\n", entryID, state, attendee->Address)) != -1) {
                        ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE);
                    }

                    break;
                }

                case ICAL_PARTSTAT_DECLINED:{
                    state = NMAP_CAL_STATE_DECLINED;
                    if ((ccode = NMAPSendCommandF(client->conn, "CSATND %lu %c %s\r\n", entryID, state, attendee->Address)) != -1) {
                        ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE);
                    }

                    break;
                }

                case ICAL_PARTSTAT_TENTATIVE:{
                    state = NMAP_CAL_STATE_TENTATIVE;
                    if ((ccode = NMAPSendCommandF(client->conn, "CSATND %lu %c %s\r\n", entryID, state, attendee->Address)) != -1) {
                        ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE);
                    }

                    break;
                }

                case ICAL_PARTSTAT_COMPLETED:{        
                    state = NMAP_CAL_STATE_COMPLETED;    
                    if ((ccode = NMAPSendCommandF(client->conn, "CSATND %lu %c %s\r\n", entryID, state, attendee->Address)) != -1) {
                        ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE);
                    }

                    /* note: removed logic to autocomplete originator if all attendees complete task */
                    /*       Decided that it is easier for user to maintain his completed tasks with respect to his own calendar */
                    break;
                }

                case ICAL_PARTSTAT_IN_PROCESS: {
                    state = NMAP_CAL_STATE_INPROCESS;
                    break;
                }

                case ICAL_PARTSTAT_DELEGATED: {
                    ccode = SetAttendeeDelegationStatus(client, iCal, entryID, attendee, saddr, user);
                    break;
                }
            }
        }
    }

    NMAPQuit(client->conn);
    ConnFree(client->conn);
    client->conn = conn;

    return(ccode);
}

static int 
DeleteEntry(CalClient *client, ICalObject *iCal, struct sockaddr_in *saddr, unsigned char *user, unsigned char *from)
{
    int ccode;
    unsigned long entryID;
    unsigned long recurID;
    unsigned char *ptr;
    Connection *conn = client->conn;

    client->conn = NMAPConnect(NULL, saddr);
    if (client->conn) {
        if (NMAPAuthenticate(client->conn, client->line, CONN_BUFSIZE)) {
            ccode = NMAPSendCommandF(client->conn, "USER %s\r\n", user);
        } else {
            NMAPQuit(client->conn);
            ConnFree(client->conn);
            client->conn = conn;
            return(0);
        }
    } else {
        client->conn = conn;
        return(0);
    }

    if ((ccode != -1) 
            && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE)) == 1000) 
            && ((ccode = NMAPSendCommand(client->conn, "CSOPEN MAIN\r\n", 13)) != -1) 
            && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE)) == 1000) 
            && ((ccode = NMAPSendCommand(client->conn, "CSUPDA\r\n", 8)) != -1)) {
        do {
            if ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE)) == 1000) {
                break;
            }
        } while (ccode != -1);

        if (ccode == 1000) {
            recurID = MsgGetUTC(iCal->RecurrenceID.Day, iCal->RecurrenceID.Month, iCal->RecurrenceID.Year, iCal->RecurrenceID.Hour, iCal->RecurrenceID.Minute, iCal->RecurrenceID.Second);
            if (recurID == (unsigned long)-1) {
                recurID = 0;
            }

            if (!recurID) {
                ccode = NMAPSendCommandF(client->conn, "CSFIND %lu %s\r\n", iCal->Sequence, iCal->UID);
            } else {
                ccode = NMAPSendCommandF(client->conn, "CSFIND %lu %s %lu\r\n", iCal->Sequence, iCal->UID, recurID);
            }

            if (ccode != -1) {
                ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE);
            }
        }

        if (ccode == 1000) {
            entryID = atol(client->line);

            if (((ccode = NMAPSendCommandF(client->conn, "CSORG %lu\r\n", entryID)) != -1) 
                    && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE)) != -1) 
                    && (ccode == 2001)) {
                ptr = strchr(client->line, ' ');
                if (ptr) {
                    *ptr = '\0';
                }

                if (QuickCmp(client->line, from) || QuickCmp(user, from)) {
                    if (!recurID) {
                        ccode = NMAPSendCommandF(client->conn, "CSDELE %lu\r\n", entryID);
                    } else {
                        ccode = NMAPSendCommandF(client->conn, "CSDELE %lu RECURRING\r\n", entryID);
                    }

                    if (ccode != -1) {
                        ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE);
                    }
                }
            }
        }
    }

    NMAPQuit(client->conn);
    ConnFree(client->conn);
    client->conn = conn;

    return(ccode);
}

static __inline int 
ProcessConnection(CalClient *client)
{
    int ccode;
    int length;
    int dsnFlags;
    unsigned long alternative = 0;
    unsigned long pos;
    unsigned long size;
    unsigned long depth = 0;
    unsigned long iCalPos = 0;
    unsigned long iCalSize = 0;
    unsigned long iCalLength = 0;
    unsigned long msgFlags;
    unsigned char *iCalBuffer;
    unsigned char temp;
    unsigned char preserve;
    unsigned char *ptr;
    unsigned char *cur;
    unsigned char *eol;
    unsigned char *line;
    unsigned char qID[16];
    unsigned char type[64];
    unsigned char subtype[64];
    unsigned char charset[64];
    unsigned char encoding[64];
    unsigned char iCalEncoding[64];
    unsigned char rtsUser[MDB_MAX_OBJECT_CHARS + 1];
    struct sockaddr_in saddr;
    BOOL copy;
    BOOL checked = FALSE;
    StreamStruct netInCodec;
    StreamStruct eCodec;
    StreamStruct memOutCodec;
    CalBounce bounce = CAL_BOUNCE_NONE;
    ICalVAttendee *attendee;

    rtsUser[0] = '\0';
    if (((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE)) != -1) 
            && (ccode == 6020) 
            && ((ptr = strchr(client->line, ' ')) != NULL)) {
        *ptr++ = '\0';

        strcpy(qID, client->line);

        length = atoi(ptr);
        client->envelope = MemMalloc(length + 3);
    } else {
        NMAPSendCommand(client->conn, "QDONE\r\n", 7);
        NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE);
        return(-1);
    }

    if (client->envelope) {
        sprintf(client->line, "Cal: %s", qID);
        XplRenameThread(XplGetThreadID(), client->line);

        client->vs = MDBCreateValueStruct(Cal.handle.directory, NULL);
    } else {
        NMAPSendCommand(client->conn, "QDONE\r\n", 7);
        NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE);
        return(-1);
    }

    if (client->vs) {
        ccode = ConnRead(client->conn, client->envelope, length);
    } else {
        NMAPSendCommand(client->conn, "QDONE\r\n", 7);
        NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE);
        return(-1);
    }

    if ((ccode != -1) 
            && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE)) == 6021)) {
        client->envelope[length] = '\0';

        cur = client->envelope;
    } else {
        NMAPSendCommand(client->conn, "QDONE\r\n", 7);
        NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE);
        return(-1);
    }

    /* PreScan the message to determine if it should be processed. */
    copy = FALSE;
    msgFlags = 0;
    eol = NULL;
    while ((ccode != -1) && *cur) {
        if (eol) {
            *eol = preserve;
        }

        line = strchr(cur, 0x0A);
        if (line) {
            if (line[-1] == 0x0D) {
                eol = line - 1;
            } else {
                eol = line;
            }

            preserve = *eol;
            *eol = '\0';

            line++;
        } else {
            eol = NULL;
            line = cur + strlen(cur);
        }

        switch (*cur) {
            case QUEUE_CALENDAR_LOCAL:
            case QUEUE_RECIP_LOCAL:
            case QUEUE_RECIP_MBOX_LOCAL: {
                copy = TRUE;
                break;
            }

            case QUEUE_FLAGS: {
                msgFlags = atol(cur + 1);

                if (!(msgFlags & MSG_FLAG_CALENDAR_OBJECT) 
                        && ((ccode = NMAPSendCommandF(client->conn, "QMIME %s\r\n", qID)) != -1) 
                        && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE)) != -1)) {
                    do {
                        switch (ccode) {
                            case 2002: {
                                if (sscanf(client->line, "%s %s %*s %*s", type, subtype) == 2) {
                                    if (QuickCmp(type, "multipart") || QuickCmp(type, "message")) {
                                        depth++;
                                        if (QuickCmp(subtype, "alternative")) {
                                            alternative++;
                                        }
                                    }

                                    if (depth < 2) {
                                        if (QuickCmp(type, "text") && QuickCmp(subtype, "calendar")) {
                                            if (depth < 1 || alternative == 1) {
                                                msgFlags |= MSG_FLAG_CALENDAR_OBJECT;
                                                copy = TRUE;
                                            }
                                        }
                                    }
                                }

                                break;
                            }

                            case 2004: {
                                alternative--;

                                /* fall through */
                            }

                            case 2003: {
                                depth--;
                                break;
                            }

                            default: {
                                break;
                            }
                        }

                        ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE);
                    } while ((ccode >= 2002) && (ccode <= 2004));
                }

                break;
            }

            case QUEUE_ADDRESS:
            case QUEUE_BOUNCE:
            case QUEUE_DATE:
            case QUEUE_FROM:
            case QUEUE_ID:
            case QUEUE_RECIP_REMOTE:
            case QUEUE_THIRD_PARTY:
            default: {
                break;
            }
        }

        cur = line;
    }

    if (eol) {
        *eol = preserve;
    }

    if ((ccode != -1) && copy && (msgFlags & MSG_FLAG_CALENDAR_OBJECT)) {
        cur = client->envelope;
    } else {
        NMAPSendCommand(client->conn, "QDONE\r\n", 7);
        NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE);
        return(ccode);
    }

    eol = NULL;
    while ((ccode != -1) && *cur) {
        copy = TRUE;

        if (eol) {
            *eol = preserve;
        }

        line = strchr(cur, 0x0A);
        if (line) {
            if (line[-1] == 0x0D) {
                eol = line - 1;
            } else {
                eol = line;
            }

            preserve = *eol;
            *eol = '\0';

            line++;
        } else {
            eol = NULL;
            line = cur + strlen(cur);
        }

        switch (cur[0]) {
            case QUEUE_FROM: {
                if (client->from) {
                    break;
                }

                ptr = cur + 1;
                while (*ptr && !isspace(*ptr)) {
                    ++ptr;
                }

                temp = *ptr;
                *ptr = '\0';
                client->from = MemStrdup(cur + 1);
                *ptr = temp;

                ccode = NMAPSendCommandF(client->conn, "QMOD RAW %s\r\n", cur);
                copy = FALSE;

                break;
            }

            case QUEUE_CALENDAR_LOCAL:
            case QUEUE_RECIP_LOCAL:
            case QUEUE_RECIP_MBOX_LOCAL: {
                ptr = strrchr(cur, ' ');
                if (ptr) {
                    dsnFlags = atol(ptr + 1);
                } else {
                    dsnFlags = 0;
                }

                if (!client->from || (dsnFlags & NO_CALENDARPROCESSING)) {
                    break;
                }

                if (!checked 
                        && ((ccode = NMAPSendCommandF(client->conn, "QMIME %s\r\n", qID)) != -1) 
                        && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE)) != -1)) {
                    checked = TRUE;

                    do {
                        switch (ccode) {
                            case 2002: {
                                if (sscanf(client->line, "%s %s %s %s", type, subtype, charset, encoding) == 4) {
                                    if ((ptr = strchr(client->line, '"')) == NULL) {
                                        break;
                                    }

                                    if ((ptr = strchr(ptr + 1, '"')) == NULL) {
                                        break;
                                    }

                                    ptr += 2;
                                    if (sscanf(ptr, "%lu %lu", &pos, &size) == 2) {
                                        if (QuickCmp(type, "multipart") || QuickCmp(type, "message")) {
                                            depth++;
                                            if (QuickCmp(subtype, "alternative")) {
                                                alternative++;
                                            }
                                        }

                                        if (depth < 2) {
                                            if (QuickCmp(type, "text") && QuickCmp(subtype, "calendar")) {
                                                if (depth < 1 || alternative == 1) {
                                                    strcpy(iCalEncoding, encoding);

                                                    iCalPos = pos;
                                                    iCalSize = size;

                                                    msgFlags |= MSG_FLAG_CALENDAR_OBJECT;
                                                    copy = TRUE;
                                                }
                                            }
                                        }
                                    }
                                }

                                break;
                            }

                            case 2004: {
                                alternative--;

                                /* fall through */
                            }

                            case 2003: {
                                depth--;
                                break;
                            }

                            default: {
                                break;
                            }
                        }

                        ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE);
                    } while ((ccode >= 2002) && (ccode <= 2004));
                }

                if ((ccode != -1) && (msgFlags & MSG_FLAG_CALENDAR_OBJECT)) {
                    ptr = cur + 1;
                    while (*ptr && !isspace(*ptr)) {
                        ++ptr;
                    }

                    temp = *ptr;
                    *ptr = '\0';

                    saddr.sin_family = AF_INET;
                    saddr.sin_port = htons(NMAP_PORT);
                } else {
                    break;
                }

                /* Find the user & get his full DN */
                if (!MsgFindObject(cur + 1, client->dn, NULL, &saddr.sin_addr, client->vs)) {
                    MDBFreeValues(client->vs);

                    *ptr = temp;
                    break;
                }

                MDBFreeValues(client->vs);

                if ((!client->iCal) 
                        && ((ccode = NMAPSendCommandF(client->conn, "QBRAW %s %lu %lu\r\n", qID, iCalPos, iCalSize)) != -1) 
                        && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE)) != -1) 
                        && ((ccode == 2021) || (ccode == 2023))) {
                    if (QuickCmp(iCalEncoding, "quoted-printable") || QuickCmp(iCalEncoding, "base64")) {
                        memset(&netInCodec, 0, sizeof(StreamStruct));
                        memset(&eCodec, 0, sizeof(StreamStruct));
                        memset(&memOutCodec, 0, sizeof(StreamStruct));

                        netInCodec.Codec = CalStreamFromNMAP;
                        netInCodec.StreamData = (void *)client;
                        netInCodec.StreamLength = atol(client->line);
                        netInCodec.Next = &eCodec;

                        eCodec.Codec = FindCodec(iCalEncoding, FALSE);
                        eCodec.Next = &memOutCodec;
                        eCodec.Start = MemMalloc(sizeof(unsigned char) * CONN_BUFSIZE);
                        if (eCodec.Start) {
                            eCodec.End = eCodec.Start + sizeof(unsigned char) * CONN_BUFSIZE;
                        } else {
                            *ptr = temp;
                            break;
                        }

                        memOutCodec.Codec = CalStreamToMemory;
                        memOutCodec.Start = MemMalloc(sizeof(unsigned char) * CONN_BUFSIZE);
                        if (memOutCodec.Start) {
                            memOutCodec.End = memOutCodec.Start + sizeof(unsigned char) * CONN_BUFSIZE;
                        } else {
                            MemFree(eCodec.Start);

                            *ptr = temp;
                            break;
                        }

                        /* Process the chain */
                        netInCodec.Codec(&netInCodec, netInCodec.Next);
                        if (memOutCodec.StreamData) {
                            iCalBuffer = memOutCodec.StreamData;
                            iCalLength = memOutCodec.StreamLength;
                        } else {
                            *ptr = temp;
                            break;
                        }

                        MemFree(eCodec.Start);
                        MemFree(memOutCodec.Start);
                    } else {
                        /* fixme - should make sure the encoding is 7bit or 8bit? */
                        iCalLength = atol(client->line);
                        iCalBuffer = MemMalloc(iCalLength + 1);
                        if (iCalBuffer) {
                            ccode = NMAPRead(client->conn, iCalBuffer, iCalLength);
                        } else {
                            while (iCalLength > 0) {
                                if (iCalLength < CONN_BUFSIZE) {
                                    ccode = NMAPRead(client->conn, client->line, iCalLength);
                                } else {
                                    ccode = NMAPRead(client->conn, client->line, CONN_BUFSIZE);
                                }

                                if (ccode != -1) {
                                    iCalLength -= ccode;
                                    continue;
                                }
                            }

                            MemFree(iCalBuffer);
                            iCalBuffer = NULL;

                            *ptr = temp;
                            break;
                        }
                    }

                    iCalBuffer[iCalLength] = '\0';
                    client->iCal = ICalParseObject(NULL, iCalBuffer, iCalLength);

                    MemFree(iCalBuffer);
                    iCalBuffer = NULL;
                }

                if (!client->iCal) {
                    *ptr = temp;
                    break;
                }

                if (client->iCal->Method == ICAL_METHOD_REPLY) {
                    attendee = client->iCal->Attendee;
                    while (attendee) {
                        if (QuickCmp(attendee->Address, client->from)) {
                            break;
                        }

                        attendee = attendee->Next;
                    }

                    if (attendee && SetAttendeeStatus(client, client->iCal, attendee, &saddr, cur + 1)) {
                            ;
                    } else {
                        /* Send a text message to the user's mailbox to let them known? */
                        sprintf(rtsUser, "%s %s",cur + 1, cur + strlen(cur) + 1);

                        if (!attendee) {
                            bounce = CAL_BOUNCE_NO_ATTENDEE;
                        } else {
                            bounce = CAL_BOUNCE_INVALID_OBJECT;
                        }
                    }

                    /* No matter what happens we do not want a copy in the mailbox */
                    copy = FALSE;
                } else if (client->iCal->Method == ICAL_METHOD_CANCEL) {
                    /* Find and delete the calendar object */
                    DeleteEntry(client, client->iCal, &saddr, cur + 1, client->from);
                    
                    /* Don't send a text message on cancel even if we don't find it */
                    /* This is a normal action when originator cancels cal item after recipients has deleted it */
                    /* because we don't check the recipients mailboxes for the cal item before we send a cancel */
                    copy = FALSE; 
                }

                /* Give them the whole cur back */
                *ptr = temp;
                break;
            }

            case QUEUE_FLAGS: {
                copy = FALSE;
                ccode = NMAPSendCommandF(client->conn, "QMOD RAW "QUEUES_FLAGS"%lu\r\n", msgFlags);
                break;
            }

            default: {
                break;
            }
        }

        if (copy) {
            ccode = NMAPSendCommandF(client->conn, "QMOD RAW %s\r\n", cur);
        }

        cur = line;
    }

    if (eol) {
        *eol = preserve;
    }

    if (ccode != -1) {
        if (bounce != CAL_BOUNCE_NONE) {
            if (CAL_BOUNCE_INVALID_OBJECT) {
                ccode = NMAPSendCommandF(client->conn, "QRTS %s %d %s\r\n", rtsUser, DELIVER_BOGUS_NAME, "The calendar item requested does not exist.");
            } else if (CAL_BOUNCE_NO_ATTENDEE) {
                ccode = NMAPSendCommandF(client->conn, "QRTS %s %d %s\r\n", rtsUser, DELIVER_BOGUS_NAME, "The address does not exist on the calendar object for the requested attendee.");
            }

            if (ccode != -1) {
                ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE);
            }
        }

        if ((ccode != -1) 
                && ((ccode = NMAPSendCommand(client->conn, "QDONE\r\n", 7)) != -1)) {
            ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE);
        }
    }

    return(ccode);
}

static void 
HandleConnection(void *param)
{
    int ccode;
    long threadNumber = (long)param;
    time_t sleep = time(NULL);
    time_t wokeup;
    CalClient *client;

    if ((client = CalClientAlloc()) == NULL) {
        XplConsolePrintf("hulacalendar: New worker failed to startup; out of memory.\r\n");

        NMAPSendCommand(client->conn, "QDONE\r\n", 7);

        XplSafeDecrement(Cal.nmap.worker.active);

        return;
    }

    do {
        XplRenameThread(XplGetThreadID(), "Cal Worker");

        XplSafeIncrement(Cal.nmap.worker.idle);

        XplWaitOnLocalSemaphore(Cal.nmap.worker.todo);

        XplSafeDecrement(Cal.nmap.worker.idle);

        wokeup = time(NULL);

        XplWaitOnLocalSemaphore(Cal.nmap.semaphore);

        client->conn = Cal.nmap.worker.tail;
        if (client->conn) {
            Cal.nmap.worker.tail = client->conn->queue.previous;
            if (Cal.nmap.worker.tail) {
                Cal.nmap.worker.tail->queue.next = NULL;
            } else {
                Cal.nmap.worker.head = NULL;
            }
        }

        XplSignalLocalSemaphore(Cal.nmap.semaphore);

        if (client->conn) {
            if (ConnNegotiate(client->conn, Cal.nmap.ssl.context)) {
                ccode = ProcessConnection(client);
            } else {
                NMAPSendCommand(client->conn, "QDONE\r\n", 7);
                NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE);
            }
        }

        if (client->conn) {
            ConnFlush(client->conn);
        }

        FreeClientData(client);

        /* Live or die? */
        if (threadNumber == XplSafeRead(Cal.nmap.worker.active)) {
            if ((wokeup - sleep) > Cal.nmap.sleepTime) {
                break;
            }
        }

        sleep = time(NULL);

        CalClientAllocCB(client, NULL);
    } while (Cal.state == CAL_STATE_RUNNING);

    FreeClientData(client);

    CalClientFree(client);

    XplSafeDecrement(Cal.nmap.worker.active);

    XplExitThread(TSR_THREAD, 0);

    return;
}

static void 
CalendarAgentServer(void *ignored)
{
    int i;
    int ccode;
    XplThreadID id;
    Connection *conn;

    XplSafeIncrement(Cal.server.active);

    XplRenameThread(XplGetThreadID(), "Calendar Agent Server");

    Cal.state = CAL_STATE_RUNNING;

    while (Cal.state < CAL_STATE_STOPPING) {
        if (ConnAccept(Cal.nmap.conn, &conn) != -1) {
            if (Cal.state < CAL_STATE_STOPPING) {
                conn->ssl.enable = FALSE;

                QUEUE_WORK_TO_DO(conn, id, ccode);
                if (!ccode) {
                    XplSignalLocalSemaphore(Cal.nmap.worker.todo);

                    continue;
                }
            }

            ConnWrite(conn, "QDONE\r\n", 7);
            ConnClose(conn, 0);

            ConnFree(conn);
            conn = NULL;

            continue;
        }

        switch (errno) {
            case ECONNABORTED:
#ifdef EPROTO
            case EPROTO: 
#endif
            case EINTR: {
                if (Cal.state < CAL_STATE_STOPPING) {
                    LoggerEvent(Cal.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_ACCEPT_FAILURE, LOG_ERROR, 0, "Server", NULL, errno, 0, NULL, 0);
                }

                continue;
            }

            default: {
                if (Cal.state < CAL_STATE_STOPPING) {
                    XplConsolePrintf("hulacalendar: Exiting after an accept() failure; error %d\r\n", errno);

                    LoggerEvent(Cal.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_ACCEPT_FAILURE, LOG_ERROR, 0, "Server", NULL, errno, 0, NULL, 0);

                    Cal.state = CAL_STATE_STOPPING;
                }

                break;
            }
        }

        break;
    }

    /* Shutting down */
    Cal.state = CAL_STATE_UNLOADING;

    XplConsolePrintf("hulacalendar: Shutting down.\r\n");

    id = XplSetThreadGroupID(Cal.id.group);

    if (Cal.nmap.conn) {
        ConnClose(Cal.nmap.conn, 1);
        Cal.nmap.conn = NULL;
    }

    if (Cal.nmap.ssl.enable) {
        Cal.nmap.ssl.enable = FALSE;

        if (Cal.nmap.ssl.conn) {
            ConnClose(Cal.nmap.ssl.conn, 1);
            Cal.nmap.ssl.conn = NULL;
        }

        if (Cal.nmap.ssl.context) {
            ConnSSLContextFree(Cal.nmap.ssl.context);
            Cal.nmap.ssl.context = NULL;
        }
    }

    ConnCloseAll(1);

    if (ManagementState() == MANAGEMENT_RUNNING) {
        ManagementShutdown();
    }

    for (i = 0; (XplSafeRead(Cal.server.active) > 1) && (i < 60); i++) {
        XplDelay(1000);
    }

    for (i = 0; (ManagementState() != MANAGEMENT_STOPPED) && (i < 60); i++) {
        XplDelay(1000);
    }

    XplConsolePrintf("hulacalendar: Shutting down %d queue threads\r\n", XplSafeRead(Cal.nmap.worker.active));

    XplWaitOnLocalSemaphore(Cal.nmap.semaphore);

    ccode = XplSafeRead(Cal.nmap.worker.idle);
    while (ccode--) {
        XplSignalLocalSemaphore(Cal.nmap.worker.todo);
    }

    XplSignalLocalSemaphore(Cal.nmap.semaphore);

    for (i = 0; XplSafeRead(Cal.nmap.worker.active) && (i < 60); i++) {
        XplDelay(1000);
    }

    if (XplSafeRead(Cal.server.active) > 1) {
        XplConsolePrintf("hulacalendar: %d server threads outstanding; attempting forceful unload.\r\n", XplSafeRead(Cal.server.active) - 1);
    }

    if (XplSafeRead(Cal.nmap.worker.active)) {
        XplConsolePrintf("hulacalendar: %d threads outstanding; attempting forceful unload.\r\n", XplSafeRead(Cal.nmap.worker.active));
    }

    LoggerClose(Cal.handle.logging);
    Cal.handle.logging = NULL;

    /* shutdown the scanning engine */

    XplCloseLocalSemaphore(Cal.nmap.worker.todo);
    XplCloseLocalSemaphore(Cal.nmap.semaphore);

    StreamioShutdown();
    MsgShutdown();

    ConnShutdown();

    MemPrivatePoolFree(Cal.nmap.pool);

    MemoryManagerClose(MSGSRV_AGENT_CALAGENT);

    XplConsolePrintf("hulacalendar: Shutdown complete\r\n");

    XplSignalLocalSemaphore(Cal.sem.main);
    XplWaitOnLocalSemaphore(Cal.sem.shutdown);

    XplCloseLocalSemaphore(Cal.sem.shutdown);
    XplCloseLocalSemaphore(Cal.sem.main);

    XplSetThreadGroupID(id);

    return;
}

static BOOL 
ReadConfiguration(void)
{
    MDBValueStruct *config;

    config = MDBCreateValueStruct(Cal.handle.directory, MsgGetServerDN(NULL));
    if (config) {
        if (MDBRead(".", MSGSRV_A_OFFICIAL_NAME, config)) {
            strcpy(Cal.officialName, config->Value[0]);

            MDBFreeValues(config);
        } else {
            Cal.officialName[0] = '\0';
        }

        MDBSetValueStructContext(NULL, config);
        if (MDBRead(MSGSRV_ROOT, MSGSRV_A_ACL, config) > 0) { 
            HashCredential(MsgGetServerDN(NULL), config->Value[0], Cal.nmap.hash);
        }
    } else {
        return(FALSE);
    }

    MDBDestroyValueStruct(config);

    return(TRUE);
}

#if defined(NETWARE) || defined(LIBC) || defined(WIN32)
static int 
_NonAppCheckUnload(void)
{
    static BOOL    checked = FALSE;
    XplThreadID    id;

    if (!checked) {
        checked = TRUE;
        Cal.state = CAL_STATE_UNLOADING;

        XplWaitOnLocalSemaphore(Cal.sem.shutdown);

        id = XplSetThreadGroupID(Cal.id.group);
        ConnClose(Cal.nmap.conn, 1);
        XplSetThreadGroupID(id);

        XplWaitOnLocalSemaphore(Cal.sem.main);
    }

    return(0);
}
#endif

static void 
SignalHandler(int sigtype)
{
    switch(sigtype) {
        case SIGHUP: {
            if (Cal.state < CAL_STATE_UNLOADING) {
                Cal.state = CAL_STATE_UNLOADING;
            }

            break;
        }

        case SIGINT:
        case SIGTERM: {
            if (Cal.state == CAL_STATE_STOPPING) {
                XplUnloadApp(getpid());
            } else if (Cal.state < CAL_STATE_STOPPING) {
                Cal.state = CAL_STATE_STOPPING;
            }

            break;
        }

        default: {
            break;
        }
    }

    return;
}

static int 
QueueSocketInit(void)
{
    Cal.nmap.conn = ConnAlloc(FALSE);
    if (Cal.nmap.conn) {
        memset(&(Cal.nmap.conn->socketAddress), 0, sizeof(Cal.nmap.conn->socketAddress));

        Cal.nmap.conn->socketAddress.sin_family = AF_INET;
        Cal.nmap.conn->socketAddress.sin_addr.s_addr = MsgGetAgentBindIPAddress();

        /* Get root privs back for the bind.  It's ok if this fails -
        * the user might not need to be root to bind to the port */
        XplSetEffectiveUserId(0);

        Cal.nmap.conn->socket = ConnServerSocket(Cal.nmap.conn, 2048);
        if (XplSetEffectiveUser(MsgGetUnprivilegedUser()) < 0) {
            XplConsolePrintf("hulacalendar: Could not drop to unprivileged user '%s'\r\n", MsgGetUnprivilegedUser());
            ConnFree(Cal.nmap.conn);
            Cal.nmap.conn = NULL;
            return(-1);
        }

        if (Cal.nmap.conn->socket == -1) {
            XplConsolePrintf("hulacalendar: Could not bind to dynamic port\r\n");
            ConnFree(Cal.nmap.conn);
            Cal.nmap.conn = NULL;
            return(-1);
        }

        if (NMAPRegister(MSGSRV_AGENT_CALAGENT, Cal.nmap.queue, Cal.nmap.conn->socketAddress.sin_port) != REGISTRATION_COMPLETED) {
            XplConsolePrintf("hulacalendar: Could not register with hulanmap\r\n");
            ConnFree(Cal.nmap.conn);
            Cal.nmap.conn = NULL;
            return(-1);
        }
    } else {
        XplConsolePrintf("hulacalendar: Could not allocate connection.\r\n");
        return(-1);
    }

    return(0);
}

XplServiceCode(SignalHandler)

int
XplServiceMain(int argc, char *argv[])
{
    int                ccode;
    XplThreadID        id;

    if (XplSetEffectiveUser(MsgGetUnprivilegedUser()) < 0) {
        XplConsolePrintf("hulacalendar: Could not drop to unprivileged user '%s', exiting.\n", MsgGetUnprivilegedUser());
        return(1);
    }

    XplSignalHandler(SignalHandler);

    Cal.id.main = XplGetThreadID();
    Cal.id.group = XplGetThreadGroupID();

    Cal.state = CAL_STATE_INITIALIZING;
    Cal.flags = 0;

    Cal.nmap.conn = NULL;
    Cal.nmap.queue = Q_FIVE;
    Cal.nmap.pool = NULL;
    Cal.nmap.sleepTime = (5 * 60);
    Cal.nmap.ssl.conn = NULL;
    Cal.nmap.ssl.enable = FALSE;
    Cal.nmap.ssl.context = NULL;
    Cal.nmap.ssl.config.options = 0;

    Cal.handle.directory = NULL;
    Cal.handle.logging = NULL;

    strcpy(Cal.nmap.address, "127.0.0.1");

    XplSafeWrite(Cal.server.active, 0);

    XplSafeWrite(Cal.nmap.worker.idle, 0);
    XplSafeWrite(Cal.nmap.worker.active, 0);
    XplSafeWrite(Cal.nmap.worker.maximum, 100000);

    if (MemoryManagerOpen(MSGSRV_AGENT_CALAGENT) == TRUE) {
        Cal.nmap.pool = MemPrivatePoolAlloc("Calendar Connections", sizeof(CalClient), 0, 3072, TRUE, FALSE, CalClientAllocCB, NULL, NULL);
        if (Cal.nmap.pool != NULL) {
            XplOpenLocalSemaphore(Cal.sem.main, 0);
            XplOpenLocalSemaphore(Cal.sem.shutdown, 1);
            XplOpenLocalSemaphore(Cal.nmap.semaphore, 1);
            XplOpenLocalSemaphore(Cal.nmap.worker.todo, 1);
        } else {
            MemoryManagerClose(MSGSRV_AGENT_CALAGENT);

            XplConsolePrintf("hulacalendar: Unable to create connection pool; shutting down.\r\n");
            return(-1);
        }
    } else {
        XplConsolePrintf("hulacalendar: Unable to initialize memory manager; shutting down.\r\n");
        return(-1);
    }

    ConnStartup(CONNECTION_TIMEOUT, TRUE);

    MDBInit();
    Cal.handle.directory = (MDBHandle)MsgInit();
    if (Cal.handle.directory == NULL) {
        XplBell();
        XplConsolePrintf("hulacalendar: Invalid directory credentials; exiting!\r\n");
        XplBell();

        MemoryManagerClose(MSGSRV_AGENT_CALAGENT);

        return(-1);
    }

    StreamioInit();
    NMAPInitialize(Cal.handle.directory);

    SetCurrentNameSpace(NWOS2_NAME_SPACE);
    SetTargetNameSpace(NWOS2_NAME_SPACE);

	Cal.handle.logging = LoggerOpen("hulacalendar");
	if (!Cal.handle.logging) {
		XplConsolePrintf("hulacalendar: Unable to initialize logging; disabled.\r\n");
	}

    ReadConfiguration();

    if (QueueSocketInit() < 0) {
        XplConsolePrintf("hulacalendar: Exiting.\r\n");

        MemoryManagerClose(MSGSRV_AGENT_CALAGENT);

        return -1;
    }

    /* initialize scanning engine here */


    if (XplSetRealUser(MsgGetUnprivilegedUser()) < 0) {
        XplConsolePrintf("hulacalendar: Could not drop to unprivileged user '%s', exiting.\r\n", MsgGetUnprivilegedUser());

        MemoryManagerClose(MSGSRV_AGENT_CALAGENT);

        return 1;
    }

    Cal.nmap.ssl.enable = FALSE;
    Cal.nmap.ssl.config.method = SSLv23_client_method;
    Cal.nmap.ssl.config.options = SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS;
    Cal.nmap.ssl.config.mode = SSL_MODE_AUTO_RETRY;
    Cal.nmap.ssl.config.cipherList = NULL;
    Cal.nmap.ssl.config.certificate.type = SSL_FILETYPE_PEM;
    Cal.nmap.ssl.config.certificate.file = MsgGetTLSCertPath(NULL);
    Cal.nmap.ssl.config.key.type = SSL_FILETYPE_PEM;
    Cal.nmap.ssl.config.key.file = MsgGetTLSKeyPath(NULL);

    Cal.nmap.ssl.context = ConnSSLContextAlloc(&(Cal.nmap.ssl.config));
    if (Cal.nmap.ssl.context) {
        Cal.nmap.ssl.enable = TRUE;
    }

    NMAPSetEncryption(Cal.nmap.ssl.context);

    if ((ManagementInit(MSGSRV_AGENT_CALAGENT, Cal.handle.directory)) 
            && (ManagementSetVariables(GetCalManagementVariables(), GetCalManagementVariablesCount())) 
            && (ManagementSetCommands(GetCalManagementCommands(), GetCalManagementCommandsCount()))) {
        XplBeginThread(&id, ManagementServer, DMC_MANAGEMENT_STACKSIZE, NULL, ccode);
    }


    if (ccode) {
        XplConsolePrintf("hulacalendar: Unable to startup the management interface.\r\n");
    }
    
    XplStartMainThread(PRODUCT_SHORT_NAME, &id, CalendarAgentServer, 8192, NULL, ccode);
    
    XplUnloadApp(XplGetThreadID());
    return(0);
}
