/****************************************************************************
 *
 * Copyright (c) 2005 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>
#include <mdb.h>

#include "forward.h"

#define PRODUCT_NAME "Hula Forward Agent"
#define PRODUCT_DESCRIPTION "Provide processing of user-created auto reply messages."
#define PRODUCT_VERSION "$Revision: 1.8 $"

static BOOL ReadForwardVariable(unsigned int variable, unsigned char *data, size_t *length);

static BOOL ForwardDMCCommandHelp(unsigned char *arguments, unsigned char **response, BOOL *closeConnection);
static BOOL ForwardShutdown(unsigned char *arguments, unsigned char **response, BOOL *closeConnection);
static BOOL SendForwardStatistics(unsigned char *arguments, unsigned char **response, BOOL *closeConnection);

ManagementCommands ForwardManagementCommands[] = {
    { DMCMC_HELP, ForwardDMCCommandHelp }, 
    { DMCMC_SHUTDOWN, ForwardShutdown },
    { DMCMC_STATS, SendForwardStatistics }, 
    { DMCMC_DUMP_MEMORY_USAGE, ManagementMemoryStats }, 
};

ManagementVariables ForwardManagementVariables[] = {
    { DMCMV_REVISIONS, DMCMV_REVISIONS_HELP, ReadForwardVariable, NULL }, 
    { DMCMV_CONNECTION_COUNT, DMCMV_CONNECTION_COUNT_HELP, ReadForwardVariable, NULL }, 
    { DMCMV_IDLE_CONNECTION_COUNT, DMCMV_IDLE_CONNECTION_COUNT_HELP, ReadForwardVariable, NULL }, 
    { DMCMV_SERVER_THREAD_COUNT, DMCMV_SERVER_THREAD_COUNT_HELP, ReadForwardVariable, NULL }, 
    { DMCMV_NMAP_ADDRESS, DMCMV_NMAP_ADDRESS_HELP, ReadForwardVariable, NULL }, 
    { DMCMV_OFFICIAL_NAME, DMCMV_OFFICIAL_NAME_HELP, ReadForwardVariable, NULL }, 
    { DMCMV_MESSAGE, FWDMV_FORWARD_MESSAGE_HELP, ReadForwardVariable, NULL }, 
    { DMCMV_VERSION, DMCMV_VERSION_HELP, ReadForwardVariable, NULL }, 
};

ManagementVariables *
GetForwardManagementVariables(void)
{
    return(ForwardManagementVariables);
}

int 
GetForwardManagementVariablesCount(void)
{
    return(sizeof(ForwardManagementVariables) / sizeof(ManagementVariables));
}

ManagementCommands *
GetForwardManagementCommands(void)
{
    return(ForwardManagementCommands);
}

int 
GetForwardManagementCommandsCount(void)
{
    return(sizeof(ForwardManagementCommands) / sizeof(ManagementCommands));
}

static BOOL 
ForwardShutdown(unsigned char *arguments, unsigned char **response, BOOL *closeConnection)
{
    XplThreadID id;

    if (response) {
        if (!arguments) {
            if (Forward.nmap.conn) {
                *response = MemStrdup("Shutting down.\r\n");
                if (*response) {
                    id = XplSetThreadGroupID(Forward.id.group);

                    Forward.state = FORWARD_STATE_UNLOADING;

                    if (Forward.nmap.conn) {
                        ConnClose(Forward.nmap.conn, 1);
                        Forward.nmap.conn = NULL;
                    }

                    if (closeConnection) {
                        *closeConnection = TRUE;
                    }

                    XplSetThreadGroupID(id);
                }
            } else if (Forward.state != FORWARD_STATE_RUNNING) {
                *response = MemStrdup("Shutdown in progress.\r\n");
            }

            if (*response) {
                return(TRUE);
            }

            return(FALSE);
        }

        *response = MemStrdup("arguments not allowed.\r\n");
        return(TRUE);
    }

    return(FALSE);
}

static BOOL 
ForwardDMCCommandHelp(unsigned char *arguments, unsigned char **response, BOOL *closeConnection)
{
    BOOL responded = FALSE;

    if (response) {
        if (arguments) {
            switch(toupper(arguments[0])) {
                case 'M': {
                    if (XplStrCaseCmp(arguments, DMCMC_DUMP_MEMORY_USAGE) == 0) {
                        if ((*response = MemStrdup(DMCMC_DUMP_MEMORY_USAGE_HELP)) != NULL) {
                            responded = TRUE;
                        }

                        break;
                    }
                }

                case 'S': {
                    if (XplStrCaseCmp(arguments, DMCMC_SHUTDOWN) == 0) {
                        if ((*response = MemStrdup(DMCMC_SHUTDOWN_HELP)) != NULL) {
                            responded = TRUE;
                        }

                        break;
                    } else if (XplStrCaseCmp(arguments, DMCMC_STATS) == 0) {
                        if ((*response = MemStrdup(DMCMC_STATS_HELP)) != NULL) {
                            responded = TRUE;
                        }

                        break;
                    }
                }

                default: {
                    break;
                }
            }
        } else if ((*response = MemStrdup(DMCMC_HELP_HELP)) != NULL) {
            responded = TRUE;
        }

        if (responded || ((*response = MemStrdup(DMCMC_UNKOWN_COMMAND)) != NULL)) {
            return(TRUE);
        }
    }

    return(FALSE);
}

static BOOL 
SendForwardStatistics(unsigned char *arguments, unsigned char **response, BOOL *closeConnection)
{
    MemStatistics poolStats;

    if (!arguments && response) {
        memset(&poolStats, 0, sizeof(MemStatistics));

        *response = MemMalloc(sizeof(PRODUCT_NAME)
                            + sizeof(PRODUCT_SHORT_NAME)
                            + 124);

        MemPrivatePoolStatistics(Forward.nmap.pool, &poolStats);

        if (*response) {
            sprintf(*response, "%s (%s: v%d.%d.%d)\r\n%lu:%lu:%lu:%lu:%d:%d:%d\r\n", 
                    PRODUCT_NAME, 
                    PRODUCT_SHORT_NAME, 
                    PRODUCT_MAJOR_VERSION, 
                    PRODUCT_MINOR_VERSION, 
                    PRODUCT_LETTER_VERSION, 
                    poolStats.totalAlloc.count, 
                    poolStats.totalAlloc.size, 
                    poolStats.pitches, 
                    poolStats.strikes, 
                    XplSafeRead(Forward.server.active), 
                    XplSafeRead(Forward.nmap.worker.active), 
                    XplSafeRead(Forward.nmap.worker.idle));

            return(TRUE);
        }

        if ((*response = MemStrdup("Out of memory.\r\n")) != NULL) {
            return(TRUE);
        }
    } else if ((arguments) && ((*response = MemStrdup("arguments not allowed.\r\n")) != NULL)) {
        return(TRUE);
    }

    return(FALSE);
}

static BOOL 
ReadForwardVariable(unsigned int variable, unsigned char *data, size_t *length)
{
    size_t count;
    unsigned char *ptr;

    switch (variable) {
        case 0: {
            unsigned char	version[30];

            PVCSRevisionToVersion(PRODUCT_VERSION, version);
            count = strlen(version) + 13;

            if (data && (*length > count)) {
                ptr = data;

                PVCSRevisionToVersion(PRODUCT_VERSION, version);
                ptr += sprintf(ptr, "forward.c: %s\r\n", version);

                *length = ptr - data;
            } else {
                *length = count;
            }

            break;
        }

        case 1: {
            if (data && (*length > 12)) {
                sprintf(data, "%010lu\r\n", (long unsigned int)XplSafeRead(Forward.nmap.worker.active));
            }

            *length = 12;
            break;
        }

        case 2: {
            if (data && (*length > 12)) {
                sprintf(data, "%010lu\r\n", (long unsigned int)XplSafeRead(Forward.nmap.worker.idle));
            }

            *length = 12;
            break;
        }

        case 3: {
            if (data && (*length > 12)) {
                sprintf(data, "%010lu\r\n", (long unsigned int)XplSafeRead(Forward.server.active));
            }

            *length = 12;
            break;
        }

        case 4: {
            count = strlen(Forward.nmap.address) + 2;
            if (data && (*length > count)) {
                sprintf(data, "%s\r\n", Forward.nmap.address);
            }

            *length = count;
            break;
        }

        case 5: {
            count = strlen(Forward.officialName) + 2;
            if (data && (*length > count)) {
                sprintf(data, "%s\r\n", Forward.officialName);
            }

            *length = count;
            break;
        }

        case 6: {
            if (Forward.replyMessage) {
                count = strlen(Forward.replyMessage) + 2;
                if (data && (*length > count)) {
                    sprintf(data, "%s\r\n", Forward.replyMessage);
                }
            } else {
                count = 2;
                data[0] = '\r';
                data[1] = '\n';
                data[2] = '\0';
            }
            *length = count;
            break;
        }

        case 7: {
            DMC_REPORT_PRODUCT_VERSION(data, *length);
            break;
        }
    }

    return(TRUE);
}
