/****************************************************************************
 *
 * Copyright (c) 2001-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>

#include <openssl/ssl.h>
#include <openssl/err.h>
#include <hulautil.h>
#include <mdb.h>

#include "webadminp.h"

#if defined(DEBUG)
#define	ModuleDebugInit()		WAModuleScreen=XplCreateScreen("WA Module Debug Screen", 0)
#else
#define	ModuleDebugInit()		WAModuleScreen=XplCreateScreen("System Console", 0);
#endif

int						TGid;
int						WAModuleScreen;

unsigned long			TModuleCount = 0;
unsigned long			OModuleCount = 0;
unsigned long			SModuleCount = 0;
unsigned long			DModuleCount = 0;
unsigned long			RModuleCount = 0;

TemplateModuleStruct	*TModules	= NULL;
ObjectModuleStruct	*OModules	= NULL;
SaveModuleStruct		*SModules	= NULL;
DeleteModuleStruct	*DModules	= NULL;
RenameModuleStruct	*RModules	= NULL;

typedef struct {
	unsigned char			*Name;
	XplPluginHandle		Handle;
	WAModuleShutdownFunc	Shutdown;
} ModuleStruct;

ModuleStruct			*Modules		= NULL;
unsigned long			ModuleCount	= 0;

#if defined(SOLARIS) || defined(LINUX) || defined(WIN32) || defined(S390RH) || defined(REDHAT)
	WAAPIStruct	WAAPI;
#else
	unsigned long	*WAAPI=NULL;
#endif

static void
AddTModule(TemplateModuleStruct *Functions)
{
	TModules=realloc(TModules, (TModuleCount+1) * sizeof(TemplateModuleStruct));
	if (!TModules) {
		// syslog(LOG_ERR, "Could not add T-module, out of memory.");
		return;
	}

	memcpy(&(TModules[TModuleCount]), Functions, sizeof(TemplateModuleStruct));
	TModuleCount++;
	ModuleDebug("Added T-module\n");
}

static void
FreeTModules(void)
{
	ModuleDebug("Unloading T-modules\n");

	if (TModules) {
		free(TModules);
	}

	TModules=NULL;
	TModuleCount=0;
}


static void
AddOModule(ObjectModuleStruct *Functions)
{
	OModules=realloc(OModules, (OModuleCount + 1) * sizeof(ObjectModuleStruct));
	if (!OModules) {
		// syslog(LOG_ERR, "Could not add O-module, out of memory.");
		return;
	}

	memcpy(&(OModules[OModuleCount]), Functions, sizeof(ObjectModuleStruct));

	/* do some sanity checking, later code relies on this */
	if (OModules[OModuleCount].Flags & WAOBJ_PRIMARY_HANDLER) {
		if (!OModules[OModuleCount].CreateFunction) {
			OModules[OModuleCount].Flags &= ~WAOBJ_PRIMARY_HANDLER;
		}
	}
	OModuleCount++;
	ModuleDebug("Added O-module\n");
}

static void
FreeOModules(void)
{
	ModuleDebug("Unloading O-modules\n");

	if (OModules) {
		free(OModules);
	}

	OModules=NULL;
	OModuleCount=0;
}

static void
AddSModule(SaveModuleStruct *Functions)
{
	SModules=realloc(SModules, (SModuleCount + 1) * sizeof(SaveModuleStruct));
	if (!SModules) {
		// syslog(LOG_ERR, "Could not add S-module, out of memory.");
		return;
	}

	memcpy(&(SModules[SModuleCount]), Functions, sizeof(SaveModuleStruct));

	SModuleCount++;
	ModuleDebug("Added S-module\n");
}

static void
FreeSModules(void)
{
	ModuleDebug("Unloading S-modules\n");

	if (SModules) {
		free(SModules);
	}

	SModules=NULL;
	SModuleCount=0;
}

static void
AddDModule(DeleteModuleStruct *Functions)
{
	DModules = realloc(DModules, (DModuleCount + 1) * sizeof(DeleteModuleStruct));
	if (!DModules) {
		// syslog(LOG_ERR, "Could not add D-module, out of memory.");
		return;
	}

	memcpy(&(DModules[DModuleCount]), Functions, sizeof(DeleteModuleStruct));

	DModuleCount++;
	ModuleDebug("Added D-module\n");
}

static void
FreeDModules(void)
{
	ModuleDebug("Unloading D-modules\n");

	if (DModules) {
		free(DModules);
	}

	DModules = NULL;
	DModuleCount = 0;
}

static void
AddRModule(RenameModuleStruct *Functions)
{
	RModules = realloc(RModules, (RModuleCount + 1) * sizeof(RenameModuleStruct));
	if (!RModules) {
		// syslog(LOG_ERR, "Could not add R-module, out of memory.");
		return;
	}

	memcpy(&(RModules[RModuleCount]), Functions, sizeof(RenameModuleStruct));

	RModuleCount++;
	ModuleDebug("Added R-module\n");
}

static void
FreeRModules(void)
{
	ModuleDebug("Unloading R-modules\n");

	if (RModules) {
		free(RModules);
	}

	RModules = NULL;
	RModuleCount = 0;
}

void
AddModule(unsigned char *Name, XplPluginHandle Handle, WAModuleShutdownFunc Shutdown)
{
	Modules=realloc(Modules, (ModuleCount+1) * sizeof(ModuleStruct));
	if (!Modules) {
		// syslog(LOG_ERR, "Could not add module, out of memory.");
		return;
	}

	Modules[ModuleCount].Handle=Handle;
	Modules[ModuleCount].Shutdown=Shutdown;
	Modules[ModuleCount].Name=strdup(Name);
	if (!Modules[ModuleCount].Name) {
		// syslog(LOG_ERR, "Could not add module, out of memory.");
		return;
	}
	ModuleCount++;
}

void
FreeModules(void)
{
	unsigned long i;

	ModuleDebug("Unloading all modules\n");

	FreeTModules();
	FreeOModules();
	FreeSModules();
	FreeDModules();
	FreeRModules();

	for (i=0; i<ModuleCount; i++) {
		ModuleDebug("Calling module %s shutdown code\n", Modules[i].Name);
		Modules[i].Shutdown();
		XplUnloadDLL(Modules[i].Name, Modules[i].Handle);
		free(Modules[i].Name);
	}

	if (Modules) {
		free(Modules);
	}

	Modules=NULL;
	ModuleCount=0;
}

static BOOL
LoadWebAdminModule(unsigned char *ModulePath, unsigned char *ModuleName, unsigned char *Arguments)
{ 
    unsigned char			Path[XPL_MAX_PATH+1];
	unsigned char			Name[XPL_MAX_PATH+1];
	XplPluginHandle		ModuleHandle;
	WAModuleInitFunc		ModuleInitFunc;
	WAModuleShutdownFunc	ModuleShutdownFunc;
	unsigned char			*ptr;
TryAgain:                                                                                                                           
   if ((strlen(ModuleName)<5) || (WAQuickNCmp(ModuleName+strlen(ModuleName)-strlen(MODULE_EXTENSION), MODULE_EXTENSION, strlen(MODULE_EXTENSION))==FALSE)) {                                                                                                            
      ptr=strrchr(ModuleName, '.');                                                                                                 
      if (ptr && WAQuickNCmp(ptr,".NLM", 4)) {                                                                                      
         strcpy(ptr, MODULE_EXTENSION);                                                                                             
         ptr=ModuleName;                                                                                                            
         while (*ptr) {                                                                                                             
            *ptr=tolower(*ptr);                                                                                                     
            ptr++;                                                                                                                  
         }                                                                                                                          
         goto TryAgain;                                                                                                             
      }                                                                                                                             
      ModuleDebug("Ignoring file %s, module extension must be %s\n", ModuleName, MODULE_EXTENSION);                                 
      return(FALSE);                                                                                                                
   }                                                                                                                                

	/* Load the module, then call it's init function */
#undef MODULE_MEMORY_DEBUG
#ifdef MODULE_MEMORY_DEBUG
	snprintf(Path, sizeof(Path), "-m %s\\%s\r", ModulePath, ModuleName);
	XplSetCurrentScreen(XplCreateScreen("System Console", 0));
	ptr=Path;
	while (*ptr) {
		ungetch(*ptr);
		ptr++;
	}
	XplDelay(1000);
	snprintf(Path, sizeof(Path), "%s", ModuleName);

	ModuleHandle=(XplPluginHandle)FindNLMHandle(Path);
#else
	/* We want to make sure that any output from the loading is sent to the System Console */
	XplSetCurrentScreen(XplCreateScreen("System Console", 0));

	snprintf(Path, sizeof(Path), "%s/%s", ModulePath, ModuleName);
	ModuleHandle=XplLoadDLL(Path);
#endif
	if (!ModuleHandle) {
		ModuleDebug("Could not load module %s, errno:%d\n", Path, errno);
		return(FALSE);
	}

	/* Give the module a chance to initialize */
	HulaStrNCpy(Name, ModuleName, sizeof(Name) - strlen("Init"));
	Name[strlen(Name)-strlen(MODULE_EXTENSION)]='\0';
	ptr=Name;
	while (*ptr) {
		*ptr=toupper(*ptr);
		ptr++;
	}

	strcat(Name, "Init");
	ModuleInitFunc=(WAModuleInitFunc)XplGetDLLFunction(ModuleName, Name, ModuleHandle);

	if (!ModuleInitFunc) {
		ModuleDebug("Could not load module %s, missing Init function '%s'\n", ModuleName, Name);
		return(FALSE);
	}

	/* Grab the shutdown function */
	HulaStrNCpy(Name, ModuleName, sizeof(Name) - strlen("Shutdown"));
	Name[strlen(Name)-strlen(MODULE_EXTENSION)]='\0';
	ptr=Name;
	while (*ptr) {
		*ptr=toupper(*ptr);
		ptr++;
	}

	strcat(Name, "Shutdown");
	ModuleShutdownFunc=(WAModuleShutdownFunc)XplGetDLLFunction(ModuleName, Name, ModuleHandle);
	if (!ModuleShutdownFunc) {
		ModuleDebug("Could not load module %s, missing Shutdown function '%s'\n", ModuleName, Name);
		return(FALSE);
	}

	if (!ModuleInitFunc(&WAAPI)) {
		ModuleDebug("Could not load module %s, Init returned FALSE\n", ModuleName);
		return(FALSE);
	}

	AddModule(ModuleName, ModuleHandle, ModuleShutdownFunc);

	return(TRUE);
}


BOOL
LoadModules(unsigned char *ModulePath, BOOL ScanDir)
{
	ModuleDebugInit();

	WAAPISetup;

	TGid=XplGetThreadGroupID();
	if (ScanDir) {
		XplDir			*Directory;
		XplDir			*DirectoryEntry;
		unsigned char	Path[XPL_MAX_PATH+1];

		snprintf(Path, sizeof(Path), "%s", ModulePath);
		if ((Directory=XplOpenDir(Path))!=NULL) {
			while ((DirectoryEntry=XplReadDir(Directory))!=NULL) {
				if (!WAQuickCmp("webadmin.nlm", DirectoryEntry->d_nameDOS) 
                        && (WAQuickNCmp("wa", DirectoryEntry->d_nameDOS, 2) || WAQuickNCmp("libwa", DirectoryEntry->d_nameDOS, 5))) {
					LoadWebAdminModule(ModulePath, DirectoryEntry->d_nameDOS, NULL);
				}
			}
			XplCloseDir(Directory);
		}
	}
	return(TRUE);
}


BOOL
WARegisterModule(ModuleRegisterStruct *Register)
{
	int	OldTGid;

	OldTGid=XplSetThreadGroupID(TGid);

	switch(Register->ModuleType) {
		case MODULE_TEMPLATE: {
			AddTModule((TemplateModuleStruct *)&(Register->Module.Template));
			break;
		}

		case MODULE_OBJECT: {
			AddOModule((ObjectModuleStruct *)&(Register->Module.Object));
			break;
		}

		case MODULE_SAVE: {
			AddSModule((SaveModuleStruct *)&(Register->Module.Save));
			break;
		}

		case MODULE_DELETE: {
			AddDModule((DeleteModuleStruct *)&(Register->Module.Delete));
			break;
		}

		case MODULE_RENAME: {
			AddRModule((RenameModuleStruct *)&(Register->Module.Rename));
			break;
		}
	}

	XplSetThreadGroupID(OldTGid);
	return(TRUE);
}
