/****************************************************************************
 *
 * Copyright (c) 2004-2005 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>

#include "hula-mgrp.h"

#include <mwtempl.h>
#include <webadmin.h>

#define PRODUCT_NAME              "Hula Manager"
#define PRODUCT_SHORT_NAME        "Hula"
#define PRODUCT_DESCRIPTION       ""

HulaManagerConfig                  HulaManager;

/*  fixme - the Exiting boolean is misued!
*/
BOOL                              Exiting;

static void 
HulaManagerSortAgentsByPriority(HulaManagerAgent **Head, HulaManagerAgent **Tail)
{
    HulaManagerAgent	*head = *Head;
    HulaManagerAgent	*tail = *Tail;
    HulaManagerAgent	*agent = *Head;

    while (agent->next != NULL) {
        if (agent->next->priority < agent->priority) {
            if ((agent->next->previous = agent->previous) != NULL) {
                agent->previous->next = agent->next;
            } else {
                head = agent->next;
            }

            agent->previous = agent->next;

            if ((agent->next = agent->previous->next) != NULL) {
                agent->next->previous = agent;
            } else {
                tail = agent;
            }

            agent->previous->next = agent;

            agent = head;

            continue;
        }

        agent = agent->next;
    }

    *Head = head;
    *Tail = tail;

    return;
}

static BOOL 
HulaManagerCheckConfiguration(void)
{
    int i;
    BOOL result = FALSE;
    MDBValueStruct *config = NULL;

    config = MDBCreateValueStruct(HulaManager.directory.handle, NULL);
    if (config) {
        HulaManager.directory.tree[0] = '\\';

        result = MDBGetServerInfo(HulaManager.directory.host, HulaManager.directory.tree + 1, config);
    } else {
        XplConsolePrintf("Hula directory driver not responding; shutting down.\r\n");
    }

    if (result) {
        i = MDBReadDN(HulaManager.directory.host, MSGSRV_A_HULA_MESSAGING_SERVER, config);
        if (i) {
            strcpy(HulaManager.directory.dn, config->Value[0]);
        } else {
            i = MDBReadDN(HulaManager.directory.host, MSGSRV_A_MESSAGING_SERVER, config);
            if (i) {
                strcpy(HulaManager.directory.dn, config->Value[0]);

                i = MDBWriteDN(HulaManager.directory.host, MSGSRV_A_HULA_MESSAGING_SERVER, config);
                if (i) {
                    MDBClear(HulaManager.directory.host, MSGSRV_A_MESSAGING_SERVER, config);
                } else {
                    result = FALSE;

                    XplConsolePrintf("Hula messaging server not upgraded; shutting down.\r\n");
                }
            } else {
                result = FALSE;

                XplConsolePrintf("Hula messaging server not configured; shutting down.\r\n");
            }
        }

        MDBFreeValues(config);

        if (result) {
            HulaManager.directory.rdn = strrchr(HulaManager.directory.dn, '\\') + 1;

            i = MDBRead(HulaManager.directory.dn, MSGSRV_A_SERVER_STATUS, config);
            if (i) {
                HulaManager.flags |= HULAMRG_FLAG_FAILED_SHUTDOWN;
            }
        }

        MDBFreeValues(config);

        if (result) {
            i = MDBReadDN(HulaManager.directory.host, WA_A_CONFIG_DN, config);
            if (i) {
                strcpy(HulaManager.directory.webAdmin, config->Value[0]);
            } else {
                XplConsolePrintf("Hula web administration server not configured; skipping.\r\n");

                HulaManager.directory.webAdmin[0] = '\0';
            }
        }
    } else {
        XplConsolePrintf("Hula directory host not configured; shutting down.\r\n");
    }

    if (config) {
        MDBDestroyValueStruct(config);
        config = NULL;
    }

    return(result);
}

static BOOL 
HulaManagerProcessAgentList(MDBValueStruct *List, HulaManagerAgent **Head, HulaManagerAgent **Tail, MDBValueStruct *V)
{
    unsigned long i;
    unsigned long j;
    BOOL result = TRUE;
    HulaManagerAgent *agent;

    for (i = 0; i < List->Used; i++) {
        if (MDBRead(List->Value[i], MSGSRV_A_MODULE_NAME, V)) {
            HulaManager.agents.count++;

            agent = (HulaManagerAgent *)MemMalloc(sizeof(HulaManagerAgent));
            if (agent) {
                agent->flags = 0;

                agent->priority = HULAMRG_AGENT_DEFAULT_PRIORITY;

                agent->data = NULL;
                agent->next = NULL;

                if ((agent->previous = *Tail) != NULL) {
                    (*Tail)->next = agent;
                } else {
                    *Head = agent;
                }

                *Tail = agent;

                strcpy(agent->name, V->Value[0]);

                MDBFreeValues(V);

                if (XplStrCaseCmp(agent->name, MSGSRV_NLM_NMAP) == 0) {
                    agent->flags |= HULAMRG_FLAG_MODULE_NMAP;
                    agent->priority = HULAMRG_AGENT_NMAP_PRIORITY;
                } else if (XplStrCaseCmp(agent->name, MSGSRV_NLM_SMTP) == 0) {
                    agent->flags |= HULAMRG_FLAG_MODULE_SMTP;
                    agent->priority = HULAMRG_AGENT_SMTP_PRIORITY;
                }

                if (HulaManagerAgentPrep(agent) == TRUE) {
                    if (MDBRead(List->Value[i], MSGSRV_A_CONFIGURATION, V)) {
                        for (j = 0; j < V->Used; j++) {
                            if (strncmp(V->Value[j], "LoadPriority:", 13) == 0) {
                                agent->priority = atoi(V->Value[j] + 13);

                                if (agent->priority > HULAMRG_AGENT_DEFAULT_PRIORITY) {
                                    agent->priority = HULAMRG_AGENT_DEFAULT_PRIORITY;
                                } else if (agent->priority < HULAMRG_AGENT_MINIMUM_PRIORITY) {
                                    agent->priority = HULAMRG_AGENT_MINIMUM_PRIORITY;
                                }

                                break;
                            }
                        }

                        MDBFreeValues(V);
                    }

                    if ((MDBRead(List->Value[i], MSGSRV_A_DISABLED, V) == 0) || (V->Value[0][0] != '1')) {
                        agent->flags |= HULAMRG_FLAG_MODULE_ENABLED;
                    } else {
                        agent->flags &= ~HULAMRG_FLAG_MODULE_ENABLED;
                    }

                    MDBFreeValues(V);
                } else {
                    agent->flags &= ~HULAMRG_FLAG_MODULE_ENABLED;
                }

            } else {
                result = FALSE;
                break;
            }
        }

        continue;
    }

    return(result);
}

static BOOL 
HulaManagerLoadConfiguration(void)
{
    BOOL result = FALSE;
    MDBValueStruct *list = NULL;
    MDBValueStruct *config = NULL;
    HulaManagerAgent *head = NULL;
    HulaManagerAgent *tail = NULL;
    HulaManagerAgent *agent;

    list = MDBCreateValueStruct(HulaManager.directory.handle, NULL);
    config = MDBCreateValueStruct(HulaManager.directory.handle, NULL);
    if (list && config) {
        result = MDBEnumerateObjects(HulaManager.directory.dn, NULL, NULL, list);
    }

    if (result && list->Used) {
        result = HulaManagerProcessAgentList(list, &head, &tail, config);

        MDBFreeValues(list);
    }

    if (result && HulaManager.directory.webAdmin[0]) {
        agent = MemMalloc(sizeof(HulaManagerAgent));
        if (agent) {
            agent->flags = 0;

            agent->priority = HULAMRG_AGENT_DEFAULT_PRIORITY;

            agent->data = NULL;
            agent->previous = NULL;

            if ((agent->next = head) != NULL) {
                head->previous = agent;
            } else {
                tail = agent;
            }

            head = agent;

            strcpy(agent->name, MSGSRV_NLM_WEBADMIN);

            if (HulaManagerAgentPrep(agent) != FALSE) {
                if ((MDBRead(HulaManager.directory.webAdmin, MSGSRV_A_DISABLED, config) == 0) || (config->Value[0][0] != '1')) {
                    agent->flags |= HULAMRG_FLAG_MODULE_ENABLED;
                } else {
                    agent->flags &= ~HULAMRG_FLAG_MODULE_ENABLED;
                }
            }

            MDBFreeValues(config);
        } else {
            result = FALSE;
        }
    }

    if (result) {
        agent = MemMalloc(sizeof(HulaManagerAgent));
        if (agent) {
            agent->flags = HULAMRG_FLAG_MODULE_ENABLED | HULAMRG_FLAG_MODULE_DMC;

            agent->priority = HULAMRG_AGENT_DMC_PRIORITY;

            agent->data = NULL;
            agent->previous = NULL;

            if ((agent->next = head) != NULL) {
                head->previous = agent;
            } else {
                tail = agent;
            }

            head = agent;

            strcpy(agent->name, MSGSRV_NLM_DMC);

            result = HulaManagerAgentPrep(agent);
        } else {
            result = FALSE;
        }
    }

    if (result) {
        HulaManagerSortAgentsByPriority(&head, &tail);

        HulaManager.agents.head = head;
        HulaManager.agents.tail = tail;
    }

    if (config) {
        MDBDestroyValueStruct(config);
        config = NULL;
    }

    if (list) {
        MDBDestroyValueStruct(list);
        list = NULL;
    }

    return(result);
}

static void 
HulaManagerUnloadConfiguration(void)
{
    HulaManagerAgent *agent;

    agent = HulaManager.agents.head;
    HulaManager.agents.head = NULL;
    HulaManager.agents.tail = NULL;
    HulaManager.agents.count = 0;

    while (agent) {
        HulaManagerAgentRelease(agent);

        agent = agent->next;
    }

    return;
}

void 
HulaManagerStartAgents(void)
{
    HulaManagerAgent	*agent = HulaManager.agents.head;

    while (agent) {
        if ((agent->flags & HULAMRG_FLAG_MODULE_ENABLED) 
                && !(agent->flags & HULAMRG_FLAG_MODULE_LOADED)) {
            if (HulaManagerLoadAgent(agent) == TRUE) {
                agent->flags |= HULAMRG_FLAG_MODULE_LOADED;
            }
        }

        agent = agent->next;
    }

    return;
}

void 
HulaManagerStopAgents(void)
{
    HulaManagerAgent	*agent = HulaManager.agents.tail;

    while (agent) {
        if (agent->flags & HULAMRG_FLAG_MODULE_LOADED) {
            HulaManagerUnloadAgent(agent);
        }

        agent = agent->previous;
    }

    return;
}

void 
HulaManagerUnload(void)
{
    HulaManagerUnloadConfiguration();

    if (HulaManager.directory.handle) {
        MsgShutdown();
    }

    ConnShutdown();

    if (HulaManager.logging) {
        LoggerClose(HulaManager.logging);
        HulaManager.logging = NULL;
    }

    /*
        Cleanup SSL
    */
    if (HulaManager.ssl.client) {
        SSL_CTX_free(HulaManager.ssl.client);
        HulaManager.ssl.client = NULL;
    }


    if (HulaManager.ssl.server) {
        SSL_CTX_free(HulaManager.ssl.server);
        HulaManager.ssl.server = NULL;
    }

    ERR_free_strings();
    ERR_remove_state(0);
    EVP_cleanup();

	XplRWLockDestroy(&HulaManager.lock);

    XPLCryptoLockDestroy();

    IPCleanup();

    return;
}

BOOL 
HulaManagerLoad(HulaManagerFlags Flags)
{
    unsigned long i;
    unsigned long err;
    unsigned char buffer[256];
    BOOL result;
    MD5_CTX context;

    HulaManager.directory.handle = NULL;

    HulaManager.flags = Flags;
    HulaManager.state = HULAMRG_STATE_INITIALIZING;

    HulaManager.version = 0;

    HulaManager.id.main = (XplThreadID)XplGetThreadID();
    HulaManager.id.group = (XplThreadID)XplGetThreadGroupID();

    HulaManager.ssl.client = NULL;
    HulaManager.ssl.server = NULL;

    XplOpenLocalSemaphore(HulaManager.sem.main, 0);
    XplOpenLocalSemaphore(HulaManager.sem.shutdown, 1);

    HulaManager.agents.count = 0;
    HulaManager.agents.head = NULL;
    HulaManager.agents.tail = NULL;

    HulaManager.times.reload = 5 * 60;

    HulaManager.signo = SIGTERM;

    memset(&HulaManager.lock, 0, sizeof(HulaManager.lock));
    XplRWLockInit(&HulaManager.lock);

    strcpy(HulaManager.paths.bin, XPL_DEFAULT_BIN_DIR);
    strcpy(HulaManager.paths.dbf, XPL_DEFAULT_DBF_DIR);
    strcpy(HulaManager.paths.certificate, XPL_DEFAULT_CERT_PATH);
    strcpy(HulaManager.paths.privateKey, XPL_DEFAULT_KEY_PATH);
    strcpy(HulaManager.paths.work, XPL_DEFAULT_WORK_DIR);

    result = MemoryManagerOpen(PRODUCT_SHORT_NAME);
    if (result) {
        IPInit();

        XplGetHighResolutionTime(i);

        SSL_load_error_strings();
        SSL_library_init();

        XPLCryptoLockInit();

        srand((unsigned int)i);

        MD5_Init(&context);

        for (i = 0; i < sizeof(buffer); ) {
            buffer[i] = (unsigned char)(rand());

            if (buffer[i]) {
                i++;
            }
        }

        buffer[sizeof(buffer) - 1] = '\0';

        MD5_Update(&context, buffer, sizeof(buffer) - 1);

        MD5_Final(buffer, &context);

        RAND_seed(buffer, 16);

    	HulaManager.logging = LoggerOpen("hulamanager");
        if (HulaManager.logging == NULL) {
            err = 0;
            XplConsolePrintf("Hula manager failed to acquire a logging handle; error %lu.\r\n", err);
        }

        result = ConnStartup(15 * 60, FALSE);
    }

    if (result) {
        result = MDBInit();
    } else {
        XplConsolePrintf("Hula connection management library failed to initialize.\r\n");
    }

    if (result) {
        HulaManager.directory.handle = (MDBHandle *)MsgInit();
        if (HulaManager.directory.handle) {
            result = HulaManagerCheckConfiguration();
        } else {
            XplConsolePrintf("Hula messaging library failed to initialize.\r\n");
        }
    } else {
        XplConsolePrintf("Hula directory interface library failed to initialize.\r\n");
    }

    if (result) {
        result = HulaManagerLoadConfiguration();
    } else {
        XplConsolePrintf("Host not configured for Hula\r\n");
    }

    if (result) {
        HulaManager.state = HULAMRG_STATE_RUNNING;
    } else {
        HulaManager.state = HULAMRG_STATE_UNLOADING;
    }

    return(result);
}
