/****************************************************************************
 *
 * Copyright (c) 2004 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#if defined(SUSE) || defined(LINUX) || defined(S390RH) || defined(REDHAT) || defined(SOLARIS) || defined(MACOSX)
#include <sys/time.h>
#include <sys/types.h>
#include <sys/resource.h>
#include <sys/wait.h>
#include <unistd.h>

#include "hula-mgrp.h"

typedef struct _HULAMGR_AGENT_DATA {
    pid_t pid;
    unsigned char filename[XPL_MAX_PATH + 1];
} HulaManagerAgentData;

static void 
HulaManagerSignalHandler(int sigtype)
{
    switch(sigtype) {
        case SIGHUP: {
            if (HulaManager.state < HULAMRG_STATE_UNLOADING) {
                HulaManager.state = HULAMRG_STATE_UNLOADING;
            }

            break;
        }

        case SIGINT:
        case SIGTERM: {
            if (HulaManager.state == HULAMRG_STATE_STOPPING) {
                HulaManager.signo = SIGKILL;
            } else if (HulaManager.state < HULAMRG_STATE_STOPPING) {
                HulaManager.state = HULAMRG_STATE_STOPPING;
            }

            break;
        }

        default: {
            break;
        }
    }

    return;
}

BOOL 
HulaManagerAgentPrep(HulaManagerAgent *Agent)
{
    unsigned char *ptr;

    if (Agent) {
        if (!Agent->data) {
            Agent->data = (void *)MemMalloc(sizeof(HulaManagerAgentData));
        }

        if (Agent->data) {
            ((HulaManagerAgentData *)(Agent->data))->pid = 0;

            sprintf(((HulaManagerAgentData *)(Agent->data))->filename, "%s/%s", XPL_DEFAULT_BIN_DIR, Agent->name);
            ptr = ((HulaManagerAgentData *)(Agent->data))->filename + strlen(XPL_DEFAULT_BIN_DIR) + 1;
            while (*ptr != '\0') {
                if (*ptr != '.') {
                    *ptr = tolower(*ptr);
                } else if (XplStrCaseCmp(ptr, ".NLM") == 0) {
                    *ptr = '\0';

                   break;
                }

                ptr++;
            }

            return(TRUE);
        }
    }

    return(FALSE);
}

void 
HulaManagerAgentRelease(HulaManagerAgent *Agent)
{
    if (Agent) {
        MemFree(Agent->data);
        Agent->data = NULL;
    }

    return;
}

BOOL 
HulaManagerLoadAgent(HulaManagerAgent *Agent)
{
    pid_t pid;
    time_t loadTime;
    BOOL result = FALSE;
    unsigned char *ptr;

    if (Agent) {
        loadTime = Agent->load;

	XplSetEffectiveUserId(0);

        pid = fork();

	if (XplSetEffectiveUser(MsgGetUnprivilegedUser()) < 0) {
	    XplConsolePrintf("hulamanager: Could not drop to unprivileged user '%s', exiting.\n", MsgGetUnprivilegedUser());
	    HulaManagerUnload();
	    exit(1);
	}

        if (pid == 0) {
            if ((loadTime) 
                    && ((loadTime + HulaManager.times.reload) > time(NULL))) {
                XplConsolePrintf("Agent %s has already been started in the last %d seconds.  Waiting to load again.\n", Agent->name, HulaManager.times.reload);

                XplDelay((loadTime + HulaManager.times.reload) - time(NULL));
            }

            ptr = strrchr(((HulaManagerAgentData *)(Agent->data))->filename, '/');
            XplConsolePrintf("  loading %s\r\n", ptr + 1);

            execl(((HulaManagerAgentData *)(Agent->data))->filename, ((HulaManagerAgentData *)(Agent->data))->filename, NULL);

            perror("agent execl");	

            exit(-1);	
        } else if (pid > 0) {
            Agent->load = time(NULL);

            ((HulaManagerAgentData *)(Agent->data))->pid = pid;

            result = TRUE;
        } else {
            perror("fork");

            result = FALSE;
        }
    }

    return(result);
}

BOOL 
HulaManagerUnloadAgent(HulaManagerAgent *Agent)
{
    unsigned char *ptr;

    if (Agent) {
        ptr = strrchr(((HulaManagerAgentData *)(Agent->data))->filename, '/');

        if (HulaManager.signo == SIGTERM) {
            XplConsolePrintf("  unloading %s\r\n", ptr + 1);
        } else {
            XplConsolePrintf("  killing %s\r\n", ptr + 1);
        }

	XplSetEffectiveUser(0);
	
        kill(((HulaManagerAgentData *)(Agent->data))->pid, HulaManager.signo);

	if (XplSetEffectiveUser(MsgGetUnprivilegedUser()) < 0) {
	    XplConsolePrintf("hulamanager: Could not drop to unprivileged user '%s', exiting.\n", MsgGetUnprivilegedUser());
	    HulaManagerUnload();
	    exit(1);
	}

        Agent->load = 0;

        return(TRUE);
    }

    return(FALSE);
}

static void 
HulaManagerUsage(void)
{
    XplConsolePrintf("HulaManager - Hula Management Utility\r\n");
    XplConsolePrintf("usage: hulamanager\r\n");
    XplConsolePrintf("  -h | --help            Help.  Display this screen.\r\n");
    XplConsolePrintf("  -r | --restart         Restart running Hula agents.\r\n");
    XplConsolePrintf("  --status               Report on running Hula agents.\r\n");
    XplConsolePrintf("  -s | --stop            Terminate running Hula agents.\r\n");
    XplConsolePrintf("  -d                     Run as a daemon.\r\n");

    return;
}

int 
main(int argc, char *argv[])
{
    int i;
    unsigned char line[1024];
    unsigned char path[XPL_MAX_PATH + 1];
    unsigned char *ptr;
    struct rlimit rlim;
    sigset_t SignalSet;
    pid_t pid;
    BOOL result = TRUE;
    BOOL daemonize = FALSE;
    FILE *fh;
    HulaManagerAgent *agent;
    HulaManagerStates state = HULAMRG_STATE_INITIALIZING;

    if (XplSetEffectiveUser(MsgGetUnprivilegedUser()) < 0) {
	XplConsolePrintf("hulamanager: Could not drop to unprivileged user '%s', exiting.\n", MsgGetUnprivilegedUser());
	return 1;
    }

    for (i = 1; i < argc; i++) {
        if ((XplStrCaseCmp(argv[i], "--help") == 0) 
                || (XplStrCaseCmp(argv[i], "-h") == 0) 
                || (XplStrCaseCmp(argv[i], "/h") == 0) 
                || (XplStrCaseCmp(argv[i], "-?") == 0) 
                || (XplStrCaseCmp(argv[i], "/?") == 0)) {
            result = FALSE;
        } else if ((strcmp(argv[i], "-r") == 0) || (strcmp(argv[i], "--restart") == 0)) {
            state = HULAMRG_STATE_RELOADING;
        } else if (strcmp(argv[i], "--status") == 0) {
            state = HULAMRG_STATE_RUNNING;
        } else if ((strcmp(argv[i], "-s") == 0) || (strcmp(argv[i], "--stop") == 0)) {
            state = HULAMRG_STATE_STOPPING;
        } else if (strcmp(argv[i], "-d") == 0) {
            daemonize = TRUE;
        } else {
            result = FALSE;

            XplConsolePrintf("Unknown argument: %s\r\n", argv[i]);
        }
    }

    if (!result) {
        HulaManagerUsage();

        return(0);
    }

    strcpy(path, XPL_DEFAULT_WORK_DIR);
    XplMakeDir(path);

    sprintf(path, "%s/hulamanager.pid", XPL_DEFAULT_WORK_DIR);
    fh = fopen(path, "r");
    if (fh == NULL) {
        pid = -1;
    } else {
        fread(line, sizeof(unsigned char), sizeof(line), fh);
        fclose(fh);

        pid = strtol(line, NULL, 10);
        if (kill(pid, 0) != 0) {
            pid = -1;
        }
    }

    if (pid != -1) {
        switch (state) {
            case HULAMRG_STATE_INITIALIZING: {
                XplConsolePrintf("Hula Manager is currently running; process id %d.\n", pid);

                exit(0);

                break;
            }

            case HULAMRG_STATE_RUNNING: {
                XplConsolePrintf("Hula Manager is currently running; requesting status.\r\n");

                if (kill(pid, SIGUSR1) != 0) {
                    perror("kill");
                }

                exit(0);

                break;
            }

            case HULAMRG_STATE_RELOADING: {
                XplConsolePrintf("Hula Manager is currently running; attempting to restart.\r\n");

                if (kill(pid, SIGHUP) != 0) {
                    perror("kill");
                }

                exit(0);

                break;
            }

            case HULAMRG_STATE_STOPPING: {
                XplConsolePrintf("Hula Manager is currently running; attempting to stop.\r\n");

                if (kill(pid, SIGTERM) != 0) {
                    perror("kill");
                }

                exit(0);

                break;
            }

            case HULAMRG_STATE_LOADING:
            case HULAMRG_STATE_STARTING:
            case HULAMRG_STATE_UNLOADING:
            case HULAMRG_STATE_DONE:
            case HULAMRG_STATE_MAX_STATES:
            default: {
                XplConsolePrintf("Unknown Hula manager state!\r\n");

                exit(-1);

                break;
            }
        }
    }

    state = HULAMRG_STATE_INITIALIZING;

    fh = fopen(path, "w+");
    if (fh != NULL) {
        fprintf(fh, "%d", getpid());
        fclose(fh);
    } else {
        XplConsolePrintf("Hula Manager could not create \"%s\"; error %d.\n", path, errno);

        exit(-1);
    }

    if (daemonize) {
        int i;
        
        pid = fork();
        if (pid < 0) {
            perror("daemonization");
            exit(-1);
        } else if (pid != 0) {
            exit(0);
        }
        setsid();
        chdir("/");
        for (i = getdtablesize(); i>-1; i--) { close(i); }
        i = open("/dev/null", O_RDWR);
        dup(i);
        dup(i);
    }

    rlim.rlim_max = 20480;
    rlim.rlim_cur = 20480;
    if (setrlimit(RLIMIT_NOFILE, &rlim) == -1) {
        perror("setrlimit");
    }

    result = HulaManagerLoad(0);
    if (result) {
        XplConsolePrintf("Loading Hula agents for server %s.\r\n", HulaManager.directory.rdn);

        sigfillset(&SignalSet);

        sigdelset(&SignalSet, SIGQUIT);
        sigdelset(&SignalSet, SIGILL);
        sigdelset(&SignalSet, SIGABRT);
        sigdelset(&SignalSet, SIGFPE);
        sigdelset(&SignalSet, SIGKILL);
        sigdelset(&SignalSet, SIGSTOP);
        sigdelset(&SignalSet, SIGSYS);
        sigdelset(&SignalSet, SIGTRAP);
        sigdelset(&SignalSet, SIGURG);
        sigdelset(&SignalSet, SIGINT);
        sigdelset(&SignalSet, SIGTERM);
        sigdelset(&SignalSet, SIGUSR1);
        sigdelset(&SignalSet, SIGUSR2);
        sigdelset(&SignalSet, SIGHUP);

        pthread_sigmask(SIG_SETMASK, &SignalSet, NULL);

        setsid();

        HulaManagerStartAgents();

        XplSignalHandler(HulaManagerSignalHandler);

        while (HulaManager.state < HULAMRG_STATE_DONE) {
            pid = waitpid(-1, &i, 0);
            if (pid > 0) {
                agent = HulaManager.agents.head;
                while (agent) {
                    if (((HulaManagerAgentData *)(agent->data))->pid != pid) {
                        agent = agent->next;
                        continue;
                    }

                    ptr = strrchr(((HulaManagerAgentData *)(agent->data))->filename, '/');

                    if (WIFEXITED(i)) {
                        /*
                            Agent terminated by calling either exit or _exit.
                            printf("Agent %s self terminated; restarting.\r\n", agent->Name);

                            HulaManagerLoadAgent(agent);
                        */

                        agent->flags &= ~HULAMRG_FLAG_MODULE_LOADED;
                        break;
                    } else if (WIFSIGNALED(i)) {
                        if (WTERMSIG(i) == SIGTERM) {
                            /*
                                Agent terminated upon a SIGTERM.
                            */

                            agent->flags &= ~HULAMRG_FLAG_MODULE_LOADED;
                            break;
                        } else if (WTERMSIG(i) == SIGKILL) {
                            /* 
                                Agent terminated upon a SIGKILL
                            */

                            agent->flags &= ~HULAMRG_FLAG_MODULE_LOADED;
                            break;
                        } else {
                            /*
                                Agent terminated due to an unhandled signal; restart.
                            */

                            HulaManagerLoadAgent(agent);
                            break;
                        }
                    } else if (WIFSTOPPED(i)) {
                        /*
                            The child has been stopped!
                        */

                        break;
                    }
                }

                continue;
            }

            if ((pid != 0) && (errno != ECHILD) && (errno != EINTR)) {
                continue;
            }

            switch (HulaManager.state) {
                case HULAMRG_STATE_RUNNING: { 
                    if (errno != EINTR) {
                        HulaManager.state = HULAMRG_STATE_DONE;
                    }

                    continue;
                }

                case HULAMRG_STATE_UNLOADING: {
                    HulaManagerStopAgents();

                    HulaManager.state = HULAMRG_STATE_RELOADING;

                    continue;
                }

                case HULAMRG_STATE_RELOADING: {
                    HulaManager.state = HULAMRG_STATE_RUNNING;

                    HulaManagerStartAgents();

                    continue;
                }

                case HULAMRG_STATE_STOPPING: {
                    if (errno != ECHILD) {
                        HulaManagerStopAgents();
                    } else {
                        HulaManager.state = HULAMRG_STATE_DONE;
                    }

                    continue;
                }

                case HULAMRG_STATE_DONE: {
                    break;
                }

                case HULAMRG_STATE_STARTING:
                case HULAMRG_STATE_INITIALIZING:
                case HULAMRG_STATE_LOADING:
                case HULAMRG_STATE_MAX_STATES:
                default: {
                    break;
                }
            }

            break;
        }

        HulaManagerStopAgents();
    }

    HulaManagerUnload();

    XplConsolePrintf("Hula Manager shutdown.\r\n");

    return(0);

}

#endif  /*  defined(SUSE) || defined(LINUX) || defined(S390RH) || defined(REDHAT) || defined(SOLARIS) || defined(MACOSX)    */
