#!/usr/bin/perl -w

use strict;

my @CommonStrings;
my @CommonStringValues;
my @Strings;
my @Languages;

sub ReadLinesToTable {
    my $filename = shift;
    my $table = shift;
    
    open INCLUDE, $filename;
    while (<INCLUDE>) {
	chomp;
	s/\r$//;
	push @{$table}, ($_);
    }
    close INCLUDE;
}

sub ReadStringSection {
    my $table = shift;

    while (<TEMPLATE>) {
	chomp;
	s/\r$//;
	if ($_ eq "") {
	    next;
	} elsif (/^\[/) {
	    return $_;
	} elsif (/^\#include\W*[\"<](.*)[\">]/) {
	    ReadLinesToTable($1, $table);
	} elsif (/^\#include\W+(.*)/) {
	    ReadLinesToTable($1, $table);	    
	} else {
	    push @{$table}, $_;
	}
    }
}

sub ReadLanguageSection {
    while (<TEMPLATE>) {
	chomp;
	s/\r$//;

	if ($_ eq "") {
	    next;
	} elsif (/^\[/) {
	    return $_;
	} elsif (/^(\w+)\[(\d+)\]\W?\=\W?(.*)/) {
	    my %language = ();
	    
	    $language{name} = $1;
	    $language{number} = $2;
	    
	    my $files = $3;

	    my @filelist = split /\W*\,\W*/, $files;
	    my @strings;
	    
	    foreach my $file (@filelist) {
		ReadLinesToTable ($file, \@strings);
	    }

	    $language{filename} = $filelist[0];
	    $language{strings} = \@strings;

	    my $hashref = \%language;
	    push @Languages, $hashref;
	}
    }
}

sub ReadCommonStringFiles {
    while (<TEMPLATE>) {
	chomp;
	s/\r$//;

	if ($_ eq "") {
	    next;
	} elsif (/^\[/) {
	    return $_;
	} elsif (/^Strings\W?\=\W?(.*)/) {
	    my @filelist = split /\W*\,\W*/, $1;
	    foreach my $file (@filelist) {
		ReadLinesToTable ($file, \@CommonStringValues);
	    }
	}
    }

}

sub ScanTemplateFile {
    open TEMPLATE, "template.def";
	
    $_ = <TEMPLATE>;
    while (!(eof TEMPLATE)) {
	if (/\[CommonStrings\]/) {
	    $_ = ReadStringSection (\@CommonStrings);
	} elsif (/\[CommonStringFiles\]/) {
	    $_ = ReadCommonStringFiles ();
	} elsif (/\[Strings\]/) {
	    $_ = ReadStringSection (\@Strings);
	} elsif (/\[Languages\]/) {
	    $_ = ReadLanguageSection();
	} else {
	    $_ = <TEMPLATE>;
	}
    }
    close TEMPLATE;
}

sub WriteCommonStrings {
    print TEMPLATEOUT "[CommonStrings]\n";
    print TEMPLATEOUT "#include common.def\n";
    print TEMPLATEOUT "\n";

    open STRINGS, ">common.def";
    for (my $i = 0; $i < scalar @CommonStrings; $i++) {
	print STRINGS "$CommonStrings[$i]|$CommonStringValues[$i]\n";
    }
    close STRINGS;
}

sub WriteStrings {
    print TEMPLATEOUT "[Strings]\n";
    print TEMPLATEOUT "#include strings.def\n";
    print TEMPLATEOUT "\n";

    open STRINGS, ">strings.def";
    for (my $i = 0; $i < scalar @Strings; $i++) {	
	print STRINGS "$Strings[$i]|$Languages[0]->{strings}->[$i]\n";
    }
    close STRINGS;
}

sub WriteLanguages
{
    print TEMPLATEOUT "[Languages]\n";
    for(my $i = 0; $i < scalar @Languages; $i++) {
	my $language = $Languages[$i];
	print TEMPLATEOUT "$language->{name}\[$language->{number}\]=$language->{filename}\n";
	open LANGUAGE, ">$language->{filename}";

	# The default language is the default in strings.def
	if ($i != 0) {
	    for (my $j = 0; $j < scalar @Strings; $j++) {	
		print LANGUAGE "$Strings[$j]|$Languages[$i]->{strings}->[$j]\n";
	    }
	}
	close LANGUAGE;
    }
    print TEMPLATEOUT "\n";
}

sub IgnoreSection
{
    $_ = <TEMPLATE>;
    while (!eof TEMPLATE) {
	if (/^\[/) {
	    return $_;
	}
	$_ = <TEMPLATE>;
    }
    return $_;
}

sub WriteTemplateFile {
    open TEMPLATE, "template.def";
    open TEMPLATEOUT, ">template.def.new";

    $_ = <TEMPLATE>;
    
    while (!(eof TEMPLATE)) {
	if (/\[CommonStrings\]/) {
	    WriteCommonStrings();
	    $_ = IgnoreSection();
	} elsif (/\[CommonStringFiles\]/) {
	    $_ = IgnoreSection();
	} elsif (/\[Strings\]/) {
	    WriteStrings();
	    $_ = IgnoreSection();
	} elsif (/\[Languages\]/) {
	    WriteLanguages();
	    $_ = IgnoreSection();
	} else {
	    print TEMPLATEOUT $_;
	    $_ = <TEMPLATE>;
	}
    }
    print TEMPLATEOUT $_;

    close TEMPLATE;
    close TEMPLATEOUT;

    `cp template.def.new template.def`;
}

ScanTemplateFile();

WriteTemplateFile()
