/****************************************************************************
 *
 * Copyright (c) 2005 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>
#include <hulautil.h>

#define ICAL2_C

#include <libical2.h>

/*
    RFC 2445 Secion 4.2.1 Alternate Text Representation

    altrepparam
        "ALTREP" "=" DQUOTE uri DQUOTE
*/
BOOL 
ICal2AltRepParameterParser(ICal2Object *ical)
{
    unsigned char *ptr;
    ICal2Parameter *parameter;
    ICal2ParameterDetail *detail;
    ICal2Property *property;
    ICal2Component *component;

    component = ical->parse.component;
    property = ical->parse.property;

    parameter = (ICal2Parameter *)MemMalloc(sizeof(ICal2Parameter));
    if (parameter) {
        memset (parameter, 0, sizeof(ICal2Parameter));

        parameter->type = ICAL2_PARAMETER_ALTREP;

        ICAL2_ADD_PROPERTY_PARAMETER(property, parameter);

        ptr = parameter->name = ical->parse.param;

        ptr += 6;
        *ptr++ = '\0';

        detail = &parameter->items.detail[parameter->items.count++];
        detail->altRep.uri = ptr;

        ptr = strchr(ptr + 1, '"');
        if (ptr) {
            ptr++;

            switch (*ptr) {
                case ':': {
                    *ptr++ = '\0';
                    ical->parse.param = NULL;
                    ical->parse.value = ptr;

                    return(TRUE);
                }

                case ';': {
                    *ptr++ = '\0';
                    ical->parse.param = ptr;

                    return(TRUE);
                }

                default: {
                    break;
                }
            }
        }
    }

    return(FALSE);
}

/*
    RFC 2445 Secion 4.2.2 Common Name

    cnparam
        "CN" "=" param-value
*/
BOOL 
ICal2CnParameterParser(ICal2Object *ical)
{
    unsigned char *ptr;
    ICal2Parameter *parameter;
    ICal2ParameterDetail *detail;
    ICal2Property *property;
    ICal2Component *component;

    component = ical->parse.component;
    property = ical->parse.property;

    parameter = (ICal2Parameter *)MemMalloc(sizeof(ICal2Parameter));
    if (parameter) {
        memset (parameter, 0, sizeof(ICal2Parameter));

        parameter->type = ICAL2_PARAMETER_CN;

        ICAL2_ADD_PROPERTY_PARAMETER(property, parameter);

        ptr = parameter->name = ical->parse.param;

        ptr += 2;
        *ptr++ = '\0';

        do {
            ICAL2_PULL_PARAMETER_DETAIL(property, parameter, detail);
            if (detail) {
                detail->cn.value = ptr;

                while (ICal2IsParamText(*ptr)) {
                    ptr++;
                }

                switch (*ptr) {
                    case '"': {
                        ptr = strchr(ptr + 1, '"');
                        if (ptr) {
                            ptr++;
                            continue;
                        }

                        break;
                    }

                    case ',': {
                        *ptr++ = '\0';
                        continue;
                    }

                    default: {
                        break;
                    }
                }

                break;
            }

            return(FALSE);
        } while ((*ptr != ':') && (*ptr != ';'));

        if (*ptr == ':') {
            *ptr++ = '\0';
            ical->parse.param = NULL;
            ical->parse.value = ptr;

            return(TRUE);
        } else if (*ptr == ';') {
            *ptr++ = '\0';
            ical->parse.param = ptr;

            return(TRUE);
        }
    }

    return(FALSE);
}

/*
    RFC 2445 Secion 4.2.3 Calendar User Type

    cutypeparam
        ;Default is "INDIVIDUAL"
        "CUTYPE" "=" (
            "INDIVIDUAL"    ; An individual
            / "GROUP"       ; A group of individuals
            / "RESOURCE"    ; A physical resource
            / "ROOM"        ; A room resource
            / "UNKNOWN"     ; Otherwise not known
            / x-name        ; Experimental type
            / iana-token    ; Other IANA registered
                            ; type
        )
*/
BOOL 
ICal2CuTypeParameterParser(ICal2Object *ical)
{
    unsigned char delim;
    unsigned char *ptr;
    ICal2Parameter *parameter;
    ICal2ParameterDetail *detail;
    ICal2Property *property;
    ICal2Component *component;

    component = ical->parse.component;
    property = ical->parse.property;

    parameter = (ICal2Parameter *)MemMalloc(sizeof(ICal2Parameter));
    if (parameter) {
        memset (parameter, 0, sizeof(ICal2Parameter));

        parameter->type = ICAL2_PARAMETER_CUTYPE;

        ICAL2_ADD_PROPERTY_PARAMETER(property, parameter);

        ptr = parameter->name = ical->parse.param;

        ptr += 6;
        *ptr++ = '\0';

        detail = &parameter->items.detail[parameter->items.count++];
        detail->cuType.type = ICAL2_CALENDAR_USER_TYPE_IANA;
        detail->cuType.value = ptr;

        while (ICal2IsXName(*ptr)) {
            ptr++;
        }

        delim = *ptr;
        *ptr++ = '\0';

        switch (toupper(detail->cuType.value[0])) {
            case 'G': {
                if (XplStrCaseCmp(detail->cuType.value, "GROUP") == 0) {
                    detail->cuType.type = ICAL2_CALENDAR_USER_TYPE_GROUP;
                }

                break;
            }

            case 'I': {
                if (XplStrCaseCmp(detail->cuType.value, "INDIVIDUAL") == 0) {
                    detail->cuType.type = ICAL2_CALENDAR_USER_TYPE_INDIVIDUAL;
                }

                break;
            }

            case 'R': {
                switch (toupper(detail->cuType.value[1])) {
                    case 'E': {
                        if (XplStrCaseCmp(detail->cuType.value, "RESOURCE") == 0) {
                            detail->cuType.type = ICAL2_CALENDAR_USER_TYPE_RESOURCE;
                        }

                        break;
                    }

                    case 'O': {
                        if (XplStrCaseCmp(detail->cuType.value, "ROOM") == 0) {
                            detail->cuType.type = ICAL2_CALENDAR_USER_TYPE_ROOM;
                        }

                        break;
                    }

                    default: {
                        break;
                    }
                }

                break;
            }

            case 'U': {
                if (XplStrCaseCmp(detail->cuType.value, "UNKNOWN") == 0) {
                    detail->cuType.type = ICAL2_CALENDAR_USER_TYPE_UNKNOWN;
                }

                break;
            }

            case 'X': {
                if (detail->cuType.value[1] == '-') {
                    detail->cuType.type = ICAL2_CALENDAR_USER_TYPE_X;
                }

                break;
            }

            default: {
                break;
            }
        }

        switch (delim) {
            case ':': {
                ical->parse.param = NULL;
                ical->parse.value = ptr;

                return(TRUE);
            }

            case ';': {
                ical->parse.param = ptr;

                return(TRUE);
            }

            default: {
                break;
            }
        }
    }

    return(FALSE);
}

/*
    RFC 2445 Secion 4.2.4 Delgators

    delfromparam
        "DELEGATED-FROM" "=" DQUOTE cal-address DQUOTE *("," DQUOTE cal-address DQUOTE)
*/
BOOL 
ICal2DelegatedFromParameterParser(ICal2Object *ical)
{
    unsigned char *ptr;
    ICal2Parameter *parameter;
    ICal2ParameterDetail *detail;
    ICal2Property *property;
    ICal2Component *component;

    component = ical->parse.component;
    property = ical->parse.property;

    parameter = (ICal2Parameter *)MemMalloc(sizeof(ICal2Parameter));
    if (parameter) {
        memset (parameter, 0, sizeof(ICal2Parameter));

        parameter->type = ICAL2_PARAMETER_DELEGATED_FROM;

        ICAL2_ADD_PROPERTY_PARAMETER(property, parameter);

        ptr = parameter->name = ical->parse.param;

        ptr += 14;
        *ptr++ = '\0';

        do {
            ICAL2_PULL_PARAMETER_DETAIL(property, parameter, detail);
            if (detail) {
                detail->delFrom.calAddress = ptr;

                ptr = strchr(ptr + 1, '"');
                if (ptr) {
                    ptr++;

                    switch (*ptr) {
                        case ',': {
                            *ptr++ = '\0';
                            continue;
                        }

                        case ':': {
                            *ptr++ = '\0';
                            ical->parse.param = NULL;
                            ical->parse.value = ptr;

                            return(TRUE);
                        }

                        case ';': {
                            *ptr++ = '\0';
                            ical->parse.param = ptr;

                            return(TRUE);
                        }

                        default: {
                            break;
                        }
                    }
                }
            }

            break;
        } while ((*ptr != ':') && (*ptr != ';'));
    }

    return(FALSE);
}

/*
    RFC 2445 Secion 4.2.5 Delagatees

    deltoparam
        "DELEGATED-TO" "=" DQUOTE cal-address DQUOTE *("," DQUOTE cal-address DQUOTE)
*/
BOOL 
ICal2DelegatedToParameterParser(ICal2Object *ical)
{
    unsigned char *ptr;
    ICal2Parameter *parameter;
    ICal2ParameterDetail *detail;
    ICal2Property *property;
    ICal2Component *component;

    component = ical->parse.component;
    property = ical->parse.property;

    parameter = (ICal2Parameter *)MemMalloc(sizeof(ICal2Parameter));
    if (parameter) {
        memset (parameter, 0, sizeof(ICal2Parameter));

        parameter->type = ICAL2_PARAMETER_DELEGATED_TO;

        ICAL2_ADD_PROPERTY_PARAMETER(property, parameter);

        ptr = parameter->name = ical->parse.param;

        ptr += 12;
        *ptr++ = '\0';

        do {
            ICAL2_PULL_PARAMETER_DETAIL(property, parameter, detail);
            if (detail) {
                detail->delTo.calAddress = ptr;

                ptr = strchr(ptr + 1, '"');
                if (ptr) {
                    ptr++;

                    switch (*ptr) {
                        case ',': {
                            *ptr++ = '\0';
                            continue;
                        }

                        case ':': {
                            *ptr++ = '\0';
                            ical->parse.param = NULL;
                            ical->parse.value = ptr;

                            return(TRUE);
                        }

                        case ';': {
                            *ptr++ = '\0';
                            ical->parse.param = ptr;

                            return(TRUE);
                        }

                        default: {
                            break;
                        }
                    }
                }
            }

            break;
        } while ((*ptr != ':') && (*ptr != ';'));
    }

    return(FALSE);
}

/*
    RFC 2445 Secion 4.2.6 Directory Entry Reference

    dirparam
        "DIR" "=" DQUOTE uri DQUOTE
*/
BOOL 
ICal2DirParameterParser(ICal2Object *ical)
{
    unsigned char *ptr;
    ICal2Parameter *parameter;
    ICal2ParameterDetail *detail;
    ICal2Property *property;
    ICal2Component *component;

    component = ical->parse.component;
    property = ical->parse.property;

    parameter = (ICal2Parameter *)MemMalloc(sizeof(ICal2Parameter));
    if (parameter) {
        memset (parameter, 0, sizeof(ICal2Parameter));

        parameter->type = ICAL2_PARAMETER_DIR;

        ICAL2_ADD_PROPERTY_PARAMETER(property, parameter);

        ptr = parameter->name = ical->parse.param;

        ptr += 3;
        *ptr++ = '\0';

        detail = &parameter->items.detail[parameter->items.count++];
        detail->dir.uri = ptr;

        ptr = strchr(ptr + 1, '"');
        if (ptr) {
            ptr++;

            switch (*ptr) {
                case ':': {
                    *ptr++ = '\0';
                    ical->parse.param = NULL;
                    ical->parse.value = ptr;

                    return(TRUE);
                }

                case ';': {
                    *ptr++ = '\0';
                    ical->parse.param = ptr;

                    return(TRUE);
                }

                default: {
                    break;
                }
            }
        }
    }

    return(FALSE);
}

/*
    RFC 2445 Secion 4.2.7 Inline Encoding

    encodingparam
        "ENCODING" "=" (
            "8BIT" /        ; "8bit" text encoding is defined in [RFC 2045]
            "BASE64" /      ; "BASE64" binary encoding format is defined in [RFC 2045]
            iana-token /    ; Some other IANA registered iCalendar encoding type
            x-name          ; A non-standard, experimental encoding type
        )

*/
BOOL 
ICal2EncodingParameterParser(ICal2Object *ical)
{
    unsigned char delim;
    unsigned char *ptr;
    ICal2Parameter *parameter;
    ICal2ParameterDetail *detail;
    ICal2Property *property;
    ICal2Component *component;

    component = ical->parse.component;
    property = ical->parse.property;

    parameter = (ICal2Parameter *)MemMalloc(sizeof(ICal2Parameter));
    if (parameter) {
        memset (parameter, 0, sizeof(ICal2Parameter));

        parameter->type = ICAL2_PARAMETER_ENCODING;

        ICAL2_ADD_PROPERTY_PARAMETER(property, parameter);

        ptr = parameter->name = ical->parse.param;

        ptr += 8;
        *ptr++ = '\0';

        detail = &parameter->items.detail[parameter->items.count++];
        detail->encoding.type = ICAL2_ENCODING_TYPE_IANA;
        detail->encoding.value = ptr;

        while (ICal2IsXName(*ptr)) {
            ptr++;
        }

        delim = *ptr;
        *ptr++ = '\0';

        switch (toupper(detail->encoding.value[0])) {
            case '8': {
                if (XplStrCaseCmp(detail->encoding.value, "8BIT") == 0) {
                    detail->encoding.type = ICAL2_ENCODING_TYPE_8BIT;
                }

                break;
            }

            case 'B': {
                if (XplStrCaseCmp(detail->encoding.value, "BASE64") == 0) {
                    detail->encoding.type = ICAL2_ENCODING_TYPE_BASE64;
                }

                break;
            }

            case 'X': {
                if (detail->encoding.value[1] == '-') {
                    detail->encoding.type = ICAL2_ENCODING_TYPE_X;
                }

                break;
            }

            default: {
                break;
            }
        }

        switch (delim) {
            case ':': {
                ical->parse.param = NULL;
                ical->parse.value = ptr;

                return(TRUE);
            }

            case ';': {
                ical->parse.param = ptr;

                return(TRUE);
            }

            default: {
                break;
            }
        }
    }

    return(FALSE);
}

/*
    RFC 2445 Secion 4.2.8 Format Type
    fmttypeparam
        "FMTTYPE" "=" (
            iana-token /    ; A IANA registered content type
            x-name          ; A non-standard content type
        )
*/
BOOL 
ICal2FmtTypeParameterParser(ICal2Object *ical)
{
    unsigned char delim;
    unsigned char *ptr;
    ICal2Parameter *parameter;
    ICal2ParameterDetail *detail;
    ICal2Property *property;
    ICal2Component *component;

    component = ical->parse.component;
    property = ical->parse.property;

    parameter = (ICal2Parameter *)MemMalloc(sizeof(ICal2Parameter));
    if (parameter) {
        memset (parameter, 0, sizeof(ICal2Parameter));

        parameter->type = ICAL2_PARAMETER_FMTTYPE;

        ICAL2_ADD_PROPERTY_PARAMETER(property, parameter);

        ptr = parameter->name = ical->parse.param;

        ptr += 7;
        *ptr++ = '\0';

        detail = &parameter->items.detail[parameter->items.count++];
        detail->format.type = ICAL2_FORMAT_TYPE_IANA;
        detail->format.value = ptr;

        while (ICal2IsXName(*ptr)) {
            ptr++;
        }

        delim = *ptr;
        *ptr++ = '\0';

        switch (toupper(detail->format.value[0])) {
            case 'X': {
                if (detail->format.value[1] == '-') {
                    detail->format.type = ICAL2_FORMAT_TYPE_X;
                }

                break;
            }

            default: {
                break;
            }
        }

        switch (delim) {
            case ':': {
                ical->parse.param = NULL;
                ical->parse.value = ptr;

                return(TRUE);
            }

            case ';': {
                ical->parse.param = ptr;

                return(TRUE);
            }

            default: {
                break;
            }
        }
    }

    return(FALSE);
}

/*
    RFC 2445 Secion 4.2.9 Free/Busy Time Type

    fbtypeparam
        "FBTYPE" "=" (
            "FREE" / 
            "BUSY" / 
            "BUSY-UNAVAILABLE" / 
            "BUSY-TENTATIVE" / 
            x-name /                ; Some experimental iCalendar data type.
            iana-token              ; Some other IANA registered iCalendar data type.
        )
*/
BOOL 
ICal2FbTypeParameterParser(ICal2Object *ical)
{
    unsigned char delim;
    unsigned char *ptr;
    ICal2Parameter *parameter;
    ICal2ParameterDetail *detail;
    ICal2Property *property;
    ICal2Component *component;

    component = ical->parse.component;
    property = ical->parse.property;

    parameter = (ICal2Parameter *)MemMalloc(sizeof(ICal2Parameter));
    if (parameter) {
        memset (parameter, 0, sizeof(ICal2Parameter));

        parameter->type = ICAL2_PARAMETER_FBTYPE;

        ICAL2_ADD_PROPERTY_PARAMETER(property, parameter);

        ptr = parameter->name = ical->parse.param;

        ptr += 6;
        *ptr++ = '\0';

        detail = &parameter->items.detail[parameter->items.count++];
        detail->freeBusy.type = ICAL2_FREE_BUSY_TYPE_IANA;
        detail->freeBusy.value = ptr;

        while (ICal2IsXName(*ptr)) {
            ptr++;
        }

        delim = *ptr;
        *ptr++ = '\0';

        switch (toupper(detail->freeBusy.value[0])) {
            case 'B': {
                if (XplStrCaseCmp(detail->freeBusy.value, "BUSY") == 0) {
                    detail->freeBusy.type = ICAL2_FREE_BUSY_TYPE_BUSY;
                } else if (XplStrCaseCmp(detail->freeBusy.value, "BUSY-UNAVAILABLE") == 0) {
                    detail->freeBusy.type = ICAL2_FREE_BUSY_TYPE_BUSY_UNAVAILABLE;
                } else if (XplStrCaseCmp(detail->freeBusy.value, "BUSY-TENTATIVE") == 0) {
                    detail->freeBusy.type = ICAL2_FREE_BUSY_TYPE_BUSY_TENTATIVE;
                }

                break;
            }

            case 'F': {
                if (XplStrCaseCmp(detail->freeBusy.value, "FREE") == 0) {
                    detail->freeBusy.type = ICAL2_FREE_BUSY_TYPE_FREE;
                }

                break;
            }

            case 'X': {
                if (detail->freeBusy.value[1] == '-') {
                    detail->freeBusy.type = ICAL2_FREE_BUSY_TYPE_X;
                }

                break;
            }

            default: {
                break;
            }
        }

        switch (delim) {
            case ':': {
                ical->parse.param = NULL;
                ical->parse.value = ptr;

                return(TRUE);
            }

            case ';': {
                ical->parse.param = ptr;

                return(TRUE);
            }

            default: {
                break;
            }
        }
    }

    return(FALSE);
}

/*
    RFC 2445 Secion 4.2.10 Language

    languageparam
        "LANGUAGE" "=" language

    language
        <Text identifying a language, as defined in [RFC 1766]>
*/
BOOL 
ICal2LanguageParameterParser(ICal2Object *ical)
{
    unsigned char *ptr;
    ICal2Parameter *parameter;
    ICal2ParameterDetail *detail;
    ICal2Property *property;
    ICal2Component *component;

    component = ical->parse.component;
    property = ical->parse.property;

    parameter = (ICal2Parameter *)MemMalloc(sizeof(ICal2Parameter));
    if (parameter) {
        memset (parameter, 0, sizeof(ICal2Parameter));

        parameter->type = ICAL2_PARAMETER_LANGUAGE;

        ICAL2_ADD_PROPERTY_PARAMETER(property, parameter);

        ptr = parameter->name = ical->parse.param;

        ptr += 6;
        *ptr++ = '\0';

        detail = &parameter->items.detail[parameter->items.count++];
        detail->language.value = ptr;

        while (!ICal2IsPropertyDelim(*ptr)) {
            ptr++;
        }

        switch (*ptr) {
            case ':': {
                *ptr++ = '\0';

                ical->parse.param = NULL;
                ical->parse.value = ptr;

                return(TRUE);
            }

            case ';': {
                *ptr++ = '\0';

                ical->parse.param = ptr;

                return(TRUE);
            }

            default: {
                break;
            }
        }
    }

    return(FALSE);
}

/*
    RFC 2445 Secion 4.2.11 Group or List Membership

    memberparam
        "MEMBER" "=" DQUOTE cal-address DQUOTE *("," DQUOTE cal-address DQUOTE)

*/
BOOL 
ICal2MemberParameterParser(ICal2Object *ical)
{
    unsigned char *ptr;
    ICal2Parameter *parameter;
    ICal2ParameterDetail *detail;
    ICal2Property *property;
    ICal2Component *component;

    component = ical->parse.component;
    property = ical->parse.property;

    parameter = (ICal2Parameter *)MemMalloc(sizeof(ICal2Parameter));
    if (parameter) {
        memset (parameter, 0, sizeof(ICal2Parameter));

        parameter->type = ICAL2_PARAMETER_MEMBER;

        ICAL2_ADD_PROPERTY_PARAMETER(property, parameter);

        ptr = parameter->name = ical->parse.param;

        ptr += 6;
        *ptr++ = '\0';

        do {
            ICAL2_PULL_PARAMETER_DETAIL(property, parameter, detail);
            if (detail) {
                detail->member.calAddress = ptr;

                ptr = strchr(ptr + 1, '"');
                if (ptr) {
                    ptr++;

                    switch (*ptr) {
                        case ',': {
                            *ptr++ = '\0';
                            continue;
                        }

                        case ':': {
                            *ptr++ = '\0';
                            ical->parse.param = NULL;
                            ical->parse.value = ptr;

                            return(TRUE);
                        }

                        case ';': {
                            *ptr++ = '\0';
                            ical->parse.param = ptr;

                            return(TRUE);
                        }

                        default: {
                            break;
                        }
                    }
                }
            }

            break;
        } while ((*ptr != ':') && (*ptr != ';'));
    }

    return(FALSE);
}

/*
    RFC 2445 Secion 4.2.12 Participant Status

    partstatparam
        ;These are the participation statuses for a "VEVENT".
        ;Default is "NEEDS-ACTION"
        "PARTSTAT" "=" (
            "NEEDS-ACTION" /    ;Event needs action
            "ACCEPTED" /        ;Event accepted
            "DECLINED" /        ;Event declined
            "TENTATIVE" /       ;Event tentatively accepted
            "DELEGATED" /       ;Event delegated
            x-name /            ;Experimental status
            iana-token          ;Other IANA registered status
        )

    partstatparam /= 
        ;These are the participation statuses for a "VTODO".
        ;Default is "NEEDS-ACTION"
        "PARTSTAT" "=" (
            "NEEDS-ACTION" /    ;To-do needs action
            "ACCEPTED" /        ;To-do accepted
            "DECLINED" /        ;To-do declined
            "TENTATIVE" /       ;To-do tentatively accepted
            "DELEGATED" /       ;To-do delegated
            "COMPLETED" /       ;To-do completed.
                                ;COMPLETED property has
                                ;date/time completed.
            "IN-PROCESS" /      ;To-do in process of
                                ;being completed
            x-name /            ;Experimental status
            iana-token          ;Other IANA registered status
        )

    partstatparam /=
        ;These are the participation statuses for a "VJOURNAL".
        ;Default is "NEEDS-ACTION"
        "PARTSTAT" "=" (
            "NEEDS-ACTION" /    ;Journal needs action
            "ACCEPTED" /        ;Journal accepted
            "DECLINED" /        ;Journal declined
            x-name /            ;Experimental status
            iana-token          ; Other IANA registered status
        )

*/
BOOL 
ICal2PartStatParameterParser(ICal2Object *ical)
{
    unsigned char delim;
    unsigned char *ptr;
    ICal2Parameter *parameter;
    ICal2ParameterDetail *detail;
    ICal2Property *property;
    ICal2Component *component;

    component = ical->parse.component;
    property = ical->parse.property;

    parameter = (ICal2Parameter *)MemMalloc(sizeof(ICal2Parameter));
    if (parameter) {
        memset (parameter, 0, sizeof(ICal2Parameter));

        parameter->type = ICAL2_PARAMETER_PARTSTAT;

        ICAL2_ADD_PROPERTY_PARAMETER(property, parameter);

        ptr = parameter->name = ical->parse.param;

        ptr += 8;
        *ptr++ = '\0';

        detail = &parameter->items.detail[parameter->items.count++];
        detail->partStat.type = ICAL2_PART_STATUS_TYPE_IANA;
        detail->partStat.status = ptr;

        while (ICal2IsXName(*ptr)) {
            ptr++;
        }

        delim = *ptr;
        *ptr++ = '\0';

        switch (toupper(detail->partStat.status[0])) {
            case 'A': {
                if (XplStrCaseCmp(detail->partStat.status, "ACCEPTED") == 0) {
                    detail->partStat.type = ICAL2_PART_STATUS_TYPE_ACCEPTED;
                }

                break;
            }

            case 'C': {
                if (XplStrCaseCmp(detail->partStat.status, "COMPLETED") == 0) {
                    detail->partStat.type = ICAL2_PART_STATUS_TYPE_COMPLETED;
                }

                break;
            }

            case 'D': {
                if (toupper(detail->partStat.status[1]) == 'E') {
                    switch (toupper(detail->partStat.status[2])) {
                        case 'C': {
                            if (XplStrCaseCmp(detail->partStat.status, "DECLINED") == 0) {
                                detail->partStat.type = ICAL2_PART_STATUS_TYPE_DECLINED;
                            }

                            break;
                        }

                        case 'L': {
                            if (XplStrCaseCmp(detail->partStat.status, "DELEGATED") == 0) {
                                detail->partStat.type = ICAL2_PART_STATUS_TYPE_DELEGATED;
                            }

                            break;
                        }

                        default: {
                            break;
                        }
                    }
                }

                break;
            }

            case 'I': {
                if (XplStrCaseCmp(detail->partStat.status, "IN_PROCESS") == 0) {
                    detail->partStat.type = ICAL2_PART_STATUS_TYPE_IN_PROCESS;
                }

                break;
            }

            case 'N': {
                if (XplStrCaseCmp(detail->partStat.status, "NEEDS_ACTION") == 0) {
                    detail->partStat.type = ICAL2_PART_STATUS_TYPE_NEEDS_ACTION;
                }

                break;
            }

            case 'T': {
                if (XplStrCaseCmp(detail->partStat.status, "TENTATIVE") == 0) {
                    detail->partStat.type = ICAL2_PART_STATUS_TYPE_TENTATIVE;
                }

                break;
            }

            case 'X': {
                /* X */
                if (detail->partStat.status[1] == '-') {
                    detail->partStat.type = ICAL2_PART_STATUS_TYPE_X;
                }

            }
        }

        switch (delim) {
            case ':': {
                ical->parse.param = NULL;
                ical->parse.value = ptr;

                return(TRUE);
            }

            case ';': {
                ical->parse.param = ptr;

                return(TRUE);
            }

            default: {
                break;
            }
        }
    }

    return(FALSE);
}

/*
    RFC 2445 Secion 4.2.13 Recurrence Identifier Range

    rangeparam
        "RANGE" "=" (
            ;To specify all instances prior to the recurrence identifier
            "THISANDPRIOR" /

            ;To specify the instance specified by the recurrence identifier
            ;and all subsequent recurrence instances
            "THISANDFUTURE"
        )

*/
BOOL 
ICal2RangeParameterParser(ICal2Object *ical)
{
    unsigned char delim;
    unsigned char *ptr;
    ICal2Parameter *parameter;
    ICal2ParameterDetail *detail;
    ICal2Property *property;
    ICal2Component *component;

    component = ical->parse.component;
    property = ical->parse.property;

    parameter = (ICal2Parameter *)MemMalloc(sizeof(ICal2Parameter));
    if (parameter) {
        memset (parameter, 0, sizeof(ICal2Parameter));

        parameter->type = ICAL2_PARAMETER_RANGE;

        ICAL2_ADD_PROPERTY_PARAMETER(property, parameter);

        ptr = parameter->name = ical->parse.param;

        ptr += 5;
        *ptr++ = '\0';

        detail = &parameter->items.detail[parameter->items.count++];
        detail->range.value = ptr;

        while (ICal2IsXName(*ptr)) {
            ptr++;
        }

        delim = *ptr;
        *ptr++ = '\0';

        if (XplStrCaseCmp(detail->range.value, "THISANDPRIOR") == 0) {
            detail->range.type = ICAL2_RANGE_TYPE_THISANDPRIOR;
        } else if (XplStrCaseCmp(detail->range.value, "THISANDFUTURE") == 0) {
            detail->range.type = ICAL2_RANGE_TYPE_THISANDFUTURE;
        } else {
            return(FALSE);
        }

        switch (delim) {
            case ':': {
                ical->parse.param = NULL;
                ical->parse.value = ptr;

                return(TRUE);
            }

            case ';': {
                ical->parse.param = ptr;

                return(TRUE);
            }

            default: {
                break;
            }
        }
    }

    return(FALSE);
}

/*
    RFC 2445 Secion 4.2.14 Alarm Trigger Relationship

    trigrelparam
        "RELATED" "=" (
            "START" /   ;Trigger off of start
            "END"       ;Trigger off of end
        )

*/
BOOL 
ICal2RelatedParameterParser(ICal2Object *ical)
{
    unsigned char delim;
    unsigned char *ptr;
    ICal2Parameter *parameter;
    ICal2ParameterDetail *detail;
    ICal2Property *property;
    ICal2Component *component;

    component = ical->parse.component;
    property = ical->parse.property;

    parameter = (ICal2Parameter *)MemMalloc(sizeof(ICal2Parameter));
    if (parameter) {
        memset (parameter, 0, sizeof(ICal2Parameter));

        parameter->type = ICAL2_PARAMETER_RANGE;

        ICAL2_ADD_PROPERTY_PARAMETER(property, parameter);

        ptr = parameter->name = ical->parse.param;

        ptr += 7;
        *ptr++ = '\0';

        detail = &parameter->items.detail[parameter->items.count++];
        detail->related.value = ptr;

        while (ICal2IsXName(*ptr)) {
            ptr++;
        }

        delim = *ptr;
        *ptr++ = '\0';

        if (XplStrCaseCmp(detail->related.value, "START") == 0) {
            detail->related.type = ICAL2_RELATED_TYPE_START;
        } else if (XplStrCaseCmp(detail->related.value, "END") == 0) {
            detail->related.type = ICAL2_RELATED_TYPE_END;
        } else {
            return(FALSE);
        }

        switch (delim) {
            case ':': {
                ical->parse.param = NULL;
                ical->parse.value = ptr;

                return(TRUE);
            }

            case ';': {
                ical->parse.param = ptr;

                return(TRUE);
            }

            default: {
                break;
            }
        }
    }

    return(FALSE);
}

/*
    RFC 2445 Secion 4.2.15 Relationship Type

    reltypeparam
        ;Default is "PARENT"
        "RELTYPE" "=" (
            "PARENT" /      ;Parent relationship.
            "CHILD" /       ;Child relationship
            "SIBLING" /     ;Sibling relationship
            iana-token /    ;Some other IANA registered iCalendar relationship type
            x-name          ;A non-standard, experimental relationship type
        )
*/
BOOL 
ICal2RelTypeParameterParser(ICal2Object *ical)
{
    unsigned char delim;
    unsigned char *ptr;
    ICal2Parameter *parameter;
    ICal2ParameterDetail *detail;
    ICal2Property *property;
    ICal2Component *component;

    component = ical->parse.component;
    property = ical->parse.property;

    parameter = (ICal2Parameter *)MemMalloc(sizeof(ICal2Parameter));
    if (parameter) {
        memset (parameter, 0, sizeof(ICal2Parameter));

        parameter->type = ICAL2_PARAMETER_RELTYPE;

        ICAL2_ADD_PROPERTY_PARAMETER(property, parameter);

        ptr = parameter->name = ical->parse.param;

        ptr += 7;
        *ptr++ = '\0';

        detail = &parameter->items.detail[parameter->items.count++];
        detail->relType.type = ICAL2_RELTYPE_TYPE_IANA;
        detail->relType.value = ptr;

        while (ICal2IsXName(*ptr)) {
            ptr++;
        }

        delim = *ptr;
        *ptr++ = '\0';

        switch (toupper(detail->relType.value[0])) {
            case 'C': {
                if (XplStrCaseCmp(detail->relType.value, "CHILD") == 0) {
                    detail->relType.type = ICAL2_RELTYPE_TYPE_CHILD;
                }

                break;
            }

            case 'P': {
                if (XplStrCaseCmp(detail->relType.value, "PARENT") == 0) {
                    detail->relType.type = ICAL2_RELTYPE_TYPE_PARENT;
                }

                break;
            }

            case 'S': {
                if (XplStrCaseCmp(detail->relType.value, "SIBLING") == 0) {
                    detail->relType.type = ICAL2_RELTYPE_TYPE_SIBLING;
                }

                break;
            }

            case 'X': {
                if (detail->relType.value[1] == '-') {
                    detail->relType.type = ICAL2_RELTYPE_TYPE_X;
                }

                break;
            }

            default: {
                break;
            }
        }

        switch (delim) {
            case ':': {
                ical->parse.param = NULL;
                ical->parse.value = ptr;

                return(TRUE);
            }

            case ';': {
                ical->parse.param = ptr;

                return(TRUE);
            }

            default: {
                break;
            }
        }
    }

    return(FALSE);
}

/*
    RFC 2445 Secion 4.2.16 Participation Role

    roleparam
        ;Default is "REQ-PARTICIPANT"
        "ROLE" "=" (
            "CHAIR" /           ;Indicates chair of the calendar entity
            "REQ-PARTICIPANT" / ;Indicates a participant whose participation is required
            "OPT-PARTICIPANT" / ;Indicates a participant whose participation is optional
            "NON-PARTICIPANT" / ;Indicates a participant who is copied for information
                                ;purposes only
            iana-token /        ;Other IANA role
            x-name              ;Experimental role
        )
*/
BOOL 
ICal2RoleParameterParser(ICal2Object *ical)
{
    unsigned char delim;
    unsigned char *ptr;
    ICal2Parameter *parameter;
    ICal2ParameterDetail *detail;
    ICal2Property *property;
    ICal2Component *component;

    component = ical->parse.component;
    property = ical->parse.property;

    parameter = (ICal2Parameter *)MemMalloc(sizeof(ICal2Parameter));
    if (parameter) {
        memset (parameter, 0, sizeof(ICal2Parameter));

        parameter->type = ICAL2_PARAMETER_ROLE;

        ICAL2_ADD_PROPERTY_PARAMETER(property, parameter);

        ptr = parameter->name = ical->parse.param;

        ptr += 4;
        *ptr++ = '\0';

        detail = &parameter->items.detail[parameter->items.count++];
        detail->role.type = ICAL2_ROLE_TYPE_IANA;
        detail->role.value = ptr;

        while (ICal2IsXName(*ptr)) {
            ptr++;
        }

        delim = *ptr;
        *ptr++ = '\0';

        switch (toupper(detail->role.value[0])) {
            case 'C': {
                if (XplStrCaseCmp(detail->role.value, "CHAIR") == 0) {
                    detail->role.type = ICAL2_ROLE_TYPE_CHAIR;
                }

                break;
            }

            case 'N': {
                if (XplStrCaseCmp(detail->role.value, "NON-PARTICIPANT") == 0) {
                    detail->role.type = ICAL2_ROLE_TYPE_NON_PARTICIPANT;
                }

                break;
            }

            case 'O': {
                if (XplStrCaseCmp(detail->role.value, "OPT-PARTICIPANT") == 0) {
                    detail->role.type = ICAL2_ROLE_TYPE_OPT_PARTICIPANT;
                }

                break;
            }

            case 'R': {
                if (XplStrCaseCmp(detail->role.value, "REQ-PARTICIPANT") == 0) {
                    detail->role.type = ICAL2_ROLE_TYPE_REQ_PARTICIPANT;
                }

                break;
            }

            case 'X': {
                if (detail->role.value[1] == '-') {
                    detail->role.type = ICAL2_ROLE_TYPE_X;
                }

                break;
            }

            default: {
                break;
            }
        }

        switch (delim) {
            case ':': {
                ical->parse.param = NULL;
                ical->parse.value = ptr;

                return(TRUE);
            }

            case ';': {
                ical->parse.param = ptr;

                return(TRUE);
            }

            default: {
                break;
            }
        }
    }

    return(FALSE);
}

/*
    RFC 2445 Secion 4.2.17 RSVP Expectation

    rsvpparam
        ;Default is FALSE
        "RSVP" "=" ("TRUE" / "FALSE")
*/
BOOL 
ICal2RsvpParameterParser(ICal2Object *ical)
{
    unsigned char delim;
    unsigned char *ptr;
    ICal2Parameter *parameter;
    ICal2ParameterDetail *detail;
    ICal2Property *property;
    ICal2Component *component;

    component = ical->parse.component;
    property = ical->parse.property;

    parameter = (ICal2Parameter *)MemMalloc(sizeof(ICal2Parameter));
    if (parameter) {
        memset (parameter, 0, sizeof(ICal2Parameter));

        parameter->type = ICAL2_PARAMETER_RSVP;

        ICAL2_ADD_PROPERTY_PARAMETER(property, parameter);

        ptr = parameter->name = ical->parse.param;

        ptr += 4;
        *ptr++ = '\0';

        detail = &parameter->items.detail[parameter->items.count++];
        detail->rsvp.value = ptr;

        while (ICal2IsXName(*ptr)) {
            ptr++;
        }

        delim = *ptr;
        *ptr++ = '\0';

        if (XplStrCaseCmp(detail->rsvp.value, "TRUE") == 0) {
            detail->rsvp.yes = TRUE;
        } else if (XplStrCaseCmp(detail->rsvp.value, "FALSE") == 0) {
            detail->rsvp.yes = FALSE;
        } else {
            return(FALSE);
        }

        switch (delim) {
            case ':': {
                ical->parse.param = NULL;
                ical->parse.value = ptr;

                return(TRUE);
            }

            case ';': {
                ical->parse.param = ptr;

                return(TRUE);
            }

            default: {
                break;
            }
        }
    }

    return(FALSE);
}

/*
    RFC 2445 Secion 4.2.18 Sent By

    sentbyparam
        "SENT-BY" "=" DQUOTE cal-address DQUOTE
*/
BOOL 
ICal2SentByParameterParser(ICal2Object *ical)
{
    unsigned char *ptr;
    ICal2Parameter *parameter;
    ICal2ParameterDetail *detail;
    ICal2Property *property;
    ICal2Component *component;

    component = ical->parse.component;
    property = ical->parse.property;

    parameter = (ICal2Parameter *)MemMalloc(sizeof(ICal2Parameter));
    if (parameter) {
        memset (parameter, 0, sizeof(ICal2Parameter));

        parameter->type = ICAL2_PARAMETER_SENT_BY;

        ICAL2_ADD_PROPERTY_PARAMETER(property, parameter);

        ptr = parameter->name = ical->parse.param;

        ptr += 7;
        *ptr++ = '\0';

        detail = &parameter->items.detail[parameter->items.count++];
        detail->sentBy.calAddress = ptr;

        ptr = strchr(ptr + 1, '"');
        if (ptr) {
            ptr++;

            switch (*ptr) {
                case ':': {
                    *ptr++ = '\0';
                    ical->parse.param = NULL;
                    ical->parse.value = ptr;

                    return(TRUE);
                }

                case ';': {
                    *ptr++ = '\0';
                    ical->parse.param = ptr;

                    return(TRUE);
                }

                default: {
                    break;
                }
            }
        }
    }

    return(FALSE);
}

/*
    RFC 2445 Secion 4.2.19 Time Zone Identifier

    tzidparam
        "TZID" "=" [tzidprefix] paramtext CRLF

    tzidprefix
        "/"
*/
BOOL 
ICal2TzIDParameterParser(ICal2Object *ical)
{
    unsigned char *ptr;
    ICal2Parameter *parameter;
    ICal2ParameterDetail *detail;
    ICal2Property *property;
    ICal2Component *component;

    component = ical->parse.component;
    property = ical->parse.property;

    parameter = (ICal2Parameter *)MemMalloc(sizeof(ICal2Parameter));
    if (parameter) {
        memset (parameter, 0, sizeof(ICal2Parameter));

        parameter->type = ICAL2_PARAMETER_TZID;

        ICAL2_ADD_PROPERTY_PARAMETER(property, parameter);

        ptr = parameter->name = ical->parse.param;

        ptr += 4;
        *ptr++ = '\0';

        detail = &parameter->items.detail[parameter->items.count++];
        detail->tzID.value = ptr;

        /*
            fixme

            This parser does not handle the tzidprefix char '/'.

            See RFC 2445 Section 4.2.19 Time Zone Identifier:

            The presence of the SOLIDUS character (US-ASCII decimal 47) as a
            prefix, indicates that this TZID represents a unique ID in a globally
            defined time zone registry (when such registry is defined).
        */
        if (*ptr != '/') {
            while (ICal2IsParamText(*ptr)) {
                ptr++;
            }

            switch (*ptr) {
                case ':': {
                    *ptr++ = '\0';
                    ical->parse.param = NULL;
                    ical->parse.value = ptr;

                    return(TRUE);
                }

                case ';': {
                    *ptr++ = '\0';
                    ical->parse.param = ptr;

                    return(TRUE);
                }

                default: {
                    break;
                }
            }
        }
    }

    return(FALSE);
}

/*
    RFC 2445 Secion 4.2.20 Value Data Type

    valuetypeparam
        "VALUE" "=" valuetype

    valuetype = (
            "BINARY" /
            "BOOLEAN" /
            "CAL-ADDRESS" /
            "DATE" /
            "DATE-TIME" /
            "DURATION" /
            "FLOAT" /
            "INTEGER" /
            "PERIOD" /
            "RECUR" /
            "TEXT" /
            "TIME" /
            "URI" /
            "UTC-OFFSET" /
            x-name /            ;Some experimental iCalendar data type.
            iana-token          ;Some other IANA registered iCalendar data type.
        )

*/
BOOL 
ICal2ValueParameterParser(ICal2Object *ical)
{
    unsigned char delim;
    unsigned char *ptr;
    ICal2Parameter *parameter;
    ICal2ParameterDetail *detail;
    ICal2Property *property;
    ICal2Component *component;

    component = ical->parse.component;
    property = ical->parse.property;

    parameter = (ICal2Parameter *)MemMalloc(sizeof(ICal2Parameter));
    if (parameter) {
        memset (parameter, 0, sizeof(ICal2Parameter));

        parameter->type = ICAL2_PARAMETER_VALUE;

        ICAL2_ADD_PROPERTY_PARAMETER(property, parameter);

        ptr = parameter->name = ical->parse.param;

        ptr += 5;
        *ptr++ = '\0';

        detail = &parameter->items.detail[parameter->items.count++];
        detail->value.type = ICAL2_VALUE_TYPE_IANA;
        detail->value.value = ptr;

        while (ICal2IsXName(*ptr)) {
            ptr++;
        }

        delim = *ptr;
        *ptr++ = '\0';

        switch (toupper(detail->value.value[0])) {
            case 'B': {
                switch (toupper(detail->value.value[1])) {
                    case 'I': {
                        if (XplStrCaseCmp(detail->value.value, "BINARY") == 0) {
                            detail->value.type = ICAL2_VALUE_TYPE_BINARY;
                        }

                        break;
                    }

                    case 'O': {
                        if (XplStrCaseCmp(detail->value.value, "BOOLEAN") == 0) {
                            detail->value.type = ICAL2_VALUE_TYPE_BOOLEAN;
                        }

                        break;
                    }

                    default: {
                        break;
                    }
                }

                break;
            }

            case 'C': {
                if (XplStrCaseCmp(detail->value.value, "CAL-ADDRESS") == 0) {
                    detail->value.type = ICAL2_VALUE_TYPE_CAL_ADDRESS;
                }

                break;
            }

            case 'D': {
                switch (toupper(detail->value.value[1])) {
                    case 'A': {
                        if (XplStrCaseCmp(detail->value.value, "DATE") == 0) {
                            detail->value.type = ICAL2_VALUE_TYPE_DATE;
                        } else if (XplStrCaseCmp(detail->value.value, "DATE-TIME") == 0) {
                            detail->value.type = ICAL2_VALUE_TYPE_DATE_TIME;
                        }

                        break;
                    }

                    case 'U': {
                        if (XplStrCaseCmp(detail->value.value, "DURATION") == 0) {
                            detail->value.type = ICAL2_VALUE_TYPE_DURATION;
                        }

                        break;
                    }

                    default: {
                        break;
                    }
                }

                break;
            }

            case 'F': {
                if (XplStrCaseCmp(detail->value.value, "FLOAT") == 0) {
                    detail->value.type = ICAL2_VALUE_TYPE_FLOAT;
                }

                break;
            }

            case 'I': {
                if (XplStrCaseCmp(detail->value.value, "INTEGER") == 0) {
                    detail->value.type = ICAL2_VALUE_TYPE_INTEGER;
                }

                break;
            }

            case 'P': {
                if (XplStrCaseCmp(detail->value.value, "PERIOD") == 0) {
                    detail->value.type = ICAL2_VALUE_TYPE_PERIOD;
                }

                break;
            }

            case 'R': {
                if (XplStrCaseCmp(detail->value.value, "RECUR") == 0) {
                    detail->value.type = ICAL2_VALUE_TYPE_RECUR;
                }

                break;
            }

            case 'T': {
                switch (toupper(detail->value.value[1])) {
                    case 'E': {
                        if (XplStrCaseCmp(detail->value.value, "TEXT") == 0) {
                            detail->value.type = ICAL2_VALUE_TYPE_TEXT;
                        }

                        break;
                    }

                    case 'I': {
                        if (XplStrCaseCmp(detail->value.value, "TIME") == 0) {
                            detail->value.type = ICAL2_VALUE_TYPE_TIME;
                        }

                        break;
                    }

                    default: {
                        break;
                    }
                }

                break;
            }

            case 'U': {
                switch (toupper(detail->value.value[1])) {
                    case 'R': {
                        if (XplStrCaseCmp(detail->value.value, "URI") == 0) {
                            detail->value.type = ICAL2_VALUE_TYPE_URI;
                        }

                        break;
                    }

                    case 'T': {
                        if (XplStrCaseCmp(detail->value.value, "UTC-OFFSET") == 0) {
                            detail->value.type = ICAL2_VALUE_TYPE_UTC_OFFSET;
                        }

                        break;
                    }

                    default: {
                        break;
                    }
                }

                break;
            }

            case 'X': {
                if (detail->value.value[1] == '-') {
                    detail->value.type = ICAL2_VALUE_TYPE_X;
                }

                break;
            }

            default: {
                break;
            }
        }

        switch (delim) {
            case ':': {
                ical->parse.param = NULL;
                ical->parse.value = ptr;

                return(TRUE);
            }

            case ';': {
                ical->parse.param = ptr;

                return(TRUE);
            }

            default: {
                break;
            }
        }
    }

    return(FALSE);
}

/*
    RFC 2445 Section 4.2 Property Parameters

    xparam
        x-name "=" param-value *("," param-value)
*/

BOOL 
ICal2XParameterParser(ICal2Object *ical)
{
    unsigned char *ptr;
    ICal2Parameter *parameter;
    ICal2ParameterDetail *detail;
    ICal2Property *property;
    ICal2Component *component;

    component = ical->parse.component;
    property = ical->parse.property;

    parameter = (ICal2Parameter *)MemMalloc(sizeof(ICal2Parameter));
    if (parameter) {
        memset (parameter, 0, sizeof(ICal2Parameter));

        parameter->type = ICAL2_PARAMETER_X;

        ICAL2_ADD_PROPERTY_PARAMETER(property, parameter);

        ptr = parameter->name = ical->parse.param;
        while (ICal2IsXName(*ptr)) {
            ptr++;
        }

        if (*ptr == '=') {
            *ptr++ = '\0';
        } else {
            return(FALSE);
        }

        do {
            ical->parse.param = ptr;

            ICAL2_PULL_PARAMETER_DETAIL(property, parameter, detail);
            if (detail) {
                detail->x.value = ical->parse.param;

                while (ICal2IsParamText(*ptr)) {
                    ptr++;
                }

                switch (*ptr) {
                    case '"': {
                        ptr = strchr(ptr + 1, '"');
                        if (ptr) {
                            ptr++;
                            continue;
                        }

                        break;
                    }

                    case ',': {
                        *ptr++ = '\0';
                        continue;
                    }

                    default: {
                        break;
                    }
                }
            }

            break;
        } while ((*ptr != ':') && (*ptr != ';'));

        if (*ptr == ':') {
            *ptr++ = '\0';
            ical->parse.param = NULL;
            ical->parse.value = ptr;

            return(TRUE);
        } else if (*ptr == ';') {
            *ptr++ = '\0';
            ical->parse.param = ptr;

            return(TRUE);
        }
    }

    return(FALSE);
}

BOOL 
ICal2IanaParameterParser(ICal2Object *ical)
{
    unsigned char *ptr;
    ICal2Parameter *parameter;
    ICal2ParameterDetail *detail;
    ICal2Property *property;
    ICal2Component *component;

    component = ical->parse.component;
    property = ical->parse.property;

    parameter = (ICal2Parameter *)MemMalloc(sizeof(ICal2Parameter));
    if (parameter) {
        memset (parameter, 0, sizeof(ICal2Parameter));

        parameter->type = ICAL2_PARAMETER_IANA;

        ICAL2_ADD_PROPERTY_PARAMETER(property, parameter);

        ptr = parameter->name = ical->parse.param;
        while (ICal2IsXName(*ptr)) {
            ptr++;
        }

        if (*ptr == '=') {
            *ptr++ = '\0';

            ical->parse.param = ptr;
        } else {
            return(FALSE);
        }

        do {
            ICAL2_PULL_PARAMETER_DETAIL(property, parameter, detail);
            if (detail) {
                detail->iana.value = ical->parse.param;

                while (ICal2IsParamText(*ptr)) {
                    ptr++;
                }

                switch (*ptr) {
                    case '"': {
                        ptr = strchr(ptr + 1, '"');
                        if (ptr) {
                            ptr++;
                            continue;
                        }

                        break;
                    }

                    case ',': {
                        *ptr++ = '\0';
                        continue;
                    }

                    default: {
                        break;
                    }
                }
            }

            break;
        } while ((*ptr != ':') && (*ptr != ';'));

        if (*ptr == ':') {
            *ptr++ = '\0';
            ical->parse.param = NULL;
            ical->parse.value = ptr;

            return(TRUE);
        } else if (*ptr == ';') {
            *ptr++ = '\0';
            ical->parse.param = ptr;

            return(TRUE);
        }
    }

    return(FALSE);
}
