/****************************************************************************
 *
 * Copyright (c) 2005 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>
#include <hulautil.h>

#define ICAL2_C

#include <libical2.h>

BOOL 
ICal2PropertyParser(ICal2Object *ical)
{
    unsigned long index;
    unsigned char *ptr;
    BOOL result = TRUE;
    ICal2Token *token;
    ICal2Property *property;
    ICal2Component *component = ical->parse.component;

    property = (ICal2Property *)MemMalloc(sizeof(ICal2Property));
    if (property) {
        memset(property, 0, sizeof(ICal2Property));

        property->name = ical->parse.name;
        property->type = ical->parse.token->type;

        ICAL2_ADD_COMPONENT_PROPERTY(component, property);

        component->counts.length += ical->parse.dest - ical->parse.name;
        component->counts.prop++;

        ical->parse.property = property;

        token = ical->parse.token;
        while (result && ical->parse.param) {
            ptr = ical->parse.param;
            index = HulaKeywordFind(ICal2ParameterIndex, ptr);
            if (index != -1) {
                ical->parse.token = &ICal2ParameterTokens[index];

                ptr += ical->parse.token->nameLen;
                if (*ptr == '=') {
                    result = ical->parse.token->parser.value(ical);
                } else if (ical->parse.token->type == ICAL2_PARAMETER_X) {
                    result = ICal2XParameterParser(ical);
                } else {
                    result = ICal2IanaParameterParser(ical);
                }
            } else {
                result = ICal2IanaParameterParser(ical);
            }
        }

        ical->parse.token = token;
        result = ical->parse.token->parser.value(ical);
    } else {
        result = FALSE;
    }

    if (!result) {
        if (property) {
            MemFree(property);
            property = NULL;
        }
    }

    return(result);
}

/*
    RFC 2445 Section 4.6 Calendar Components

    component
        1*(
            eventc / 
            todoc / 
            journalc / 
            freebusyc / 
            timezonec / 
            iana-comp / 
            x-comp
        )

    iana-comp
        "BEGIN" ":" iana-token CRLF

        1*contentline

        "END" ":" iana-token CRLF

    x-comp
        "BEGIN" ":" x-name CRLF

        1*contentline

        "END" ":" x-name CRLF
*/
BOOL 
ICal2BeginPropertyParser(ICal2Object *ical)
{
    long index;
    unsigned long length = ical->parse.dest - ical->parse.name;
    unsigned char *ptr;
    ICal2Token *token;
    ICal2Component *child;

    child = (ICal2Component *)MemMalloc(sizeof(ICal2Component));
    if (child) {
        memset(child, 0, sizeof(ICal2Component));

        child->counts.length = length;
        if ((child->parent = ical->parse.component) != NULL) {
            if ((child->prev = child->parent->tail) != NULL) {
                child->prev->next = child;
            } else {
                child->parent->head = child;
            }

            child->parent->tail = child;
        } else {
            ical->components.head = ical->components.tail = child;
        }

        ical->parse.component = child;

        ptr = child->name = ical->parse.value;
        index = HulaKeywordFind(ICal2ComponentIndex, ptr);
        if (index != -1) {
            token = &ICal2ComponentTokens[index];

            ptr += token->nameLen;
            if (*ptr == '\0') {
                child->type = token->type;
            } else if (token->type == ICAL2_COMPONENT_X) {
                child->type = ICAL2_COMPONENT_X;
            } else {
                child->type = ICAL2_COMPONENT_IANA;
            }
        } else {
            child->type = ICAL2_COMPONENT_IANA;
        }

        return(TRUE);
    }

    return(FALSE);
}

/*
    RFC 2445 Section 4.6 Calendar Components

    component
        1*(
            eventc / 
            todoc / 
            journalc / 
            freebusyc / 
            timezonec / 
            iana-comp / 
            x-comp
        )

    iana-comp
        "BEGIN" ":" iana-token CRLF

        1*contentline

        "END" ":" iana-token CRLF

    x-comp
        "BEGIN" ":" x-name CRLF

        1*contentline

        "END" ":" x-name CRLF
*/
BOOL 
ICal2EndPropertyParser(ICal2Object *ical)
{
    unsigned long type;
    BOOL result = FALSE;
    ICal2Token *token;
    ICal2Component *child;
    ICal2Component *parent;

    child = ical->parse.component;
    if (XplStrCaseCmp(ical->parse.value, child->name) == 0) {
        type = child->type;
        token = &ICal2ComponentTokens[0];
        while (token->name && (token->type != type)) {
            token++;
        }

        if (token->name) {
            ical->parse.token = token;
            result = token->parser.property(ical);

            child->counts.length += ical->parse.dest - ical->parse.name;

            parent = child->parent;
            if (parent) {
                parent->counts.comp += child->counts.comp + 1;
                parent->counts.length += child->counts.length;
            } else {
                ical->components.count += child->counts.comp + 1;
                ical->length += child->counts.length;
            }

            ical->parse.component = parent;
        }
    }

    return(result);
}

/*
    RFC 2445 Section 4.8.8.1 Non-standard properites

    x-prop
        x-name *(";" xparam) [";" languageparam] ":" text CRLF
*/
BOOL 
ICal2XPropertyParser(ICal2Object *ical)
{
    unsigned char *ptr;
    BOOL result = TRUE;
    ICal2Token *token;
    ICal2Property *property;
    ICal2Component *component = ical->parse.component;

    property = (ICal2Property *)MemMalloc(sizeof(ICal2Property));
    if (property) {
        memset(property, 0, sizeof(ICal2Property));

        property->name = ical->parse.name;
        property->type = ICAL2_PROPERTY_X;

        ICAL2_ADD_COMPONENT_PROPERTY(component, property);

        component->counts.length += ical->parse.dest - ical->parse.name;

        component->counts.prop++;
        ical->parse.property = property;

        ptr = property->name;
        while (ICal2IsXName(*ptr)) {
            ptr++;
        }

        if (*ptr == ':') {
            *ptr++ = '\0';

            ical->parse.param = NULL;
            ical->parse.value = ptr;
        } else if (*ptr == ';') {
            *ptr++ = '\0';

            ical->parse.param = ptr;
            ical->parse.value = NULL;
        } else {
            return(FALSE);
        }

        token = ical->parse.token;
        while (result && ical->parse.param) {
            ptr = ical->parse.param;
            if ((toupper(ptr[0]) == 'X') && (ptr[1] == '-')) {
                result = ICal2XParameterParser(ical);
            } else {
                result = ICal2IanaParameterParser(ical);
            }
        }

        ical->parse.token = token;
        result = ICal2TextValueParser(ical);
    } else {
        result = FALSE;
    }

    if (!result) {
        if (property) {
            MemFree(property);
            property = NULL;
        }
    }

    return(result);
}

BOOL 
ICal2IanaPropertyParser(ICal2Object *ical)
{
    unsigned char *ptr;
    BOOL result = TRUE;
    ICal2Token *token;
    ICal2Property *property;
    ICal2Component *component = ical->parse.component;

    property = (ICal2Property *)MemMalloc(sizeof(ICal2Property));
    if (property) {
        memset(property, 0, sizeof(ICal2Property));

        property->name = ical->parse.name;
        property->type = ICAL2_PROPERTY_IANA;

        ICAL2_ADD_COMPONENT_PROPERTY(component, property);

        component->counts.length += ical->parse.dest - ical->parse.name;

        component->counts.prop++;
        ical->parse.property = property;

        ptr = property->name;
        while (ICal2IsXName(*ptr)) {
            ptr++;
        }

        if (*ptr == ':') {
            *ptr++ = '\0';

            ical->parse.param = NULL;
            ical->parse.value = ptr;
        } else if (*ptr == ';') {
            *ptr++ = '\0';

            ical->parse.param = ptr;
            ical->parse.value = NULL;
        } else {
            return(FALSE);
        }

        token = ical->parse.token;
        while (result && ical->parse.param) {
            result = ICal2IanaParameterParser(ical);
        }

        ical->parse.token = token;
        result = ICal2ValueParser(ical);
    } else {
        result = FALSE;
    }

    if (!result) {
        if (property) {
            MemFree(property);
            property = NULL;
        }
    }

    return(result);
}
