/****************************************************************************
 *
 * Copyright (c) 2001-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <stdlib.h>
#include <xpl.h>

#include <mwtempl.h>
#include <webadmin.h>

#include <errno.h>
#include <hulautil.h>
#include <connio.h>
#include <management.h>

#include <mdb.h>
#include <msgapi.h>

#include <wastats.tok>
#include <wastats.ary>

EXPORT BOOL LIBWASTATSInit(WAAPIArg);
EXPORT BOOL LIBWASTATSShutdown(void);

WAAPIDefine;

/* General stuff */
struct {
    BOOL unload;

    struct {
        XplThreadID main;
        XplThreadID group;
    } id;

    struct {
        XplSemaphore shutdown;
    } sem;
} WAStats;

typedef struct {
    MDBValueStruct *V;
    unsigned long offset;

    struct {
        Connection *conn;

        unsigned char dn[MDB_MAX_OBJECT_CHARS + 1];
        unsigned char rdn[MDB_MAX_OBJECT_CHARS + 1];
        unsigned char line[CONN_BUFSIZE + 1];
        unsigned long time;
    } selected;

    struct {
        StatisticsStruct cleared;
        StatisticsStruct current;
    } server;

    struct {
        AntispamStatisticsStruct cleared;
        AntispamStatisticsStruct current;
    } spam;

    struct {
        BOOL enable;
        ConnSSLConfiguration config;
        SSL_CTX *context;
        Connection *conn;
    } ssl;
} WAstatsSessionStruct;

static int
ReleaseMessagingServer(WAstatsSessionStruct *statsSession)
{
    int ccode = 0;

    if (statsSession->selected.conn) {
        if ((ccode = ConnWrite(statsSession->selected.conn, "QUIT\r\n", 6)) != -1) {
            ccode = ConnFlush(statsSession->selected.conn);
        }

        ConnClose(statsSession->selected.conn, 0);
        ConnFree(statsSession->selected.conn);

        statsSession->selected.conn = NULL;
    }

    return(ccode);
}

static int
ConnectToMessagingServer(unsigned char *dn, SessionStruct *session, WAstatsSessionStruct *statsSession)
{
    int ccode;
    unsigned long used;
    unsigned long port[2];
    MDBValueStruct *config;

    if (statsSession->selected.conn) {
        ReleaseMessagingServer(statsSession);
    }

    port[0] = DMC_MANAGER_SSL_PORT;
    port[1] = DMC_MANAGER_PORT;

    config = MDBCreateValueStruct(session->AuthHandle, NULL);
    if (config) {
        MDBRead(dn, MSGSRV_A_CONFIGURATION, config);
    } else {
        return(FALSE);
    }

    statsSession->selected.conn = ConnAlloc(TRUE);

    for (used = 0; used < config->Used; used++) {
        if (XplStrNCaseCmp(config->Value[used], "DMC: ManagerPort=", 17) == 0) {
            if (isdigit(config->Value[used][17])) {
                port[1] = atol(config->Value[used] + 17);
            } else {
                port[1] = -1;
            }
        } else if (XplStrNCaseCmp(config->Value[used], "DMC: ManagerSSLPort=", 20) == 0) {
            if (isdigit(config->Value[used][20])) {
                port[0] = atol(config->Value[used] + 20);
            } else {
                port[0] = -1;
            }
        }
    }

    MDBFreeValues(config);

    if (((port[0] != (unsigned long)-1) || (port[1] != (unsigned long)-1)) && (MDBRead(dn, MSGSRV_A_IP_ADDRESS, config))) {
        ccode = 0;

        statsSession->selected.conn->socketAddress.sin_family = AF_INET;
        statsSession->selected.conn->socketAddress.sin_addr.s_addr = inet_addr(config->Value[0]);
    } else {
        ccode = -1;
    }

    MDBDestroyValueStruct(config);
    config = NULL;

    if (!ccode) {
        if (statsSession->ssl.enable && ((statsSession->selected.conn->socketAddress.sin_port = htons((unsigned short)port[0])) != (unsigned short)-1)) {
            ccode = ConnConnect(statsSession->selected.conn, NULL, 0, statsSession->ssl.context);
        } else if ((statsSession->selected.conn->socketAddress.sin_port = htons((unsigned short)port[1])) != (unsigned short)-1) {
            ccode = ConnConnect(statsSession->selected.conn, NULL, 0, NULL);
        } else {
            ccode = -1;
        }

        if (ccode != -1) {
            ccode = ConnRead(statsSession->selected.conn, statsSession->selected.line, CONN_BUFSIZE);
        }
    }

    return(ccode);
}

static int
LoadStatistics(SessionStruct *session, WAstatsSessionStruct *statsSession)
{
    int ccode;
    unsigned long i;
    unsigned long *statistic;

    if (statsSession->selected.conn) {
        if (((ccode = ConnWrite(statsSession->selected.conn, "STAT0\r\n", 7)) != -1)
                && ((ccode = ConnFlush(statsSession->selected.conn)) != -1)
                && ((ccode = ConnReadLine(statsSession->selected.conn, statsSession->selected.line, CONN_BUFSIZE)) != -1)
                && (XplStrNCaseCmp(statsSession->selected.line, DMCMSG1001RESPONSES_COMMING, 5) == 0)
                && (atol(statsSession->selected.line + 5) == (sizeof(StatisticsStruct) / sizeof(unsigned long)))) {
            statistic = &(statsSession->server.current.ModulesLoaded);

            for (i = 0; (ccode != -1) && (i < sizeof(StatisticsStruct) / sizeof(unsigned long)); i++, statistic++) {
                if ((ccode = ConnReadLine(statsSession->selected.conn, statsSession->selected.line, CONN_BUFSIZE)) != -1) {
                    *statistic = atol(statsSession->selected.line);
                }
            }

            if (ccode != -1) {
                if (i == sizeof(StatisticsStruct) / sizeof(unsigned long)) {
                    ccode = ConnRead(statsSession->selected.conn, statsSession->selected.line, CONN_BUFSIZE);
                } else {
                    ccode = -1;
                }
            }
        } else if (ccode != -1) {
            ccode = -1;
        }
    } else {
        ccode = -1;
    }

    return(ccode);
}

static int
LoadSpamStatistics(SessionStruct *session, WAstatsSessionStruct *statsSession)
{
    int ccode;
    unsigned long i;
    unsigned long *statistic;

    if (statsSession->selected.conn) {
        if (((ccode = ConnWrite(statsSession->selected.conn, "STAT1\r\n", 7)) != -1)
                && ((ccode = ConnFlush(statsSession->selected.conn)) != -1)
                && ((ccode = ConnReadLine(statsSession->selected.conn, statsSession->selected.line, CONN_BUFSIZE)) != -1)
                && (XplStrNCaseCmp(statsSession->selected.line, DMCMSG1001RESPONSES_COMMING, 5) == 0)
                && (atol(statsSession->selected.line + 5) == (sizeof(AntispamStatisticsStruct) / sizeof(unsigned long)))) {
            statistic = &(statsSession->spam.current.QueueSpamBlocked);

            for (i = 0; (ccode != -1) && (i < sizeof(AntispamStatisticsStruct) / sizeof(unsigned long)); i++, statistic++) {
                if ((ccode = ConnReadLine(statsSession->selected.conn, statsSession->selected.line, CONN_BUFSIZE)) != -1) {
                    *statistic = atol(statsSession->selected.line);
                }
            }

            if (ccode != -1) {
                if (i == sizeof(AntispamStatisticsStruct) / sizeof(unsigned long)) {
                    ccode = ConnRead(statsSession->selected.conn, statsSession->selected.line, CONN_BUFSIZE);
                } else {
                    ccode = -1;
                }
            }
        } else if (ccode != -1) {
            ccode = -1;
        }
    } else {
        ccode = -1;
    }

    return(ccode);
}

static int
PrintFloat(unsigned long number, unsigned long divideBy, unsigned char *buffer)
{
    register long x;
    register long y;

    if (divideBy == 0) {
        x = 0;
        y = 0;
    } else {
        x = (number * 100) / divideBy;
        y = x % 100;
    }

    return(sprintf(buffer, "%lu.%02lu", x / 100, y));
}

static BOOL
WAStatsInitSession(SessionStruct *session, void **moduleData)
{
    WAstatsSessionStruct *statsSession = MemMalloc(sizeof(WAstatsSessionStruct));

    if (!statsSession) {
        return(FALSE);
    }

    memset(statsSession, 0, sizeof(WAstatsSessionStruct));
    statsSession->V = MDBCreateValueStruct(session->AuthHandle, NULL);

    statsSession->ssl.conn = NULL;
    statsSession->ssl.context = NULL;
    statsSession->ssl.config.options = 0;

    statsSession->selected.time = 0;
    statsSession->selected.conn = NULL;

    ConnStartup(3 * 60, TRUE);

    statsSession->ssl.enable = FALSE;
    statsSession->ssl.config.method = SSLv23_client_method;
    statsSession->ssl.config.options = SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS;
    statsSession->ssl.config.mode = SSL_MODE_AUTO_RETRY;
    statsSession->ssl.config.cipherList = NULL;
    statsSession->ssl.config.certificate.type = SSL_FILETYPE_PEM;
    statsSession->ssl.config.certificate.file = MsgGetTLSCertPath(NULL);
    statsSession->ssl.config.key.type = SSL_FILETYPE_PEM;
    statsSession->ssl.config.key.file = MsgGetTLSKeyPath(NULL);

    statsSession->ssl.context = ConnSSLContextAlloc(&(statsSession->ssl.config));
    if (statsSession->ssl.context) {
        statsSession->ssl.enable = TRUE;
    }

    *moduleData = (void *)statsSession;

    return(TRUE);
}

static BOOL
WAStatsDestroySession(SessionStruct *session, void *moduleData)
{
    WAstatsSessionStruct *statsSession = (WAstatsSessionStruct *)moduleData;

    if (statsSession) {
        MDBDestroyValueStruct(statsSession->V);

        ReleaseMessagingServer(statsSession);

        MemFree(statsSession);
    }

    return(TRUE);
}

static BOOL
WAStatsHandleTemplate(ConnectionStruct *client, SessionStruct *session, unsigned long page, TokenOverlayStruct *token, unsigned long *gotoToken, void *moduleData)
{
    WAstatsSessionStruct *statsSession = (WAstatsSessionStruct *)moduleData;
    unsigned char Buffer[512];

    switch(token->TokenID) {
        case T_GET_STATS: {
            if (XplStrCaseCmp(session->CurrentClass, MSGSRV_C_SERVER) == 0 &&
                ConnectToMessagingServer(session->CurrentObject, session, statsSession) != -1)
            {
                LoadStatistics(session, statsSession);
                LoadSpamStatistics(session, statsSession);
            }

            break;
        }

        case T_STAT_VALUE: {
            unsigned long seconds = statsSession->server.current.Uptime;

            switch (token->ArgumentOffsetOrID[0]) {
                case AA_MSGS_QUEUE_LOCAL: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->server.current.MsgQueuedLocal));

                    break;
                }

                case AA_MSGS_QUEUE_REMOTE: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->server.current.MsgQueuedRemote));

                    break;
                }

                case AA_MSGS_RECEIVED_LOCAL: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->server.current.MsgReceivedLocal));

                    break;
                }

                case AA_MSGS_RECEIVED_REMOTE: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->server.current.MsgReceivedRemote));

                    break;
                }

                case AA_MSGS_RECEIVED_PER_S: {
                    WASendClient(client, Buffer, PrintFloat(
                        statsSession->server.current.MsgReceivedLocal +
                        statsSession->server.current.MsgReceivedRemote,
                        seconds, Buffer
                    ));

                    break;
                }

                case AA_MSGS_DELIVERED_LOCAL: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->server.current.MsgStoredLocal));

                    break;
                }

                case AA_MSGS_DELIVERED_REMOTE: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->server.current.MsgStoredRemote));

                    break;
                }

                case AA_MSGS_DELIVERED_PER_S: {
                    WASendClient(client, Buffer, PrintFloat(
                        statsSession->server.current.MsgStoredLocal +
                        statsSession->server.current.MsgStoredRemote,
                        seconds, Buffer
                    ));

                    break;
                }

                case AA_MSGS_FAILED_LOCAL: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->server.current.DeliveryFailedLocal));

                    break;
                }

                case AA_MSGS_FAILED_REMOTE: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->server.current.DeliveryFailedRemote));

                    break;
                }

                case AA_MSGS_FAILED_PER_S: {
                    WASendClient(client, Buffer, PrintFloat(
                        statsSession->server.current.DeliveryFailedLocal +
                        statsSession->server.current.DeliveryFailedRemote,
                        seconds, Buffer
                    ));

                    break;
                }

                case AA_RECIPIENTS_LOCAL: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->server.current.RcptStoredLocal));

                    break;
                }

                case AA_RECIPIENTS_REMOTE: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->server.current.RcptStoredRemote));

                    break;
                }

                case AA_RECIPIENTS_PER_S: {
                    WASendClient(client, Buffer, PrintFloat(
                        statsSession->server.current.RcptStoredLocal +
                        statsSession->server.current.RcptStoredRemote,
                        seconds, Buffer
                    ));

                    break;
                }

                case AA_KB_RECEIVED_LOCAL: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->server.current.ByteReceivedLocal));

                    break;
                }

                case AA_KB_RECEIVED_REMOTE: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->server.current.ByteReceivedRemote));

                    break;
                }

                case AA_KB_RECEIVED_PER_S: {
                    WASendClient(client, Buffer, PrintFloat(
                        statsSession->server.current.ByteReceivedLocal +
                        statsSession->server.current.ByteReceivedRemote,
                        seconds, Buffer
                    ));

                    break;
                }

                case AA_KB_RECEIVED_AVERAGE_LOCAL: {
                    if (statsSession->server.current.ByteReceivedCount > 0) {
                        WASendClient(client, Buffer, sprintf(Buffer, "%lu",
                            statsSession->server.current.ByteReceivedLocal /
                            statsSession->server.current.ByteReceivedCount
                        ));
                    } else {
                        WASendClient(client, "0", 1);
                    }

                    break;
                }

                case AA_KB_RECEIVED_AVERAGE_REMOTE: {
                    if (statsSession->server.current.ByteReceivedCount > 0) {
                        WASendClient(client, Buffer, sprintf(Buffer, "%lu",
                            statsSession->server.current.ByteReceivedRemote /
                            statsSession->server.current.ByteReceivedCount
                        ));
                    } else {
                        WASendClient(client, "0", 1);
                    }

                    break;
                }

                case AA_KB_STORED_LOCAL: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->server.current.ByteStoredLocal));

                    break;
                }

                case AA_KB_STORED_REMOTE: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->server.current.ByteStoredRemote));

                    break;
                }

                case AA_KB_STORED_PER_S: {
                    WASendClient(client, Buffer, PrintFloat(
                        statsSession->server.current.ByteStoredLocal +
                        statsSession->server.current.ByteStoredRemote,
                        seconds, Buffer
                    ));

                    break;
                }

                case AA_KB_STORED_AVERAGE_LOCAL: {
                    if (statsSession->server.current.ByteStoredCount > 0) {
                        WASendClient(client, Buffer, sprintf(Buffer, "%lu",
                            statsSession->server.current.ByteStoredLocal /
                            statsSession->server.current.ByteStoredCount
                        ));
                    } else {
                        WASendClient(client, "0", 1);
                    }

                    break;
                }

                case AA_KB_STORED_AVERAGE_REMOTE: {
                    if (statsSession->server.current.ByteStoredCount > 0) {
                        WASendClient(client, Buffer, sprintf(Buffer, "%lu",
                            statsSession->server.current.ByteStoredRemote /
                            statsSession->server.current.ByteStoredCount
                        ));
                    } else {
                        WASendClient(client, "0", 1);
                    }

                    break;
                }

                case AA_CLIENT_CONNS_CURR: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->server.current.IncomingClients));

                    break;
                }

                case AA_CLIENT_CONNS_TOTAL: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->server.current.ServicedClients));

                    break;
                }

                case AA_CLIENT_CONNS_PER_S: {
                    WASendClient(client, Buffer, PrintFloat(
                        statsSession->server.current.IncomingClients +
                        statsSession->server.current.ServicedClients,
                        seconds, Buffer
                    ));

                    break;
                }

                case AA_SERVER_CONNS_CURR: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->server.current.IncomingServers));

                    break;
                }

                case AA_SERVER_CONNS_TOTAL: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->server.current.ServicedServers));

                    break;
                }

                case AA_SERVER_CONNS_PER_S: {
                    WASendClient(client, Buffer, PrintFloat(
                        statsSession->server.current.IncomingServers +
                        statsSession->server.current.ServicedServers,
                        seconds, Buffer
                    ));

                    break;
                }

                case AA_OUTG_CONNS_CURR: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->server.current.OutgoingServers));

                    break;
                }

                case AA_OUTG_CONNS_TOTAL: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->server.current.ClosedOutServers));

                    break;
                }

                case AA_OUTG_CONNS_PER_S: {
                    WASendClient(client, Buffer, PrintFloat(
                        statsSession->server.current.OutgoingServers +
                        statsSession->server.current.ClosedOutServers,
                        seconds, Buffer
                    ));

                    break;
                }

                case AA_NMAP_CONNS_CURR: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->server.current.CurrentStoreAgents));

                    break;
                }

                case AA_NMAP_CONNS_TOTAL: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->server.current.ServicedStoreAgents));

                    break;
                }

                case AA_NMAP_CONNS_PER_S: {
                    WASendClient(client, Buffer, PrintFloat(
                        statsSession->server.current.CurrentStoreAgents +
                        statsSession->server.current.ServicedStoreAgents,
                        seconds, Buffer
                    ));

                    break;
                }

                case AA_NMAP_TO_NMAP_CONNS_CURR: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->server.current.CurrentNMAPtoNMAPAgents));

                    break;
                }

                case AA_NMAP_TO_NMAP_CONNS_TOTAL: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->server.current.ServicedNMAPtoNMAPAgents));

                    break;
                }

                case AA_NMAP_TO_NMAP_CONNS_PER_S: {
                    WASendClient(client, Buffer, PrintFloat(
                        statsSession->server.current.CurrentNMAPtoNMAPAgents +
                        statsSession->server.current.ServicedNMAPtoNMAPAgents,
                        seconds, Buffer
                    ));

                    break;
                }


                case AA_WRONG_PASSWORDS: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->server.current.WrongPassword));

                    break;
                }

                case AA_UNAUTHORIZED_ATTEMPTS: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->server.current.UnauthorizedAccess));

                    break;
                }

                case AA_UPTIME: {
                    long d = seconds / 86400;
                    long a = d * 86400;
                    long h = (seconds - a) / 3600;
                    long b = h * 3600;
                    long m = (seconds - (a + b)) / 60;
                    long s = seconds - (a + b + m * 60);

                    WASendClient(client, Buffer, sprintf(Buffer, "%3lud %02luh %02lum %02lus", d, h, m, s));

                    break;
                }


                case AA_BOUNCES_REFUSED: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->spam.current.QueueSpamBlocked));

                    break;
                }

                case AA_ACCESS_FROM_BLOCKED_ADDR: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->spam.current.AddressBlocked));

                    break;
                }

                case AA_ACCESS_BLOCKED_BY_RBL: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->spam.current.MAPSBlocked));

                    break;
                }

                case AA_REMOTE_ROUTING_ATTEMPT: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->spam.current.DeniedRouting));

                    break;
                }

                case AA_ACCESS_BLOCKED_BY_NO_DNS: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->spam.current.NoDNSEntry));

                    break;
                }

                case AA_MESSAGES_SCANNED: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->spam.current.MessagesScanned));

                    break;
                }

                case AA_ATTACHMENTS_SCANNED: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->spam.current.AttachmentsScanned));

                    break;
                }

                case AA_MESSAGES_WITH_ATTACH_BLOCKED: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->spam.current.AttachmentsBlocked));

                    break;
                }

                case AA_MESSAGES_WITH_VIRUSES_FOUND: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->spam.current.VirusesFound));

                    break;
                }

                case AA_MESSAGES_WITH_VIRUSES_BLOCKED: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->spam.current.VirusesBlocked));

                    break;
                }

                case AA_MESSAGES_WITH_VIRUSES_CURED: {
                    WASendClient(client, Buffer, sprintf(Buffer, "%lu", statsSession->spam.current.VirusesCured));

                    break;
                }
            }
            break;
        }

        default: {
            return(FALSE);
        }
    }

    return(TRUE);
}

static BOOL
WAStatsHandleURL(ConnectionStruct *client, SessionStruct *session, URLStruct *url, void *moduleData)
{
    return(FALSE);
}


/************************************************************************
    Now comes the "stock" stuff any module needs
*************************************************************************/

EXPORT BOOL
LIBWASTATSInit(WAAPIArg)
{
    ModuleRegisterStruct reg;

    WAStats.unload = FALSE;

    WAAPISet;

    /* This lets us access the msg api functions */
    MsgInit();

    /* Register user handler */
    memset(&reg, 0, sizeof(ModuleRegisterStruct));

    reg.ModuleType = MODULE_TEMPLATE;
    reg.Module.Template.InitSession = WAStatsInitSession;
    reg.Module.Template.DestroySession = WAStatsDestroySession;
    reg.Module.Template.HandleURL = WAStatsHandleURL;
    reg.Module.Template.HandleTemplate = WAStatsHandleTemplate;
    reg.Module.Template.TokenRangeStart = WASTATS_TOKEN_START;
    reg.Module.Template.TokenRangeEnd = WASTATS_TOKEN_END;

    WARegisterModule(&reg);

    return(TRUE);
}

EXPORT BOOL
LIBWASTATSShutdown(void)
{
    MsgShutdown();
    WAStats.unload = TRUE;
    return(TRUE);
}

/************************************************************************
    Below are all the things that make loading and unloading possible :-)
*************************************************************************/

#if defined(NETWARE) || defined(LIBC)
int
RequestUnload(void)
{
    if (!WAStats.unload) {
        int OldTGid;

        OldTGid = XplSetThreadGroupID(WAStats.id.group);

        XplConsolePrintf("\rThis NLM will automatically be unloaded by the thread that loaded it.\n");
        XplConsolePrintf("\rIt does not allow manual unloading.\n");
        SetCurrentScreen(CreateScreen("System Console", 0));

        XplUngetCh('n');

        XplSetThreadGroupID(OldTGid);

        return(1);
    }

    return(0);
}

void
WAStatsSigHandler(int Signal)
{
    int OldTGid;

    OldTGid = XplSetThreadGroupID(WAStatsObject.id.group);

    XplSignalLocalSemaphore(WAStandardObject.sem.shutdown);    /* The signal will release main() */
    XplWaitOnLocalSemaphore(WAStandardObject.sem.shutdown);    /* The wait will wait until main() is gone */

    /* Do any required cleanup */

    XplCloseLocalSemaphore(WAStats.sem.shutdown);
    XplSetThreadGroupID(OldTGid);
}

int
main(int argc, char *argv[])
{
    WAStatOsbject.unload = TRUE;
    WAStatsObject.id.group = XplGetThreadGroupID();

    signal(SIGTERM, WAStatsSigHandler);

    MDBInit();

    XplOpenLocalSemaphore(WAStat.sem.shutdown, 0);
    XplWaitOnLocalSemaphore(WAStat.sem.shutdown);
    XplSignalLocalSemaphore(WAStat.sem.shutdown);

    return(0);
}
#endif
