/***************************************************************************
 *   Copyright (C) 2004-2006 by Jim Campbell                               *
 *   ifpgui@gmail.com                                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <fts.h>

#include <qapplication.h>
#include <qmessagebox.h>
#include <qfileinfo.h>
#include <qdir.h>
#include <qdeepcopy.h>

#include "gui_user.h"
#include "iointerface.h"
#include "ifpinterface.h"
#include "progressdlg.h"

QPtrList<FileInfos> new_ifp_file_tree;
QString iFPNotConnectedMsg = QObject::tr("Could not communicate with iRiver device.\n"
   "Try unplugging the USB link, powering off the device, and plugging it back in.\n\n"
   "After you have reconnected, double-click the root ('\\') directory in the iRiver Filesystem panel.");


//-----------------------------------------------------------
// helper functions
//-----------------------------------------------------------
int get_next_filedir_basename(char *str);

void slash2backslash(QString& str);
int ifp_get_size_recursive(QPtrList<FileInfos> *ifp_fi, int *filecnt);



//===========================================================================
// file_dir_callbk()
//   This is the ifp list callback.  This is used to build the file tree
//   which is stored in ifp_file_tree;
//
//   context  : the current node to populate in the file tree.
//   type     : the type of node (file or directory)
//   filename : the name of the node
//   filesize : the size of the file if the node is a file
//
//   Returns: 0 on success
//===========================================================================
static int file_dir_callbk(void * context, int type, const char * filename, int filesize)
{
   QPtrList<FileInfos> *list;

   list = (QPtrList<FileInfos> *)context;

   if (list == NULL)
      return(-1);

   list->prepend(new FileInfos(QString::fromUtf8(filename, -1), type, filesize, NULL));
   return 0;
}

//===========================================================================
// ifp_get_file_list()
//   This function retrieves all files in the directory specified by
//   @filename and fills in the @file_info_ptr parameter with the infos
//   about every entry.  Also, it goes through the list and gets the info
//   (files and directories) for each subdirectiry.
//
//   dev            : the ifp_device structure
//   filename       : the name of the directory to get the list of
//   file_finfo_ptr : the address of the structure to place the info into
//
//   Returns: IFP_OK if successful,
//            IFP_DIRECTORY_NOT_FOUND if filename does not exist
//            IFP_LOCAL_FILE_OPEN_ERROR if dir list fails
//            ENOMEM on memory error
//===========================================================================
int ifp_get_file_list(struct ifp_device * dev, QString& filename, QPtrList<FileInfos> *list)
   {
   int ret;
   QString fullname, tmp_str;
   //char fullname[IFP_MAXPATHLEN];
   FileInfos *fi_temp;

   slash2backslash(filename);

   // check if directory exists
   if (ifp_exists(dev, filename) != IFP_DIR)
      {
      return (-ENOENT);
      }

   ret = ifp_list_dirs(dev, filename, file_dir_callbk, list);
   if (ret)
      {
      ifp_err_i(ret, "couldn't get basic directory listing.");
      return IFP_ERR_DEV_FUBAR;
      }

   // check if directory is empty
   if (list->isEmpty())
      {
      return(IFP_OK);
      }

   // now look through results and get the subdirectories
   for (fi_temp = list->first(); fi_temp; fi_temp = list->next())
      {
      if (fi_temp->FileType() == IO_INTF_DIR)
         {
         fullname = filename;
         if (!fullname.endsWith("\\"))
            fullname.append("\\");
         fullname.append(fi_temp->FileName());
         fullname.truncate(IFP_MAXPATHLEN);
         // get the subdirectiry info
         fi_temp->sub_file_tree = new QPtrList<FileInfos>;
         ret = ifp_get_file_list(dev, fullname, fi_temp->sub_file_tree);
         if (ret != IFP_OK)
            return(ret);
         }
      }
   return(IFP_OK);
   }

//===========================================================================
// ifp_read_file_tree()
//   This function retrieves all files on the iFP device.  It stores them
//   in the ifp_file_tree structure.
//
//   Returns: IFP_OK if successful,
//            IFP_DIRECTORY_NOT_FOUND if filename does not exist
//            IFP_LOCAL_FILE_OPEN_ERROR if dir list fails
//            ENOMEM on memory error
//===========================================================================
int ifp_read_file_tree()
{
   int ret;
   QString start_dir;
   FileInfos *fi_temp;

   new_ifp_file_tree.clear();

   // the list owns the objects
   new_ifp_file_tree.setAutoDelete(TRUE);

   // Add the first element "/"
   fi_temp = new FileInfos("/", IO_INTF_DIR, 0, NULL);
   new_ifp_file_tree.append(fi_temp);
   fi_temp->sub_file_tree = new QPtrList<FileInfos>;
   start_dir = "/";
   ret = ifp_get_file_list(&ifp_dev, start_dir, fi_temp->sub_file_tree);

   return (ret);
}

//===========================================================================
// get_next_filedir_basename()
//   This function returns the first position of a dir seperator char.
//===========================================================================
int get_next_filedir_basename(QString str)
   {
   int ret;

   ret = str.find("\\", 0);
   if (ret == -1)
      ret = str.find("/", 0);
   return(ret);
   }

//===========================================================================
// ifp_get_info()
//   This function gets the FileInfos for the given @filename.
//
//   Returns: IFP_OK on success
//            -ENOENT for filename not found
//===========================================================================
int ifp_get_info(QString filename, FileInfos **finfo)
   {
   QString tmp_name, scratch_name, split_str;
   FileInfos *fi_temp;
   QPtrList<FileInfos> *list;

   if ((filename == "\\") || (filename == "/"))
      {
      *finfo = new_ifp_file_tree.first();
      return(IFP_OK);
      }
   if (new_ifp_file_tree.isEmpty())
      return(-ENOENT);

   // create the path list
   split_str = filename.left(1);
   QStringList lst(QStringList::split(split_str, filename));
   // start with the node after the root node (\)
   fi_temp = new_ifp_file_tree.first();

   QStringList::Iterator it2 = lst.begin();
   for ( ; it2 != lst.end(); ++it2)
      {
      list = fi_temp->sub_file_tree;
      if (list == NULL)
         return(-ENOENT);
      if (list->isEmpty())
         return(-ENOENT);

      // get the name of the node to find
      tmp_name = *it2;
      // find the node in the list
      *finfo = NULL;
      for (fi_temp = list->first(); fi_temp; fi_temp = list->next())
         {
         if (fi_temp->FileName() == tmp_name)
            {
            *finfo = fi_temp;
            break;
            }
         }
      if (*finfo == NULL)
         return(-ENOENT);
      }
   if (*finfo == NULL)
      return(-ENOENT);
   return(IFP_OK);
   }

//===========================================================================
// ifp_get_size_recursive()
//   This function gets size of the directory @ifp_fi recursively.
//
//   Returns: size on success
//===========================================================================
int ifp_get_size_recursive(FileInfos *finfo, int *filecnt)
   {
   int size;
   QPtrList<FileInfos> *list;
   FileInfos *fi_temp;

   size = 0;
   list = finfo->sub_file_tree;
   for (fi_temp = list->first(); fi_temp; fi_temp = list->next())
      {
      if (fi_temp->FileType() == IFP_FILE)
         {
         size += fi_temp->FileSize();
         (*filecnt) += 1;
         }
      else if (fi_temp->FileType() == IFP_DIR)
         {
         size += ifp_get_size_recursive(fi_temp, filecnt);
         }
      }
   return(size);
   }

//===========================================================================
// ifp_get_dir_size()
//   This function gets size of the directory @filename.  All its files
//   and subdirectories with their files.
//
//   Returns: filesize on success
//            -ENOENT for filename not found
//===========================================================================
int ifp_get_dir_size(QString filename, int *filecnt)
   {
   int ret;
   FileInfos *fi_temp;

   fi_temp = NULL;
   ret = ifp_get_info(filename, &fi_temp);
   if (ret != IFP_OK)
      return(ret);

   if (fi_temp == NULL)
      return(-ENOENT);

   if (fi_temp->FileType() == IFP_FILE)
      return(-ENOENT);

   // get the size
   return(ifp_get_size_recursive(fi_temp, filecnt));
   }

//===========================================================================
// local_get_dir_size()
//   This function gets size of the directory @filename.  All its files
//   and subdirectories with their files.
//
//   Returns: filesize on success
//            -ENOENT for filename not found
//===========================================================================
int local_get_dir_size(QString dirName, int *filecnt)
   {
   int filesize, tmpcnt;

   QDir dir(dirName, "", QDir::Name | QDir::IgnoreCase, QDir::All);
   if (!dir.exists())
      return(0);

   const QFileInfoList *list = dir.entryInfoList();
   QFileInfoListIterator it( *list );
   QFileInfo *finfo;

   filesize = 0;
   while ( (finfo = it.current()) != 0 )
      {
      if (finfo->isFile())
         {
         filesize += finfo->size();
         *filecnt += 1;
         }
      else
         {
         if ((finfo->fileName() != ".") && (finfo->fileName() != ".."))
            {
            tmpcnt = 0;
            filesize += local_get_dir_size(finfo->filePath(), &tmpcnt);
            *filecnt += tmpcnt;
            }
         }
      ++it;
      }
   return(filesize);
   }


int iFPgetFileList(const QString& filename, QPtrList<FileInfos> *list)
   {
   int ret;
   FileInfos *finfo, *fi_temp;
   QPtrList<FileInfos> *tmp_list;

   finfo = NULL;
   ret = ifp_get_info(filename, &finfo);
   if (ret != IFP_OK)
      return(ret);
   if (finfo == NULL)
      return(-1);

   tmp_list = finfo->sub_file_tree;

   if (tmp_list == NULL)
      return(IFP_OK);

   for (fi_temp = tmp_list->first(); fi_temp; fi_temp = tmp_list->next())
      {
      list->append(new FileInfos(fi_temp->FileName(),
                                 fi_temp->FileType(),
                                 fi_temp->FileSize(),
                                 NULL));
      }
   return(ret);
   }

/**
 * Download the file sfilename from the iFP device to the local directory destDir
 * @param sfilename the source filename on the iFP device.  Must be / delimited.
 * @param destDir the local directory; the place to save the downloaded file
 * @param fn_context the current context of the transfer.  User defined structure
 * usually used to keep track of the progress of the download.
 * @return IFP_OK on success (0).  Otherwise return an error code.
 */
int iFPdownloadFile(const QString& sfilename, const QString& destDir, void* fn_context)
   {
   int ret, pos;
   QCString sbuffer, dbuffer;
   QString Message, destination, baseName, err, ifpSourceName;

   if (ifp_dh == NULL)
      return(-1);

   baseName = sfilename;
   pos = baseName.findRev('/');
   baseName.remove(0, pos + 1);
   destination = destDir + baseName;

   ifpSourceName = sfilename;
   slash2backslash(ifpSourceName);

   sbuffer = ifpSourceName.utf8();
   dbuffer = destination.utf8();
   ret = ifpgui_ifp_download_file(&ifp_dev, sbuffer, dbuffer, progress, fn_context);

   return(ret);
   }

/**
 * Download the directory sdirname from the iFP device to the local directory destDir
 * @param sdirname the source directory name on the iFP device.  Must be / delimited.
 * @param destDir the local directory; the place to save the downloaded directory
 * @param fn_context the current context of the transfer.  User defined structure
 * usually used to keep track of the progress of the download.
 * @return IFP_OK on success (0).  Otherwise return an error code.
 */
int iFPdownloadDirectory(const QString& sdirname, const QString& destDir, void* fn_context)
   {
   int ret, pos;
   QCString sbuffer, dbuffer;
   QString Message, destination, baseName, err, ifpSourceName;
   ifp_progress_info pginfo;

   ret = -1;
   if (ifp_dh == NULL)
      return(ret);

   baseName = sdirname;
   pos = baseName.findRev('/');
   baseName.remove(0, pos + 1);
   destination = destDir + baseName;

   ifpSourceName = sdirname;
   slash2backslash(ifpSourceName);

   sbuffer = ifpSourceName.utf8();
   dbuffer = destination.utf8();
   ret = ifpgui_ifp_download_dir(&ifp_dev, sbuffer, dbuffer, progress, fn_context);

   return(ret);
   }
/**
 * Upload the file sfilename from the iFP device to the iRiver directory destDir
 * @param sfilename the source filename on the local harddrive.  Must be / delimited.
 * @param destDir the iRiver directory; the place to put the uploaded file
 * @param fn_context the current context of the transfer.  User defined structure
 * usually used to keep track of the progress of the upload.
 * @return IFP_OK on success (0).  Otherwise return an error code.
 */
int iFPuploadFile(const QString& sfilename, const QString& destDir, void* fn_context)
   {
   int ret, pos;
   QCString sbuffer, dbuffer;
   QString Message, destination, baseName, err, ifpDestName;

   if (ifp_dh == NULL)
      return(-1);

   baseName = sfilename;
   pos = baseName.findRev('/');
   baseName.remove(0, pos + 1);
   destination = destDir + baseName;

   ifpDestName = destination;
   slash2backslash(ifpDestName);

   sbuffer = sfilename.utf8();
   dbuffer = ifpDestName.utf8();
   ret = ifpgui_ifp_upload_file(&ifp_dev, sbuffer, dbuffer, progress, fn_context);

   return(ret);
   }

/**
 * Upload the directory sdirname to the iRiver device directory destDir
 * @param sdirname the source directory name on the local harddrive.  Must be / delimited.
 * @param destDir the iRiver directory; the place to put the uploaded directory
 * @param fn_context the current context of the transfer.  User defined structure
 * usually used to keep track of the progress of the upload.
 * @return IFP_OK on success (0).  Otherwise return an error code.
 */
int iFPuploadDirectory(const QString& sdirname, const QString& destDir, void* fn_context)
   {
   int ret, pos;
   QCString sbuffer, dbuffer;
   QString Message, destination, baseName, err, ifpDestName;
   ifp_progress_info pginfo;

   ret = -1;
   if (ifp_dh == NULL)
      return(ret);

   baseName = sdirname;
   pos = baseName.findRev('/');
   baseName.remove(0, pos + 1);
   destination = destDir + baseName;

   ifpDestName = destination;
   slash2backslash(ifpDestName);

   sbuffer = sdirname.utf8();
   dbuffer = ifpDestName.utf8();
   ret = ifpgui_ifp_upload_dir(&ifp_dev, sbuffer, dbuffer, progress, fn_context);

   return(ret);
   }

int showYesNoAllMsgBox(const QString& msgH, const QString& msgB)
   {
   int mbret;

   QMessageBox mb(msgH, msgB,
                  QMessageBox::Question,
                  QMessageBox::No  | QMessageBox::Escape,
                  QMessageBox::Yes | QMessageBox::Default,
                  QMessageBox::YesAll);
   mbret = mb.exec();
   return(mbret);
   }

void showInfoMsgBox(const QString& msgH, const QString& msgB)
   {
   QMessageBox mb(msgH, msgB,
                  QMessageBox::Information,
                  QMessageBox::Ok, 0, 0);
   mb.exec();
   return;
   }

void backslash2slash(QString& str)
   {
   str.replace('\\', '/');
   }

void slash2backslash(QString& str)
   {
   str.replace('/', '\\');
   }
