/*
 *  IHU -- I Hear U, easy VoIP application using Speex and Qt
 *
 *  Copyright (C) 2003-2006 Matteo Trotta - <mrotta@users.sourceforge.net>
 *
 *  http://ihu.sourceforge.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 */

#include <stdlib.h>
#include <time.h>

#include <qstring.h>
 
#include "Error.h"
#include "IhuNoGui.hpp"

IhuNoGui::IhuNoGui() : config(Config::instance())
{
}

IhuNoGui::~IhuNoGui()
{
	delete rsa;
	delete player;
	delete receiver;
	delete recorder;
	delete transmitter;
	delete logger;
}

void IhuNoGui::initIhu()
{
	try
	{
		fromFile = false;
		listening = false;
		muteMic = false;
		muteSpk = false;
		crypt = false;
		
		rsa = new Rsa(RSA_STRENGTH);
		player = new Player();
		receiver = new Receiver(player, rsa);
		recorder = new Recorder();
		transmitter = new Transmitter(recorder, rsa);
		logger = new Logger();
	
		connect( receiver, SIGNAL(newSocket(int,int,struct sockaddr_in)), this, SLOT(newConnection(int,int,struct sockaddr_in)) );
		connect( receiver, SIGNAL(keyRequest()), this, SLOT(sendKeyRequest()) );
		connect( receiver, SIGNAL(sendNewKey()), this, SLOT(sendKey()) );
		connect( receiver, SIGNAL(newKey(QString)), this, SLOT(receivedNewKey(QString)) );
		connect( receiver, SIGNAL(finish()), this, SLOT(stopSignal()) );
		connect( receiver, SIGNAL(warning(QString)), this, SLOT(warning(QString)) );
		connect( receiver, SIGNAL(error(QString)), this, SLOT(abortAll(QString)) );
		connect( receiver, SIGNAL(ringReply()), transmitter, SLOT(sendRingReplyPacket()) );
		connect( transmitter, SIGNAL(ringMessage()), this, SLOT(ringMessage()) );
		connect( transmitter, SIGNAL(finish()), this, SLOT(stopSignal()) );
		connect( transmitter, SIGNAL(error(QString)), this, SLOT(abortAll(QString)) );
		connect( transmitter, SIGNAL(message(QString)), this, SLOT(message(QString)) );
		connect( player, SIGNAL(warning(QString)), this, SLOT(warning(QString)) );	
		connect( recorder, SIGNAL(warning(QString)), this, SLOT(warning(QString)) );	
		
		receiver->resetCalls();
		
		applySettings();
	}
	catch (Error e)
	{
		abortAll(e.getText());
	}
}

void IhuNoGui::message(QString text)
{
	warning(text);
}

void IhuNoGui::warning(QString text)
{
	text = logger->log(text);
	if (!text.isEmpty())
		qWarning(text);
}

void IhuNoGui::abortAll(QString text)
{
	qWarning("Error: " + text);
	exit(1);
}

void IhuNoGui::waitForCalls()
{
	try
	{
		int inport = config.readNumEntry("/ihu/net/inport");
		warning(QString("Waiting for calls on port %1...").arg(inport));
		listening = true;
		receiver->listen(inport, (bool) config.readBoolEntry("/ihu/net/udp"), (bool) config.readBoolEntry("/ihu/net/tcp"));
	}
	catch (Error e)
	{
		abortAll(e.getText());
	}
}

void IhuNoGui::agcRefresh(bool on)
{
	float step = (float) (config.readNumEntry("/ihu/sound/agcstep")/1000.f);
	float level = (float) (config.readNumEntry("/ihu/sound/agclevel")/100.f);
	float min = (float) config.readNumEntry("/ihu/sound/agcmin");
	float max = (float) config.readNumEntry("/ihu/sound/agcmax");
	player->setAgc(on, step, level, min, max);
	config.writeEntry("/ihu/sound/agc", on);
}

void IhuNoGui::adrRefresh(bool on)
{
	float mindelay = (float) (config.readNumEntry("/ihu/sound/adrmindelay")/1000.f);
	float maxdelay = (float) (config.readNumEntry("/ihu/sound/adrmaxdelay")/1000.f);
	float stretch = (float) config.readNumEntry("/ihu/sound/adrstretch");
	player->setAdr(on, mindelay, maxdelay, stretch);
	config.writeEntry("/ihu/sound/adr", on);
}

void IhuNoGui::newConnection(int sd, int protocol, struct sockaddr_in sa)
{
	try
	{
		if (listening)
		{
			transmitter->newConnection(sd, sa, protocol);
			transmitter->answer();
			warning(QString("Connected with %1 (%2).").arg(receiver->getCallerName()).arg(receiver->getIp()));
		}
		else
			ringOn(false);
	}
	catch (Error e)
	{
		abortAll(e.getText());
	}
}

void IhuNoGui::stopSignal()
{
	if (fromFile)
	{
		message(tr("End of file"));
	}
	else
	{
		QString text = QString("%1 ").arg(receiver->getCallerName());
		if (receiver->refused())
			text += tr("refused the call.");
		else
		if (receiver->getTotal() > 0)
			text += tr("closed the call.");
		else
			text = QString("%1 rejected the call.").arg(receiver->getIp());
		warning(text);
	}
	stopAll();
}

void IhuNoGui::stopAll()
{
	try
	{
		ringOn(false);
		if (!fromFile)
			qWarning(logger->logStop(receiver->getTotal(), transmitter->getTotal()));
		fromFile = false;
		transmitter->end();
		receiver->end();
		if (listening)
			waitForCalls();
		else
			exit(0);
	}
	catch (Error e)
	{
		abortAll(e.getText());
	}
}

void IhuNoGui::playFile(QString name)
{
	if (!name.isEmpty())
	{
		try
		{
			fromFile = true;
			warning("Playing " + name + "...");
			receiver->playFile(name);
		}
		catch (Error e)
		{
			abortAll(e.getText());
		}
	}
}

void IhuNoGui::call(QString host)
{
	try {
		if (host.isEmpty())
			throw Error(tr("No host specified!"));
		int callPort = config.readNumEntry("/ihu/net/outport");
		QString callHost = host;
		int tmpInd = host.findRev(':');
		if (tmpInd > 0)
		{
			callHost = host.left(tmpInd);
			callPort = host.right(host.length() - tmpInd - 1).toInt();
		}
		int sd = transmitter->call(callHost, callPort, config.readNumEntry("/ihu/net/protocol"));
		qWarning(logger->logOutgoingCall(host, transmitter->getIp()));
		receiver->start(sd, config.readNumEntry("/ihu/net/protocol"));
		ringOn(true);
		receiver->waitConnection();
	}
	catch (Error e)
	{
		abortAll(e.getText());
	}
}

void IhuNoGui::cryptOn()
{
	try
	{
		srand(time(NULL));
		changeKey();
		crypt = true;
		warning("Encryption enabled.");
	}
	catch (Error e)
	{
		abortAll(e.getText());
	}
}

void IhuNoGui::disableIn()
{
	transmitter->changeStatus(Transmitter::TRANSMITTER_STATUS_MUTE);
	muteMic = true;
}

void IhuNoGui::disableOut()
{
	receiver->changeStatus(Receiver::RECEIVER_STATUS_MUTE);
	muteSpk = true;
}

void IhuNoGui::changeKey()
{
	int i, len = config.readNumEntry("/ihu/security/keylen")/8;
	char key[len];
	for (i=0; i<len; i++)
		key[i] = (char)((rand()%256)-128);
	transmitter->enableCrypt(key, len);
}

void IhuNoGui::sendKey()
{
	transmitter->sendKeyPacket();
}

void IhuNoGui::sendKeyRequest()
{
	if (transmitter->isWorking())
		transmitter->sendKeyRequestPacket();
	else
		abortAll("stream is crypted but the decryption key is not available.");
}

void IhuNoGui::applySettings()
{
	int srate, quality, vbr, abr, complexity, vad, dtx, txstop;
	float vbrquality;

	transmitter->setName(config.readEntry("/ihu/general/name"));
	
	switch (config.readNumEntry("/ihu/speex/mode"))
	{
		case 0:
				srate = 8000;
				break;
		case 1:
				srate = 16000;
				break;
		case 2:
				srate = 32000;
				break;
		default:
				srate = 8000;
				break;
	}
	quality = config.readNumEntry("/ihu/speex/quality");
	abr = config.readNumEntry("/ihu/speex/abr")*1000;
	vbr = config.readNumEntry("/ihu/speex/bitratemode");
	switch (vbr)
	{
		case 0:
				abr = 0;
				break;
		case 1:
				abr = 0;
				break;
		case 2:
				vbr = 0;
				break;
	}
	vbrquality = (float) config.readNumEntry("/ihu/speex/vbrquality");
	complexity = config.readNumEntry("/ihu/speex/complexity");
	vad = config.readBoolEntry("/ihu/speex/vad");
	dtx = config.readBoolEntry("/ihu/speex/dtx");
	txstop = config.readNumEntry("/ihu/sound/txstop");
	
	int th = 0;
	if (dtx)
		th = config.readNumEntry("/ihu/sound/threshold");
	transmitter->setThreshold(th);
	transmitter->setup(srate, quality, abr, vbr, vbrquality, complexity, vad, dtx, txstop);

	recorder->setup(config.readNumEntry("/ihu/sound/inputdriver"), config.readEntry("/ihu/sound/inputinterface"));
	player->setup(config.readNumEntry("/ihu/sound/outputdriver"), config.readEntry("/ihu/sound/outputinterface"), 
			config.readNumEntry("/ihu/sound/ringvolume"), config.readNumEntry("/ihu/sound/prepackets"));
	
	adrRefresh(config.readBoolEntry("/ihu/sound/adr"));
	agcRefresh(config.readBoolEntry("/ihu/sound/agc"));

	QString nname;

	try
	{
		nname = QString("/ihu/security/logfile");
		logger->enable(config.readEntry(nname));
		nname = QString("/ihu/general/dumpout");
		transmitter->dump(config.readEntry(nname));
		nname = QString("/ihu/general/dumpin");
		receiver->dump(config.readEntry(nname));
	}
	catch (Error e)
	{
		warning(e.getText());
		config.writeEntry(nname, "");
	}

}

void IhuNoGui::ringOn(bool on)
{
	transmitter->ring(on);
}

void IhuNoGui::ringMessage()
{
	QString text = QString("Ringing %1").arg(transmitter->getIp());
	if (receiver->replied())
		text += QString(" (%1)").arg(receiver->getCallerName());
	else
		text += QString(" (waiting for reply...)");
	warning(text);
}

void IhuNoGui::receivedNewKey(QString text)
{
	if (config.readBoolEntry("/ihu/security/showkey"))
	{
		warning(QString("New decryption key: %1").arg(text));
	}
	if (!crypt)
		cryptOn();	
}
