/*
Copyright 1990-2001 Sun Microsystems, Inc. All Rights Reserved.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions: The above copyright notice and this
permission notice shall be included in all copies or substantial
portions of the Software.


THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE OPEN GROUP OR SUN MICROSYSTEMS, INC. BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH
THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE EVEN IF
ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH DAMAGES.


Except as contained in this notice, the names of The Open Group and/or
Sun Microsystems, Inc. shall not be used in advertising or otherwise to
promote the sale, use or other dealings in this Software without prior
written authorization from The Open Group and/or Sun Microsystems,
Inc., as applicable.


X Window System is a trademark of The Open Group

OSF/1, OSF/Motif and Motif are registered trademarks, and OSF, the OSF
logo, LBX, X Window System, and Xinerama are trademarks of the Open
Group. All other trademarks and registered trademarks mentioned herein
are the property of their respective owners. No right, title or
interest in or to any trademark, service mark, logo or trade name of
Sun Microsystems, Inc. or its licensors is granted.

*/


package sun.awt.im.iiimp;

import java.awt.*;
import java.awt.event.*;
import java.awt.geom.*;
import java.awt.font.*;
import java.text.*;
import com.sun.iiim.*;

class LookupPane extends Panel {

    // START information
    private int master;
    private int choicePerWindow;
    private Dimension rowColumn;
    private int direction;
    private int labelOwner;

    // DRAW information
    private int firstCandidate;
    private int lastCandidate;
    private int currentCandidate;
    private String[] candidates;
    private String[] labelList;
    private int processType;
    private int indexPage;
    private String title;

    // Label cache
    private FocusLabel[] awtLabels;
    private int cacheSize = 32;
    private static final int CANDIDATE_GAP = 5;
    private int oldCandidate;
    private String[] oldCandidates;

    public LookupPane() {
	awtLabels = new FocusLabel[cacheSize];
	for (int i = 0; i < awtLabels.length; i++) {
	    awtLabels[i] = new FocusLabel();
	}

        enableInputMethods(false);
    }

    private void checkCache(int size) {
	if (size > cacheSize) {
	    FocusLabel[] tmp = new FocusLabel[size];
	    System.arraycopy(awtLabels, 0, tmp, 0, cacheSize);
	    for (int i = cacheSize; i < size; i++) {
		tmp[i] = new FocusLabel();
	    }
	    awtLabels = tmp;
	    cacheSize = size;
	}
    }

    private boolean same(String[] sa1, String[] sa2) {
	if (sa1 == null || sa2 == null || sa1.length != sa2.length) {
	    return false;
	}

	for (int i = 0; i < sa1.length; i++) {
	    if (!sa1[i].equals(sa2[i])) {
		return false;
	    }
	}
	return true;
    }

    private boolean same(CharacterIterator[] it1, CharacterIterator[] it2) {
	if (it1 == null || it2 == null || it1.length != it2.length) {
	    return false;
	}

	for (int i = 0; i < it1.length; i++) {
	    for (char c1 = it1[i].first(), c2 = it2[i].first();
		 c1 != CharacterIterator.DONE;
		 c1 = it1[i].next(), c2 = it2[i].next()) {
		if (c1 != c2) {
		    return false;
		}
	    }
	}
	return true;
    }

    void handleEvent(IIIMLookupEvent e) {
	switch(e.getType()) {
	  case IIIMLookupEvent.START:
	    start(e);
	    break;
	  case IIIMLookupEvent.DRAW:
	    if (Manager.COLOR_SUPPORT) {
		draw2(e);
	    } else {
		draw(e);
	    }
	    break;
	  case IIIMLookupEvent.PROCESS:
	    process(e);
	    break;
	  default:
	    break;
        }
    }

    private void start(IIIMLookupEvent e) {
	// For now, only SERVER_MASTER is supported
	// master = e.getMaster();
	master = IIIMLookupEvent.SERVER_MASTER;
	choicePerWindow = e.getChoicePerWindow();
	rowColumn = e.getRowColum();
	direction = e.getDirection();
	labelOwner = e.getLabelOwner();

	oldCandidates = null;
	oldCandidate = -1;
	GridLayout gl = new GridLayout(rowColumn.width, rowColumn.height);
	gl.setHgap(CANDIDATE_GAP);
	setLayout(gl);
    }

    private void draw(IIIMLookupEvent e) {
	currentCandidate = e.getCurrentCandidate();
	candidates = e.getCandidates();
	labelList = e.getLabelList();

	if (candidates.length != labelList.length) {
	    throw new IllegalArgumentException(" Lookup Error.");
	}
	
	if (master == IIIMLookupEvent.SERVER_MASTER) {
	    if (!same(candidates, oldCandidates)) {
		removeAll();
		checkCache(candidates.length);
		for (int i = 0; i < candidates.length; i++) {
		    FocusLabel fl = awtLabels[i];
		    fl.setText(labelList[i] + " : " + candidates[i]);
		    add(fl);
		    fl.setTarget(this, i);
		    if (i == currentCandidate) {
			fl.setFocus(true);
		    }
		}
		repaint();
		awtLabels[currentCandidate].requestFocus();
	    } else if (currentCandidate != oldCandidate) {
		awtLabels[currentCandidate].setFocus(true);
		awtLabels[currentCandidate].repaint();
		if (oldCandidate >= 0) { 
		    awtLabels[oldCandidate].setFocus(false);
		    awtLabels[oldCandidate].repaint();
		}
	    }
		
	} else {
	    // CLIENT_MASTER
	    firstCandidate = e.getFirstCandidate();
	    lastCandidate = e.getLastCandidate();
	    checkCache(lastCandidate - firstCandidate);
	    for (int i = firstCandidate; i < lastCandidate; i++) {
		awtLabels[i - firstCandidate].setText(labelList[i] +
						      " : " + candidates[i]);
		add(awtLabels[i - firstCandidate]);
		awtLabels[i - firstCandidate].setTarget(this, i);
	    }
	}

	oldCandidate = currentCandidate;
	oldCandidates = candidates;
    }

    private AttributedCharacterIterator[] candidateIterator;
    private AttributedCharacterIterator[] labelIterator;
    private AttributedCharacterIterator[] oldCandidateIterator;

    private void draw2(IIIMLookupEvent e) {
	candidateIterator = e.getCandidateIterator();
	labelIterator = e.getLabelIterator();
	if (candidateIterator.length != labelIterator.length ||
	    master != IIIMLookupEvent.SERVER_MASTER) {
	    throw new IllegalArgumentException(" Lookup Error.");
	}

	if (!same(candidateIterator, oldCandidateIterator)) {
	    checkCache(candidateIterator.length);
	    for (int i = 0; i < candidateIterator.length; i++) {
		FocusLabel fl = awtLabels[i];
		fl.setIterator(labelIterator[i], candidateIterator[i]);
		add(fl);
		fl.setTarget(this, i);
		if (i == currentCandidate) {
		    fl.setFocus(true);
		}
		repaint();
		awtLabels[currentCandidate].requestFocus();
	    }
	} else {
	    awtLabels[currentCandidate].setFocus(true);
	    awtLabels[currentCandidate].repaint();
	    if (oldCandidate >= 0) {
		awtLabels[oldCandidate].setFocus(false);
		awtLabels[oldCandidate].repaint();
	    }
	}
	oldCandidate = currentCandidate;
	oldCandidateIterator = candidateIterator;
    }

    private void process(IIIMLookupEvent e) {
	processType = e.getProcessType();
	indexPage = e.getProcessIndexPage();

	// currently this protocol never used....
	repaint();
    }
        
    void forwardEvent(AWTEvent e) {
	target.dispatchEvent(e);
    }

    void changeFocus(int index) {
	if (currentCandidate != index) {
	    awtLabels[currentCandidate].setFocus(false);
	    awtLabels[currentCandidate].repaint();
	    currentCandidate = index;
	}
    }

    void clicked(int index) {
	IIIMActionEvent e =
	    new IIIMActionEvent(IIIMActionEvent.LOOKUP_PROCESSED,
				awtLabels[index].getText());
	window.dispatchActionEvent(e);
    }

    private java.awt.Component target;
    void setTarget(java.awt.Component target) {
	this.target = target;
    }

    private LookupWindow window;
    void setLookupWindow(LookupWindow window) {
	this.window = window;
    }
}

class FocusLabel extends Label implements KeyListener, MouseListener {

    private boolean focused = false;
    private boolean pressed = false;
    private LookupPane target;
    private int index;
    private AttributedCharacterIterator labelIterator;
    private AttributedCharacterIterator candidateIterator;

    FocusLabel(String s) {
	super(s);
	setAlignment(LEFT);
	enableInputMethods(false);
	addMouseListener(this);
	addKeyListener(this);
    }

    FocusLabel() {
	this(null);
    }

    void setFocus(boolean b) {
        focused = b;
    }
    
    void setTarget(LookupPane target, int index) {
	this.target = target;
	this.index = index;
    }
    
    public void setText(String text) {
	super.setText(text);
	setFocus(false);
    }

    public void setIterator(AttributedCharacterIterator labelIterator,
			    AttributedCharacterIterator candidateIterator) {
	super.setText(iteratorToString(labelIterator) + " : " +
		      iteratorToString(candidateIterator));
	this.labelIterator = labelIterator;
	this.candidateIterator = candidateIterator;
    }

    public static String iteratorToString(CharacterIterator iterator) {
        StringBuffer sb = new StringBuffer();
        for (char c = iterator.first(); c != CharacterIterator.DONE;
             c = iterator.next()) {
            sb.append(c);
        }
        return sb.toString();
    }

    public void paint(Graphics g) {
	Dimension size = getSize();
	Color bg = getBackground();
	Color fg = getForeground();

	if (candidateIterator == null) {
	    super.paint(g);
	} else {
	    Graphics2D g2 = (Graphics2D)g;
	    g2.setColor(bg);
	    g2.fillRect(0, 0, size.width, size.height);
	    g2.setColor(fg);

	    FontRenderContext frc = g2.getFontRenderContext();
	    Font font = g2.getFont();
	    Rectangle2D b1 =
		font.getStringBounds(iteratorToString(labelIterator), frc);
	    Rectangle2D b2 = font.getStringBounds(" : ", frc);
	    Rectangle2D b3 =
		font.getStringBounds(iteratorToString(candidateIterator), frc);

	    int startX = (int)
		(size.width - (b1.getWidth() + b2.getWidth() + b3.getWidth()))
		/ 2;
	    int startY = (int)
		(size.height +
		 (b1.getHeight() + b2.getHeight() + b3.getHeight())) / 2;

	    g.drawString(labelIterator, startX, startY);
	    g.drawString(" : ", startX + (int)b1.getWidth(), startY);
	    g.drawString(candidateIterator,
			 startX + (int)(b1.getWidth() + b2.getWidth()),
			 startY);
	}

	if (focused) {
	    if (pressed) {
		g.setColor(bg.darker());
	    } else {
		g.setColor(bg.brighter());
	    }
	    g.drawLine(1, 1, size.width - 1, 1);
	    g.drawLine(1, 1, 1, size.height - 1);
	    if (pressed) {
		g.setColor(bg.brighter());
	    } else {
		g.setColor(bg.darker());
	    }
	    g.drawLine(1, size.height - 1, size.width - 1, size.height - 1);
	    g.drawLine(size.width - 1, 1, size.width - 1, size.height - 1);
	    requestFocus();
	} else {
	    g.setColor(bg);
	    g.drawRect(1, 1, size.width - 2, size.height - 2);
	}
    }

    public void mouseClicked(MouseEvent e) {
	if (target != null) {
	    target.clicked(index);
	}
    }

    public void mouseEntered(MouseEvent e) {
        if (target != null) {
	    setFocus(true);
	    repaint();
	    target.changeFocus(index);
	}
    }

    public void mouseExited(MouseEvent e) {
	if (target != null) {
	    setFocus(false);
	    pressed = false;
	    repaint();
	    target.forwardEvent(e);
	}
    }

    public void mousePressed(MouseEvent e) {
	pressed = true;
	repaint();
    }

    public void mouseReleased(MouseEvent e) {
	pressed = false;
	repaint();
    }
    
    public void keyPressed(KeyEvent e) {
	if (target != null) {
	    target.forwardEvent(e);
	}
    }
    
    public void keyReleased(KeyEvent e) {
	if (target != null) {
	    target.forwardEvent(e);
	}
    }
    
    public void keyTyped(KeyEvent e) {
	if (target != null) {
	    target.forwardEvent(e);
	}
    }
}
