/*************************************************************************
 * 
 * irmp3 - Multimedia Audio Jukebox for Linux
 * http://irmp3.sourceforge.net
 *
 * $Source: /cvsroot/irmp3/irmp3/src/irmp3d/mod_cd.c,v $ -- support for playing audio CDs
 * $Id: mod_cd.c,v 1.22 2004/02/17 20:40:36 boucman Exp $
 *
 * Copyright (C) by Vladimir Nadvornik
 *
 * Please contact the current maintainer, Jeremy Rosen <jeremy.rosen@enst-bretagne.fr
 * for information and support regarding irmp3.
 *
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/wait.h>
#include <sys/mount.h>
#include <linux/cdrom.h>

#include "config.h"
#include "irmp3config.h"
#include "irmp3tools.h"
#include "irmp3log.h"
#include "irmp3mod.h"
#include "mod_cd.h"




/*************************************************************************
 * GLOBALS
 */
char *mod_cd_scandir_pattern;
int mod_cd_externmount = 1;
typedef struct trkindex {
    int start;
    int length;
} trkindex_t;    



typedef struct mod_cd_player_s {
	char *device;
	char *mountpoint;
	int fd;
	int  status;
	trkindex_t trkindex[100];
	int oldpos;			//remember last displayed track time
	int current_track;		//remember current track
} mod_cd_player_t;
#define CD_STATUS_HALT	0 // must be zero for init
#define CD_STATUS_STOP	1
#define CD_STATUS_PLAY	2
#define CD_STATUS_PAUSE	3


#define MOD_CD_MAX_DEV 100
mod_cd_player_t mod_cd_player[MOD_CD_MAX_DEV];
int mod_cd_num_player = 1;
/*************************************************************************
 * MODULE INFO
 */
mod_t mod_cd = {
	"mod_cd",
	mod_cd_deinit,		// deinit
	NULL,			// reload
	0,			// watchfd
	NULL,			// poll
	mod_cd_update,			// update
	mod_cd_message,		// message
	NULL,			// SIGCHLD handler
	mod_cd_init,
	NULL,			// avoid warning
};



/*************************************************************************/
int mod_cd_play_sub(int trk,int player)
{
	struct cdrom_ti ti;
        struct cdrom_msf msf;

        int start = mod_cd_player[player].trkindex[trk].start;
	int stop = start + mod_cd_player[player].trkindex[trk].length;
	
	
        /* try CDROMPLAYMSF */
	msf.cdmsf_min0 = start / (75 * 60);
	msf.cdmsf_sec0 = (start % (75 * 60)) / 75;
	msf.cdmsf_frame0 = start % 75;
	
	msf.cdmsf_min1 = stop / (75 * 60);
	msf.cdmsf_sec1 = (stop % (75 * 60)) / 75;
	msf.cdmsf_frame1 = stop % 75;
	
	if (ioctl(mod_cd_player[player].fd, CDROMPLAYMSF, &msf) == -1) {
		/* CDROMPLAYMSF failed, try CDROMPLAYTRKIND*/
		ti.cdti_trk0 = trk;
		ti.cdti_trk1 = trk;

		ti.cdti_ind0 = 1;
		ti.cdti_ind1 = 99;

		if (ioctl(mod_cd_player[player].fd, CDROMPLAYTRKIND, &ti) == -1) {
			log_printf(LOG_ERROR, "mod_cd_play(): failed both ioctls, CDROMPLAYMSF and CDROMPLAYTRKIND \n");
			return -1;
		}
	}	
	log_printf(LOG_DEBUG, "mod_cd_play(): playing track %d\n", trk);

	return 0;
}

int mod_cd_stop_sub(int player)
{
	struct cdrom_subchnl subchnl;

	subchnl.cdsc_format = CDROM_MSF;

	if (ioctl(mod_cd_player[player].fd, CDROMSUBCHNL, &subchnl) == -1) {
		log_printf(LOG_ERROR,
				"mod_cd_stop(): ioctl cdromsubchnl\n");
		return 1;
	}

	if (subchnl.cdsc_audiostatus != CDROM_AUDIO_PLAY) {
		log_printf(LOG_DEBUG, "mod_cd_stop(): not playing\n");
		return 0;
	}

	if (ioctl(mod_cd_player[player].fd, CDROMSTOP) == -1) {
		log_printf(LOG_ERROR, "mod_cd_stop(): ioctl cdromstop\n");
		return 1;
	}
	//mod_sendmsg(MSGTYPE_PLAYER, "stop");
	return 0;
}

int mod_cd_pause_sub(int player)
{
	struct cdrom_subchnl subchnl;

	subchnl.cdsc_format = CDROM_MSF;

	if (ioctl(mod_cd_player[player].fd, CDROMSUBCHNL, &subchnl) == -1) {
		log_printf(LOG_ERROR, "mod_cd_pause(): ioctl cdromsubchnl\n");
		return -1;
	}

	switch (subchnl.cdsc_audiostatus) {
		case CDROM_AUDIO_PAUSED:
			if (ioctl(mod_cd_player[player].fd, CDROMRESUME) == -1) {
				log_printf(LOG_ERROR, "mod_cd_pause(): ioctl cdromresume\n");
				return -1;
			}
			//mod_sendmsg(MSGTYPE_PLAYER, "play");
			return 0;
		case CDROM_AUDIO_PLAY:
			if (ioctl(mod_cd_player[player].fd, CDROMPAUSE) == -1) {
				log_printf(LOG_ERROR, "mod_cd_pause(): ioctl cdrompause\n");
				return -1;
			}
			//mod_sendmsg(MSGTYPE_PLAYER, "pause");
			return 0;
	}
	return 0;
}

int mod_cd_seek_trkind(int seek, int abs,int player)
{
	struct cdrom_ti ti;
	struct cdrom_subchnl subchnl;

	subchnl.cdsc_format = CDROM_MSF;

	if (ioctl(mod_cd_player[player].fd, CDROMSUBCHNL, &subchnl) == -1) {
		log_printf(LOG_ERROR, "mod_cd_seek(): ioctl cdromsubchnl\n");
		return 1;
	}

	if (subchnl.cdsc_audiostatus != CDROM_AUDIO_PLAY) {
		log_printf(LOG_DEBUG, "mod_cd_seek(): not playing\n");
		return 0;
	}

	ti.cdti_trk0 = subchnl.cdsc_trk;
	ti.cdti_trk1 = subchnl.cdsc_trk;

	if (abs)
		ti.cdti_ind0 = seek;
	else
		ti.cdti_ind0 = subchnl.cdsc_ind + seek;

	if (ti.cdti_ind0 < 1)
		ti.cdti_ind0 = 1;
	else if (ti.cdti_ind0 > 99)
		ti.cdti_ind0 = 99;

	ti.cdti_ind1 = 99;

	log_printf(LOG_DEBUG, "mod_cd_seek(): seek to %d\n", ti.cdti_ind0);

	if (ioctl(mod_cd_player[player].fd, CDROMPLAYTRKIND, &ti) == -1) {
		log_printf(LOG_ERROR, "mod_cd_seek(): ioctl cdromplaytrkind\n");
		return 1;
	}
	return 0;
}

int mod_cd_seek_msf(int seek, int abs, int player)
{
	struct cdrom_subchnl subchnl;
        struct cdrom_msf msf;

        int start, stop, pos;
	subchnl.cdsc_format = CDROM_MSF;

	if (ioctl(mod_cd_player[player].fd, CDROMSUBCHNL, &subchnl) == -1) {
		log_printf(LOG_ERROR, "mod_cd_seek(): ioctl cdromsubchnl\n");
		return 1;
	}

	if (subchnl.cdsc_audiostatus != CDROM_AUDIO_PLAY) {
		log_printf(LOG_DEBUG, "mod_cd_seek(): not playing\n");
		return 0;
	}

        start = mod_cd_player[player].trkindex[subchnl.cdsc_trk].start;
	stop = start + mod_cd_player[player].trkindex[subchnl.cdsc_trk].length;
	
	if (abs)
		pos = start + seek * 75;
	else 
		pos = subchnl.cdsc_absaddr.msf.minute * 75 * 60 +
		      subchnl.cdsc_absaddr.msf.second * 75 +
		      subchnl.cdsc_absaddr.msf.frame +
		      seek * 75;
		      
        if (pos < start) 
		pos = start;

        if (pos > stop) 
		pos = stop - 75; /* 1 second before the end of track */

	msf.cdmsf_min0 = pos / (75 * 60);
	msf.cdmsf_sec0 = (pos % (75 * 60)) / 75;
	msf.cdmsf_frame0 = pos % 75;
	
	msf.cdmsf_min1 = stop / (75 * 60);
	msf.cdmsf_sec1 = (stop % (75 * 60)) / 75;
	msf.cdmsf_frame1 = stop % 75;
	
	if (ioctl(mod_cd_player[player].fd, CDROMPLAYMSF, &msf) == -1) {

		log_printf(LOG_ERROR, "mod_cd_seek(): ioctl cdromplaymsf\n");
		return 1;
	}
	return 0;
}

int mod_cd_seek_sub(int seek, int abs, int player)
{
	log_printf(LOG_DEBUG, "mod_cd_seek(): seek: %d  abs: %d\n", seek,
		   abs);
	if (mod_cd_seek_msf(seek, abs, player))
		if (mod_cd_seek_trkind(seek, abs,player)) {
			log_printf(LOG_ERROR,"mod_cd_seek(): both cdromplaytrkind and cdromplaymsf failed\n");
		return -1;
		}
	return 0;
}

int mod_cd_toc(int player)
{
	struct cdrom_tochdr tochdr;
	struct cdrom_tocentry te;

	int i;
	int type = -2;
	// -2 = none
	// -1 = data
	//  0 = audio
	int time = 0, last_time = 0;

	if (ioctl(mod_cd_player[player].fd, CDROMREADTOCHDR, &tochdr) == -1) {
		return -2;
	}


	for (i = tochdr.cdth_trk0; i < tochdr.cdth_trk1 + 1; i++) {
		te.cdte_track = i;
		te.cdte_format = CDROM_MSF;
		if (ioctl(mod_cd_player[player].fd, CDROMREADTOCENTRY, &te) == -1) {
			return -2;
		}

		time =
			te.cdte_addr.msf.minute * 60 * 75 + 
			te.cdte_addr.msf.second * 75 +
			te.cdte_addr.msf.frame;
			
		if (i > 0) {
			mod_cd_player[player].trkindex[i - 1].start  = last_time;
			mod_cd_player[player].trkindex[i - 1].length = time - last_time;
		}
		last_time = time;

		if (type == -2) {
			if (te.cdte_ctrl == CDROM_DATA_TRACK)
				type = -1; //data
			else
				type = 0; //audio
		}
	}

	te.cdte_track = CDROM_LEADOUT;
	te.cdte_format = CDROM_MSF;


	if (ioctl(mod_cd_player[player].fd, CDROMREADTOCENTRY, &te) == -1) {
		return type; // 0 for audio,<0 for data or unknown
	}

	time =
		te.cdte_addr.msf.minute * 60 * 75 + 
		te.cdte_addr.msf.second * 75 +
		te.cdte_addr.msf.frame;

	mod_cd_player[player].trkindex[i - 1].start  = last_time;
	mod_cd_player[player].trkindex[i - 1].length = time - last_time;

	return type;
}




/*******
 * ANSWER A HALT REQUEST
 *
 */
void mod_cd_halt (int player)
{
	
	if (mod_cd_player[player].status == CD_STATUS_HALT) return;
	mod_cd_stop_sub(player);
	if (mod_cd_player[player].fd >= 0) {
		close(mod_cd_player[player].fd);
		mod_cd_player[player].fd = -1;
	}
	mod_cd_player[player].current_track = -1;
	log_printf(LOG_NOISYDEBUG,"mod_cd : entered halt state\n");
	mod_cd_player[player].status = CD_STATUS_HALT;
}




/******************************************
 * ANSWER A STOP REQUEST
 *
 */

void mod_cd_stop(int player) 
{
	switch(mod_cd_player[player].status) {
		case CD_STATUS_HALT:
		case CD_STATUS_STOP:
			return;
		case CD_STATUS_PLAY:
		case CD_STATUS_PAUSE:
			if(mod_cd_stop_sub(player)) {
				mod_sendmsg(MSGTYPE_PLAYER,"error");
				mod_cd_halt(player);
				return;
			}
			mod_cd_player[player].status = CD_STATUS_STOP;
			log_printf(LOG_NORMAL, "Playing stopped\n");
			log_printf(LOG_NOISYDEBUG,"mod_cd : entered stop state\n");
			return;
		default:
			log_printf(LOG_ERROR,"mod_mpg123: unknown status %d\n",mod_cd_player[player].status);
			return;
	}
}
		

/******************************************
 * ANSWER A PLAY REQUEST
 *
 */
	
void mod_cd_play(int track,int player) {
	switch(mod_cd_player[player].status) {
		case CD_STATUS_HALT:
			if ((mod_cd_player[player].fd = open(mod_cd_player[player].device,  O_RDONLY | O_NONBLOCK)) < 0) {
				mod_sendmsg(MSGTYPE_PLAYER, "error");
				mod_cd_halt(player);
				return;
			} else if(mod_cd_toc(player)) {
				mod_sendmsg(MSGTYPE_PLAYER, "error");
				mod_cd_halt(player);
				return;
			}
		case CD_STATUS_PAUSE:
		case CD_STATUS_STOP:
		case CD_STATUS_PLAY:
			if(mod_cd_play_sub(track,player)) {
				mod_sendmsg(MSGTYPE_PLAYER, "error");
				mod_cd_halt(player);
				return;
			}
			log_printf(LOG_NOISYDEBUG,"mod_cd : entered play state\n");
			log_printf(LOG_NORMAL, "Playing track %d\n", track);
			mod_cd_player[player].status= CD_STATUS_PLAY;
			mod_cd_player[player].current_track = track;
			mod_cd_player[player].oldpos= -1;
			return;

		default:
			log_printf(LOG_ERROR,"mod_cd: unknown status %d\n",mod_cd_player[player].status);
			return;
	}
}

/******************************************
 * ANSWER A PAUSE REQUEST
 *
 */
	
void mod_cd_pause(int player) {
	switch(mod_cd_player[player].status) {
		case CD_STATUS_HALT:
		case CD_STATUS_STOP:
			return;
			
		case CD_STATUS_PAUSE:
			if(mod_cd_pause_sub(player)) {
				mod_sendmsg(MSGTYPE_PLAYER, "error");
				mod_cd_halt(player);
				return;
			}
			log_printf(LOG_NOISYDEBUG,"mod_mpg123 : entered play state\n");
			mod_cd_player[player].status= CD_STATUS_PLAY;
			return;
			
		case CD_STATUS_PLAY:
			if(mod_cd_pause_sub(player)) {
				mod_sendmsg(MSGTYPE_PLAYER, "error");
				mod_cd_halt(player);
				return;
			}
			log_printf(LOG_NOISYDEBUG,"mod_mpg123 : entered pause state\n");
			mod_cd_player[player].status= CD_STATUS_PAUSE;
			return;
			
		default:
			log_printf(LOG_ERROR,"mod_mpg123: unknown status %d\n",mod_cd_player[player].status);
			return;

	}
}
/******************************************
 * ANSWER A SEEK REQUEST
 *
 */
	
void mod_cd_seek(int seek, int abs,int player) {
	switch(mod_cd_player[player].status) {
		case CD_STATUS_HALT:
		case CD_STATUS_STOP:
			return;

		case CD_STATUS_PAUSE:
			if(mod_cd_pause_sub(player)) {
				mod_sendmsg(MSGTYPE_PLAYER, "error");
				mod_cd_halt(player);
				return;
			}
			log_printf(LOG_NOISYDEBUG,"mod_mpg123 : entered play state\n");
			mod_cd_player[player].status= CD_STATUS_PLAY;
		case CD_STATUS_PLAY:
			if(mod_cd_seek_sub(seek,abs,player)) {
				mod_sendmsg(MSGTYPE_PLAYER, "error");
				mod_cd_halt(player);
				return;
			}
			return;

		default:
			log_printf(LOG_ERROR,"mod_mpg123: unknown status %d\n",mod_cd_player[player].status);
			return;
	}
}

void mod_cd_update(void)
{
	struct cdrom_subchnl subchnl;
	int pos, remain;
	int player;
	for (player = 0; player < mod_cd_num_player; player++) {
		if(mod_cd_player[player].status != CD_STATUS_PLAY) continue;

		subchnl.cdsc_format = CDROM_MSF;

		if (ioctl(mod_cd_player[player].fd, CDROMSUBCHNL, &subchnl) == -1) {
			log_printf(LOG_ERROR, "mod_cd_update(): ioctl cdromsubchnl\n");
			return;
		}
		pos =
			subchnl.cdsc_reladdr.msf.minute * 60 +
			subchnl.cdsc_reladdr.msf.second;
		remain = (mod_cd_player[player].trkindex[subchnl.cdsc_trk].length +74) / 75  - pos;
		if (pos != mod_cd_player[player].oldpos) {     // only send track time if it has changed
			if(mod_cd_player[player].oldpos == -1) {
				mod_sendmsgf(MSGTYPE_PLAYER,"playing");
			}
			mod_sendmsgf(MSGTYPE_PLAYER, "time %d %d", pos, remain);
			mod_cd_player[player].oldpos=pos;
		}

		if (subchnl.cdsc_audiostatus == CDROM_AUDIO_COMPLETED ||
				subchnl.cdsc_audiostatus == CDROM_AUDIO_NO_STATUS) {
			mod_sendmsg(MSGTYPE_PLAYER, "endofsong");
		}
	}
}

/**********************
 * TRY TO EJECT THE CDROM
 */


int mod_cd_eject(int force, int player)
{
	if(mod_cd_player[player].status == CD_STATUS_HALT) {
		log_printf(LOG_DEBUG, "mod_cd_eject(): eject force=%d\n", force);
		if (mod_cd_externmount) {
			int retcode;
			retcode = system_block(0,NULL,NULL,"umount %s", mod_cd_player[player].device);
			if (retcode != 0 && retcode != 2) { //don't take "not mounted" as an error
				log_printf(LOG_DEBUG, "mod_cd_eject(): umount failed %d\n",retcode);
				free_message(mod_query(MSGTYPE_EVENT, "teststopall"));	/* try to stop player before umount */
				if (force) {
					free_message(mod_query(MSGTYPE_INPUT, "stopall"));	/* try to stop player before umount */
					free_message(mod_query(MSGTYPE_INPUT, "cd eject2"));
				}
				return -1;
			}
		} else {
			if (umount(mod_cd_player[player].mountpoint) != 0 && errno != 22) {// not mounted isn't an error
				log_printf(LOG_DEBUG, "mod_cd_eject(): umount failed %d\n");

				if (force) {
					free_message(mod_query(MSGTYPE_INPUT, "stopall"));	/* try to stop player before umount */
					free_message(mod_query(MSGTYPE_INPUT, "cd eject2"));
				}
				return -1;
			}
		}
	} else  {
		mod_cd_halt(player);
		log_printf(LOG_DEBUG, "mod_cd_eject(): eject force=%d CD\n", force);
	}
	if ((mod_cd_player[player].fd = open(mod_cd_player[player].device, O_RDONLY | O_NONBLOCK)) < 0) {
		log_printf(LOG_DEBUG, "mod_cd_eject(): open %s", mod_cd_player[player].device);
		return -1;
	}

	if (ioctl(mod_cd_player[player].fd, CDROMEJECT) == -1) {
		log_printf(LOG_DEBUG, "mod_cd_eject(): ioctl cdromeject %s\n", strerror(errno));
	}

	if (mod_cd_player[player].fd >= 0) {
		close(mod_cd_player[player].fd);
		mod_cd_player[player].fd = -1;
	}
	return 0;
}

int mod_cd_load(int clear,int player)
{
	log_printf(LOG_DEBUG, "mod_cd_load():\n");

	if(mod_cd_player[player].status != CD_STATUS_HALT) {
		mod_sendmsg(MSGTYPE_PLAYER,"error");
		mod_cd_halt(player);
	}
      	if ((mod_cd_player[player].fd = open(mod_cd_player[player].device,  O_RDONLY | O_NONBLOCK)) < 0) {
		log_printf(LOG_DEBUG, "mod_cd_load(): no cd in %s, (%s)\n", mod_cd_player[player].device, strerror(errno));
		return -1;
	}
	if (ioctl(mod_cd_player[player].fd, CDROMCLOSETRAY, NULL) == -1) {
		log_printf(LOG_ERROR, "mod_cd_load(): ioctl cdromclosetray\n");
	}
	switch (mod_cd_toc(player)) {
	    case -1: { //DATA
		log_printf(LOG_DEBUG, "mod_cd_load(): mp3, mounting\n");
		if (mod_cd_externmount) {
			system_block(0,NULL,NULL,"mount %s",mod_cd_player[player].mountpoint);
		} else {
			mount(mod_cd_player[player].device, mod_cd_player[player].mountpoint, "iso9660",
			      MS_RDONLY | MS_MGC_VAL, NULL);
		}

		if(clear)
			mod_sendmsgf(MSGTYPE_INPUT, "playlist loaddir %s/%s",mod_cd_player[player].mountpoint, mod_cd_scandir_pattern);
		
		else
			mod_sendmsgf(MSGTYPE_INPUT, "playlist loaddirplus %s/%s",mod_cd_player[player].mountpoint, mod_cd_scandir_pattern);
		break;
	    } 
	    case 0: { //AUDIO
		struct cdrom_tochdr tochdr;
		int i;
		log_printf(LOG_DEBUG, "mod_cd_load(): audio cd\n");

		if (ioctl(mod_cd_player[player].fd, CDROMREADTOCHDR, &tochdr) == -1) {
			log_printf(LOG_ERROR, "mod_cd_load(): ioctl cdromreadtochdr\n");
			return 1;
		}

		if(clear) mod_sendmsg(MSGTYPE_INPUT, "playlist clear");
		for (i = tochdr.cdth_trk0; i < tochdr.cdth_trk1 + 1; i++) {
			struct cdrom_tocentry te;

			te.cdte_track = i;
			te.cdte_format = CDROM_MSF;
			if (ioctl(mod_cd_player[player].fd, CDROMREADTOCENTRY, &te) == -1) {
				log_printf(LOG_ERROR, "mod_cd_load(): ioctl cdromreadtocentry\n");
				return 1;
			}
			if (te.cdte_ctrl == CDROM_DATA_TRACK)
				continue;	/* skip data tracks */

			mod_sendmsgf(MSGTYPE_INPUT, "playlist addtype audiotrack AUDIOCD_TRACK# %02d %d",i,player);
		}
		mod_sendmsg(MSGTYPE_INPUT, "playlist done");
		break;
	    }
	    default:	//no cd
	        log_printf(LOG_DEBUG, "mod_cd_load(): no cd\n"); 
        }
	close(mod_cd_player[player].fd);
	mod_cd_player[player].fd = -1;
	return 0;
}


/*************************************************************************
 * RECEIVE MESSAGE
 */
void mod_cd_message(int msgtype, char *msg,const char __attribute__((unused))*sender)
{
	char *c1, *c2, *c3, *c4, *c5;
	int allplayer;
	int player;
	// handle input messages
	if (msgtype == MSGTYPE_INPUT) {

		c1 = strtok(msg, " \t");	// it works!
		if(!c1) return;

		if ( !strcasecmp(c1, "cd")) {
			c2 = strtok(NULL, " \t");
			if(!c2) return;
			if (!strcasecmp(c2, "load")){
				c3 = strtok(NULL, "");
				if(c3){
					player = atoi(c3);
					if(player < 0 || player >= mod_cd_num_player) return;
					mod_cd_load(1,player );
				}else{
					mod_cd_load(1,0);
				}
			} else if (!strcasecmp(c2, "loadplus")){
				c3 = strtok(NULL, "");
				if(c3){
					player = atoi(c3);
					if(player < 0 || player >= mod_cd_num_player) return;
					mod_cd_load(0,player );
				}else{
					mod_cd_load(0,0);
				}
			}else if (!strcasecmp(c2, "eject")){
				c3 = strtok(NULL, "");
				if(c3){
					player = atoi(c3);
					if(player < 0 || player >= mod_cd_num_player) return;
					mod_cd_eject(1,player );
				}else{
					mod_cd_eject(1,0);
				}
			}else if (!strcasecmp(c2, "eject2")){
				c3 = strtok(NULL, "");
				if(c3){
					player = atoi(c3);
					if(player < 0 || player >= mod_cd_num_player) return;
					mod_cd_eject(0,player );
				}else{
					mod_cd_eject(0,0);
				}
			}
		}else{
			return;
		}
	} else if (msgtype == MSGTYPE_PLAYER) {

		c1 = strtok(msg, " \t");	// it works!
		if(!c1) return;
		// see if we can play the requested song
		if (!strcasecmp(c1, "play")) {
			c2 =  strtok(NULL, " \t");
			if(!c2) return;
			c3 =  strtok(NULL, " \t");
			c4 = strtok(NULL, " \t");
			c5 = strtok(NULL, "");
			
			if(!strcasecmp(c2,"mod_cd")) {
				for (allplayer = 0; allplayer < mod_cd_num_player; allplayer++) {
					if(atoi(c5) == allplayer) {
						if (!strcasecmp(c3, "AUDIOCD_TRACK#") ) {
							mod_cd_play(atoi(c4), allplayer);
						} else {
							log_printf(LOG_DEBUG, "mod_cd_message(): unable to play '%s'\n", c3);
							mod_sendmsg(MSGTYPE_PLAYER,"error");
							mod_cd_halt(allplayer);
						}
					} else {
						log_printf(LOG_DEBUG, "mod_cd_message(), cd player %d: not for me '%s'\n",allplayer, c2);
						mod_cd_halt(allplayer);
					}
				}
			} else {
				player = mod_cd_num_player; // no player
				for (allplayer = 0; allplayer < mod_cd_num_player; allplayer++) {
						mod_cd_halt(allplayer);
				}
			}
		} else if ( !strcasecmp(c1, "seek")) {
			c2 =  strtok(NULL, " \t");
			if(!c2) return;
			if (*c2 == '-' || *c2 == '+') {
				for (allplayer = 0; allplayer < mod_cd_num_player; allplayer++) {
					mod_cd_seek(atoi(c2), 0,allplayer);	//relative
				}
			} else {
				for (allplayer = 0; allplayer < mod_cd_num_player; allplayer++) {
					mod_cd_seek(atoi(c2), 1,allplayer);
				}
			}

		} else if (!strcasecmp(c1, "stop")) {
			for (allplayer = 0; allplayer < mod_cd_num_player; allplayer++) {
				mod_cd_stop(allplayer);
			}
		} else if (!strcasecmp(c1, "pause")) {
			for (allplayer = 0; allplayer < mod_cd_num_player; allplayer++) {
				mod_cd_pause(allplayer);
			}
		}	       
	} else if( msgtype == MSGTYPE_QUERY) {
		c1 = strtok(msg, " \t");	// it works!
		if(!c1) return;
		c2 = strtok(NULL, "");
		if (!strcasecmp(c1,"whocanplay") && !strcasecmp(c2,"audiotrack")) {
			mod_sendmsgf(MSGTYPE_INFO,"canplay mod_cd %s",c2);
		} else if (!strcasecmp(c1,"titleguess") ) {
			for (allplayer = 0; allplayer < mod_cd_num_player; allplayer++) {
				if(mod_cd_player[allplayer].status !=  CD_STATUS_HALT ) {
					mod_sendmsgf(MSGTYPE_INFO, "title Track %02d Device %02d", mod_cd_player[allplayer].current_track,allplayer);
				}
			}
		} else if (!strcasecmp(c1,"artistguess") ) {
			for (allplayer = 0; allplayer < mod_cd_num_player; allplayer++) {
				if(mod_cd_player[allplayer].status !=  CD_STATUS_HALT ) {
					mod_sendmsgf(MSGTYPE_INFO, "artist Audio-CD");
				}
			}
		}
	}
}


/*************************************************************************
 * MODULE INIT FUNCTION
 */
char *mod_cd_init(void)
{
	int player;
	char tmp[100];
	bzero(mod_cd_player,sizeof(mod_cd_player_t)*MOD_CD_MAX_DEV);
	log_printf(LOG_DEBUG, "mod_cd_init(): initializing\n");
	mod_cd_num_player = config_getnum("num_cd_device",1);
	for(player=0; player< mod_cd_num_player; player++) {
		mod_cd_player[player].fd = -1;
		mod_cd_player[player].oldpos = -1;
		mod_cd_player[player].current_track = -1;
		sprintf(tmp,"cd_device_%d",player);
		mod_cd_player[player].device = config_getstr(tmp, "/dev/cdrom");
		sprintf(tmp,"cd_mountpoint_%d",player);
		mod_cd_player[player].mountpoint = config_getstr(tmp, "/mnt/cdrom");
	}
	mod_cd_scandir_pattern = config_getstr("cd_scandir_pattern", "*.[Mm][Pp]3");
	mod_cd_externmount = (strcasecmp("yes", config_getstr("cd_externmount", "yes")) == 0);



	return NULL;
}


/*************************************************************************
 * MODULE DEINIT FUNCTION
 */
void mod_cd_deinit(void)
{
	int player;
	for(player=0; player< mod_cd_num_player; player++) {
		mod_cd_halt(player);
	}
	log_printf(LOG_DEBUG, "mod_cd_deinit(): deinitialized\n");
}


/*************************************************************************
 * EOF
 */
