/*
 * client_manager.h - include file for client-manager
 *
 * iTALC
 * Copyright (c) 2004-2005 Tobias Doerffel <tobias@doerffel.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program (see COPYING); if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */


#ifndef _CLIENT_MANAGER_H
#define _CLIENT_MANAGER_H

#include <qlistview.h>
#include <qstring.h>
#include <qvaluevector.h>
#include <qdom.h>
#include <qspinbox.h>

#include "client.h"
#include "side_bar_widget.h"



/*!
  \brief The Client-Manager

  The client-manager is one of the most important widgets in iTALC. It provides a browser for managing clients.
  Managing means, that you can add classrooms and clients. Then you can either run an action for a whole classroom or you
  can do this on single clients.
  This class contains a lot of internal data, which is neccessary for managing the clients. But it's needed nowhere out of this
  class, so there are almost no member-acces-methods.

  \sa   client for detailed description of actual functionality.
*/


class QPopupMenu;
class QToolButton;

class classRoom;
class classRoomItem;
class clientPropertiesEdit;
class configWidget;


class clientManager : public sideBarWidget
{
	Q_OBJECT
public:
	static clientManager * inst( void )
	{
		if (s_instOfMe == NULL)
			return( s_instOfMe = new clientManager );
		return( s_instOfMe );
	}

	void doCleanupWork( void );

	void loadGlobalClientConfig( void );
	void saveGlobalClientConfig( void );
	void loadPersonalConfig( void );
	void savePersonalConfig( void );

	QValueVector<client *> visibleClients( void ) const;
	static void getVisibleClients( QListViewItem * _p, QValueVector<client *> & _vv );

	void createActionMenu( QPopupMenu * _m );
	void createActionMenuForClassRooms( QPopupMenu * _m );

	bool demoRunning( void ) const;
	inline int updateInterval( void ) const
	{
		return( m_clientUpdateInterval );
	}
	inline const QString & winCfg( void ) const
	{
		return( m_winCfg );
	}
	inline void setUpdateIntervalSpinBox( QSpinBox * _update_interval_sb )
	{
		m_updateIntervalSpinBox = _update_interval_sb;
		m_updateIntervalSpinBox->setValue( m_clientUpdateInterval );
	}
	inline QPopupMenu * quickSwitchMenu( void )
	{
		return( m_quickSwitchMenu );
	}

	void setStateOfClassRoom( classRoom * _cr, bool _shown );
	int addClassRoomToQuickSwitchMenu( classRoom * _cr );
	void removeClassRoomFromQuickSwitchMenu( int _id );


public slots:
	void updateClients( void );

	// slots for toolbar-actions
	void startFullScreenDemo( void );
	void startDemo( void );
	void stopDemo( void );
	void lockScreens( void );
	void unlockScreens( void );
	void sendMessage( void );
	void distributeFile( void );
	void collectFiles( void );
	void killGames( void );
	void killBrowsers( void );
	void powerOnClients( void );
	void rebootClients( void );
	void powerOffClients( void );
	void execCmds( void );
	void restartX( void );

	// slots for actions in view-menu
	void optimizeClientSize( void );
	void increaseClientSize( void );
	void decreaseClientSize( void );

	// slots for config-widget in side-bar
	void updateIntervalChanged( int _value );

	void hideAllClassRooms( void );


private slots:
	void itemDoubleClicked( QListViewItem * _i, const QPoint &, int );
	void itemDoubleClicked( QListViewItem * _i );
	void contextMenuRequest( QListViewItem * _i, const QPoint &, int );

	// slots for client-actions in context-menu
	void showHideClient( void );
	void editClientProperties( void );
	void removeClient( void );

	// slots for classroom-actions in context-menu
	void showSelectedClassRooms( void );
	void hideSelectedClassRooms( void );
	void editClassRoomName( void );
	void removeClassRoom( void );

	// slots for general actions in context-menu
	void addClient( void );
	void addClassRoom( void );


private:
	clientManager();
	// only for hiding copy-constructor, because clientManager is a singleton-class
	clientManager( const clientManager & ) :
		sideBarWidget( "", QPixmap( "" ), NULL )
	{
	}
	virtual ~clientManager();

	void cmdToActiveClients( client::clientCmds _cmd, const QString & _u_data = CONFIRM_NO );

	void closeEvent( QCloseEvent * _ce );

	void saveSettingsOfChildren( QDomDocument & _doc, QDomElement & _root, QListViewItem * _lvi, bool _is_global_config );
	void createActionMenu( QListViewItem * _p, QPopupMenu * _m );
	
	void getHeaderInformation( const QDomElement & _header );
	void loadTree( classRoom * _parentItem, const QDomElement & _parentElement, bool _is_global_config );

	QValueVector<classRoomItem *> selectedItems( void );
	void getSelectedItems( QListViewItem * _p, QValueVector<classRoomItem *> & _vv, bool _add_all = FALSE );

	void resizeClients( const int _new_width );


	QListView * m_list;

	QValueVector<classRoom *> m_classRooms;
	QValueVector<client *> m_clientsToRemove;
	QValueVector<classRoom *> m_classRoomsToRemove;
	QDomDocument m_domTree;

	const QString m_personalConfiguration;
	const QString m_globalClientConfiguration;

	QSpinBox * m_updateIntervalSpinBox;
	QPopupMenu * m_quickSwitchMenu;

	int m_clientUpdateInterval;
	QString m_winCfg;

	friend class clientPropertiesEdit;
	friend class configWidget;

	static clientManager * s_instOfMe;

} ;






class classRoom : public QObject, public QListViewItem
{
	Q_OBJECT
public:
	classRoom( QListView * _parent, const QString & _name );
	classRoom( QListViewItem * _parent, const QString & _name );
	virtual ~classRoom();

	void setOpen( bool );
        void setup( void );

	inline const QPixmap * pixmap( int _id ) const
	{
		if( _id == 0 )
			return( m_pix );
		return( NULL );
	}

	void createActionMenu( QPopupMenu * _m, bool _add_sub_menu = TRUE );


public slots:
	void switchToClassRoom( void );


private slots:
	void processCmdOnAllClients( int _cmd );


private:
	void initPixmaps( void );
	void setPixmap( QPixmap * _pix );

	QPixmap * m_pix;
	int m_qsMenuId;

	static QPixmap * s_classRoomClosedPixmap;
	static QPixmap * s_classRoomOpenedPixmap;

} ;




class classRoomItem : public QListViewItem
{
public:
	classRoomItem( client * _client, QListViewItem * _parent, const QString & _name );
	virtual ~classRoomItem();

	inline const QPixmap * pixmap( int _id ) const
	{
		if( _id == 0 )
			return( m_pix );
		return( NULL );
	}
	inline client * getClient( void )
	{
		return( m_client );
	}

	void setObserved( bool _obs );


private:
	void setPixmap( QPixmap * _pix );

	QPixmap * m_pix;
	client * m_client;

	static QPixmap * s_clientPixmap;
	static QPixmap * s_clientObservedPixmap;

} ;


#endif
