/*
 * client_manager.cpp - implementation of client-manager (partly based on qt-clientManager-example)
 *
 * iTALC
 * Copyright (c) 2004-2005 Tobias Doerffel <tobias@doerffel.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program (see COPYING); if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */


#include <math.h>
#include <qmessagebox.h>
#include <qtimer.h>
#include <qfiledialog.h>
#include <qpopupmenu.h>
#include <qlabel.h>
#include <qworkspace.h>
#include <qvbox.h>
#include <qtoolbutton.h>
#include <qsplitter.h>
#include <qwhatsthis.h>
#include <qdatetime.h>
#if QT_VERSION >= 0x030200
#include <qsplashscreen.h>
#endif

#include "italc.h"
#include "client_manager.h"
#include "client.h"
#include "client_properties_edit.h"
#include "cmd_input_dialog.h"
#include "msg_input_dialog.h"
#include "text_input_dialog.h"
#include "embed.h"
#include "version.h"
#include "paths.h"
#include "progress_information.h"
#include "italc_side_bar.h"
#include "system_environment.h"
#include "qt3_compat.h"

#include "client_manager.moc"


template<typename T>
inline T roundCorrect( T _val )
{
	if( _val-floor( _val ) < 0.5 )
	{
		return( floor( _val ) );
	}
	return( ceil( _val ) );
}




const int widths[] = { 128, 192, 256, 320, 384, 448, 512, /*576, 640, 704, 768, 832, 896, 960, 1024, 1088, 1152, 1216, 1280, */0 };



QPixmap * classRoomItem::s_clientPixmap = NULL;
QPixmap * classRoomItem::s_clientObservedPixmap = NULL;


clientManager * clientManager::s_instOfMe = NULL;


clientManager::clientManager() :
	sideBarWidget( tr( "Client-Manager" ), embed::getIconPixmap( "client_manager" ), italc::inst()->m_splitter ),
	m_personalConfiguration( QDir::home().path() + "/" + QString( ITALC_CONFIG_PATH ) + "/personal-config.xml" ),
	m_globalClientConfiguration( QDir::home().path() + "/" + QString( ITALC_CONFIG_PATH ) + "/global-client-config.xml" ),
	m_quickSwitchMenu( new QPopupMenu( this ) ),
	m_clientUpdateInterval( 1 )
{
	s_instOfMe = this;

	m_list = new QListView( contentParent() );
	QWhatsThis::add( m_list, tr( "In this list clients and classrooms are managed. You can add clients or "
					"classrooms by clicking with the right mousebutton in this list." ) );

	m_list->addColumn( tr("Clients") );
	m_list->addColumn( tr("IP-address") );
	m_list->setSelectionMode( QListView::Extended );
	m_list->setTreeStepSize( 12 );
	m_list->setDefaultRenameAction( QListView::Accept );
	m_list->setSorting( 0 );
	m_list->setRootIsDecorated( TRUE );
	m_list->setShowToolTips( TRUE );
	m_list->setCaption( tr( "Client-Manager" ) );
#if QT_VERSION >= 0x030200
	connect( m_list, SIGNAL( doubleClicked( QListViewItem *, const QPoint &, int ) ), this,
			SLOT( itemDoubleClicked( QListViewItem *, const QPoint &, int ) ) );
#else
	connect( m_list, SIGNAL( doubleClicked( QListViewItem * ) ), this,
			SLOT( itemDoubleClicked( QListViewItem * ) ) );
#endif
	connect( m_list, SIGNAL( contextMenuRequested( QListViewItem *, const QPoint &, int ) ), this,
			SLOT( contextMenuRequest( QListViewItem *, const QPoint &, int ) ) );


	QFont f;
	f.setPixelSize( 12 );
	QLabel * help_txt = new QLabel( tr( "Use the context-menu (right mouse-button) for adding/removing clients and/or "
						"classrooms. Once you added clients you can show or hide them by "
						"double-clicking them. You can also show or hide a whole classroom by "
						"clicking right at it and selecting \"Show/hide all clients in room\"." ),
					contentParent() );
	help_txt->setAlignment( help_txt->alignment()|WordBreak );
	help_txt->setFont( f );

	loadGlobalClientConfig();
	loadPersonalConfig();

	show();
}




clientManager::~clientManager()
{
}




void clientManager::doCleanupWork( void )
{
	while( m_clientsToRemove.size() )
	{
		delete m_clientsToRemove[0];
		m_clientsToRemove.erase( m_clientsToRemove.begin() );
	}

	while( m_classRoomsToRemove.size() )
	{
		QValueVector<classRoom *>::iterator it = qFind( m_classRooms.begin(), m_classRooms.end(),
									m_classRoomsToRemove[0] );
		if( it != m_classRooms.end() )
		{
			m_classRooms.erase( it );
		}
		delete m_classRoomsToRemove[0];
		m_classRoomsToRemove.erase( m_classRoomsToRemove.begin() );
	}
	savePersonalConfig();
	saveGlobalClientConfig();

	//saveSettings();
}




void clientManager::saveGlobalClientConfig( void )
{
	QDomDocument doc( "italc-config-file" );

	QDomElement italc_config = doc.createElement( "globalclientconfig" );
	italc_config.setAttribute( "version", VER_STRING );
	doc.appendChild( italc_config );

	QDomElement root = doc.createElement( "body" );
	italc_config.appendChild( root );

	QListViewItem * lvi = m_list->firstChild();

	for( int i = 0; i < m_list->childCount(); ++i )
	{
		saveSettingsOfChildren( doc, root, lvi, TRUE );
		lvi = lvi->nextSibling();
	}

#if QT_VERSION >= 0x030100
	QString xml = "<?xml version=\"1.0\"?>\n" + doc.toString( 2 );
#else
	QString xml = "<?xml version=\"1.0\"?>\n" + doc.toString();
#endif
	if( italc::ensureConfigPathExists() == FALSE )
	{
		qFatal( "Could not read/write or create directory " + QString( ITALC_CONFIG_PATH ) + "! For running iTALC, make sure you have "
			"write-access to your home-directory and to " + QString( ITALC_CONFIG_PATH ) + " (if already existing)." );
	}

	QFile outfile( m_globalClientConfiguration );
	outfile.open( IO_WriteOnly | IO_Truncate );

	outfile.writeBlock( xml.ascii(), xml.length() );
	outfile.close();
}




void clientManager::savePersonalConfig( void )
{
	QDomDocument doc( "italc-config-file" );

	QDomElement italc_config = doc.createElement( "personalconfig" );
	italc_config.setAttribute( "version", VER_STRING );
	doc.appendChild( italc_config );

	QDomElement head = doc.createElement( "head" );
	italc_config.appendChild( head );

	QDomElement globalsettings = doc.createElement( "globalsettings" );
	globalsettings.setAttribute( "client-update-interval", m_clientUpdateInterval );
	globalsettings.setAttribute( "use-big-icons", italc::inst()->usesBigPixmaps() );
	globalsettings.setAttribute( "win-width", italc::inst()->width() );
	globalsettings.setAttribute( "win-height", italc::inst()->height() );
	globalsettings.setAttribute( "opened-tab", italc::inst()->m_sideBar->openedTab() );

	QTextStream ts( m_winCfg, IO_WriteOnly );
	ts << *( italc::inst() );
	globalsettings.setAttribute( "wincfg", m_winCfg );

	QMap<QString, QString> ip_addr_map = systemEnvironment::localIPAddresses();
	int idx = ip_addr_map.values().findIndex( MASTER_HOST );
	if( idx != -1 )
	{
		globalsettings.setAttribute( "net-iface", ip_addr_map.keys()[idx] );
	}

	head.appendChild( globalsettings );


	QDomElement root = doc.createElement( "body" );
	italc_config.appendChild( root );

	QListViewItem * lvi = m_list->firstChild();

	for( int i = 0; i < m_list->childCount(); ++i )
	{
		saveSettingsOfChildren( doc, root, lvi, FALSE );
		lvi = lvi->nextSibling();
	}


#if QT_VERSION >= 0x030100
	QString xml = "<?xml version=\"1.0\"?>\n" + doc.toString( 2 );
#else
	QString xml = "<?xml version=\"1.0\"?>\n" + doc.toString();
#endif
	if( italc::ensureConfigPathExists() == FALSE )
	{
		qFatal( "Could not read/write or create directory " + QString( ITALC_CONFIG_PATH ) + "! For running iTALC, make sure you have "
			"write-access to your home-directory and to " + QString( ITALC_CONFIG_PATH ) + " (if already existing)." );
	}

	QFile outfile( m_personalConfiguration );
	outfile.open( IO_WriteOnly | IO_Truncate );

	outfile.writeBlock( xml.ascii(), xml.length() );
	outfile.close();
}




void clientManager::saveSettingsOfChildren( QDomDocument & _doc, QDomElement & _root, QListViewItem * _parent, bool _is_global_config )
{
	QDomElement classroom = _doc.createElement( "classroom" );
	classroom.setAttribute( "name", _parent->text( 0 ) );
	_root.appendChild( classroom );

	QListViewItem * lvi = _parent->firstChild();

	for( int i = 0; i < _parent->childCount(); ++i )
	{
		if( lvi->childCount() || dynamic_cast<classRoom *>( lvi ) != NULL )
		{
			saveSettingsOfChildren( _doc, classroom, lvi, _is_global_config );
		}
		else
		{
			if( dynamic_cast<classRoomItem *>( lvi ) != NULL )
			{
				client * c = dynamic_cast<classRoomItem *>( lvi )->getClient();
				QDomElement client_element = _doc.createElement( "client" );
				client_element.setAttribute( "id", c->id() );
				if( _is_global_config == TRUE )
				{
					client_element.setAttribute( "name", c->name() );
					client_element.setAttribute( "ip", c->ip() );
					client_element.setAttribute( "mac", c->mac() );
				}
				else
				{
					client_element.setAttribute( "visible", ( c->isShown() == TRUE ) ? "yes" : "no" );
/*					if( m_rotating )
					{
						client_element.setAttribute( "x", QString::number( static_cast<int>( c->m_rasterX ) ) );
 						client_element.setAttribute( "y", QString::number( static_cast<int>( c->m_rasterY ) ) );
					}
					else
					{*/
						client_element.setAttribute( "x", QString::number( c->parentWidget()->pos().x() ) );
 						client_element.setAttribute( "y", QString::number( c->parentWidget()->pos().y() ) );
						client_element.setAttribute( "w", QString::number( c->width() ) );
 						client_element.setAttribute( "h", QString::number( c->height() ) );
					//}
				}
				classroom.appendChild( client_element );
			}
		}
		lvi = lvi->nextSibling();
	}
}




// routine that returns m_list of all visible clients
QValueVector<client *> clientManager::visibleClients( void ) const
{
	QValueVector<client *> vv;
	QListViewItem * lvi = m_list->firstChild();

	for( int i = 0; i < m_list->childCount(); ++i )
	{
		getVisibleClients( lvi, vv );
		lvi = lvi->nextSibling();
	}

	return( vv );
}




void clientManager::getVisibleClients( QListViewItem * _p, QValueVector<client *> & _vv )
{
	QListViewItem * lvi = _p->firstChild();

	classRoomItem * l = NULL;

	for( int i = 0; i < _p->childCount(); ++i )
	{
		if( lvi->childCount() )
		{
			getVisibleClients( lvi, _vv );
		}
		else if( ( l = dynamic_cast<classRoomItem *>( lvi ) ) != NULL && l->getClient()->isShown() )
		{
			_vv.push_back( l->getClient() );
		}
		lvi = lvi->nextSibling();
	}
}




void clientManager::getHeaderInformation( const QDomElement & _header )
{
	QDomNode node = _header.firstChild();

	while( !node.isNull() )
	{
		if( node.isElement() && node.nodeName() == "globalsettings" )
		{
			m_clientUpdateInterval = node.toElement().attribute( "client-update-interval" ).toInt();
			if( node.toElement().attribute( "use-big-icons" ) != QString::null )
			{
				italc::inst()->setUsesBigPixmaps( node.toElement().attribute( "use-big-icons" ).toInt() );
			}
			if( node.toElement().attribute( "win-width" ) != QString::null &&
				node.toElement().attribute( "win-height" ) != QString::null )
			{
				italc::inst()->resize( node.toElement().attribute( "win-width" ).toInt(),
							node.toElement().attribute( "win-height" ).toInt() );
			}
			if( node.toElement().attribute( "opened-tab" ) != QString::null )
			{
				italc::inst()->m_openedTabInSideBar = node.toElement().attribute( "opened-tab" ).toInt();
			}
			if( node.toElement().attribute( "wincfg" ) != QString::null )
			{
				m_winCfg = node.toElement().attribute( "wincfg" );
			}

			// get list with all network-interfaces with according IP-addresses
			QMap<QString, QString> ip_addr_map = systemEnvironment::localIPAddresses();
#ifdef DEBUG
			for( int i = 0; i < ip_addr_map.count(); ++i )
			{
				printf( "%s: %s\n", ip_addr_map.keys()[i].ascii(), ip_addr_map.values()[i].ascii() );
			}
#endif
			// iface already set in configuration?
			if( node.toElement().attribute( "net-iface" ) != QString::null && ip_addr_map.count() &&
				ip_addr_map.find( node.toElement().attribute( "net-iface" ) ) != ip_addr_map.end() )
			{
				MASTER_HOST = ip_addr_map[node.toElement().attribute( "net-iface" )];
			}
			else if( ip_addr_map.count() )
			{
				if( ip_addr_map.find( "eth0" ) != ip_addr_map.end() )
				{
					MASTER_HOST = ip_addr_map["eth0"];
				}
				else
				{
					MASTER_HOST = ip_addr_map.values()[0];
				}
			}
			else
			{
				// the guy running iTALC seems to have no network-ifaces configured, not even loopback!!!
				qFatal( "You have no network-interfaces configured!!! iTALC can't run without at least a "
					"local loopback-interface.\n" );
			}

			// if the attr did not exist, we got zero as value, which is not acceptable
			if( m_clientUpdateInterval < 1 )
			{
				m_clientUpdateInterval = 1;
			}
		}
		node = node.nextSibling();
        }
}




void clientManager::loadTree( classRoom * _parent_item, const QDomElement & _parent_element, bool _is_global_config )
{
	for( QDomNode node = _parent_element.firstChild(); node.isNull() == FALSE; node = node.nextSibling() )
	{
		if( node.isElement() == FALSE )
		{
			continue;
		}

		if( node.nodeName() == "classroom"
#warning TODO: remove this compat-code
			 || node.nodeName() == "clientgroup"
							)
		{
			classRoom * cur_item = NULL;
			if( _is_global_config == TRUE )
			{
				// add new classroom
				QString name = node.toElement().attribute( "name" );
				if( _parent_item == NULL )
				{
					cur_item = new classRoom( m_list, name );
				}
				else
				{
					cur_item = new classRoom( _parent_item, name );
				}

				if( node.toElement().attribute( "opened" ) == "yes" )
				{
					cur_item->setOpen( TRUE );
				}

				m_classRooms.push_back( cur_item );
			}

			// recursive build of the tree
			loadTree( cur_item, node.toElement(), _is_global_config );
		}
		else if( node.nodeName() == "client" )
		{
			if( _is_global_config == TRUE )
			{
				// add new client
				QString ip = node.toElement().attribute( "ip" );
				QString mac = node.toElement().attribute( "mac" );
				QString name = node.toElement().attribute( "name" );

				/*client * c = */
				(void) new client( ip, mac, name, _parent_item, node.toElement().attribute( "id" ).toInt() );
			}
			else
			{
				client * c = client::clientFromID( node.toElement().attribute( "id" ).toInt() );
				if( c == NULL )
				{
					continue;
				}
				QPoint pos = c->mapFrom( c->parentWidget()->parentWidget(),
							QPoint( node.toElement().attribute("x").toInt(),
								node.toElement().attribute("y").toInt() ) );
				// workaround for Qt-Bug
				if( pos.x() == 0 )
				{
					pos.setX( 1 );
				}
				c->move( pos );
				//c->parentWidget()->move (node.toElement().attribute("x").toInt(), node.toElement().attribute("y").toInt());
				c->m_rasterX = node.toElement().attribute( "x" ).toInt();
				c->m_rasterY = node.toElement().attribute( "y" ).toInt();
				c->setFixedSize( node.toElement().attribute( "w" ).toInt(),
							node.toElement().attribute( "h" ).toInt() );
				//c->setFixedSize( m_stdWidth, m_stdWidth*3/4 );

				if( node.toElement().attribute( "visible" ) == "yes" )
				{
					c->show();
				}
				else
				{
					c->hide();
				}
			}
		}
        }
}




void clientManager::loadGlobalClientConfig( void )
{
	m_list->clear();

	// read the XML file and create DOM tree
	QFile cfg_file( m_globalClientConfiguration );
	if( !cfg_file.open( IO_ReadOnly ) )
	{
#if QT_VERSION >= 0x030200
		if( splashScreen != NULL )
		{
			splashScreen->close();
		}
#endif
		QMessageBox::warning( this, tr( "No configuration-file found" ), tr( "Could not open configuration file %1.\n"
											"You will have to add at least one "
											"classroom and clients by using the "
											"context-menu in the client-manager."
											).arg( m_globalClientConfiguration ) );
		return;
	}

	if( !m_domTree.setContent( &cfg_file ) )
	{
#if QT_VERSION >= 0x030200
		if( splashScreen != NULL )
		{
			splashScreen->close();
		}
#endif
		QMessageBox::critical( 0, tr( "Error in configuration-file" ), tr( "Error while parsing configuration-file %1.\n"
											"Please edit it. Otherwise you should "
											"delete this file and have to add all "
											"classrooms and clients again."
											).arg( m_globalClientConfiguration ) );
		cfg_file.close();
		return;
	}
	cfg_file.close();


	// get the head information from the DOM
	QDomElement root = m_domTree.documentElement();
	QDomNode node = root.firstChild();

	// create the tree view out of the DOM
	node = root.firstChild();
	while( !node.isNull() )
	{
		if( node.isElement() && node.nodeName() == "body" )
		{
			loadTree( NULL, node.toElement(), TRUE );
			break;
		}
		node = node.nextSibling();
	}
}




void clientManager::loadPersonalConfig( void )
{
	// read the XML file and create DOM tree
	QFile cfg_file( m_personalConfiguration );
	if( !cfg_file.open( IO_ReadOnly ) )
	{
		return;
	}

	if( !m_domTree.setContent( &cfg_file ) )
	{
#if QT_VERSION >= 0x030200
		if( splashScreen != NULL )
		{
			splashScreen->close();
		}
#endif
		QMessageBox::critical( 0, tr( "Error in configuration-file" ), tr( "Error while parsing configuration-file %1.\n"
											"Please edit it. Otherwise you should "
											"delete this file."
											).arg( m_personalConfiguration ) );
		cfg_file.close();
		return;
	}
	cfg_file.close();


	// get the head information from the DOM
	QDomElement root = m_domTree.documentElement();
	QDomNode node = root.firstChild();

	while( !node.isNull() )
	{
		if( node.isElement() && node.nodeName() == "head" )
		{
			getHeaderInformation( node.toElement() );
			break;
		}
		node = node.nextSibling();
	}

	// create the tree view out of the DOM
	node = root.firstChild();
	while( !node.isNull() )
	{
		if( node.isElement() && node.nodeName() == "body" )
		{
			loadTree( NULL, node.toElement(), FALSE );
			break;
		}
		node = node.nextSibling();
	}
}



void clientManager::createActionMenu( QPopupMenu * _m )
{
	QListViewItem * lvi = m_list->firstChild();

	for( int i = 0; i < m_list->childCount(); ++i )
	{
		if( lvi->childCount() )
		{
			createActionMenu( lvi, _m );
		}
		lvi = lvi->nextSibling();
	}
}




void clientManager::createActionMenu( QListViewItem * _p, QPopupMenu * _m )
{
	QPopupMenu * root = new QPopupMenu( _m );
	_m->insertItem( _p->text( 0 ), root );

	QListViewItem * lvi = _p->firstChild();
	classRoomItem * l = NULL;

	for( int i = 0; i < _p->childCount(); ++i )
	{
		if( lvi->childCount() )
		{
			createActionMenu( lvi, root );
		}
		else if( ( l = dynamic_cast<classRoomItem *>( lvi ) ) != NULL )
		{
			QPopupMenu * client_submenu = new QPopupMenu( root );
			l->getClient()->createActionMenu( client_submenu );
			root->insertItem( l->getClient()->fullName(), client_submenu );
		}
		lvi = lvi->nextSibling();
	}
}




void clientManager::createActionMenuForClassRooms( QPopupMenu * _m )
{
	for( unsigned int i = 0; i < m_classRooms.size(); ++i )
	{
		m_classRooms[i]->createActionMenu( _m );
	}
}




bool clientManager::demoRunning( void ) const
{
	QValueVector<client *> vc = visibleClients();

	for( unsigned int i = 0; i < vc.size(); ++i )
	{
		if( vc[i]->demoRunning() )
			return( TRUE );
	}
	return( FALSE );
}




void clientManager::updateClients( void )
{
	QValueVector<client *> clients = visibleClients();

	//unsigned int cur_client = static_cast<unsigned int>( clients.size() * ( rand() * ( 1.0f/RAND_MAX ) ) );

	// loop through all clients
	for( unsigned int cl = 0; cl < clients.size(); ++cl )
	{
/*		if( m_rotating && cl == cur_client )
		{
			// set focus -> client zoomes automatically if zoom-res. != standard-res.
			clients[cl]->setFocus();
		}
		else
		{*/
			// update current client
			clients[cl]->update();
		//}
	}

	//QTimer::singleShot (50, this, SLOT(updateClients()));
	QTimer::singleShot( m_clientUpdateInterval*1000, this, SLOT( updateClients() ) );
}







inline void clientManager::cmdToActiveClients( client::clientCmds _cmd, const QString & _u_data )
{
	QValueVector<client *> clients = visibleClients();
	for( unsigned int cl = 0; cl < clients.size(); ++cl )
	{
		clients[cl]->processCmd( _cmd, _u_data );
		// there're sometimes problems, when action on all clients are started simulanous (especially demo, because all clients
		// connect to the same IVS on master), so add a little delay between each client...
		QTime t;
		t.start();
		while( t.elapsed() < 300 )
		{
		}
	}
}





void clientManager::startDemo( void )
{
	progressInformation pi( tr( "Please wait, while the demo is started." ) );
	cmdToActiveClients( client::START_DEMO );
}




void clientManager::startFullScreenDemo( void )
{
	progressInformation pi( tr( "Please wait, while the demo is started." ) );
	cmdToActiveClients( client::START_FULLSCREEN_DEMO );
}




void clientManager::stopDemo(void )
{
	progressInformation pi( tr( "Please wait, while the demo is stopped." ) );
	cmdToActiveClients( client::STOP_DEMO );
}




void clientManager::powerOnClients( void )
{
	progressInformation pi( tr( "Please wait, while the clients are being turned on." ) );
	cmdToActiveClients( client::POWER_ON );
}




void clientManager::rebootClients( void )
{
	if( QMessageBox::question( this, tr( "Reboot clients" ), tr( "Are you sure want to reboot all visible clients?" ),
					 QMessageBox::Yes, QMessageBox::No ) == QMessageBox::Yes )
	{
		progressInformation pi( tr( "Please wait, while the clients are being rebooted." ) );
		cmdToActiveClients( client::REBOOT );
	}
}




void clientManager::powerOffClients( void )
{
	if( QMessageBox::question( this, tr( "Power off clients" ), tr( "Are you sure want to power off all visible clients?" ),
					QMessageBox::Yes, QMessageBox::No ) == QMessageBox::Yes )
	{
		progressInformation pi( tr( "Please wait, while the clients are being powered off." ) );
		cmdToActiveClients( client::POWER_OFF );
	}
}




void clientManager::lockScreens( void )
{
//	if (QMessageBox::question(this, tr("Lock screen on clients"), tr("Are you sure want to lock the screen on all visible clients?"), QMessageBox::Yes, QMessageBox::No) == QMessageBox::No)
//		return;
	progressInformation pi( tr( "Please wait, while the screens are locked." ) );
	cmdToActiveClients( client::LOCK_X );
}




void clientManager::unlockScreens( void )
{
//	if (QMessageBox::question(this, tr("Unlock screen on all clients"), tr("Are you sure want to unlock the screen on all visible clients?"), QMessageBox::Yes, QMessageBox::No) == QMessageBox::No)
//		return;
	progressInformation pi( tr( "Please wait, while the screens are unlocked." ) );
	cmdToActiveClients( client::UNLOCK_X );
}




void clientManager::killGames( void )
{
	progressInformation pi( tr( "Please wait, while games are killed." ) );
	cmdToActiveClients( client::KILL_GAMES );
}




void clientManager::killBrowsers( void )
{
	if( QMessageBox::question( this, tr( "Kill Mozilla/FireFox on clients" ), tr( "Are you sure want to kill Mozilla/FireFox on all "
						"visible clients?" ), QMessageBox::Yes, QMessageBox::No ) == QMessageBox::Yes )
	{
		progressInformation pi( tr( "Please wait, while browsers are killed." ) );
		cmdToActiveClients( client::KILL_BROWSERS );
	}
}




void clientManager::execCmds( void )
{
	QString cmds;

	cmdInputDialog cmd_input_dialog( cmds, this );
	if( cmd_input_dialog.exec() == QDialog::Accepted && cmds != "" )
	{
		cmdToActiveClients( client::EXEC_CMDS, cmds );
	}
}



void clientManager::sendMessage( void )
{
	QString msg;

	msgInputDialog msg_input_dialog( msg, this );
	if( msg_input_dialog.exec() == QDialog::Accepted && msg != "" )
	{
		progressInformation pi( tr( "Please wait, while the message is being sent." ) );
		cmdToActiveClients( client::SEND_MESSAGE, msg );
	}
}



void clientManager::distributeFile( void )
{
/*	QMessageBox::information (this, tr("Function not implemented yet."), tr("This function is not completely implemented yet. This is why it is disabled at the moment."), QMessageBox::Ok);
	return;*/

	QFileDialog ofd( QDir::home().path(), QString::null, this, "", TRUE );
	ofd.setMode( QFileDialog::ExistingFile );
	ofd.setCaption( client::tr( "Select file to distribute" ) );
	if( ofd.exec() == QDialog::Accepted )
	{
		cmdToActiveClients( client::DISTRIBUTE_FILE, ofd.selectedFile() );
	}
}




void clientManager::collectFiles( void )
{
/*	QMessageBox::information (this, tr("Function not implemented yet."), tr("This function is not completely implemented yet. This is why it is disabled at the moment."), QMessageBox::Ok);
	return;*/

	QString fn;
	textInputDialog file_name_input( client::tr( "Please enter the name(s) of the file(s) to be collected (wildcards are "
							"allowed).\nThe base-directory is HOME/PUBLIC." ), fn, this );
	file_name_input.setCaption( client::tr( "Collect files" ) );

	if( file_name_input.exec() == QDialog::Accepted && fn != "" )
	{
		cmdToActiveClients( client::COLLECT_FILES, fn );
	}
}



#ifdef COMPILE_LINUX

void clientManager::restartX( void )
{
	if( QMessageBox::question( this, tr( "Restart X on clients" ), tr( "Are you sure want to restart X on all visible clients?" ),
					QMessageBox::Yes, QMessageBox::No ) == QMessageBox::Yes )
	{
		cmdToActiveClients( client::RESTART_X );
	}
}

#endif


const int decor_w = 4;
const int decor_h = 22;


void clientManager::optimizeClientSize( void )
{
	QValueVector<client *> vc = visibleClients();
	if( vc.size() )
	{
		const int avail_w = italc::inst()->workspace()->width();
		const int avail_h = italc::inst()->workspace()->height();
		float cw = vc[0]->width()+decor_w;	// add width of decoration
		float ch = vc[0]->height()+decor_h;	// add height of titlebar

		// later we divide by cw, so assume standard-value if zero
		if( static_cast<int>( cw ) == 0 )
		{
			cw = 256.0f;
		}
		// later we divide by ch, so assume standard-value if zero
		if( static_cast<int>( ch ) == 0 )
		{
			ch = 192.0f;
		}
		int x_offset = vc[0]->parentWidget()->pos().x();
		int y_offset = vc[0]->parentWidget()->pos().y();

		for( unsigned int i = 0; i < vc.size(); ++i )
		{
			if( vc[i]->parentWidget()->pos().x() < x_offset )
			{
				x_offset = vc[i]->parentWidget()->pos().x();
			}
			if( vc[i]->parentWidget()->pos().y() < y_offset )
			{
				y_offset = vc[i]->parentWidget()->pos().y();
			}
		}

		float max_rx = 0.0;
		float max_ry = 0.0;
		for( unsigned int i = 0; i < vc.size(); ++i )
		{
			vc[i]->m_rasterX = roundCorrect( ( vc[i]->parentWidget()->pos().x()-x_offset ) / cw );
			vc[i]->m_rasterY = roundCorrect( ( vc[i]->parentWidget()->pos().y()-y_offset ) / ch );
			if( vc[i]->m_rasterX > max_rx )
			{
				max_rx = vc[i]->m_rasterX;
			}
			if( vc[i]->m_rasterY > max_ry )
			{
				max_ry = vc[i]->m_rasterY;
			}
		}
		++max_rx;
		++max_ry;
		// now we have length of col and length of row and can calculate a width and a height (independent from each other)
		// divide available width by max length of rows
		int nw = static_cast<int>( floor( avail_w / max_rx ) );
		// calculate according height
		int nh = (nw-decor_w)*3/4+decor_h;
		// is this height fit max_ry times into available height?
		if( nh*max_ry >= avail_h )
		{
			// no, then divide available height by max length of cols
			nh = static_cast<int>( floor( avail_h / max_ry ) );
			// and calculate according width
			nw = (nh-decor_h)*4/3+decor_w;
		}
		for( unsigned int i = 0; i < vc.size(); ++i )
		{
			vc[i]->setFixedSize( nw-decor_w, nh-decor_h );
			vc[i]->parentWidget()->move( static_cast<int>( vc[i]->m_rasterX*nw ), static_cast<int>( vc[i]->m_rasterY*nh ) );
		}
	}
}




void clientManager::resizeClients( const int _new_width )
{
	QValueVector<client *> vc = visibleClients();

	if( vc.size() )
	{
		const int _new_height = _new_width*3/4;
		float cw = vc[0]->width()+decor_w;	// add width of decoration
		float ch = vc[0]->height()+decor_h;	// add height of titlebar
		// later we divide by cw, so assume standard-value if zero
		if( static_cast<int>( cw ) == 0 )
			cw = 256.0f;
		// later we divide by ch, so assume standard-value if zero
		if( static_cast<int>( ch  )== 0 )
			ch = 192.0f;

		int x_offset = vc[0]->parentWidget()->pos().x();
		int y_offset = vc[0]->parentWidget()->pos().y();

		for( unsigned int i = 0; i < vc.size(); ++i )
		{
			if( vc[i]->parentWidget()->pos().x() < x_offset )
			{
				x_offset = vc[i]->parentWidget()->pos().x();
			}
			if( vc[i]->parentWidget()->pos().y() < y_offset )
			{
				y_offset = vc[i]->parentWidget()->pos().y();
			}
		}

		for( unsigned int i = 0; i < vc.size(); ++i )
		{
			vc[i]->setFixedSize( _new_width, _new_height );
			const int xp = static_cast<int>( ( vc[i]->parentWidget()->pos().x() - x_offset ) / cw *
							( _new_width + decor_w ) ) + x_offset;
			const int yp = static_cast<int>( ( vc[i]->parentWidget()->pos().y() - y_offset ) / ch *
							( _new_height + decor_h ) ) + y_offset;
			vc[i]->parentWidget()->move( xp, yp );
		}
	}
}




void clientManager::increaseClientSize( void )
{
	QValueVector<client *> vc = visibleClients();

	if( vc.size() )
	{
		const int cw = vc[0]->width();
		int i = 0;
		// seek to first width which is greater than current client-width
		while( widths[i] > 0 && cw >= widths[i] )
		{
			++i;
		}
		if( widths[i] > 0 )
		{
			resizeClients( widths[i] );
		}
	}
}




void clientManager::decreaseClientSize( void )
{
	QValueVector<client *> vc = visibleClients();

	if( vc.size() )
	{
		const int cw = vc[0]->width();
		int i = 0;
		// seek to last width
		while( widths[i] > 0 )
		{
			++i;
		}
		--i;
		// seek to first width which is smaller than current client-width
		while( i > 0 && cw <= widths[i] )
		{
			--i;
		}
		if( i >= 0 && widths[i] > 0 )
		{
			resizeClients( widths[i] );
		}
	}
}




void clientManager::updateIntervalChanged( int _value )
{
	m_clientUpdateInterval = _value;
}




void clientManager::closeEvent( QCloseEvent * _ce )
{
	_ce->ignore();
}




#if QT_VERSION >= 0x030200

void clientManager::itemDoubleClicked( QListViewItem * _i )
{
}

void clientManager::itemDoubleClicked( QListViewItem * _i, const QPoint &, int )

#else

void clientManager::itemDoubleClicked( QListViewItem * _i, const QPoint &, int )
{
}
void clientManager::itemDoubleClicked( QListViewItem * _i )

#endif
{
	classRoomItem * cri = dynamic_cast<classRoomItem *>( _i );

	if( cri != NULL )
	{
		if( cri->getClient()->isShown() )
		{
			cri->getClient()->hide();
		}
		else
		{
			cri->getClient()->show();
		}
	}
}




void clientManager::contextMenuRequest( QListViewItem * _i, const QPoint &, int )
{
	QPopupMenu * context_menu = new QPopupMenu( this );

	classRoomItem * cri = dynamic_cast<classRoomItem *>( _i );
	classRoom * cr = dynamic_cast<classRoom *>( _i );

	if( cri != NULL )
	{
		context_menu->insertItem( embed::getIconPixmap( "client_show" ), tr( "Show/hide" ), this, SLOT( showHideClient() ) );
		context_menu->insertItem( embed::getIconPixmap( "client_settings" ), tr( "Edit properties" ), this,
								SLOT( editClientProperties() ) );
		context_menu->insertItem( embed::getIconPixmap( "client_remove" ), tr( "Remove" ), this, SLOT( removeClient() ) );
		context_menu->insertSeparator();

		QPopupMenu * single_clients_submenu = new QPopupMenu( context_menu );

		cri->getClient()->createActionMenu( single_clients_submenu );

		context_menu->insertItem( embed::getIconPixmap( "client" ), tr( "Actions" ), single_clients_submenu );
		context_menu->insertSeparator();
	}
	else if( cr != NULL )
	{
		context_menu->insertItem( embed::getIconPixmap( "classroom_show" ), tr( "Show all clients in classroom" ), this,
						SLOT( showSelectedClassRooms() ) );
		context_menu->insertItem( embed::getIconPixmap( "classroom_closed" ), tr( "Hide all clients in classroom" ), this,
						SLOT( hideSelectedClassRooms() ) );
		context_menu->insertItem( embed::getIconPixmap( "client_settings" ), tr( "Edit name" ), this, SLOT( editClassRoomName() ) );
		context_menu->insertItem( embed::getIconPixmap( "classroom_remove" ), tr("Remove classroom" ), this,
						SLOT( removeClassRoom() ) );

		context_menu->insertSeparator();

		QPopupMenu * actions_for_classroom_submenu = new QPopupMenu( context_menu );

		cr->createActionMenu( actions_for_classroom_submenu, FALSE );

		context_menu->insertItem( embed::getIconPixmap( "classroom_opened" ), tr( "Actions for %1" ).arg( cr->text( 0 ) ),
						actions_for_classroom_submenu );
		context_menu->insertSeparator();
	}
	else
	{ 
		QPopupMenu * actions_for_classroom_submenu = new QPopupMenu( context_menu );

		for( unsigned int i = 0; i < m_classRooms.size(); ++i )
		{
			m_classRooms[i]->createActionMenu( actions_for_classroom_submenu );
		}

		context_menu->insertItem( embed::getIconPixmap( "classroom_closed" ), italc::tr( "Action for whole classroom" ),
						actions_for_classroom_submenu );
		context_menu->insertSeparator();
	}

	context_menu->insertItem( embed::getIconPixmap( "client_add" ), tr( "Add client" ), this, SLOT( addClient() ) );
	context_menu->insertItem( embed::getIconPixmap( "classroom_add" ), tr( "Add classroom" ), this, SLOT( addClassRoom() ) );

	context_menu->exec( QCursor::pos() );

	delete context_menu;
}




QValueVector<classRoomItem *> clientManager::selectedItems( void )
{
	QValueVector<classRoomItem *> vv;
	QListViewItem * lvi = m_list->firstChild();

	for( int i = 0; i < m_list->childCount(); ++i )
	{
		getSelectedItems( lvi, vv );
		lvi = lvi->nextSibling();
	}

	return( vv );
}




void clientManager::getSelectedItems( QListViewItem * _p, QValueVector<classRoomItem *> & _vv, bool _add_all )
{
	QListViewItem * lvi = _p->firstChild();

	for( int i = 0; i < _p->childCount(); ++i )
	{
		if( lvi->childCount() )
		{
			getSelectedItems( lvi, _vv, m_list->isSelected( lvi ) );
		}
		else if( _add_all || ( dynamic_cast<classRoomItem *>( lvi ) != NULL && m_list->isSelected( lvi ) ) )
		{
			_vv.push_back( dynamic_cast<classRoomItem *>( lvi ) );
		}
		lvi = lvi->nextSibling();
	}
}




// slots for client-actions in context-menu
void clientManager::showHideClient( void )
{
	QValueVector<classRoomItem *> si = selectedItems();

	if( si.size() > 0 )
	{
		bool all_shown = TRUE;
		for( unsigned int i = 0; i < si.size(); ++i )
		{
			if( si[i]->getClient()->isShown() == FALSE )
			{
				all_shown = FALSE;
				break;
			}
		}
		for( unsigned int i = 0; i < si.size(); ++i )
		{
			if( all_shown )
			{
				si[i]->getClient()->hide();
			}
			else
			{
				si[i]->getClient()->show();
			}
		}
	}
}




void clientManager::editClientProperties( void )
{
	QValueVector<classRoomItem *> si = selectedItems();

	if( si.size() > 0 )
	{
		for( unsigned int i = 0; i < si.size(); ++i )
		{
			clientPropertiesEdit edit( si[i]->getClient(), si[i]->parent()->text( 0 ), this );
			edit.exec();
		}
	}
}




void clientManager::removeClient( void )
{
	QValueVector<classRoomItem *> si = selectedItems();

	if( si.size() > 0 )
	{
		for( unsigned int i = 0; i < si.size(); ++i )
		{
			si[i]->getClient()->hide();
			m_clientsToRemove.push_back( si[i]->getClient() );
			si[i]->setVisible (FALSE);
		}
	}
}




void clientManager::setStateOfClassRoom( classRoom * _cr, bool _shown )
{
	QListViewItem * cri = _cr->firstChild();

	// If all clients are shown, we hide them all. Otherwise we show all.
	for( int i = 0; i < _cr->childCount(); ++i )
	{
		if( dynamic_cast<classRoomItem *>( cri ) != NULL )
		{
			if( _shown )
			{
				dynamic_cast<classRoomItem *>( cri )->getClient()->show();
			}
			else
			{
				dynamic_cast<classRoomItem *>( cri )->getClient()->hide();
			}
		}
		cri = cri->nextSibling();
	}
}




int clientManager::addClassRoomToQuickSwitchMenu( classRoom * _cr )
{
	return( m_quickSwitchMenu->insertItem( _cr->text( 0 ), _cr, SLOT( switchToClassRoom() ) ) );
}




void clientManager::removeClassRoomFromQuickSwitchMenu( int _id )
{
	m_quickSwitchMenu->removeItem( _id );
}




void clientManager::showSelectedClassRooms( void )
{
	for( unsigned int i = 0; i < m_classRooms.size(); ++i )
	{
		classRoom * cr = NULL;
		if( m_list->isSelected( m_classRooms[i] ) )
		{
			cr = m_classRooms[i];
		}

		if( cr != NULL && cr->childCount() )
		{
			setStateOfClassRoom( cr, TRUE );
		}
	}
}




void clientManager::hideSelectedClassRooms( void )
{
	for( unsigned int i = 0; i < m_classRooms.size(); ++i )
	{
		if( m_list->isSelected( m_classRooms[i] ) && m_classRooms[i]->childCount() )
		{
			setStateOfClassRoom( m_classRooms[i], FALSE );
		}
	}
}




void clientManager::hideAllClassRooms( void )
{
	for( unsigned int i = 0; i < m_classRooms.size(); ++i )
	{
		setStateOfClassRoom( m_classRooms[i], FALSE );
	}
}




void clientManager::editClassRoomName( void )
{
	for( unsigned int i = 0; i < m_classRooms.size(); ++i )
	{
		classRoom * cr = NULL;
		if( m_list->isSelected( m_classRooms[i] ) )
		{
			cr = m_classRooms[i];
		}

		if( cr != NULL )
		{
			QString classroom_name = cr->text( 0 );

			textInputDialog classroom_name_input( tr( "Please enter a new name for classroom \"%1\"."
								).arg( classroom_name ), classroom_name, this );
			classroom_name_input.setCaption( tr( "New name for classroom" ) );
			if( classroom_name_input.exec() == QDialog::Accepted && classroom_name != "" )
			{
				cr->setText( 0, classroom_name );
			}
		}
	}
}




void clientManager::removeClassRoom( void )
{
	for( unsigned int i = 0; i < m_classRooms.size(); ++i )
	{
		classRoom * cr = NULL;
		if( m_list->isSelected( m_classRooms[i] ) )
		{
			cr = m_classRooms[i];
		}
		if( cr != NULL )
		{
			if( QMessageBox::question( this, tr( "Remove classroom" ),
							tr( "Are you sure want to remove classroom \"%1\"?\nAll clients in it will also be "
								"removed!" ).arg( cr->text( 0 ) ), QMessageBox::Yes, QMessageBox::No )
				== QMessageBox::No )
			{
				continue;
			}

			cr->setVisible( FALSE );

			QValueVector<client *> vv;
			getVisibleClients( cr, vv );
			for( unsigned int i = 0; i < vv.size(); ++i )
			{
				vv[i]->hide();
				m_clientsToRemove.push_back( vv[i] );
			}
			m_classRoomsToRemove.push_back( cr );
		}
	}
}




// slots for general actions in context-menu
void clientManager::addClient( void )
{
	if( m_classRooms.size() == 0 )
	{
		QMessageBox::information( this, tr( "Missing classroom" ), tr( "Before you add clients, please add at least one "
											"classroom!" ), QMessageBox::Ok );
		return;
	}

	QString classroom_name = "";

	for( unsigned int i = 0; i < m_classRooms.size(); ++i )
	{
		if( m_list->isSelected( m_classRooms[i] ) )
		{
			classroom_name = m_classRooms[i]->text( 0 );
			break;
		}
	}

	clientPropertiesEdit edit( NULL, classroom_name, this );
	edit.setCaption( tr( "Add client" ) );
	edit.exec();
}




void clientManager::addClassRoom( void )
{
	QString classroom_name = tr( "New classroom" );

	textInputDialog classroom_name_input( tr( "Please enter the name of the classroom you want to create." ), classroom_name, this );
	classroom_name_input.setCaption( tr( "Name for new classroom" ) );

	if( classroom_name_input.exec() == QDialog::Accepted && classroom_name != "" )
	{
		classRoom * cr = NULL;
		for( unsigned int i = 0; i < m_classRooms.size(); ++i )
		{
			if( m_list->isSelected( m_classRooms[i] ) )
			{
				cr = m_classRooms[i];
				break;
			}
		}
		if( cr != NULL )
		{
			m_classRooms.push_back( new classRoom( cr, classroom_name ) );
		}
		else
		{
			m_classRooms.push_back( new classRoom( m_list, classroom_name ) );
		}
	}
}







/*****************************************************************************
 *
 * Class classRoom
 *
 *****************************************************************************/


QPixmap * classRoom::s_classRoomClosedPixmap = NULL;
QPixmap * classRoom::s_classRoomOpenedPixmap = NULL;


classRoom::classRoom( QListViewItem * _parent, const QString & _name ) :
	QListViewItem(_parent, _name ),
	m_pix( NULL ),
	m_qsMenuId( clientManager::inst()->addClassRoomToQuickSwitchMenu( this ) )
{
	initPixmaps();
}




classRoom::classRoom( QListView * _parent, const QString & _name ) :
	QListViewItem( _parent, _name ),
	m_pix( NULL ),
	m_qsMenuId( clientManager::inst()->addClassRoomToQuickSwitchMenu( this ) )
{
	initPixmaps();
}




classRoom::~classRoom()
{
	clientManager::inst()->removeClassRoomFromQuickSwitchMenu( m_qsMenuId );
}




void classRoom::initPixmaps( void )
{
	if( s_classRoomClosedPixmap == NULL )
	{
		s_classRoomClosedPixmap = new QPixmap( embed::getIconPixmap( "classroom_closed" ) );
	}

	if( s_classRoomOpenedPixmap == NULL )
	{
		s_classRoomOpenedPixmap = new QPixmap( embed::getIconPixmap( "classroom_opened" ) );
	}

	setPixmap( s_classRoomClosedPixmap );
}




void classRoom::createActionMenu( QPopupMenu * _m, bool _add_sub_menu )
{
	QPopupMenu * this_classroom_submenu = _m;
	QPopupMenu * orig_root = _m;

	if( _add_sub_menu == TRUE )
	{
		this_classroom_submenu = new QPopupMenu( _m );
		orig_root = this_classroom_submenu;
		_m->insertItem( text( 0 ), this_classroom_submenu );
	}
	QPopupMenu * admin_submenu = new QPopupMenu( this_classroom_submenu );

	for( int i = client::START_FULLSCREEN_DEMO; i < client::CMD_COUNT; ++i )
	{
		if( i >= client::EXEC_CMDS )
			this_classroom_submenu = admin_submenu;

		int id = this_classroom_submenu->insertItem( embed::getIconPixmap( client::s_commands[i].m_icon ),
						client::tr( client::s_commands[i].m_name ), this, SLOT( processCmdOnAllClients( int ) ) );
		this_classroom_submenu->setItemParameter( id, i );

		if( client::s_commands[i].m_insertSep == TRUE )
		{
			this_classroom_submenu->insertSeparator();
		}
	}

	orig_root->insertItem( embed::getIconPixmap( "client_settings" ), tr( "Administation" ), admin_submenu );
}




void classRoom::processCmdOnAllClients( int _cmd )
{
	QValueVector<client *> vv;

	// get all visible clients for this classroom
	clientManager::getVisibleClients( this, vv );

	QString u_data = CONFIRM_NO;

	// may be we have to do something (e.g. confirm) before executing cmd
	switch( static_cast<client::clientCmds>( _cmd ) )
	{
		case client::SEND_MESSAGE:
		{
			msgInputDialog msg_input_dialog( u_data, NULL );
			if( msg_input_dialog.exec() != QDialog::Accepted || u_data == "" )
			{
				return;
			}
			break;
		}
		case client::RESTART_X:
		{
			if( QMessageBox::question( NULL, clientManager::tr( "Restart X on clients" ),
				clientManager::tr( "Are you sure want to restart X on all visible clients?" ),
				QMessageBox::Yes, QMessageBox::No ) == QMessageBox::No )
			{
				return;
			}
			break;
		}
		case client::KILL_BROWSERS:
		{
			if( QMessageBox::question( NULL, clientManager::tr( "Kill Mozilla/FireFox on clients" ),
				clientManager::tr( "Are you sure want to kill Mozilla/FireFox on all visible clients?" ),
				QMessageBox::Yes, QMessageBox::No ) == QMessageBox::No )
			{
				return;
			}
			break;
		}
		case client::EXEC_CMDS:
		{
			cmdInputDialog cmd_input_dialog( u_data, NULL );
			if( cmd_input_dialog.exec() != QDialog::Accepted || u_data == "" )
			{
				return;
			}
			break;
		}
		case client::RUN_X_APP:
		{
			textInputDialog app_input( client::tr( "Please enter the name of the application you want to run "
								"and redirect." ), u_data, NULL );
			app_input.setCaption( client::tr( "Run application..." ) );
			if( app_input.exec() != QDialog::Accepted || u_data == "" )
			{
				return;
			}
			break;
		}
		case client::SSH_LOGIN:
		{
			u_data = "root";
			textInputDialog user_input( client::tr( "Please enter the user-name you want to login with." ), u_data, NULL );
			user_input.setCaption( client::tr( "SSH-Login" ) );
			if( user_input.exec() != QDialog::Accepted || u_data == "" )
			{
				return;
			}
			break;
		}
		case client::REBOOT:
		{
			if( QMessageBox::question( NULL, clientManager::tr( "Reboot clients" ),
				clientManager::tr( "Are you sure want to reboot all visible clients?" ),
				QMessageBox::Yes, QMessageBox::No ) == QMessageBox::No )
			{
				return;
			}
			break;
		}
		case client::POWER_OFF:
		{
			if( QMessageBox::question( NULL, clientManager::tr( "Power off clients" ),
				clientManager::tr( "Are you sure want to power off all visible clients?" ),
				QMessageBox::Yes, QMessageBox::No ) == QMessageBox::No )
			{
				return;
			}
			break;
		}
		default:
			break;
	}

	for( unsigned int i = 0; i < vv.size(); ++i )
	{
		vv[i]->processCmd( static_cast<client::clientCmds>( _cmd ), u_data );
	}
}




void classRoom::setPixmap( QPixmap * _pix )
{
	m_pix = _pix;

	setup();
	widthChanged( 0 );
	invalidateHeight();
	repaint();
}




void classRoom::setOpen( bool _o )
{
	if( _o )
	{
		setPixmap( s_classRoomOpenedPixmap );
	}
	else
	{
		setPixmap( s_classRoomClosedPixmap );
	}

	QListViewItem::setOpen( _o );
}




void classRoom::setup( void )
{
	setExpandable( TRUE );
	QListViewItem::setup();
}




void classRoom::switchToClassRoom( void )
{
	clientManager::inst()->hideAllClassRooms();
	clientManager::inst()->setStateOfClassRoom( this, TRUE );
}










classRoomItem::classRoomItem( client * _client, QListViewItem * _parent, const QString & _name ) :
	QListViewItem( _parent, _name ),
	m_client( _client )
{
	if( s_clientPixmap == NULL )
	{
		s_clientPixmap = new QPixmap( embed::getIconPixmap( "client" ) );
	}
	if( s_clientObservedPixmap == NULL )
	{
		s_clientObservedPixmap = new QPixmap( embed::getIconPixmap( "client_observed" ) );
	}

	m_pix = s_clientPixmap;
	setText( 1, m_client->ip() );
}




classRoomItem::~classRoomItem()
{
	m_client->setClassRoomItem( NULL );
}




void classRoomItem::setObserved( bool _obs )
{
	if( _obs == FALSE )
	{
		setPixmap( s_clientPixmap );
	}
	else
	{
		setPixmap( s_clientObservedPixmap );
	}
}




void classRoomItem::setPixmap( QPixmap * _pix )
{
	m_pix = _pix;

	setup();
	widthChanged( 0 );
	invalidateHeight();
	repaint();
}
