/****************************************************************************
 *                             FileAccObj.cc
 *
 * Author: Matthew Ballance
 * Desc:   
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "FileAccObj.h"
#include <string.h>


/********************************************************************
 * FileAccObj()
 *
 * Default constructor...
 ********************************************************************/
FileAccObj::FileAccObj() :
    fileName("")
{
    fp = 0;
    okay = 1;
    ownFp = 0;
}

/********************************************************************
 * FileAccObj()
 ********************************************************************/
FileAccObj::FileAccObj(const char *filename, FileOpenMode openFlags) :
    fileName(filename), errString(128)
{
    Uint32 idx = 0;
    Char   mode[4];
    okay   = 0;

    openMode = openFlags;

    if (openFlags & OpenMode_Read) {
        mode[idx++] = 'r';
    } 
    if (openFlags & OpenMode_Write) {
        mode[idx++] = 'w';
    }

    mode[idx++] = 'b';

    mode[idx] = 0;

    if (!(fp = fopen(filename, mode))) {
        errString = "Cannot open file ";
        errString += filename;
        errString += "\n";
        return;
    }

    bigEndian = 0;
    currPos = 0;
    okay    = 1;
    ownFp   = 1;
}

/********************************************************************
 * ~FileAccObj()
 ********************************************************************/
FileAccObj::~FileAccObj()
{
    if (fp) {
        fclose(fp);
    }
}

/********************************************************************
 * operator ()
 ********************************************************************/
void FileAccObj::operator () (const char *filename, enum FileOpenMode openFl)
{
    Uint32 idx = 0;
    Char   mode[4];
    okay = 0;

    if (fp) {
        close();
    }

    openMode = openFl;

    if (openFl & OpenMode_Read) {
        mode[idx++] = 'r';
    } 
    if (openFl & OpenMode_Write) {
        mode[idx++] = 'w';
    }

    mode[idx++] = 'b';

    mode[idx] = 0;

    if (!(fp = fopen(filename, mode))) {
        errString = "Cannot open file ";
        errString += filename;
        errString += "\n";
        return;
    }

    bigEndian = 0;
    fileName = filename;

    ownFp   = 1;
    okay    = 1;
}

/********************************************************************
 * operator =
 ********************************************************************/
void FileAccObj::operator = (FileAccObj &rhs)
{
    if (fp && ownFp) {
        close();
    }

    ownFp = 0;
    fp = rhs.fp;
    openMode = rhs.openMode;
    currPos  = rhs.currPos;
    fileName = rhs.fileName;
    bigEndian = rhs.bigEndian;

    okay = 1;
}


/********************************************************************
 * close
 ********************************************************************/
int FileAccObj::close()
{
    if (fp) {
        fclose(fp);
    }

    fp = 0;
}

/********************************************************************
 * print()
 ********************************************************************/
void FileAccObj::print(const char *fmt, ...)
{
    va_list ap;

    va_start(ap, fmt);

    vsprintf(d_buf, fmt, ap);
    write_bytes((Uint8 *)d_buf, strlen(d_buf));

    va_end(ap);
}

/********************************************************************
 * write_bytes()
 ********************************************************************/
int FileAccObj::write_bytes(Uint8 *buf, Uint32 len)
{
    if (!fp) {
        return -1;
    }

    if (!fwrite(buf, len, 1, fp)) {
        return -1;
    } 

    return 0;
}

/********************************************************************
 * read_bytes()
 ********************************************************************/
int FileAccObj::read_bytes(Uint8 *buf, Uint32 len)
{
    if (!fp) {
        return -1;
    }

    if (!fread(buf, len, 1, fp)) {
        return -1;
    } 

    return 0;
}

/********************************************************************
 * write_uint32()
 ********************************************************************/
int FileAccObj::write_uint32(Uint32 val)
{
    Uint8    buf[4];

    if (bigEndian) {
        buf[0] = val >> 24;
        buf[1] = val >> 16;
        buf[2] = val >> 8;
        buf[3] = val;
    } else {
        buf[3] = val >> 24;
        buf[2] = val >> 16;
        buf[1] = val >> 8;
        buf[0] = val;
    }

    if (write_bytes(buf, 4) < 0) {
        return -1;
    }

    return 0;
}

/********************************************************************
 * read_uint32()
 ********************************************************************/
int FileAccObj::read_uint32(Uint32 *val_ptr)
{
    Uint8    buf[4];

    if (read_bytes(buf, 4) < 0) {
        return -1;
    }

    if (bigEndian) {
        *val_ptr = (buf[0] << 24) | (buf[1] << 16) | 
            (buf[2] << 8) | (buf[3]);
    } else {
        *val_ptr = (buf[3] << 24) | (buf[2] << 16) | 
            (buf[1] << 8) | (buf[0]);
    }

    return 0;
}

/********************************************************************
 * write_uint8()
 ********************************************************************/
int FileAccObj::write_uint8(Uint8 val)
{
    if (!fp) {
        return -1;
    }
    fputc(val, fp);

    return 0;
}

/********************************************************************
 * read_uint8()
 ********************************************************************/
int FileAccObj::read_uint8(Uint8 *val)
{
    int ret;

    if (!fp) {
        return -1;
    }

    ret = fgetc(fp);

    *val = ret;

    if (ret == EOF) {
        return 0;
    }

    return 1;
}

/********************************************************************
 * eof()
 ********************************************************************/
int FileAccObj::eof()
{
    return (feof(fp));
}

char FileAccObj::d_buf[8192];

