/****************************************************************************
 *                              MenuMgr.cc
 *
 * Author: Matthew Ballance
 * Desc:   Implements a menu manager. Creates menus from a description
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "MenuMgr.h"
#include "CmdSwitcher.h"
#include "TclListObj.h"
#include "Tokenizer.h"
#include "StrAccObj.h"
#include <string.h>
#include "vector.h"
#include <ctype.h>

#undef DEBUG_MENU_MGR

#define FP stderr
#ifdef DEBUG_MENU_MGR
#define DBG_MSG(x)   fprintf x
#else
#define DBG_MSG(x)
#endif

typedef enum {
    MCS_Menu = 1,
    MCS_MenuButton,
    MCS_MenuBar,
    MCS_ToolBar,
    MCS_Popup
};

/********************************************************************
 * MenuCmd
 *
 * This class holds one level of menu-command.
 ********************************************************************/
class MenuCmd  {

    public:
        Vector<MenuCmd>    d_children;
        Tcl_Obj           *cmdList;
        String             d_instName;

        Tcl_Obj *getCmdList() {return cmdList;}
        MenuCmd(Tcl_Interp *interp, int objc, Tcl_Obj *const objv[]) :
            d_instName(16) {
            cmdList = Tcl_NewListObj(0, 0);

            for (Uint32 i=0; i<objc; i++) {
                Tcl_ListObjAppendElement(interp, cmdList, 
                        Tcl_DuplicateObj(objv[i]));
            }
            d_instName = Tcl_GetString(objv[1]);
        }
};

/********************************************************************
 * MenuDesc
 ********************************************************************/
class MenuDesc {

    public:
        String              d_cmdName;
        String              d_instName;
        Vector<MenuDesc>    d_children;
        Tcl_Obj            *d_optlist;

        MenuDesc() : d_cmdName(16), d_instName(16), d_optlist(0) { ; }
        MenuDesc(char *cmd, char *inst, Tcl_Obj *opt_list) : 
            d_cmdName(cmd), d_instName(inst), d_optlist(opt_list) { ; }

};

class MenuInfo {

    private:
        Tcl_Obj        *cmdList;

    public:
        Tcl_Obj *getCmdList() {return cmdList;}
        MenuInfo(Tcl_Interp *interp, int objc, Tcl_Obj *const objv[]) {
            Uint32 i;

            cmdList = Tcl_NewListObj(0, 0);
            for (i=0; i<objc; i++) {
                Tcl_ListObjAppendElement(interp, cmdList, 
                        Tcl_DuplicateObj(objv[i]));
            }
        }
};

/********************************************************************
 * MenuMgr_InstCmd()
 ********************************************************************/
static int MenuMgr_InstCmd(
        ClientData         clientData,
        Tcl_Interp        *interp,
        int                argc,
        Tcl_Obj           *const objv[])
{
    MenuMgr   *menum = (MenuMgr *)clientData;
    return menum->InstCmd(argc-1, &objv[1]);
}

/********************************************************************
 * Menu-element commands
 ********************************************************************/
typedef enum {
    MLC_MenuBar = 1,
    MLC_ButtonBar,
    MLC_Separator,
    MLC_Menu,
    MLC_MenuButton,
    MLC_CmdButton,
    MLC_ToolBar,
    MLC_Popup,
    MLC_Cmd,
    MLC_Combo,
    MLC_Button,
    MLC_Cascade,
    MLC_Check,
    MLC_RadioBox,
    MLC_Radio,
    MLC_Script
};

static CmdSwitchStruct   ml_cmds[] = {
    { "menubar",     MLC_MenuBar    },
    { "buttonbar",   MLC_ButtonBar  },
    { "separator",   MLC_Separator  },
    { "menu",        MLC_Menu       },
    { "menubutton",  MLC_MenuButton },
    { "toolbar",     MLC_ToolBar    },
    { "cmdbutton",   MLC_CmdButton  },
    { "popup",       MLC_Popup      },
    { "command",     MLC_Cmd        },
    { "combo",       MLC_Combo      },
    { "button",      MLC_Button     },
    { "cascade",     MLC_Cascade    },
    { "check",       MLC_Check      },
    { "radiobox",    MLC_RadioBox   },
    { "radio",       MLC_Radio      },
    { "script",      MLC_Script     },
    { "",            0              }
};

/********************************************************************
 * MenuMgr()
 ********************************************************************/
MenuMgr::MenuMgr(
        Tcl_Interp *interp,
        Char         *inst_name) : 
    instName(inst_name), pathTmp(128), groupTmp(128), dlgPath(128),
    d_parseBuf(512), packList(interp)
{
    Int32    idx;
    LogRegionType    *lrt;

    this->interp = interp;

    loadMode    = 0;
    createMode  = 0;
    menuBarMode = 0;


    Tcl_InitHashTable(&hashTable, TCL_STRING_KEYS);

    Tcl_CreateObjCommand(interp, inst_name, MenuMgr_InstCmd, this, 0);

    if (LogMgr::FindRegionType("MenuMgr", &lrt) < 0) {
        Tcl_AppendResult(interp, "cannot find MenuMgr Log Region", 0);
        return;
    }

    LogDestination *ld = new LogDestination();

    log = new LogInstance(lrt, "menu_mgr");
    LogMgr::AddInstance(log);

    log_low(log, LogMgr::DebugLevel_Low);
    log_med(log, LogMgr::DebugLevel_Med);
    log_high(log, LogMgr::DebugLevel_High);

    ld->setLogLevel(LogMgr::DebugLevel_Off);
    log->AddDestination(ld);
}

/********************************************************************
 * getName
 ********************************************************************/
Char *MenuMgr::getName(void)
{
    return instName.value();
}

/********************************************************************
 * ProcessLabel()
 *
 * Processes a label string looking for an accelerator char (&)
 *
 * NOTE :: It is assumed that -text or -label has already been 
 * appended to the cmd_list. The first thing we append is the label
 * name...
 ********************************************************************/
int MenuMgr::ProcessLabel(
        Tcl_Interp        *interp,
        TclListObj        &cmd_list,
        Tcl_Obj           *string)
{
    return ProcessLabel(interp, cmd_list, Tcl_GetString(string));
} 

int MenuMgr::ProcessLabel(
        Tcl_Interp        *interp,
        TclListObj        &cmd_list,
        Char              *ch_str)
{
    char  buf[128], *ptr;
    Uint32 len = strlen(ch_str);

    if ((ptr = strchr(ch_str, '&'))) {
        if (ptr > ch_str) {
            memcpy(buf, ch_str, ptr-ch_str);
            buf[ptr-ch_str] = 0;
            strcat(buf, ptr+1);
        } else {
            strcpy(&buf[ptr-ch_str], ptr+1);
        }

        cmd_list << buf;
        cmd_list << "-underline" << (ptr-ch_str);
    } else {
        cmd_list << ch_str;
    }

    return TCL_OK;
}

/********************************************************************
 * RemoveAmp()
 ********************************************************************/
char *MenuMgr::RemoveAmp(char *buf, char *str)
{
    int idx = 0;

    while (*str) {
        if (*str != '&') {
            buf[idx++] = *str;
        } 
        str++;
    }

    buf[idx] = 0;

    return buf;
}

/********************************************************************
 * FindList()
 ********************************************************************/
Tcl_Obj *MenuMgr::FindList(char *spec_p)
{
    Tcl_Obj          *ret = 0, *cmdList, *tmp;
    static char       spec[1024];
    char             *ptr;
    char             *menu_name;
    Tcl_HashEntry    *entry;
    MenuInfo         *info;
    int               llen;

    strcpy(spec, spec_p);

    /**** First, find the menu-name (first path element) *****/
    if (!(ptr = strchr(spec, '.'))) {
        goto find_exit;
    }
    *ptr = 0;
    ptr++;
    menu_name = spec;

    /**** See if this menu exists ****/
    if (!(entry = Tcl_FindHashEntry(&hashTable, menu_name))) {
        Tcl_AppendResult(interp, "no menu named ", menu_name, 0);
        goto find_exit;
    }

    /**** Send the last element of the cmdList to the helper. This is
     **** because the hashEntry was named by the <id> on this list
     ****/
    llen = 0;

find_exit:
    return ret;
}

/********************************************************************
 * Identify()
 *
 * Turns a string into an identifier... Removes:
 * - Removes '&'
 * - Replaces ' ' with '_'
 * - Ensures first char is a lower-case alpha char
 ********************************************************************/
char *MenuMgr::Identify(char *id_str)
{
    static char  buf[1024];
    Uint32 idx = 0, o_idx = 0;

    DBG_MSG((FP, "Identify %s\n", id_str));

    while (id_str[idx]) {
        if (id_str[idx] == '&') {
            idx++;
        } else if (id_str[idx] == ' ' || id_str[idx] == '.') {
            buf[o_idx++] = '_';
            idx++;
        } else {
            buf[o_idx++] = id_str[idx++];
        }
    }

    buf[o_idx] = 0;

    buf[0] = tolower(buf[0]);

    DBG_MSG((FP, "Identify => %s\n", buf));

    return buf;
}

typedef enum {
    MIO_Text = 1,
    MIO_Label,
    MIO_Side,
    MIO_HelpText,
    MIO_NumOpts
} MenuItemOpts;

static CmdSwitchStruct mi_opts[] = {
    { "-text",                MIO_Text      },
    { "-label",               MIO_Label     },
    { "-side",                MIO_Side      },
    { "-helptext",            MIO_HelpText  },
    { "",                     0             }
};

/********************************************************************
 * expand_string()
 ********************************************************************/
char *MenuMgr::expand_string(Tcl_Interp *interp_p, char *str)
{
    Uint32         idx = 0, len;
    char          *start, var[64], *val;
    Tcl_HashEntry *entry;

    while (*str) {
        if (*str == '$') {
            /**** Scan to the end of the string ****/
            str++;
            start = str;
            while ((!isspace(*str) && *str != '}' && *str != '"' && *str)) {
                str++;
            }
            memcpy(var, start, str-start);
            var[str-start] = 0;

            if ((entry = Tcl_FindHashEntry(&d_varTable, var))) {
                val = (char *)Tcl_GetHashValue(entry);
                len = strlen(val);
                log_high("    d_varBuf[idx] = %c ; d_varBuf[idx-1] = %c\n", 
                        d_varBuf[idx], d_varBuf[idx-1]);

                memcpy(&d_varBuf[idx], val, len);
                (&d_varBuf[idx])[len] = 0;

                log_high("    after expand - d_varBuf = %s\n", d_varBuf);
                idx += len;
                d_varBuf[idx++] = ' ';
            } else {
                fprintf(stderr, "Menu Error: No variable \"%s\"\n", var);
            }
        } else {
            d_varBuf[idx++] = *str;
        }

        if (!(*str)) {
            break;
        }
        str++;
    }

    d_varBuf[idx] = 0;
    return d_varBuf;
}

/********************************************************************
 * ParseMenuFile()
 ********************************************************************/
MenuDesc *MenuMgr::ParseMenuFile(char *filename)
{
    FileAccObj         f;
    Tokenizer          t;
    char              *tok;
    MenuDesc          *desc;
    Tcl_HashEntry     *hashEntry;
    Int32              itmp;

    log_high.enter("ParseMenuFile(%s)\n", filename);

    f(filename, FileAccObj::OpenMode_Read);

    if (!f.okay) {
        Tcl_AppendResult(interp, f.get_error(), 0);
        fprintf(stderr, "ERROR: %s\n", f.get_error());
        return 0;
    }
    t(&f);

    t.nextToken();

    while ((tok = t.getToken())) {
        log_high("token = %s\n", tok);
        desc = ParseSpec(t);

        log_high("Adding instance \"%s\"\n",
                desc->d_instName.value());

        hashEntry = Tcl_CreateHashEntry(&hashTable, 
                desc->d_instName.value(), &itmp);
        Tcl_SetHashValue(hashEntry, desc);
    }

    log_high.leave("ParseMenuFile(%s)\n", filename);

    return desc;
}

/********************************************************************
 * ParseSpec()
 *
 * All lines are:
 * <cmd> <inst> ?(list)? ?{ sub }?
 *
 * Entry:
 *     - must be positioned on first token.
 *
 * Exit:
 *     - will be on:
 *         - next cmd token
 *         - '}', if leaving a sub-level
 *         - null
 ********************************************************************/
MenuDesc *MenuMgr::ParseSpec(Tokenizer &t)
{
    char      *cmd, *inst, *b, *o, *ov;
    Tcl_Obj   *opt_list;
    MenuDesc  *c, *p = 0;

    log_high.enter("ParseSpec()\n");

    if (!(cmd = t.getToken()) || !strcmp(cmd, "}")) {
        return 0;
    }

    p = new MenuDesc();
    if (!(cmd = t.getToken())) {
        goto parse_exit;
    }
    p->d_cmdName = cmd;

    if (!(inst = t.nextToken())) {
        goto parse_exit;
    }
    p->d_instName = inst;

    /**** Process any options list ****/
    if (!(b = t.nextToken())) { goto parse_exit; }

    opt_list = Tcl_NewListObj(0, 0);
    p->d_optlist = opt_list;

    log_high("new desc: cmd=%s inst=%s\n", 
            p->d_cmdName.value(), p->d_instName.value());

    if (b[0] == '(') {
        log_high.enter("Specified options\n");
        while ((o=t.nextToken()) && o[0] != ')') {

            Tcl_ListObjAppendElement(interp, opt_list, 
                    Tcl_NewStringObj(o, -1));
            log_high("opt=%s\n", o);

            if (!(ov = t.nextToken())) { 
                p = 0;
                goto parse_exit;
            }
            if (ov[0] == ')') {
                fprintf(stderr, "Expecting option value ; got ')'\n");
                /**** mismatched option-value ****/
                p = 0;
                goto parse_exit;
            }

            log_high("optv=%s\n", ov);

            Tcl_ListObjAppendElement(interp, opt_list, 
                    Tcl_NewStringObj(ov, -1));
        }
        if (!o) { 
            p = 0;
            goto parse_exit;
        }

        log_high.leave("Specified options\n");
        t.nextToken();
    } 

    /**** See if we must recurse... ****/
    if ((cmd = t.getToken()) && cmd[0] == '{') {
        t.nextToken();
        do {
            if ((c = ParseSpec(t))) {
                p->d_children.append(c);

                /**** On exit, the token will either be '}' or the
                 **** next cmd
                 ****/
            }
            cmd = t.getToken();
            log_high("after ParseSpec(%s) => token=%s\n",
                    (c)?c->d_instName.value():"null", cmd);
        } while (c && cmd && cmd[0] != '}');
        if (cmd[0] == '}') {
            t.nextToken();
        }
    }

parse_exit:
    log_high.leave("ParseSpec(%s)\n", (p)?"OK":"ERROR");
    return p;
}

/********************************************************************
 * RealizeScriptItem()
 *
 ********************************************************************/
int MenuMgr::RealizeScriptItem(String &pathTmp, MenuDesc *menu)
{
    Int32         opt, i, limit;
    Tcl_Obj      *opt_list = menu->d_optlist;
    Tcl_Obj      *opt_o;
    Tcl_Obj      *script = 0;
    TclListObj    cmd_obj(interp);

    Tcl_ListObjLength(interp, opt_list, &limit);

    for (i=0; i<limit; i++) {
        const char *opt;
        Tcl_ListObjIndex(interp, opt_list, i, &opt_o);

        opt = Tcl_GetString(opt_o);

        if (!strcmp(opt, "-script")) {
            /*** Found the script arg... ***/
            i++;
            Tcl_ListObjIndex(interp, opt_list, i, &script);
        } else {
            fprintf(stderr, "ERROR: Unknown script-menu option \"%s\"\n", opt);
            return TCL_ERROR;
        }
    }

    if (!script) {
        fprintf(stderr, "ERROR: no 'script' specified\n");
        return TCL_ERROR;
    }


    /*** Okay, want to run the script... ***/
    cmd_obj << pathTmp << "add_script" << menu->d_instName.value();
    cmd_obj.append_elems(script);

    int ret = Tcl_EvalObjEx(interp, cmd_obj.end(), TCL_EVAL_GLOBAL);

    if (ret != TCL_OK) {
        fprintf(stderr, "Error while processing script: %s\n",
                Tcl_GetStringResult(interp));
        return TCL_ERROR;
    }

    return TCL_OK;
}

/********************************************************************
 * ProcessOptions()
 ********************************************************************/
int MenuMgr::ProcessOptions(
        TclListObj        &cmd_list,
        MenuDesc          *menu,
        Uint32             itemType,
        Uint32             in_flags)
{
    Int32    opt, i, limit;
    Tcl_Obj *opt_o, *opt_v, *opt_list = menu->d_optlist;
    Uint32   specified = 0;
    char    *tmp;

    Tcl_ListObjLength(interp, opt_list, &limit);

    for (i=0; i<limit; i++) {
        Tcl_ListObjIndex(interp, opt_list, i, &opt_o);

        opt = CmdSwitch(mi_opts, Tcl_GetString(opt_o));
        switch (opt) {
            case MIO_Label:
            case MIO_Text:
            case MIO_HelpText:
                if (itemType == MLC_CmdButton) {
                    cmd_list << "-helptext";
                } else {
                    cmd_list << "-label";
                }

                i++;
                Tcl_ListObjIndex(interp, opt_list, i, &opt_v);
                if (itemType == MLC_CmdButton) {
                    cmd_list << Tcl_GetString(opt_v);
                } else {
                    ProcessLabel(interp, cmd_list, opt_v);
                }
                specified |= OPT_TEXT;
                break;

            case MIO_Side:
                if (menuBarMode && itemType == MLC_MenuButton) {
                    packList << opt_o;
                    i++;
                    Tcl_ListObjIndex(interp, opt_list, i, &opt_v);
                    packList << opt_v;
                }
                break;

            default:
                /*** Just cat onto the cmd... ****/
                cmd_list << opt_o;
                i++;

                Tcl_ListObjIndex(interp, opt_list, i, &opt_v);

                tmp = expand_string(interp, Tcl_GetString(opt_v));
                log_high("expanded string = %s ; orig = %s\n",
                        tmp, Tcl_GetString(opt_v));
                cmd_list << tmp;
                break;
        }
    }

    /**** If not specified, then use the id ****/
    if (!(specified & OPT_TEXT)) {
        if (itemType == MLC_CmdButton) {
            cmd_list << "-helptext" << menu->d_instName.value();
        } else {
            cmd_list << "-label";
            ProcessLabel(interp, cmd_list, menu->d_instName.value());
        }
    }

    return TCL_OK;
}

typedef enum {
    CBO_Image,
    CBO_HelpText
} CmdButtonOpts;

static CmdSwitchStruct cmdb_opts[] = {
    { "-image",                  CBO_Image     },
    { "-helptext",               CBO_HelpText  },
    { "",                        0             }
};

/********************************************************************
 * RealizeMenu()
 ********************************************************************/
int MenuMgr::RealizeMenu(MenuDesc *menu)
{
    Int32       pre_idx = 0, ret, cmd, itmp, i;
    bool        exec = false;
    MenuCmd    *cCmd;
    Tcl_Obj    *res_obj = 0, *opt_v;
    TclListObj  cmd_obj(interp);
    char        buf[256];

    pre_idx = pathTmp.length();
    cCmd = currCmd;

    cmd = CmdSwitch(ml_cmds, menu->d_cmdName.value());

    switch (cmd) {

        /**** menubar <name> {contents} ****/
        case MLC_MenuBar:
            menuBarMode = 1;
            cmd_obj << "menu" << pathTmp << "-tearoff" << 0;

            ret = Tcl_EvalObjEx(interp, cmd_obj.end(), TCL_EVAL_GLOBAL);
            if (ret != TCL_OK) { 
                fprintf(stderr, "ERROR");
                return TCL_ERROR; 
            }
            exec = true;
            break;

        /**** toolbar <name> {contents} ****/
        case MLC_ToolBar:
            menuBarMode = 1;

            cmd_obj << "IviToolbar::IviToolbar" << pathTmp;

            ret = Tcl_EvalObjEx(interp, cmd_obj.end(), TCL_EVAL_GLOBAL);
            if (ret != TCL_OK) { 
                fprintf(stderr, "ERROR: %s\n", Tcl_GetStringResult(interp));
                return TCL_ERROR; 
            }

            exec = true;
            break;

        /**** menubutton <id> ?options? {contents} ****/
        case MLC_MenuButton:
            if (menuBarMode) {
                /**** Now, we just add a cascade to the root menubar...
                 ****/
                cmd_obj << pathTmp << "add" << "cascade" << "-label" 
                    << menu->d_instName.value();

                pathTmp += ".";
                pathTmp += Identify(menu->d_instName.value());

                cmd_obj << "-menu" << pathTmp;
            }

            /**** Copy the various options... ****/
            ProcessOptions(cmd_obj, menu, cmd, 0);

            ret = Tcl_EvalObjEx(interp, cmd_obj.end(), TCL_EVAL_GLOBAL);
            if (ret != TCL_OK) { return TCL_ERROR; }

            cmd_obj << "menu" << pathTmp << "-tearoff" << 0;
            ret = Tcl_EvalObjEx(interp, cmd_obj.end(), TCL_EVAL_GLOBAL);
            if (ret != TCL_OK) { return TCL_ERROR; }

            exec = true;
            break;

        case MLC_Combo:
            cmd_obj << pathTmp << "add_combo" << menu->d_instName.value();
            cmd_obj.append_elems(menu->d_optlist);

            ret = Tcl_EvalObjEx(interp, cmd_obj.end(), TCL_EVAL_GLOBAL);

            if (ret != TCL_OK) {
                fprintf(stderr, "ERROR: %s\n",
                        Tcl_GetStringResult(interp));
                return TCL_ERROR;
            }
            break;

        /************************************************************
         * Script
         *
         * script <inst> ( -script "Script" )
         *
         * 
         ************************************************************/
        case MLC_Script: 
            RealizeScriptItem(pathTmp, menu);
            break;

        case MLC_Popup:
            cmd_obj << "Popup::Popup" << pathTmp;
            ret = Tcl_EvalObjEx(interp, cmd_obj.end(), TCL_EVAL_GLOBAL);
            if (ret != TCL_OK) { return TCL_ERROR; }

            {
                Tcl_HashEntry *entry = Tcl_CreateHashEntry(
                        &d_varTable, "menu", &itmp);
                Tcl_SetHashValue(entry, strdup(pathTmp.value()));
            }

            exec = true;
            break;

        /**** command id ?options? ****/
        case MLC_Cmd:
            cmd_obj << pathTmp << "add" << "command";
            ProcessOptions(cmd_obj, menu, cmd, 0);

            {
                Tcl_Obj *cmd = cmd_obj.end();

                ret = Tcl_EvalObjEx(interp, cmd, TCL_EVAL_GLOBAL);
            }
            if (ret != TCL_OK) { return TCL_ERROR; }
        break;

        /**** Toolbar button ****/
        case MLC_CmdButton:
            cmd_obj << pathTmp << "add_button" << menu->d_instName.value();

            ProcessOptions(cmd_obj, menu, cmd, 0);

            ret = Tcl_EvalObjEx(interp, cmd_obj.end(), TCL_EVAL_GLOBAL);
            if (ret != TCL_OK) { 
                fprintf(stderr, "Error: %s\n", Tcl_GetStringResult(interp));
                return TCL_ERROR; 
            }
            break;

        case MLC_Separator:
            if (d_toplevItemType == MCS_MenuBar) {
                cmd_obj << pathTmp << "add" << "separator";

                { 
                    Tcl_Obj *cmd = cmd_obj.end();

                    ret = Tcl_EvalObjEx(interp, cmd, TCL_EVAL_GLOBAL);
                }
                if (ret != TCL_OK) { return TCL_ERROR; }
            } else {
                cmd_obj << pathTmp << "add_sep" << menu->d_instName.value();
                ret = Tcl_EvalObjEx(interp, cmd_obj.end(), TCL_EVAL_GLOBAL);
                if (ret != TCL_OK) { 
                    fprintf(stderr, "Error: %s\n", Tcl_GetStringResult(interp));
                    return TCL_ERROR; 
                }
            }
            break;

        /**** button id ?options? ****/
        case MLC_Button:
            pathTmp += ".";
            pathTmp += menu->d_instName.value();

            cmd_obj << "button" << pathTmp << "-relief" << "flat" << 
                "-borderwidth" << 0;

            ret = Tcl_EvalObjEx(interp, cmd_obj.end(), TCL_EVAL_GLOBAL);
            if (ret != TCL_OK) { return TCL_ERROR; }

            packList << pathTmp;
            break;

         /**** cascade id ?options? {contents} ****/
        case MLC_Cascade:
            cmd_obj << pathTmp << "add" << "cascade" << "-menu";
            pathTmp += ".";
            pathTmp += Identify(menu->d_instName.value());
            cmd_obj << pathTmp;

            ProcessOptions(cmd_obj, menu, cmd, 0);

            ret = Tcl_EvalObjEx(interp, cmd_obj.end(), TCL_EVAL_GLOBAL);
            if (ret != TCL_OK) { return TCL_ERROR; }

            cmd_obj << "menu" << pathTmp << "-tearoff" << 0;
            ret = Tcl_EvalObjEx(interp, cmd_obj.end(), TCL_EVAL_GLOBAL);
            if (ret != TCL_OK) { return TCL_ERROR; }

            exec = true;
            break;

        default:
            Tcl_AppendResult(interp, "unknown file cmd ", 
                menu->d_cmdName.value(), 0);
            return TCL_ERROR;
            break;
    }

    if (exec) {
        /**** Execute the children of this element ****/
        for (i=0; i<menu->d_children.length(); i++) {
            MenuDesc *subm = menu->d_children.idx(i);

            ret = RealizeMenu(subm);
            if (ret != TCL_OK) {
                return TCL_ERROR;
            }
        }

        if (cmd == MLC_ToolBar) {
            Tcl_Obj *el;
            char    *pack_order = "left";

            for (i=0; i<packList.length(); i++) {
                el = packList[i];

                if (String::equal(Tcl_GetString(el), "-side")) {
                    i++;
                    pack_order = Tcl_GetString(packList[i]);
                    DBG_MSG((FP, "Changing pack-order to %s\n", pack_order));
                    continue;
                }

                cmd_obj << "pack";
                cmd_obj << el;
                cmd_obj << "-side" << pack_order;

                ret = Tcl_EvalObjEx(interp, cmd_obj.end(), TCL_EVAL_GLOBAL);
                if (ret != TCL_OK) {
                    return TCL_ERROR;
                }
            }

            el = packList.end();
        }

        if (cmd == MLC_Popup) {
            Tcl_HashEntry *entry = Tcl_FindHashEntry(&d_varTable, "menu");
            free((char *)Tcl_GetHashValue(entry));
            Tcl_DeleteHashEntry(entry);
        }
    }

    pathTmp.setLen(pre_idx);
    currCmd = cCmd;

    return TCL_OK;
}

/********************************************************************
 * LoadFileHelper()
 ********************************************************************/
int MenuMgr::LoadFileHelper(
        int        objc,
        Tcl_Obj   *const objv[])
{
    return TCL_OK;
}

typedef enum {
    MMC_Create = 1,
    MMC_Load,
    MMC_Info,
    MMC_Append,
    MMC_NumCmds
};

static CmdSwitchStruct create_subcmds[] = {
    {"create",              MMC_Create},
    {"load",                MMC_Load},
    {"info",                MMC_Info},
    {"append",              MMC_Append},
    {"",                    0}
};


static CmdSwitchStruct menumgr_cmds[] = {
    {"menu",              MCS_Menu      },
    {"menubutton",        MCS_MenuButton},
    {"toolbar",           MCS_ToolBar   },
    {"menubar",           MCS_MenuBar   },
    {"popup",             MCS_Popup     },
    {"",                  0             }
};

/********************************************************************
 * InstCmd()
 ********************************************************************/
int MenuMgr::InstCmd(
        int           argc, 
        Tcl_Obj      *const objv[])
{
    Int32           cmd, ret, subcmd, i, x, itmp;
    Tcl_HashEntry  *hashEntry;
    MenuInfo       *dInfo;
    Tcl_Obj        *unset = 0;
    TclListObj      cmd_obj(interp);
    MenuDesc       *menu;

    currCmd = 0;

    DBG_MSG((FP, "MenuMgr Cmd %s\n", Tcl_GetString(objv[0])));

    cmd = CmdSwitch(create_subcmds, Tcl_GetString(objv[0]));
    switch (cmd) {

        /**** $menum create [menubar|menubutton|toolbar] 
         ****        <Name> <PathName> ?options? 
         ****/
        case MMC_Create:
            if (argc < 4) {
                Tcl_AppendResult(interp, "too few args", 0);
                return TCL_ERROR;
            }

            if (argc > 4) {
                for (i=4; i<argc; i++) {
                    if (String::equal(Tcl_GetString(objv[i]), "-var")) {
                        i++;
                        TclListObj  it(interp, objv[i]);
                        /**** Loop through each pair in the attached 
                         **** list...
                         ****/
                        unset = objv[i];
                        if ((it.length() % 2)) {
                            Tcl_AppendResult(interp, 
                                    "var-list has un-even length", 0);
                            return TCL_ERROR;
                        }

                        Tcl_InitHashTable(&d_varTable, TCL_STRING_KEYS);
                        for (x=0; x<it.length(); x+=2) {
                            Tcl_HashEntry *entry;

                            entry = Tcl_CreateHashEntry(&d_varTable, 
                                    Tcl_GetString(it[x]), &itmp);
                            Tcl_SetHashValue(entry, Tcl_GetString(it[x+1]));
                        }
                    }
                }
            }

            /**** Try to find the specified item... ****/
            if (!(hashEntry = Tcl_FindHashEntry(&hashTable, 
                            Tcl_GetString(objv[2])))) {
                Tcl_AppendResult(interp, "no menu ", 
                        Tcl_GetString(objv[2]), 0);
                return TCL_ERROR;
            }
            menu = (MenuDesc *)Tcl_GetHashValue(hashEntry);

            subcmd = CmdSwitch(menumgr_cmds, Tcl_GetString(objv[1]));
            d_toplevItemType = subcmd;
            switch (subcmd) {
                case MCS_Menu:
                    break;

                case MCS_MenuButton:
                    break;

                case MCS_ToolBar:
                    break;

                case MCS_MenuBar:
                    break;

                case MCS_Popup:
                    break;

                default:
                    Tcl_AppendResult(interp, "unknown sub-cmds ", 
                            Tcl_GetString(objv[1]), 0);
                    return TCL_ERROR;
                    break;
            }

            createMode = 1;
            pathTmp = Tcl_GetString(objv[3]);

            ret = RealizeMenu(menu);

            Tcl_SetObjResult(interp, Tcl_DuplicateObj(objv[3]));

            if (unset) {
                Tcl_DeleteHashTable(&d_varTable);
            }
            break;

        /**** $menum load <File> ****/
        case MMC_Load:
            if (argc < 2) {
                Tcl_AppendResult(interp, "too few args", 0);
                return TCL_ERROR;
            }

            if (!ParseMenuFile(Tcl_GetString(objv[1]))) {
                return TCL_ERROR;
            }
            break;

        /************************************************************
         * Info
         ************************************************************/
        case MMC_Info:
            FindList(Tcl_GetString(objv[1]));
            break;

        /************************************************************
         * Append
         *
         * append <PathElement> ...
         ************************************************************/
        case MMC_Append:
            return AppendCmd(argc, objv);
            break;

        default:
            Tcl_AppendResult(interp, "unknown sub-cmd ", 
                    Tcl_GetString(objv[0]), 0);
            return TCL_ERROR;
            break;
    }

    return TCL_OK;
}

/********************************************************************
 * GlobObjv()
 ********************************************************************/
void MenuMgr::GlobObjv(String &buf, int objc, Tcl_Obj *const objv[])
{
    Uint32 i, idx, has_wht;
    char   *str;

    buf.setLen(0);

    for (i=0; i<objc; i++) {
        has_wht=0;
        str = Tcl_GetString(objv[i]);

        idx = 0;
        while (str[idx]) {
            if (isspace(str[idx++])) {
                has_wht=1;
                break;
            }
        }

        if (has_wht) {
            buf += "\"";
            buf += str;
            buf += "\"";
        } else {
            buf += str;
        }

        buf += " ";
    }
}

/********************************************************************
 * FindNode()
 ********************************************************************/
MenuDesc *MenuMgr::FindNode(char *nodeName)
{
    char       *first=nodeName, *last;
    char        tmp[128], tmp2[128];
    int         len = strlen(nodeName);
    MenuDesc   *desc, *next;

    while (first) {
        if (!(last = strchr(first, '.'))) {
            strcpy(tmp, first);
        } else {
            memcpy(tmp, first, last-first);
            tmp[last-first] = 0;
        }

        if (first == nodeName) {
            Tcl_HashEntry *entry = Tcl_FindHashEntry(&hashTable, tmp);
            if (!entry) {
                Tcl_AppendResult(interp, "cannot find Menu named ", tmp, 0);
                return 0;
            }
            desc = (MenuDesc *)Tcl_GetHashValue(entry);
        } else {
            Uint32 i;
            for (i=0; i<desc->d_children.length(); i++) {
                next = desc->d_children.idx(i);
                RemoveAmp(tmp2, next->d_instName.value());
                if (String::equal(tmp, tmp2)) {
                    break;
                }
            }

            if (i == desc->d_children.length()) {
                Tcl_AppendResult(interp, "cannot find menu ", tmp, 0);
                return 0;
            } 
            desc = next;
        }

        if (last && *last) {
            first = (last+1);
        } else {
            first = 0;
        }
    }

    return desc;
}

/********************************************************************
 * AppendCmd()
 * append <spec> ...
 ********************************************************************/
int MenuMgr::AppendCmd(int objc, Tcl_Obj *const objv[])
{
    MenuDesc *desc, *parent;
    Tokenizer t;
    StrAccObj sacc;

    if (objc < 3) {
        Tcl_AppendResult(interp, "too few args", 0);
        return TCL_ERROR;
    }

    GlobObjv(d_parseBuf, objc-2, &objv[2]);
    sacc(d_parseBuf.value());
    t(&sacc);
    t.nextToken();

    if (!(desc = ParseSpec(t))) {
        return TCL_ERROR;
    }

    /**** Now, find the parent ****/
    if (!(parent = FindNode(Tcl_GetString(objv[1])))) {
        return TCL_ERROR;
    }

    /**** Now, append the new description to the parent indicated ****/
    parent->d_children.append(desc);

    return TCL_OK;
}

static MenuMgr *Globl_MenuMgr = 0;

/********************************************************************
 * MenuMgr_TclCmd()
 ********************************************************************/
static int MenuMgr_TclCmd(
        ClientData        clientData,
        Tcl_Interp       *interp,
        int               argc,
        char            **argv)
{
    MenuMgr *menum;
    Uint32     globl = 0, i;

    if (argc < 2) {
        Tcl_AppendResult(interp, "too few arguments", 0);
        return TCL_ERROR;
    }

    for (i=1; i<argc; i++) {
        if (String::equal(argv[i], "-global")) {
            globl = 1;
        }
    }

    if (globl) {
        if (!Globl_MenuMgr) {
            menum = new MenuMgr(interp, argv[1]);
            Globl_MenuMgr = menum;
        }
        Tcl_AppendResult(interp, Globl_MenuMgr->getName(), 0);
    } else {
        menum = new MenuMgr(interp, argv[1]);
        Tcl_AppendResult(interp, argv[1], 0);
    }

    return TCL_OK;
}

/********************************************************************
 * MenuMgr_Init()
 ********************************************************************/
extern "C" int MenuMgr_Init(Tcl_Interp *interp)
{
    LogRegionType    *lr = new LogRegionType("MenuMgr",
            "Menu Template Manager",
            "",
            "",
            "");
    LogMgr::AddRegionType(lr);

    Tcl_CreateCommand(interp, "menu_mgr", 
            (Tcl_CmdProc *)MenuMgr_TclCmd, 0, 0);
    return TCL_OK;
}


