/*****************************************************************************
 *                                SigDB.h
 * Author: Matthew Ballance
 * Desc:   Describes a signal database...
 *
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *
 *****************************************************************************/
#ifndef INCLUDED_SIG_DB_H
#define INCLUDED_SIG_DB_H

#include "types.h"
#include <tcl.h>
#include <tk.h>
#include "tclOldConfig.h"
#include "vector.h"
#include "ivi_String.h"
#include "vpi_user.h"
#include "IviSim.h"

class DesignDB;
class DFIO;
class DFIOTrace;
class SdbReader;

class SigDB {

    /****************************************************************
     * Public Methods
     ****************************************************************/
    public:

        /************************************************************
         * SigDB()
         ************************************************************/
        SigDB(
            Tcl_Interp    *interp,
            Uint32         argc,
            Char         **argv);

        /************************************************************
         * ~SigDB()
         ************************************************************/
        virtual ~SigDB();

        /************************************************************
         * InstCmd()
         ************************************************************/
        virtual Int32 InstCmd(Uint32 argc, Char **argv);

        /************************************************************
         * addSignal()
         ************************************************************/
        virtual Vector<DFIOTrace> *SigDB::addSignal(Char *path);

        /************************************************************
         * addClient()
         * Adds an SDBR as a client of this SDB
         ************************************************************/
        void addClient(SdbReader *sdbr);

        /************************************************************
         * removeClient()
         ************************************************************/
        void removeClient(SdbReader *sdbr);

        DesignDB *get_ddb(void)    { return ddb; } 
        const char *getInstName()  { return instName.value(); }

        DFIO *getDFIO() { return dfio; }

        virtual Int32 get_resolution();

        Uint32 getMaxTime();

        char *getSdbId() const { return sdbId; }

        Uint32                     okay;

    /************************************************************
     * Protected Data
     ************************************************************/
    protected:
        Vector<DFIOTrace>         *traceVector;
        Vector<DFIOTrace>          d_traces;

    /************************************************************
     * Private Data
     ************************************************************/
    private:
        Int32 Configure(Uint32 argc, Char **argc, Uint32 flags);
        void SigDB::addSimSignals(IviSim *sim, const char *path);
        void SigDB::addDFIOSignals(DFIO *dfio, const char *path);

        static int SigDB::SDB_SimStepEnd(
            ClientData         clientData,
            Tcl_Interp        *interp,
            int                objc,
            Tcl_Obj           *const objv[]);

        enum {
            OPT_DDB = 0,
            OPT_DFIO,
            OPT_NumOpt
        };

        /************************************************************
         * getConfigSpec()
         ************************************************************/
        Tk_ConfigSpec *getConfigSpec();
                
        /************************************************************
         * optionSpecified()
         ************************************************************/
        Uint32 optionSpecified(Uint32 idx) {
            return (getConfigSpec()[idx].specFlags&TK_CONFIG_OPTION_SPECIFIED);
        }


        Char                      *ddbName;
        DesignDB                  *ddb; 

        Char                      *dfioName;
        DFIO                      *dfio;

        Char                      *sdbId;

        String                     instName;

        Tcl_Interp                *interp;
        ClientData                 d_SimStepEnd_CbId;
        ClientData                 d_SimStepUpdate_CbId;

        Int32                      resolution;
        Uint32                     maxTimeValid;
        Uint32                     maxTime;

        Vector<SdbReader>          d_clients;
};

#endif /* INCLUDED_SIG_DB_H */

