/****************************************************************************
 *                             TclBitVector.cc
 *
 * Author: Matthew Ballance
 * Desc:   Implements the TCL interface to the BitVector class... 
 * Introduces a new TCL type...
 *
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *
 ****************************************************************************/
#include "BitVector.h"
#include "tcl.h"
#include <stdio.h>
#include <string.h>
#include <strings.h>
#include "TclBitVector.h"

static Tcl_ObjType  *bv_type = NULL;

/****************************************************************************
 * TclBitVector_New()
 ****************************************************************************/
int TclBitVector_New(ClientData userData, Tcl_Interp *interp, 
        int argc, char **argv)
{
    Tcl_Obj    *newObj = (Tcl_Obj *)Tcl_Alloc(sizeof(Tcl_Obj));
    BitVector  *bitv = new BitVector(String("11011000"));

    return TCL_OK;
}

/****************************************************************************
 * Tcl_NewBitVector_Str()
 ****************************************************************************/
Tcl_Obj *Tcl_NewBitVector_Str(char *str, Int32 len)
{
    BitVector *bv;
    Tcl_Obj   *obj = Tcl_NewObj();

    if (len < 0) {
        len = strlen(str);
    }
    bv = new BitVector(String(str, len));
    obj->typePtr = bv_type;
    obj->refCount = 0;
    obj->internalRep.otherValuePtr = bv;
    Tcl_InvalidateStringRep(obj);

    return obj;
}

/****************************************************************************
 * Tcl_NewBitVector_BinStr()
 ****************************************************************************/
Tcl_Obj *Tcl_NewBitVector_BinStr(char *str, Int32 len)
{
    BitVector *bv;
    Tcl_Obj   *obj = Tcl_NewObj();

    if (len < 0) {
        len = strlen(str);
    }
    bv = new BitVector(String(str, len), BitVector::Radix_Bin);
    obj->typePtr = bv_type;
    obj->refCount = 0;
    obj->internalRep.otherValuePtr = bv;
    Tcl_InvalidateStringRep(obj);

    return obj;
}

/****************************************************************************
 * Tcl_NewBitVector_DFIOVal()
 ****************************************************************************/
Tcl_Obj *Tcl_NewBitVector_DFIOVal(DFIOVal *value, Uint32 bitLen)
{
    char str[2048];

    for (Uint32 i=0; i<bitLen; i++) {
        Uint32 avl = ((value[i/32].aval & (1 << (i%32)))?2:0);
        Uint32 bvl = ((value[i/32].bval & (1 << (i%32)))?1:0);

        str[bitLen-i-1] = "01zx"[avl|bvl];
    }
    str[bitLen] = 0;

    return Tcl_NewStringObj(str, -1);
}

/****************************************************************************
 * TclBitVector_FreeInternapRep()
 ****************************************************************************/
static void TclBitVector_FreeInternalRep(struct Tcl_Obj *obj)
{
    BitVector *bv = (BitVector *)obj->internalRep.otherValuePtr;
    delete bv;
}

/****************************************************************************
 * TclBitVector_DupInternalRep()
 ****************************************************************************/
static void TclBitVector_DupInternalRep(Tcl_Obj *srcPtr, Tcl_Obj *dupPtr)
{
    BitVector    *bv1, *bv2;

    bv1 = (BitVector *)srcPtr->internalRep.otherValuePtr;
    bv2 = new BitVector(*bv1);
    dupPtr->internalRep.otherValuePtr = bv2;
    dupPtr->typePtr = bv_type;
}

/****************************************************************************
 * TclBitVector_UpdateString()
 ****************************************************************************/
static void TclBitVector_UpdateString(Tcl_Obj *obj)
{
    BitVector *bv  = (BitVector *)obj->internalRep.otherValuePtr;
    const String  &str = bv->toString();

    obj->bytes = Tcl_Alloc(str.length()+1);
    strcpy(obj->bytes, str.value());
    obj->length = str.length();
}

/****************************************************************************
 * TclBitVector_SetFromAny()
 ****************************************************************************/
static int TclBitVector_SetFromAny(Tcl_Interp *interp, Tcl_Obj *obj)
{
    Char *str = Tcl_GetString(obj);

    if (obj->typePtr && obj->typePtr->freeIntRepProc) {
        obj->typePtr->freeIntRepProc(obj);
    }
    obj->typePtr = bv_type;
    obj->internalRep.otherValuePtr = new BitVector(String(str));
    Tcl_InvalidateStringRep(obj);

    return TCL_OK;
}

/****************************************************************************
 * Tcl_GetBitVectorFromObj()
 ****************************************************************************/
int Tcl_GetBitVectorFromObj(
        Tcl_Interp        *interp,
        Tcl_Obj           *obj,       /**** Input            ****/
        BitVector        **out)       /**** BitVector output ****/
{
    int tmp;
    BitVector    *bv, *cpy;

    if (obj->typePtr == bv_type) {
        cpy = (BitVector *)obj->internalRep.otherValuePtr;
        bv = new BitVector(*cpy);
    } else {
        bv = new BitVector(String(Tcl_GetStringFromObj(obj, &tmp)));
    }

    *out = bv;

    return TCL_OK;
}

/****************************************************************************
 * TclBitVector_BitVector()
 *
 * Changes the form of an existing BitVector... 
 ****************************************************************************/
static int TclBitVector_BitVector(
        ClientData         clientData,
        Tcl_Interp        *interp,
        int                objc,
        Tcl_Obj           *const *objv)
{
    Uint32        i; 
    int           len;
    Tcl_Obj      *arg = objv[1];
    Tcl_Obj      *ret;
    BitVector    *bv;
    char         *str;

    if (Tcl_IsShared(arg)) {
        ret = Tcl_DuplicateObj(arg);
    } else {
        ret = arg;
    }

    /**** bitvector <bv> [options] ****/
    if (ret->typePtr != bv_type) {
        if (TclBitVector_SetFromAny(interp, ret) != TCL_OK) {
            Tcl_AppendResult(interp, "cannot convert to bitvector", 0);
            return TCL_ERROR;
        }
    }

    bv = (BitVector *)ret->internalRep.otherValuePtr;

    for (i=2; i<objc; i++) {
        str = Tcl_GetStringFromObj(objv[i], &len);

        if (str[0] == '-') {
            if (str[1] == 'h' && !strncmp(str, "-hex", 4)) {
                bv->setRadix((bv->getRadix() & ~(BitVector::Radix_Mask)) 
                        | BitVector::Radix_Hex | BitVector::Radix_Typed);
            } else if (str[1] == 'b' && !strncmp(str, "-bin", 4)) {
                bv->setRadix((bv->getRadix() & ~(BitVector::Radix_Mask)) 
                        | BitVector::Radix_Bin | BitVector::Radix_Typed);
            } else if (str[1] == 'o' && !strncmp(str, "-oct", 4)) {
                bv->setRadix((bv->getRadix() & ~(BitVector::Radix_Mask)) 
                        | BitVector::Radix_Oct | BitVector::Radix_Typed);
            } else if (str[1] == 'd' && !strncmp(str, "-dec", 4)) {
                bv->setRadix((bv->getRadix() & ~(BitVector::Radix_Mask)) 
                        | BitVector::Radix_Dec | BitVector::Radix_Typed);
            } else if (str[1] == 's' && !strcmp(str, "-signed")) {
                bv->setRadix(bv->getRadix() | BitVector::Radix_Signed);
            } else if (str[1] == 's' && !strncmp(str, "-str", 4)) {
                bv->setRadix((bv->getRadix() & ~(BitVector::Radix_Mask)) 
                        | BitVector::Radix_Str);
            } else {
                Tcl_AppendResult(interp, "unknown option ", 
                        str, 0);
                return TCL_ERROR;
            }
        } else {
            Tcl_AppendResult(interp, "expecting: bitvector vector ?options?",
                    0);
            return TCL_ERROR;
        }
    }

    Tcl_InvalidateStringRep(ret);

    Tcl_SetObjResult(interp, ret);
    return TCL_OK;
}


/****************************************************************************
 * TclBitVector_Init()
 ****************************************************************************/
extern "C" int TclBitVector_Init(Tcl_Interp *interp)
{
    Tcl_ObjType     *obj_type;

    Tcl_Obj    *newObj = Tcl_NewObj();
    Tcl_Obj    *ptr;

    obj_type = (Tcl_ObjType *)Tcl_Alloc(sizeof(Tcl_ObjType));
    memset(obj_type, 0, sizeof(Tcl_ObjType));

    obj_type->name = (Char *)Tcl_Alloc(strlen("BitVector")+1);
    strcpy(obj_type->name, "BitVector");
    obj_type->freeIntRepProc = TclBitVector_FreeInternalRep;
    obj_type->dupIntRepProc  = TclBitVector_DupInternalRep;
    obj_type->updateStringProc = TclBitVector_UpdateString;
    obj_type->setFromAnyProc   = TclBitVector_SetFromAny;

    bv_type = obj_type;
    Tcl_RegisterObjType(obj_type);


    Tcl_CreateObjCommand(interp, "bitvector", TclBitVector_BitVector, 0, 0);

    return TCL_OK;
}




