/****************************************************************************
 *                           TclChannelObj.cc
 * 
 * Author: Matthew Ballance
 * Desc:   Implements a basic character-oriented channel. This basic channel
 *         may be extended.
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "TclChannelObj.h"
#include <string.h>

#undef  DEBUG_TCL_CHANNEL_OBJ

#define FP stderr
#ifdef DEBUG_TCL_CHANNEL_OBJ
#define DBG_MSG(x) fprintf x
#else
#define DBG_MSG(x)
#endif

/********************************************************************
 * TclChannelObj
 ********************************************************************/
TclChannelObj::TclChannelObj(
        Tcl_Interp         *interp,
        const char         *channelName,
        const char         *channelTypeName,
        Int32               flags
        )
{
    memcpy(&d_ChannelProcs, &d_DefaultChannelProcs, sizeof(Tcl_ChannelType));
    d_ChannelProcs.typeName = strdup(channelTypeName);

    d_interp = interp;
    d_channelName = channelName;
    d_mask        = 0;

    d_channel = Tcl_CreateChannel(&d_ChannelProcs, (char *)channelName, 
            this, flags);
    Tcl_RegisterChannel(interp, d_channel);

    Tcl_CreateCommand(d_interp, d_channelName.value(), 
            &TclChannelObj::InstCmd, this, 0);

    ok = 1;
}

/********************************************************************
 * ~TclChannelObj()
 ********************************************************************/
TclChannelObj::~TclChannelObj()
{
    Tcl_UnregisterChannel(d_interp, d_channel);
}

/************************************************************
 * InputProc()
 ************************************************************/
int TclChannelObj::InputProc(
        char         *buf,
        int           bufSize,
        int          *errorCodePtr)
{
    int ret;

    DBG_MSG((stderr, "InputProc()\n"));

    if (!d_fifo.length()) {
        return 0;
    } else {
        ret = d_fifo.dequeue(buf, bufSize);
    }
    return ret;
}

/************************************************************
 * OutputProc()
 ************************************************************/
int TclChannelObj::OutputProc(
        const char   *buf,
        int           toWrite,
        int          *errorCodePtr)
{
    write((char *)buf);
    return toWrite;
}

/************************************************************
 * SeekProc()
 ************************************************************/
int TclChannelObj::SeekProc(
        long          offset,
        int           seekMode,
        int          *errorCodePtr)
{
    DBG_MSG((stderr, "SeekProc(%d, %d, %x)\n", 
            offset, seekMode, errorCodePtr));

    return TCL_OK;
}

/************************************************************
 * SetOptionProc()
 ************************************************************/
int TclChannelObj::SetOptionProc(
        Tcl_Interp   *interp,
        const char   *optionName,
        const char   *optionValue)
{
    DBG_MSG((stderr, "SetOptionProc()\n"));
    return TCL_OK;
}

/************************************************************
 * GetOptionProc()
 ************************************************************/
int TclChannelObj::GetOptionProc(
        Tcl_Interp   *interp,
        const char   *optionName,
        Tcl_DString  *dsPtr)
{
    DBG_MSG((stderr, "GetOptionProc()\n"));
    return TCL_OK;
}

/************************************************************
 * WatchProc()
 ************************************************************/
void TclChannelObj::WatchProc(int mask)
{
    DBG_MSG((stderr, "WatchProc()\n"));
    d_mask = mask;
}

/************************************************************
 * GetHandleProc()
 ************************************************************/
int TclChannelObj::GetHandleProc(
        int           direction,
        ClientData   *handlePtr)
{
    DBG_MSG((stderr, "GetHandleProc()\n"));
    return TCL_OK;
}

/************************************************************
 * BlockModeProc()
 ************************************************************/
int TclChannelObj::BlockModeProc(int mode)
{
    DBG_MSG((stderr, "BlockModeProc()\n"));
    return TCL_OK;
}

/************************************************************
 * FlushProc()
 ************************************************************/
int TclChannelObj::FlushProc()
{
    DBG_MSG((stderr, "FlushProc()\n"));
    return TCL_OK;
}

/************************************************************
 * HandlerProc()
 ************************************************************/
int TclChannelObj::HandlerProc(
        int           interestMask)
{
    DBG_MSG((stderr, "HandlerProc()\n"));
    return TCL_OK;
}

/************************************************************
 * Close2Proc()
 ************************************************************/
int TclChannelObj::Close2Proc(
        Tcl_Interp   *interp,
        int           flags)
{
    DBG_MSG((stderr, "Close2Proc()\n"));
    return TCL_OK;
}

/************************************************************
 * CloseProc()
 ************************************************************/
int TclChannelObj::CloseProc(
        Tcl_Interp   *interp)
{
    DBG_MSG((stderr, "CloseProc()\n"));
    return TCL_OK;
}

/********************************************************************
 * InstCmd()
 ********************************************************************/
int TclChannelObj::InstCmd(int argc, const char **argv)
{
    if (!strcmp(argv[1], "delete")) {
        Tcl_EventuallyFree(this, &TclChannelObj::Delete);
    } else 
    if (!strcmp(argv[1], "execute")) {
        return Execute();
    } else {
        Tcl_AppendResult(d_interp, "Unknown TclChannelObj sub-cmd ",
                argv[1], 0);
        return TCL_ERROR;
    }

    return TCL_OK;
}

/********************************************************************
 * Execute()
 ********************************************************************/
int TclChannelObj::Execute()
{
    Tcl_Obj    *line = Tcl_NewStringObj("", -1);
    int         ret = 0, cmdret;
    int         complete = 0;

    Tcl_IncrRefCount(line);

    Tcl_Flush(d_channel);

    while (ret >= 0) {
        Tcl_SetStringObj(line, "", -1);

        /**** Loop until we have a complete command ****/
        complete = 0;
        while (!complete && ret >= 0) {
            ret = Tcl_GetsObj(d_channel, line);
            if (ret < 0) {
                break;
            }

            Tcl_AppendToObj(line, "\n", -1);
            complete = Tcl_CommandComplete(Tcl_GetString(line));
        }

        if (complete > 0) {
            cmdret = Tcl_EvalEx(d_interp, Tcl_GetString(line), -1, 
                    TCL_EVAL_GLOBAL);

            if (cmdret != TCL_OK) {
                fprintf(stderr, "command execution failed - bailing\n");
                Tcl_DecrRefCount(line);
                return TCL_ERROR;
                break;
            }
        }
    }

    Tcl_DecrRefCount(line);

    return TCL_OK;
}

/********************************************************************
 * InputProc_Static()
 ********************************************************************/
int TclChannelObj::InputProc_Static(
        ClientData    clientData,
        char         *buf,
        int           bufSize,
        int          *errorCodePtr)
{
    return ((TclChannelObj *)clientData)->InputProc(
            buf, bufSize, errorCodePtr);
}

/********************************************************************
 * OutputProc_Static()
 ********************************************************************/
int TclChannelObj::OutputProc_Static(
        ClientData    clientData,
        const char   *buf,
        int           toWrite,
        int          *errorCodePtr)
{
    return ((TclChannelObj *)clientData)->OutputProc(
            buf, toWrite, errorCodePtr);
}

/********************************************************************
 * SeekProc_Static()
 ********************************************************************/
int TclChannelObj::SeekProc_Static(
        ClientData    clientData,
        long          offset,
        int           seekMode,
        int          *errorCodePtr)
{
    return ((TclChannelObj *)clientData)->SeekProc(
            offset, seekMode, errorCodePtr);
}

/********************************************************************
 * SetOptionProc_Static()
 ********************************************************************/
int TclChannelObj::SetOptionProc_Static(
        ClientData    clientData,
        Tcl_Interp   *interp,
        const char   *optionName,
        const char   *optionValue)
{
    return ((TclChannelObj *)clientData)->SetOptionProc(
            interp, optionName, optionValue);
}

/********************************************************************
 * GetOptionProc_Static()
 ********************************************************************/
int TclChannelObj::GetOptionProc_Static(
        ClientData    clientData,
        Tcl_Interp   *interp,
        const char   *optionName,
        Tcl_DString  *dsPtr)
{
    return ((TclChannelObj *)clientData)->GetOptionProc(
            interp, optionName, dsPtr);
}

/********************************************************************
 * WatchProc_Static()
 ********************************************************************/
void TclChannelObj::WatchProc_Static(
        ClientData    clientData,
        int           mask)
{
    return ((TclChannelObj *)clientData)->WatchProc(mask);
}

/********************************************************************
 * GetHandleProc_Static()
 ********************************************************************/
int TclChannelObj::GetHandleProc_Static(
        ClientData    clientData,
        int           direction,
        ClientData   *handlePtr)
{
    return ((TclChannelObj *)clientData)->GetHandleProc(
            direction, handlePtr);
}

/********************************************************************
 * BlockModeProc_Static()
 ********************************************************************/
int TclChannelObj::BlockModeProc_Static(
        ClientData    clientData,
        int           mode)
{
    return ((TclChannelObj *)clientData)->BlockModeProc(mode);
}

/********************************************************************
 * FlushProc_Static()
 ********************************************************************/
int TclChannelObj::FlushProc_Static(
        ClientData    clientData)
{
    return ((TclChannelObj *)clientData)->FlushProc();
}

/********************************************************************
 * HandlerProc_Static()
 ********************************************************************/
int TclChannelObj::HandlerProc_Static(
        ClientData    clientData,
        int           interestMask)
{
    return ((TclChannelObj *)clientData)->HandlerProc(interestMask);
}

/********************************************************************
 * Close2Proc_Static()
 ********************************************************************/
int TclChannelObj::Close2Proc_Static(
        ClientData    clientData,
        Tcl_Interp   *interp,
        int           flags)
{
    return ((TclChannelObj *)clientData)->Close2Proc(interp, flags);
}

/********************************************************************
 * CloseProc_Static()
 ********************************************************************/
int TclChannelObj::CloseProc_Static(
        ClientData    clientData,
        Tcl_Interp   *interp)
{
    return ((TclChannelObj *)clientData)->CloseProc(interp);
}

/********************************************************************
 * write()
 ********************************************************************/
void TclChannelObj::write(Char *msg)
{
    d_fifo << msg;
    if (d_mask & TCL_READABLE) {
        Tcl_NotifyChannel(d_channel, TCL_READABLE);
    }
}

Tcl_ChannelType TclChannelObj::d_DefaultChannelProcs = {
    "TclChannel",
    TCL_CHANNEL_VERSION_2,
    (Tcl_DriverCloseProc *)&TclChannelObj::CloseProc_Static,
    (Tcl_DriverInputProc *)&TclChannelObj::InputProc_Static,
    (Tcl_DriverOutputProc *)&TclChannelObj::OutputProc_Static,
    (Tcl_DriverSeekProc *)&TclChannelObj::SeekProc_Static,
    (Tcl_DriverSetOptionProc *)&TclChannelObj::SetOptionProc_Static,
    (Tcl_DriverGetOptionProc *)&TclChannelObj::GetOptionProc_Static,
    (Tcl_DriverWatchProc *)&TclChannelObj::WatchProc_Static,
    (Tcl_DriverGetHandleProc *)&TclChannelObj::GetHandleProc_Static,
    (Tcl_DriverClose2Proc *)&TclChannelObj::Close2Proc_Static,
    (Tcl_DriverBlockModeProc *)&TclChannelObj::BlockModeProc_Static,
    (Tcl_DriverFlushProc *)&TclChannelObj::FlushProc_Static,
    (Tcl_DriverHandlerProc *)&TclChannelObj::HandlerProc_Static
};

/********************************************************************
 * TclChannelObj_Create()
 ********************************************************************/
static int TclChannelObj_Create(
        ClientData        clientData,
        Tcl_Interp       *interp,
        int               argc,
        const char      **argv)
{
    TclChannelObj  *obj;
    
    obj = new TclChannelObj(interp, argv[1], "TclChannelObj", 
            TCL_WRITABLE|TCL_READABLE);

    return TCL_OK;
}

/********************************************************************
 * TclChannelObj_Init()
 ********************************************************************/
extern "C" int TclChannelObj_Init(Tcl_Interp *interp)
{
    Tcl_CreateCommand(interp, "tcl_channel_obj", TclChannelObj_Create,
            0, 0);
}



