/****************************************************************************
 *                         IviRemoteSimPlugin.h
 *
 *
 * Author: Matthew Ballance
 * Desc:   
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#ifndef INCLUDED_IVI_REMOTE_SIM_PLUGIN_H
#define INCLUDED_IVI_REMOTE_SIM_PLUGIN_H
#include "types.h"
#include "vpi_user.h"
#include <tcl.h>
#include "RemoteAppTclCmdListener.h"
#include "IviSim.h"

class IviRemoteCmdServer;
class IviRemotePluginCmdChannel;
class RemoteAppClientConnection;
class DesignDB;
class SigDB;
class DFIO;

/****************************************************************************/
/** \class IviRemoteSimPlugin
 *  \brief Implements the base class for the remote portion of the IVI
 *         remote-sim interface
 ****************************************************************************/
class IviRemoteSimPlugin {

    friend class IviRemotePluginCmdChannel;

    /****************************************************************
     * Public Methods
     ****************************************************************/
    public:

        /************************************************************
         * IviRemoteSimPlugin()
         ************************************************************/
        IviRemoteSimPlugin(Tcl_Interp *interp);

        /************************************************************
         * ok()
         ************************************************************/
        bool &ok() { return d_ok; }

        /************************************************************
         * getInstName()
         */
        /**
         ************************************************************/
        const char *getInstName() { return d_instName.value(); }

        /************************************************************
         * getInterp()
         */
        /**
         ************************************************************/
        Tcl_Interp *getInterp() { return d_interp; }

        /************************************************************
         * HandleDisconnect()
         */
        /** 
         ************************************************************/
        virtual void HandleDisconnect();

    /****************************************************************
     * Public Data
     ****************************************************************/
    public:



    /****************************************************************
     * Protected Methods
     ****************************************************************/
    protected:

        /************************************************************
         * ConstructSim()
         */
        /** Expected to construct an IviSim object of the type 
         *  used by the remote portion of this simulator
         ************************************************************/
        virtual IviSim *ConstructSim() = 0;

        /************************************************************
         * ConstructDDB()
         */
        /** Creates a DesignDB of the type used by this simulator
         ************************************************************/
        virtual DesignDB *ConstructDDB();

        /************************************************************
         * ConstructDFIO()
         */
        /** Creates a DFIO of the type used by this simulator
         ************************************************************/
        virtual DFIO *ConstructDFIO();

        /************************************************************
         * ConstructSDB()
         */
        /** Creates a SigDB of the type used by this simulator
         ************************************************************/
        virtual SigDB *ConstructSDB();

        /************************************************************
         * BeginRun()
         */
        /** Called to instruct the simulator to run. 
         *
         *  - When the run ends, SignalEndOfSimulation() should be
         *    called.
         *  - If the simulation ends before the runtime is up, 
         *     SignalEndOfSimulation() should be called
         ************************************************************/
        virtual int BeginRun(Uint32 runLen) = 0;

        /********************************************************************
         * Signal routines for plugin to call
         ********************************************************************/

        /************************************************************
         * SignalReadyForSimulation()
         */
        /** Called by the plugin when construction of the simulator
         *  is complete.
         *
         *  This function enters the command loop and waits for a
         *  command from the IVI front-end. When the function 
         *  returns, it is assumed that simulation will continue
         ************************************************************/
        virtual void SignalReadyForSimulation();

        /************************************************************
         * SignalEndOfRun()
         */
        /** Called by the plugin when the end of the simulation run
         *  has been reached. 
         *
         *  This function enters the command loop and waits for a
         *  command from the IVI front-end. When the function 
         *  returns, it is assumed that simulation will continue
         *
         *  @param simTime Current simulation time. simTime is 
         *         specified in simulator time units. simTime may 
         *         be scaled if the simulator timestep is larger 
         *         than unity (ie 100ps).
         ************************************************************/
        virtual void SignalEndOfRun(Uint64 simTime);

        /************************************************************
         * SignalEndOfSimulation()
         */
        /** Called to signal the end of simulation. 
         *
         *  This is an alternative to calling SignalEndOfRun(). 
         *
         *  This function enters the command loop and waits for a 
         *  command. The only command to be received should be
         *  close simulation.
         ************************************************************/
        virtual void SignalEndOfSimulation(Uint64 simTime);


    /****************************************************************
     * Private Methods
     ****************************************************************/
    private:

        /************************************************************
         * ConstructSimObjs()
         ************************************************************/
        int ConstructSimObjs();

        /************************************************************
         * WaitCmd()
         ************************************************************/
        int WaitCmd();

        /************************************************************
         * ProcessCmds()
         ************************************************************/
        int ProcessCmds(bool wait);

    /****************************************************************
     * Private Data
     ****************************************************************/
    private:
        RemoteAppTclCmdListener    *d_cmdServer;
        RemoteAppClientConnection  *d_conn;

        String                      d_instName;

        IviSim                     *d_iviSim;
        DesignDB                   *d_ddb;
        SigDB                      *d_sdb;
        DFIO                       *d_dfio;

        Uchar                       d_data[256];
        Uint32                      d_cmd;
        Uint32                      d_len;

        bool                        d_recv;

        bool                        d_ok;
        Tcl_Interp                 *d_interp;
};

#endif /* INCLUDED_IVI_REMOTE_SIM_PLUGIN_H */
