#*****************************************************************************
#*                             MainWindow.tcl
#*
#* Author: Matthew Ballance
#* Desc:   Launches the main IVI window
#*
#*
#* <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
#*
#*    This source code is free software; you can redistribute it
#*    and/or modify it in source code form under the terms of the GNU
#*    General Public License as published by the Free Software
#*    Foundation; either version 2 of the License, or (at your option)
#*    any later version.
#*
#*    This program is distributed in the hope that it will be useful,
#*    but WITHOUT ANY WARRANTY; without even the implied warranty of
#*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#*    GNU General Public License for more details.
#*
#*    You should have received a copy of the GNU General Public License
#*    along with this program; if not, write to the Free Software
#*    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
#*
#* </Copyright>
#*
#*
#*****************************************************************************

package provide ivi_ui "1.0"

#*****************************************************************************
#* Globals
#* installation-dir variables
#*****************************************************************************
set IVI_HOME [lindex [array get env IVI_HOME] 1]
set cmd_window 0

lappend auto_path [file join $IVI_HOME modules BWidget-1.6.0]
package require BWidget

#*********************************************************************
#* ConstructMainWin()
#*********************************************************************
proc ConstructMainWin {w} {
   global IVI_HOME
   global ivi_global
   global cmd_window
   global tcl_platform

    #*********************************************************
    #* Set the icon on the main windows
    #*********************************************************
    if {$tcl_platform(platform) == "windows"} {
        wm iconbitmap . -default \
            [file join $IVI_HOME modules ivi_ui ivi_icon.ico]
    }


   #*********************************************************
   #* Create the menu bar
   #*    - File
   #*    - View
   #*    - Help
   #*********************************************************

   #**** Create the menu-bar
   set menu [MenuMgr::create menubar IviUi::Main .menu]
   . configure -menu $menu

    set top_frame  [frame .top_frame]
    set cmd_window [GuiConsole::GuiConsole $top_frame.cmd \
       -intr_handler intr_handler \
       -log_fp $ivi_global(logfile_fp)]

    uplevel #0 {set ImgPath [file join $IVI_HOME modules ivi_ui]}
    set toolbar    [MenuMgr::create toolbar IviUi::MainToolbar \
        $top_frame.toolbar -var [list w $w cmd $cmd_window]]

    pack $toolbar -side top -expand no -fill x
    pack $cmd_window -side top -expand yes -fill both

    set bottom_frame [frame .bottom_frame -relief sunken -borderwidth 2]

    set bottom_llabel [label $bottom_frame.llabel -text "" -borderwidth 1 \
       -relief sunken -width 10]
    set bottom_rlabel [label $bottom_frame.rlabel -text "" -borderwidth 1 \
       -relief sunken]

    set ivi_global(cmd_rlabel) $bottom_rlabel
    set ivi_global(cmd_llabel) $bottom_llabel

    pack $bottom_llabel $bottom_rlabel -side left -expand yes -fill x 
    pack $bottom_frame -side bottom -expand no -fill x 

    pack $top_frame -side top -expand yes -fill both

    $bottom_llabel config -text "Time: 0"
}

#*********************************************************************
#* MainWindow_ConstructCB
#*********************************************************************
proc MainWindow_ConstructCB args {
    global IVI_HOME
    global auto_path

    wm title . "IVI - Icarus Verilog Interactive"

    if {![catch {set dir [glob "[file join $IVI_HOME lib Img]*" ]}]} {
        lappend auto_path $dir
    }

    #**** Construct command-window 
    ConstructMainWin .

    #*** Setup a stdin/stdout console if requested
    if {[IviConfig::current App.Console]} {
        StdConsole::StdConsole std_console
    }

    SetStatBarText "No Design Loaded"
}

#*********************************************************************
#* save_transcript
#*********************************************************************
proc save_transcript {filename} {
    global cmd_window

    if {$filename != ""} {
        $cmd_window save $filename
    }
}

#*********************************************************************
#* guiSaveTranscript
#*********************************************************************
proc guiSaveTranscript {} {
    set fileTypes {
            {{All Files} {.*}}
        }

    set file [tk_getSaveFile -filetypes $fileTypes -initialdir [pwd]]

    save_transcript $file
}

#*********************************************************************
#* guiLoadFile
#*********************************************************************
proc guiLoadFile {} {
    global ivi_global

    set fileTypes {
        {{Verilog Design Files} {.vvp}}
        {{All Files} {.*}}
    }
 
    set fileName [tk_getOpenFile -filetypes $fileTypes -initialdir [pwd]]

    if {$fileName != ""} {
        if {[catch {load_design $fileName} res]} {
            ivi_puts "ERROR: Cannot load \"$fileName\" - \"$res\""
        }
    } else {
#        SetStatBarText "ERROR: No design specified"
    }
}

#*********************************************************************
#* guiGetDesignFilterList
#*********************************************************************
proc guiGetDesignFilterList {} {
    set list ""

    foreach sim [sim_mgr sim_list] {
        set suffix_list [sim_mgr get_suffixes $sim]
        lappend list $suffix_list
    }

    return $list
}

#*********************************************************************
#* guiOpenDesign
#*********************************************************************
proc guiOpenDesign {} {
    set m [DialogMgr::create IviUi::OpenDesign .openDesign \
        -var {w .openDesign}]
  
    set types [sim_mgr sim_list]
    set types [linsert $types 0 "auto-detect"]

    set fc [$m subwidget designType]
    $fc configure -values $types
    $fc setvalue first

    $m popup

    if {[$m wait] != ""} {
        set filename [[$m subwidget filename] get_text]
        set type_idx [[$m subwidget designType] getvalue]
        set type_name [lindex $types $type_idx]

        set cmd [list load_design $filename]

        if {$type_name != "auto-detect"} {
            lappend cmd -type $type_name
        }

        if {[catch $cmd res]} {
            ivi_puts "ERROR: Cannot load design \"$res\""
        }

        destroy $m
    }
}

#*********************************************************************
#* guiOpenDesign_FileHelper
#*********************************************************************
proc guiOpenDesign_FileHelper {w} {
    set m .openDesign

    set fileTypes [list [list "" [guiGetDesignFilterList]] {{All Files} {*}}]

    set dfile [tk_getOpenFile -filetypes $fileTypes -initialdir [pwd]]

    if {$dfile != ""} {
        set entry [$w subwidget filename]
        $entry delete 0 end
        $entry insert 0 $dfile
    }

    raise $w
    focus $w
}

#*********************************************************************
#* guiIncludeFile
#*********************************************************************
proc guiIncludeFile {} {
    global ivi_global

    set fileTypes {
        {{IVI Command Files} {.inc .do}}
        {{All Files} {.*}}
    }
 
    set fileName [tk_getOpenFile -filetypes $fileTypes -initialdir [pwd]]

    if {$fileName != ""} {
        ivi_puts "Sourcing command-file $fileName"
        catch {ivi_do $fileName} res
    }
}

#*********************************************************************
#* guiSetCWD
#*********************************************************************
proc guiSetCWD {} {

    set fileName [tk_chooseDirectory]

    if {$fileName != ""} {
        ivi_puts "Changing CWD to $fileName"
        cd $fileName
    }
}

#*********************************************************************
#* guiGetFilterList
#*********************************************************************
proc guiGetFilterList {type} {
    set list ""

#    lappend list "auto-detect"

    foreach filter [dfio_mgr filter_list $type] {
        set filter_id [$filter id]
        lappend list $filter_id
    }

    return $list
}

#*********************************************************************
#* guiGetFileExtensions
#*********************************************************************
proc guiGetFileExtensions {} {
    set suffixes  ""
    foreach filter [dfio_mgr filter_list] {
        lappend suffixes [$filter suffixes]
    }

    return $suffixes
}

#*********************************************************************
#* guiLoadSDB_FileHelper
#*********************************************************************
proc guiLoadSDB_FileHelper {w} {

    set fileTypes [list [list "" [guiGetFileExtensions]] {{All Files} {.*}}]
    set file [tk_getOpenFile -filetypes $fileTypes -initialdir [pwd]]

    set entry [$w subwidget filename]
    $entry delete 0 end
    $entry insert 0 $file

    raise $w
    focus $w
}

#*********************************************************************
#* guiSaveSDB_FileHelper
#*********************************************************************
proc guiSaveSDB_FileHelper {w} {
    set m .saveSDB
    set ftc [$m subwidget fileType]
    set filt_list [guiGetFilterList export]
    set ft  [$ftc getvalue]

    set filt_type [lindex $filt_list $ft]
    set filt  [prv_sdb_findtype $filt_type]
    set exts  [$filt suffixes]

    set fileTypes [list [list "" $exts] {{All Files} {.*}}]

#    set fileTypes [list [list "" [guiGetFileExtensions]] {{All Files} {.*}}]
    set file [tk_getSaveFile -filetypes $fileTypes -initialdir [pwd]]

    set entry [$w subwidget filename]
    $entry delete 0 end
    $entry insert 0 $file
}

#*********************************************************************
#* guiLoadSDB
#*********************************************************************
proc guiLoadSDB {} {
    set m [DialogMgr::create IviUi::OpenSDB .loadSDB \
        -var {w .loadSDB}]

    set filt_list [guiGetFilterList import]
    set filt_list [linsert $filt_list 0 "auto-detect"]

    #**** Find the list of filter types
    set fc [$m subwidget fileType]
    $fc configure -values $filt_list
    $fc setvalue first

    #**** Select an SDB name...
    for {set i 1} {$i < 256} {incr i} {
        set sdb_name "SigData$i"

        set ok 1
        foreach sdb [sdb_mgr list] {
            if {[$sdb cget -sdb_id] == $sdb_name} {
                set ok 0
                break
            }
        }
        if {$ok == 1} {
            break
        }
    }

    set entry [$m subwidget sdb_name]
    $entry set_text $sdb_name

    $m popup

    if {[$m wait] != ""} {
        set filename [[$m subwidget filename] get_text]
        set sdb_name [[$m subwidget sdb_name] get_text]
        set type_idx [[$m subwidget fileType] getvalue]
        set type_name [lindex $filt_list $type_idx]

        set cmd [list sdb open $sdb_name $filename]

        if {$type_name != "auto-detect"} {
            lappend cmd -type $type_name
        }

        if {[catch $cmd res ]} {
            ivi_puts "ERROR :: cannot open sdb - \"$res\""
        }
        destroy $m
    }
}

#*********************************************************************
#* guiSaveSDB
#*********************************************************************
proc guiSaveSDB {} {

    set m [DialogMgr::create IviUi::SaveSDB .saveSDB \
        -var {w .saveSDB}]

    #**** Find the list of SDB's
    set sdb_list ""
    foreach sdb [sdb_mgr list] {
        lappend sdb_list [$sdb cget -sdb_id]
    }

    set sigc [$m subwidget sdbName]

    if {[llength $sdb_list] == 0} {
        $sigc configure -values {"No Signal Databases"}
    } else {
        $sigc configure -values $sdb_list
    }
    $sigc setvalue first

    #**** Find the list of filter types
    set fc [$m subwidget fileType]
    set filt_list [guiGetFilterList export]
    set filt_list [linsert $filt_list 0 "auto-detect"]
    $fc configure -values $filt_list
    $fc setvalue first

    $m popup

    if {[$m wait] != ""} {
        set filename [[$m subwidget filename] get_text]
        set sdb_idx  [[$m subwidget sdbName]  getvalue]

        set sdb_name [lindex $sdb_list $sdb_idx]

        set type_idx [[$m subwidget fileType] getvalue]
        set type_name [lindex $filt_list $type_idx]

        set cmd [list sdb save $sdb_name $filename]
        if {$type_name != "auto-detect"} {
            lappend cmd -type $type_name
        }

        if {$sdb_name != "" && $filename != ""} {
            if {[catch $cmd res]} {
                ivi_puts "$res"
            }
        } else {
            if {$sdb_name == ""} {
                ivi_puts "ERROR :: No SDB specified"
            } 
            if {$filename == ""} {
                ivi_puts "ERROR :: No filename specified"
            }
        }

        destroy $m
    }
}

#*********************************************************************
#* SetStatBarText()
#*********************************************************************
proc SetStatBarText { msg } {
    global ivi_global

    $ivi_global(cmd_rlabel) config -text $msg
}

#*********************************************************************
#* SetTimeBarText()
#********************************************************************* 
proc SetTimeBarText args {
    global ivi_global
    set cb_time [lindex $args 1]
    set cb_res  [lindex $args 0]

    $ivi_global(cmd_llabel) config -text " Time: $cb_time $cb_res"
}

#*********************************************************************
#* ViewHelpBrowser
#*********************************************************************
proc ViewHelpBrowser {} {
    global    .helpBrowser

    for {set i 0} {$i < 10} {incr i} {

        if {[info globals .helpBrowser$i.help] == ""} {
            set wnd  [toplevel .helpBrowser$i]
            set help [HelpBrowser::HelpBrowser .helpBrowser$i.help]
            bind $help <Destroy> [list destroy $wnd]
            pack $help -expand yes -fill both
            break
        }
    }
}

#*********************************************************************
#* ViewAbout
#*********************************************************************
proc ViewAbout {} {
    global IVI_HOME
    if {![winfo exists .about]} {
        IviAbout::IviAbout .about -filename \
            [file join $IVI_HOME modules ivi_ui Credits.txt]
    }
}

#*********************************************************************
#* ViewLicense
#*********************************************************************
proc ViewLicense {} {
    global IVI_HOME
    global ivi_global
    if {[info globals .license] == ""} {
        set t [toplevel .license]

        set scont [ScrollContainer::ScrollContainer $t.text_frame]
        set cf [$scont subwidget frame]
        set txt  [text $cf.text -bg white]

        set hsb [$scont subwidget hsb]
        set vsb [$scont subwidget vsb]

        $txt configure -yscrollcommand "$vsb set"
        $txt configure -xscrollcommand "$hsb set"

        $vsb configure -command "$txt yview"
        $hsb configure -command "$txt xview"

        set lic "" 
        if {[file readable [file join $IVI_HOME modules ivi_ui License.txt]]} {
            set fileid [open [file join $IVI_HOME \
                modules ivi_ui License.txt] "r"]
            set lic [read $fileid]
            close $fileid
        }

        if {$lic != ""} {
            $txt delete 1.0 end
            $txt insert end $lic
        }
        $txt config -state disabled 

        pack $txt $scont -expand yes -fill both

        set cb [button $t.close -text "Close" -command "destroy $t"]
        pack $cb

        wm title $t "IVI License"
    }
} 
   
#*********************************************************************
#* gui_puts
#*********************************************************************
proc gui_puts args {
    global cmd_window

    if {$cmd_window != 0} {
        $cmd_window puts $args
    } else {
        puts $args
    }
}

#*********************************************************************
#* intr_handler
#*********************************************************************
proc intr_handler args {
    global IVI_BREAK_REQUEST

    set IVI_BREAK_REQUEST 1
    ivi_puts "Break requested"
}

#*********************************************************************
#* SplashScreen
#*********************************************************************
proc SplashScreen { art delay artdir {parent ""}} {
    catch { [winfo] } errmsg

    if {$delay == ""} {
        set delay 2000
    }

    if {[string match invalid* $errmsg] } {
        return -code error "Splash requires Tk"
    }

    set logo [file join $artdir $art]

    if {[file exists $logo]} {
        toplevel .splash -borderwidth 0 -relief raised
        set logo [image create photo -file $logo]
        label .splash.logo -image $logo -borderwidth 0
        pack .splash.logo -fill both
        wm resizable .splash false false

        if {$parent != ""} {
            wm transient .splash $parent
            set pg [wm geometry $parent]
        }

        update idletasks
        set cg [wm geometry .splash]
        set elems [split $cg {+}]
        set sx [lindex [split [lindex $elems 0] {x}] 0 ]
        set sy [lindex [split [lindex $elems 0] {x}] 1 ]
        set ww [winfo screenwidth .splash]
        set wh [winfo screenheight .splash]

        set nx [expr ($ww / 2) - ($sx / 2)]
        set ny [expr ($wh / 2) - ($sy / 2)]
        wm geometry .splash "[lindex $elems 0]+$nx+$ny"

        after $delay destroy .splash
        raise .splash
        update idletasks
    } else {
        set msg "Splash logo missing!\n"
        append msg "(file: \"$logo\" not found)"
        puts $msg"
    }

    return .splash
}

#*********************************************************************
#* IviApp_InitRcDir()
#*
#* Checks to see if $HOME/.ivi exists. If not, creates and 
#* initializes...
#*********************************************************************
proc IviApp_InitRcDir args {
    global IVI_HOME

    set home_rc [file join ~ .ivi]
    set share_dir [file join $IVI_HOME etc]

    if {![file exists $home_rc]} {
        puts "NOTE :: Creating resource dir \"$home_rc\""
        if {[catch {file mkdir $home_rc} res]} {
            puts "    ERROR :: Cannot create \"$home_rc\""
            puts "        \"$res\""
        }

        #**** Now, copy template files over from the share dir...
        if {![catch {set template_files [glob $share_dir/*.rct]} res]} {
            foreach file $template_files {
                set path_arr [file split $file]
                regsub {\.rct} [lindex $path_arr end] {} dest_name
                set dest_name [file join $home_rc $dest_name]
                if {[catch {file copy $file $dest_name} res]} {
                    puts "    ERROR :: while copying \"$file\""
                    puts "       $res"
                }
            }
        } else {
            puts "glob failed: $res"
        }
    }
}

#*********************************************************************
#* IviApp_InitVirtualBindings
#*********************************************************************
proc IviApp_InitVirtualBindings {bind_style} {

    switch $bind_style {

        windows -
        unix {
            event add <<RMB-Popup>>         <Button-3>
            event add <<Mult-Select>>       <Control-Button-1>
            event add <<Cursor-Add-Select>> <Control-Button-1>
        }

        macosx {
            event add <<RMB-Popup>>         <Button-3>
            event add <<RMB-Popup>>         <Control-Button-1>
            event add <<Mult-Select>>       <Command-Button-1>
            event add <<Cursor-Add-Select>> <Command-Button-1>
        }
    }
}

#*********************************************************************
#* IviApp_InitGuiPrefs()
#*********************************************************************
proc IviApp_InitGuiPrefs {} {
    global tcl_platform
    global ivi_global

    switch $tcl_platform(platform) {
        windows {
            set ivi_global(menu_font)        "{MS Sans Serif} 8"
            set ivi_global(edit_font)        "{MS Sans Serif} 8"
            set ivi_global(normal_cursor)    "arrow"
            option add *TreeWidget.font      "{MS Sans Serif} 8"
            option add *TreeWidget.Font      "{MS Sans Serif} 8"
            option add *WaveWidget.font      "{MS Sans Serif} 8"
            option add *WaveWidget.Font      "{MS Sans Serif} 8"

            IviApp_InitVirtualBindings windows
        }

        macintosh {
            set menu_font [font create ivi_menu_font -family helvetica -size 12]
            set ivi_global(menu_font) $menu_font
            set ivi_global(normal_cursor)    "left_ptr"
            set edit_font [font create ivi_edit_font -family arial \
                -size 12 -weight normal]
            set ivi_global(edit_font) $edit_font

            option add *Entry.Background "white"
            option add *font $menu_font

            IviApp_InitVirtualBindings macosx
        }

        default {
            set ivi_global(normal_cursor)    "left_ptr"

            switch [tk windowingsystem] {
                aqua {
                    set ivi_global(menu_font) \
                        [font create ivi_menu_font -family helvetica -size 12]
                    set ivi_global(edit_font) \
                        [font create ivi_edit_font -family arial \
                            -size 12 -weight normal]
                    option add *Entry.Background "white"
                    option add *font $ivi_global(menu_font)

                    option add *Scrollbar.width 12
                    option add *Canvas.background white
                    option add *Entry.selectBackground "#000080"

                    option add *TreeWidget.font      $ivi_global(menu_font)
                    option add *TreeWidget.Font      $ivi_global(menu_font)
                    option add *WaveWidget.font      $ivi_global(menu_font)
                    option add *WaveWidget.Font      $ivi_global(menu_font)

                    IviApp_InitVirtualBindings macosx
                }

                default {
                    package require wm_default
                    wm_default::setup
                    wm_default::addoptions

                    IviApp_InitVirtualBindings unix
                }
            }
        }
    }
}

#*********************************************************************
#* IviApp_Init()
#*
#*
#* Initialize the application...
#*********************************************************************
proc IviApp_Init args {
    global ivi_global
    global IVI_HOME
    global CallbackTypes

    regsub {\\} $IVI_HOME {/} IVI_HOME

    #**** Load the menus for Ivi... The user is allowed to override... 
    MenuMgr::load [file join $IVI_HOME modules ivi_ui ivi_ui.mtf]
    DialogMgr::load [file join $IVI_HOME modules ivi_ui ivi_ui.dlt]

    #**** Ensure that '~/.ivi' exists...
    IviApp_InitRcDir
    IviApp_InitGuiPrefs

    # Hide the main window until we're done initializing
#    set splash [SplashScreen "ivi_splash2.gif" \
#        [IviConfig::current App.SplashScreenUptime] \
#        [file join $IVI_HOME modules ivi_ui] ""]


    #**** Setup application callbacks
    init_base_widgets

    #**** Setup callback to bring up the main window 
    callback add App.Construct null MainWindow_ConstructCB

    #**** Load all modules...
    ModuleLoader::ModuleLoader [file join $IVI_HOME modules]

    #**** Complete initialization of the database...
    IviConfig::FinishInit
}


