/****************************************************************************
 *                          WaveWidgetPrint.cc
 *
 * Author: Matthew Ballance
 * Desc:   Printing routines
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "WaveWidget.h"
#include "WidgetManager.h"
#include "WidgetCommon.h"
#include "SdbReader.h"
#include "SdbrCursor.h"
#include "CallbackMgr.h"
#include "PsPixmapObj.h"
#include "BitVector.h"
#include "BitVectorABVal.h"

#include "SigDB.h"
#include "SdbSignal.h"
#include "DFIO.h"
#include "DFIOTrace.h"

#include "LogInstance.h"
#include "LogDestination.h"
#include "LogMgr.h"

#include "TreeWidget.h"
#include "TreeWidgetNode.h"

/******************************************************************
 * SetupPsGraphics()
 ******************************************************************/
void WaveWidget::SetupPsGraphics()
{
    GCObj  *gcObjs = d_psGcObjs;

    for (Uint32 i=0; i<TC_NumColors; i++) {
        gcObjs[i].setFont(d_config.tkfont);
        gcObjs[i].setBgColor(d_config.background);
        gcObjs[i].setFgColor((Uint32)0x0);
    }

    gcObjs[TC_ScaleColor].setFgColor((Uint32)(150|(150 << 8)|(150 << 16)));
}

/******************************************************************
 * PageLayout()
 ******************************************************************/
void WaveWidget::PageLayout(
        PsPixmapObj         *pm,
        Uint32               startTime,
        Uint32               endTime,
        Double               pixPerTime,
        Uint32              &pages_w,
        Uint32              &pages_h)
{
    /**** Now, scan through to find what the dimensions are... 
     **** - we know how high the pixmap is, so scale that by the page...
     **** - we know how long
     ****/
    Uint32 delta = endTime - startTime;
    Uint32 pixels = (Uint32)(pixPerTime * delta);
    Uint32 pm_width = pm->width();
    Uint32 pm_height = pm->height();

    pm_width = GetPageAvailableWidth(pm);
    pm_height = GetPageAvailableHeight(pm);

    if ((pixels % pm_width) || !delta) {
        pages_w = (pixels/pm_width)+1;
    } else {
        pages_w = (pixels/pm_width);
    }

    pages_h = 1;
}

/******************************************************************
  PageLayout
 *
 * page_layout <pixmap> <startTime> <endTime> <scale>
 ******************************************************************/
int WaveWidget::PageLayout(int objc, Tcl_Obj *const objv[])
{
    Int32           startTime, endTime;
    Double          pixPerTime = pixelsPerTimeUnit;
    PsPixmapObj    *pm;
    Tcl_Obj        *ret;
    Uint32          w_pages, h_pages;

    if (objc < 4) {
        Tcl_AppendResult(d_interp, "too few arguments", 0);
        return TCL_ERROR;
    }

    pm = (PsPixmapObj *)WidgetMgr_GetObjHandle(
            d_interp, "PsPixmapObj", Tcl_GetString(objv[1]));

    if (!pm) {
        Tcl_AppendResult(d_interp, "no pixmap named ", 
                Tcl_GetString(objv[1]), 0);
        return TCL_ERROR;
    }

    if (Tcl_GetIntFromObj(d_interp, objv[2], &startTime) != TCL_OK) {
        return TCL_ERROR;
    }

    if (Tcl_GetIntFromObj(d_interp, objv[3], &endTime) != TCL_OK) {
        return TCL_ERROR;
    }

    if (objc > 4) {
        if (Tcl_GetDoubleFromObj(d_interp, objv[4], &pixPerTime) != TCL_OK) {
            return TCL_ERROR;
        }
    }

    ret = Tcl_NewListObj(0, 0);

    PageLayout(pm, startTime, endTime, pixPerTime, w_pages, h_pages);

    Tcl_ListObjAppendElement(d_interp, ret, Tcl_NewIntObj(w_pages));
    Tcl_ListObjAppendElement(d_interp, ret, Tcl_NewIntObj(h_pages));
    Tcl_SetObjResult(d_interp, ret);

    return TCL_OK;
}

/******************************************************************
 * PrintPageBorder()
 ******************************************************************/
void WaveWidget::PrintPageBorder(
    PsPixmapObj       *pm,
    Uint32             page_num,
    Uint32             page_total,
    Uint32            &head_border,
    Uint32            &bottom_border)
{
    char            tmp[128], *t;
    Int32           t_width, t_height;
    GCObj          *text = &d_psGcObjs[TC_CursorUnselColor];
    Tk_TextLayout   layout;

    bottom_border = 0;

    sprintf(tmp, "Page %d/%d", page_num, page_total);
    layout = text->ComputeTextLayout(tmp, -1, TK_JUSTIFY_CENTER, 
            &t_width, &t_height);

    head_border = t_height+4;

    text->DrawTextLayout(layout, pm->width()-(t_width+8), 2);
    text->FreeTextLayout(layout);
}

/******************************************************************
 * GetPageAvailableWidth()
 ******************************************************************/
Uint32 WaveWidget::GetPageAvailableWidth(PsPixmapObj *pixmap)
{
    Uint32    treeWidth = 16, treeHeight = 100, labelPad=8;
    Uint32    availWidth;

    if (d_sigTree) {
        Uint32 yOff;
        d_sigTree->getTreeDim(treeWidth, treeHeight);
    }
    availWidth = pixmap->width()-(treeWidth+labelPad);

    return availWidth;
}

/******************************************************************
 * GetPageAvailableHeight()
 ******************************************************************/
Uint32 WaveWidget::GetPageAvailableHeight(PsPixmapObj *pixmap)
{
    Uint32 availHeight = pixmap->height();

    availHeight -= d_scaleTextHeight;

    return availHeight;
}

/******************************************************************
 * CalcPagePixPerTime()
 *
 * <pixmap> <time/page>
 ******************************************************************/
int WaveWidget::CalcPagePixPerTime(int argc, Tcl_Obj *const objv[])
{
    PsPixmapObj   *pm;
    Int32          time_per_page;
    Double         tPP, ppt, pixels;

    if (!(pm = (PsPixmapObj *)WidgetMgr_GetObjHandle(
            d_interp, "PsPixmapObj", Tcl_GetString(objv[0])))) {
        Tcl_AppendResult(d_interp, "no pixmap named ",
                Tcl_GetString(objv[0]), 0);
        return TCL_ERROR;
    }

    if (Tcl_GetIntFromObj(d_interp, objv[1], &time_per_page) != TCL_OK) {
        return TCL_ERROR;
    }

    pixels = (Double)GetPageAvailableWidth(pm);

    tPP    = (Double)time_per_page;

    ppt   = pixels/tPP;
    Tcl_SetObjResult(d_interp, Tcl_NewDoubleObj(ppt));

    return TCL_OK;
}

/******************************************************************
 * CalcTimePerPage()
 *
 * <pixmap> <pix_per_time>
 ******************************************************************/
int WaveWidget::CalcTimePerPage(int objc, Tcl_Obj *const objv[])
{
    PsPixmapObj   *pm;
    Double         availWidth; 
    Int32          time_per_page;
    Double         pixPerTime, tPP, ppt, pixels;

    if (!(pm = (PsPixmapObj *)WidgetMgr_GetObjHandle(
            d_interp, "PsPixmapObj", Tcl_GetString(objv[0])))) {
        Tcl_AppendResult(d_interp, "no pixmap named ",
                Tcl_GetString(objv[0]), 0);
        return TCL_ERROR;
    }

    if (Tcl_GetDoubleFromObj(d_interp, objv[1], &pixPerTime) != TCL_OK) {
        return TCL_ERROR;
    }

    availWidth = (Double)GetPageAvailableWidth(pm);

    time_per_page = (Int32)(availWidth / pixPerTime);

    Tcl_SetObjResult(d_interp, Tcl_NewIntObj(time_per_page));

    return TCL_OK;
}

/******************************************************************
 * PrintPage() 
 ******************************************************************/
void WaveWidget::PrintPage(
        PsPixmapObj        *pm,
        Uint32              start_time,
        Uint32              end_time,
        Uint32              start_ypos,
        Uint32              ysize,
        Double              pixPerTime,
        Uint32              page_num,
        Uint32              page_total)
{
    Uint32       lMargin=0, rMargin=0, tMargin=0, bMargin=0;
    GCObj       *text = &d_psGcObjs[TC_CursorUnselColor];
    Uint32       labelPad=8;
    Uint32       mWidth=0, mHeight=0, rW, rH;
    Uint32       ml, mr, mw, mh;

    /**** Draw page boundary ****/
    pm->StartPage();

    PrintPageBorder(pm, page_num, page_total, tMargin, bMargin);

    text->rect(lMargin, tMargin, pm->width()-(lMargin+rMargin), 
            pm->height()-(tMargin+bMargin));

    if (d_sigTree) {
        Uint32 yOff;

        d_sigTree->getTreeDim( mWidth, mHeight);

        pm->setWinRect(lMargin, tMargin, mWidth+labelPad,
                pm->height()-(tMargin+bMargin));
        text->rect(0, 0, mWidth+labelPad, pm->height());
        d_sigTree->setUseXGC(false);
        d_sigTree->RedrawTree(*pm, 0, &yOff, pm->height());
        d_sigTree->setUseXGC(true);
        pm->clrWinRect();
    }

    /**** Set a clipping rectangle to cover the wave data region
     **** clipping region covers: (currX+mWidth), 0
     **** The width is currW-(mWidth+labelPad)
     ****/
    pm->setWinRect(mWidth+labelPad, tMargin, 
            pm->width()-(mWidth+labelPad+lMargin+rMargin), 
            pm->height()-(tMargin+bMargin));

    DrawWaveProper(*pm, start_ypos, ysize, 
            start_time, end_time, pixPerTime);

    Uint32 cwStart   = ysize;
    Uint32 tagCenter = cwStart+(getTraceSep()/2);
    DrawCursors(sdbr->cursorMgr, *pm, 1, tagCenter, 1);

    pm->clrWinRect();

    pm->EndPage();
}

/******************************************************************
 * Print()
 *
 * print <pixmap> <startTime> <endTime> <pixPerTime> <pagelist>
 ******************************************************************/
int WaveWidget::Print(int objc, Tcl_Obj *const objv[])
{
    PsPixmapObj *pm;
    Uint32       i, j, pages=1;
    Uint32       w_pages, h_pages;
    Int32        start_time, end_time;
    Uint32       cp_height, pm_twidth;
    Double       pixPerTime = pixelsPerTimeUnit;
    Tcl_Obj     *page_list = 0, *page;
    Uint32       treeWidth=16, treeHeight=100, labelPad=8;
   
    d_useXGC = false;

    pm = (PsPixmapObj *)WidgetMgr_GetObjHandle(
            d_interp, "PsPixmapObj", Tcl_GetString(objv[1]));

    if (!pm) {
        fprintf(stderr, "ERROR: Cannot find PsPixmap \"%s\"\n", 
                Tcl_GetString(objv[1]));
    }

    SetupPsGraphics();
    pm->addGCs(getGCs(), TC_NumColors);

    if (Tcl_GetIntFromObj(d_interp, objv[2], &start_time) != TCL_OK) {
        return TCL_ERROR;
    }

    if (Tcl_GetIntFromObj(d_interp, objv[3], &end_time) != TCL_OK) {
        return TCL_ERROR;
    }

    if (objc > 4) {
        if (Tcl_GetDoubleFromObj(d_interp, objv[4], &pixPerTime) != TCL_OK) {
            return TCL_ERROR;
        }
    }

    PageLayout(pm, start_time, end_time, pixPerTime, w_pages, h_pages);
    if (d_sigTree) {
        Uint32 yOff;
        d_sigTree->getTreeDim(treeWidth, treeHeight);
    }

    pm_twidth = PixelsToTime(pm->width()-(treeWidth+labelPad));
    cp_height = getCursorPaneHeight();

    if (objc > 5) {
        /**** Page-list supplied ****/
        page_list = objv[5];
        Int32       plen;
        Uint32      st = start_time;
        Uint32      et, yst=0, ys=(pm->height()-cp_height);
        Int32       tp=(w_pages*h_pages), cp=0;

        Tcl_ListObjLength(d_interp, page_list, &plen);
        for (i=0; i<plen; i++) {
            Tcl_ListObjIndex(d_interp, page_list, i, &page);
            Tcl_GetIntFromObj(d_interp, page, &cp);

            st = pm_twidth * (cp / w_pages);
            et = st + pm_twidth;

            if (et > end_time) {
                et = end_time;
            }

            yst = (pm->height()-cp_height)*(cp % h_pages);

            PrintPage(pm, st, et, yst, ys, pixPerTime, cp, tp);
        }

    } else {
        Uint32 st = start_time;
        Uint32 et, yst=0, ys=(pm->height()-cp_height);
        Uint32 tp=(w_pages*h_pages), cp=0;

        /**** Just paint everything... ****/
        for (i=0; i<w_pages; i++) {
            et = st + pm_twidth;
            if (et > end_time) {
                et = end_time;
            }
            for (j=0; j<h_pages; j++) {
                cp++;
                PrintPage(pm, st, et, yst, ys, pixPerTime, cp, tp);
            }
            st = et;
        }
    }

    pm->removeGCs(getGCs(), TC_NumColors);
    d_useXGC = true;
    return TCL_OK;
}
