
/****************************************************************************
 *
 * All portions copyright their respective authors.  All rights reserved.
 *
 * This file is part of IVMan (ivm).
 *
 * This file may be distributed under the terms of the Q Public License
 * as defined by Troll Tech AS of Norway and appearing in the file
 * LICENSE.QPL included in the packaging of this file.
 * 
 * See http://www.troll.no/qpl for QPL licensing information.
 *
 * $Id: IvmConfigProperties.c,v 1.22 2005/11/29 00:42:48 ro_han Exp $
 *****************************************************************************/

#include <string.h>
#include <glib.h>
#include <libxml/tree.h>
#include <libxml/parser.h>
#include <libxml/xpath.h>
#include <libxml/xpathInternals.h>


#include "IvmConfigProperties.h"
#include "IvmConfigActions.h"
#include "IvmConfigCommon.h"

#include <libhal.h>
LibHalContext *hal_ctx;

#include <dbus/dbus.h>

#include <assert.h>

IvmConfigProperties *parseIvmConfigProperties(char const *const path,
                                              char const *udi,
                                              char const *const property)
{
    DBusError dbus_error;

    dbus_error_init(&dbus_error);

    char *currentProperty = NULL;
    gboolean mountable = ivm_device_is_mountable(udi);

    LIBXML_TEST_VERSION IvmConfigProperties *ret = NULL;

    ret = (IvmConfigProperties *) calloc(1, sizeof(IvmConfigProperties));

    if ( !ret ) {
        DEBUG(_("Out of memory!\n"));
        return NULL;
    }

    ret->checkOnInit = FALSE;
    ret->exec = NULL;
    int num_exec = 0;

    xmlDocPtr doc;
    xmlNodePtr cur;

    doc = xmlParseFile(path);
    if (doc == NULL)
    {
        DEBUG(_("Document not parsed successfully."));
        return NULL;
    }

    cur = xmlDocGetRootElement(doc);
    if (cur == NULL)
    {
        DEBUG(_("Document is empty!"));
        xmlFreeDoc(doc);
        return NULL;
    }
    if (xmlStrcmp(cur->name, (const xmlChar *) "PropertiesConfig"))
    {
        DEBUG(_
              ("Incorrect document type, root node should have been %s"),
              "PropertiesConfig");
        xmlFreeDoc(doc);
        return NULL;
    }
    cur = cur->children;
    while ( cur )
    {
        // DEBUG("At XML node %s",cur->name);
        if (!xmlStrcmp(cur->name, (const xmlChar *) "Match"))
        {
            /* Check if our device matches - if so, continue iterating
               through siblings, otherwise jump back up to parent */

            gboolean matches = FALSE;

            xmlChar *name = xmlGetProp(cur, "name");
            xmlChar *value = xmlGetProp(cur, "value");

            if ( !name )
            {
                DEBUG(_("Warning: %s XML tag encountered with missing or bad attributes, ignored\n"), cur->name);
                goto nextMatch;
            }
            
            if (!xmlStrcmp(name, (const xmlChar *) "*"))
            {
                matches = TRUE;
                goto nextMatch;
            }
            
            if (value == NULL)
            {
                DEBUG(_("Warning: %s XML tag encountered with missing or bad attributes, ignored\n"), cur->name);
                goto nextMatch;
            }

            if (!xmlStrcmp(name, (const xmlChar *) "ivm.mountable"))
            {
                if (((!xmlStrcmp
                      (value, (const xmlChar *) "true"))
                     && mountable)
                    ||
                    ((!xmlStrcmp
                      (value, (const xmlChar *) "false")) && !mountable))
                    matches = TRUE;
            }

            else if (!xmlStrncmp(name, (const xmlChar *) "hal.", 4))
            {
                if (udi)
                    matches = hal_xml_property_matches(name, value, udi);
            }
            else
                DEBUG(_("Invalid match name: %s"), (char *) name);

nextMatch:
            if (matches == TRUE && cur->children != NULL)
                cur = cur->children;
            else if (cur->next != NULL)
                cur = cur->next;
            else
                cur = cur->parent->next;

            if (value) xmlFree(value);
            if (name) xmlFree(name);
        }

        else if (!xmlStrcmp(cur->name, (const xmlChar *) "Option"))
        {
            
            xmlChar *name = xmlGetProp(cur, "name");
            xmlChar *value = xmlGetProp(cur, "value");

            if ((name == NULL) || (value == NULL))
            {
                DEBUG(_("Warning: %s XML tag encountered with missing or bad attributes, ignored\n"), cur->name);
                goto nextOption;
            }
            
            if (!xmlStrcmp(name, (xmlChar *) "checkOnInit"))
            {
                if (!xmlStrcmp(value, (xmlChar *) "true"))
                    ret->checkOnInit = TRUE;
                else
                    ret->checkOnInit = FALSE;
            }

            else
                DEBUG(_("Invalid Option name: %s"), (char *) name);

nextOption:
            if (name) xmlFree(name);
            if (value) xmlFree(value);

            if (cur->next != NULL)
                cur = cur->next;
            else
                cur = cur->parent->next;
        }


        else if (!xmlStrcmp(cur->name, (const xmlChar *) "Property"))
        {
            gboolean matches = FALSE;

            xmlChar *name = xmlGetProp(cur, "name");

            if (name == NULL)
            {
                DEBUG(_("Warning: %s XML tag encountered with missing or bad attributes, ignored\n"), cur->name);
                goto nextProperty;
            }

            if (!xmlStrncmp(name, (const xmlChar *) "hal.", 4))
            {
                int length = xmlStrlen(name) - 4;

                xmlChar *confProperty = xmlStrsub(name, 4, length);

                if (!xmlStrcmp(confProperty, (const xmlChar *) property))
                {
                    matches = TRUE;
                    currentProperty = (char *) strdup((char *) name);
                }

            }
            else
                DEBUG(_("Invalid Property name: %s"), (char *) name);


nextProperty:
            if (matches == TRUE && cur->children != NULL)
                cur = cur->children;
            else
            {
                if (currentProperty != NULL)
                    free(currentProperty);
                if (cur->next != NULL)
                    cur = cur->next;
                else
                    cur = cur->parent->next;
            }

            if (name) xmlFree(name);
        }

        else if (!xmlStrcmp(cur->name, (const xmlChar *) "Action"))
        {
            gboolean matches = FALSE;

            xmlChar *value = xmlGetProp(cur, "value");
            if (value == NULL)
            {
                DEBUG(_("Warning: %s XML tag encountered with missing or bad attributes, ignored\n"), cur->name);
                goto nextAction;
            }

            if (!xmlStrcmp(value, (const xmlChar *) "*"))
                matches = TRUE;

            else if (udi)
                matches =
                    hal_xml_property_matches(currentProperty, value, udi);

            if (matches)
            {
                num_exec++;

                ret->exec = (char **) realloc(ret->exec,
                                              (num_exec +
                                               1) * sizeof(char *));
                ret->exec[num_exec - 1] = (char *) xmlGetProp(cur, "exec");
                ret->exec[num_exec] = NULL;
            }

nextAction:
            if (cur->next != NULL)
                cur = cur->next;
            else
            {
                if (currentProperty != NULL)
                    free(currentProperty);
                cur = cur->parent->next;
            }

            if (value) xmlFree(value);
        }


        else if (cur->next != NULL)
            cur = cur->next;
        else
            cur = cur->parent->next;

    }
    xmlFreeDoc(doc);

    xmlCleanupParser();

    ivm_check_dbus_error(&dbus_error);

    return ret;
}








IvmConfigProperties *IvmConfigPropertiesAll(char const *const path,
                                            char const *udi)
{
    assert(path != NULL);
    assert(udi != NULL);

    DBusError dbus_error;

    dbus_error_init(&dbus_error);

    char *currentProperty = NULL;
    gboolean mountable = ivm_device_is_mountable(udi);

    LIBXML_TEST_VERSION IvmConfigProperties *ret = NULL;

    ret = (IvmConfigProperties *) malloc(sizeof(IvmConfigProperties));

    if (ret == NULL)
    {
        DEBUG(_("Out of memory!"));
        return NULL;
    }

    memset(ret, 0, sizeof(IvmConfigProperties));

    ret->checkOnInit = FALSE;
    ret->exec = NULL;
    int num_exec = 0;

    xmlDocPtr doc;
    xmlNodePtr cur;

    doc = xmlParseFile(path);
    if (doc == NULL)
    {
        DEBUG(_("Document not parsed successfully."));
        return NULL;
    }

    cur = xmlDocGetRootElement(doc);
    if (cur == NULL)
    {
        DEBUG(_("Document is empty!"));
        xmlFreeDoc(doc);
        return NULL;
    }
    if (xmlStrcmp(cur->name, (const xmlChar *) "PropertiesConfig"))
    {
        DEBUG(_
              ("Incorrect document type, root node should have been %s"),
              "PropertiesConfig");
        xmlFreeDoc(doc);
        return NULL;
    }
    cur = cur->children;
    while (cur != NULL)
    {
        //DEBUG("At XML node %s",cur->name);
        if (!xmlStrcmp(cur->name, (const xmlChar *) "Match"))
        {
            /* Check if our device matches - if so, continue iterating
               through siblings, otherwise jump back up to parent */

            gboolean matches = FALSE;

            xmlChar *name = xmlGetProp(cur, "name");
            xmlChar *value = xmlGetProp(cur, "value");

            if (name == NULL)
            {
                DEBUG(_("Warning: %s XML tag encountered with missing or bad attributes, ignored\n"), cur->name);
                goto nextMatchAll;
            }
            
            if (!xmlStrcmp(name, (const xmlChar *) "*"))
            {
                matches = TRUE;
                goto nextMatchAll;
            }
                
            
            if (value == NULL)
            {
                DEBUG(_("Warning: %s XML tag encountered with missing or bad"
                        " attributes, ignored\n"), cur->name);
                goto nextMatchAll;
            }


            else if (!xmlStrcmp(name, (const xmlChar *) "ivm.mountable"))
                matches = mountable;

            else if (!xmlStrncmp(name, (const xmlChar *) "hal.", 4))
            {
                if (udi)
                    matches = hal_xml_property_matches(name, value, udi);
            }
            else
                DEBUG(_("Invalid match name: %s"), (char *) name);

nextMatchAll:
            if (matches == TRUE && cur->children != NULL)
                cur = cur->children;
            else if (cur->next != NULL)
                cur = cur->next;
            else
                cur = cur->parent->next;

            if (value) xmlFree(value);
            if (name) xmlFree(name);
        }

        else if (!xmlStrcmp(cur->name, (const xmlChar *) "Option"))
        {
            xmlChar *name = xmlGetProp(cur, "name");
            xmlChar *value = xmlGetProp(cur, "value");

            if ((name == NULL) || (value == NULL))
            {
                DEBUG(_("Warning: %s XML tag encountered with missing or bad attributes, ignored\n"), cur->name);
                goto nextOptionAll;
            }

            if (!xmlStrcmp(name, (xmlChar *) "checkOnInit"))
            {
                if (!xmlStrcmp(value, (xmlChar *) "true"))
                    ret->checkOnInit = TRUE;
                else
                    ret->checkOnInit = FALSE;
            }

            else
                DEBUG(_("Invalid Option name: %s"), (char *) name);

nextOptionAll:
            if (name) xmlFree(name);
            if (value) xmlFree(value);

            if (cur->next != NULL)
                cur = cur->next;
            else
                cur = cur->parent->next;
        }


        else if (!xmlStrcmp(cur->name, (const xmlChar *) "Property"))
        {
            gboolean matches = TRUE;

            xmlChar *name = xmlGetProp(cur, "name");

            if (name == NULL)
            {
                DEBUG(_("Warning: %s XML tag encountered with missing or bad attributes, ignored\n"), cur->name);
                goto nextPropertyAll;
            }

            if (!xmlStrncmp(name, (const xmlChar *) "hal.", 4))
                currentProperty = (char *) strdup((char *) name);

            else
                DEBUG(_("Invalid Property name: %s"), (char *) name);

nextPropertyAll:
            if (matches == TRUE && cur->children != NULL)
                cur = cur->children;
            else
            {
                if (currentProperty != NULL)
                    free(currentProperty);
                if (cur->next != NULL)
                    cur = cur->next;
                else
                    cur = cur->parent->next;
            }

            if (name) xmlFree(name);
        }

        else if (!xmlStrcmp(cur->name, (const xmlChar *) "Action"))
        {

            gboolean matches = FALSE;

            xmlChar *value = xmlGetProp(cur, "value");
            
            if (value == NULL)
            {
                DEBUG(_("Warning: %s XML tag encountered with missing or bad attributes, ignored\n"), cur->name);
                goto nextActionAll;
            }

            if (!xmlStrcmp(value, (const xmlChar *) "*"))
                matches = TRUE;

            else if (udi)
                matches =
                    hal_xml_property_matches(currentProperty, value, udi);

            if (matches)
            {
                num_exec++;

                ret->exec = (char **) realloc(ret->exec,
                                              (num_exec +
                                               1) * sizeof(char *));
                ret->exec[num_exec - 1] =
                    (char *) xmlGetProp(cur, "exec");;
                ret->exec[num_exec] = NULL;
            }

nextActionAll:
            if (cur->next != NULL)
                cur = cur->next;
            else
            {
                if (currentProperty != NULL)
                    free(currentProperty);
                cur = cur->parent->next;
            }

            if (value) xmlFree(value);
        }


        else if (cur->next != NULL)
            cur = cur->next;
        else
            cur = cur->parent->next;

    }
    xmlFreeDoc(doc);

    xmlCleanupParser();

    ivm_check_dbus_error(&dbus_error);

    return ret;
}
