// K-3D
// Copyright (c) 1995-2006, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Tim Shead (tshead@k-3d.com)
*/

#include "path.h"
#include "result.h"

#include <boost/tokenizer.hpp>

#ifdef K3D_PLATFORM_WIN32

	#include <windows.h>

#else // K3D_PLATFORM_WIN32

	#include <sys/stat.h>

#endif // !K3D_PLATFORM_WIN32

namespace k3d
{

namespace filesystem
{

namespace detail
{

// Define some platform-specific odds-and-ends
#ifdef K3D_PLATFORM_WIN32

	const char* NATIVE_SEARCHPATH_SEPARATOR = ";";
	const char NATIVE_PATH_SEPARATOR = '\\';

#else // K3D_PLATFORM_WIN32

	const char* NATIVE_SEARCHPATH_SEPARATOR = ":";
	const char NATIVE_PATH_SEPARATOR = '/';

#endif // !K3D_PLATFORM_WIN32

const char GENERIC_PATH_SEPARATOR = '/';

	
} // namespace detail
	
/////////////////////////////////////////////////////////////////////////////////////////////
// make_relative_path
	
const boost::filesystem::path make_relative_path(const boost::filesystem::path& AbsolutePath, const boost::filesystem::path& ReferencePath)
{
	// The AbsolutePath must actually *be* an absolute path!
	return_val_if_fail(AbsolutePath.is_complete(), boost::filesystem::path());

	// As a special-case, if the AbsolutePath and ReferencePath don't share the same root name, return the AbsolutePath (which is the best we can do)
	if(AbsolutePath.root_name() != ReferencePath.root_name())
		return AbsolutePath;

	boost::filesystem::path relative_path;

	const boost::filesystem::path root_path = ReferencePath;
	const boost::filesystem::path absolute_path(AbsolutePath);

	boost::filesystem::path::iterator a = root_path.begin();
	boost::filesystem::path::iterator b = absolute_path.begin();
	while(a != root_path.end() && b != absolute_path.end() && *a == *b)
	{
		++a;
		++b;
	}

	for(; a != root_path.end(); ++a)
		relative_path /= "..";
	for(; b != absolute_path.end(); ++b)
		relative_path /= boost::filesystem::path(*b, boost::filesystem::no_check);

	return relative_path;
}

/////////////////////////////////////////////////////////////////////////////////////////////
// path
	
path::path()
{
}

path::path(const std::string& GenericPath) :
	storage(GenericPath)
{
}

path generic_path(const std::string& GenericPath)
{
	return path(GenericPath);
}

path native_path(const std::string& NativePath)
{
	std::string generic_path(NativePath);
	std::replace(generic_path.begin(), generic_path.end(), '\\', detail::GENERIC_PATH_SEPARATOR);
	return path(generic_path);
}

path& path::operator/=(const path& rhs)
{
	/** \todo Handle extra separators here */
	storage += detail::GENERIC_PATH_SEPARATOR;
	storage += rhs.storage;
	return *this;
}

path path::operator/(const path& rhs) const
{
	path result(*this);
	result /= rhs;
	return result;
}

std::string path::string() const
{
	return storage;
}

std::string path::native_file_string() const
{
	std::string native_string(storage);
	std::replace(native_string.begin(), native_string.end(), detail::GENERIC_PATH_SEPARATOR, detail::NATIVE_PATH_SEPARATOR);
	return native_string;
}

/*
path& path::normalize()
{
}

path path::root_path() const
{
}

std::string path::root_name() const
{
}

std::string path::root_directory() const
{
}

path path::relative_path() const
{
}

std::string path::leaf() const
{
}

path path::branch_path() const
{
}
*/

bool path::empty() const
{
	return storage.empty();
}

/*
bool path::is_complete() const
{
}

bool path::has_root_path() const
{
}

bool path::has_root_name() const
{
}

bool path::has_root_directory() const
{
}

bool path::has_relative_path() const
{
}

bool path::has_leaf() const
{
}

bool path::has_branch_path() const
{
}

path::iterator path::begin() const
{
	return iterator(storage);
}

path::iterator path::end() const
{
	return iterator();
}
*/

bool path::operator==(const path& that) const
{
	return !(*this < that) && !(that < *this);
}

bool path::operator!=(const path& that) const
{
	return !(*this == that);
}

bool path::operator<(const path& that) const
{
	return storage < that.storage;
}

bool path::operator<=(const path& that) const
{
	return !(that < *this);
}

bool path::operator>(const path& that) const
{
	return that < *this;
}

bool path::operator>=(const path& that) const
{
	return !(*this < that);
}

/*
//////////////////////////////////////////////////////////////////////////////////
// path::iterator

path::iterator::iterator() :
	storage(0),
	index(0)
{
}

path::iterator::iterator(const std::string& Storage) :
	storage(&Storage),
	index(0)
{
}

std::string path::iterator::operator*()
{
	return storage->substr(index, storage->find(detail::GENERIC_PATH_SEPARATOR, index+1) - index);
}

bool path::iterator::operator==(const iterator& RHS)
{
	return storage == RHS.storage && index == RHS.index;
}

bool path::iterator::operator!=(const iterator& RHS)
{
	return !(*this == RHS);
}

path::iterator& path::iterator::operator++()
{
	index = storage->find(detail::GENERIC_PATH_SEPARATOR, index+1);
	if(index == std::string::npos)
	{
		storage = 0;
		index = 0;
	}
	
	return *this;
}
*/

//////////////////////////////////////////////////////////////////////////////////
// exists

bool exists(const path& Path)
{
#ifdef K3D_PLATFORM_WIN32
	return ::GetFileAttributesA(Path.string().c_str()) != 0xFFFFFFFF;
#else // K3D_PLATFORM_WIN32
	struct stat path_stat;
	return ::stat(Path.string().c_str(), &path_stat) == 0;
#endif // !K3D_PLATFORM_WIN32
}

//////////////////////////////////////////////////////////////////////////////////
// split_native_paths

const path_list split_native_paths(const std::string& Paths)
{
	path_list results;

	boost::char_separator<char> separator(detail::NATIVE_SEARCHPATH_SEPARATOR);
	typedef boost::tokenizer<boost::char_separator<char> > tokenizer;
	tokenizer paths(Paths, separator);
	for(tokenizer::iterator path = paths.begin(); path != paths.end(); ++path)
		results.push_back(native_path(*path));

	return results;
}

} // namespace filesystem

} // namespace k3d

