// K-3D
// Copyright (c) 2006, Romain Behar
//
// Contact: romainbehar@yahoo.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/i18n.h>
#include <k3dsdk/mesh_modifier.h>
#include <k3dsdk/mesh_selection_sink.h>
#include <k3dsdk/module.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/transformable.h>

#include <iterator>

namespace libk3ddeformation
{

/////////////////////////////////////////////////////////////////////////////
// transform_points

class transform_points :
	public k3d::mesh_selection_sink<k3d::mesh_modifier<k3d::transformable<k3d::persistent<k3d::node> > > >
{
	typedef k3d::mesh_selection_sink<k3d::mesh_modifier<k3d::transformable<k3d::persistent<k3d::node> > > > base;

public:
	transform_points(k3d::idocument& Document) :
		base(Document)
	{
		m_mesh_selection.changed_signal().connect(make_reset_mesh_slot());

		m_input_matrix.changed_signal().connect(make_update_mesh_slot());
	}

	void on_create_mesh(const k3d::mesh& InputMesh, k3d::mesh& Mesh)
	{
		k3d::deep_copy(InputMesh, Mesh);
		k3d::replace_selection(m_mesh_selection.value(), Mesh);
	}

	void on_update_mesh(const k3d::mesh& Source, k3d::mesh& Target)
	{
		// Sanity checks ...
		assert(Source.points.size() == Target.points.size());

		// Translate points ...
		const k3d::matrix4 matrix = m_input_matrix.value();

		for(unsigned long i = 0; i != Target.points.size(); ++i)
			Target.points[i]->position = k3d::mix(Source.points[i]->position, matrix * Source.points[i]->position, Target.points[i]->selection_weight);
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<transform_points>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink,
			k3d::interface_list<k3d::itransform_source,
			k3d::interface_list<k3d::itransform_sink > > > > > factory(
				k3d::uuid(0x8543cf2d, 0x170c4edc, 0x9bf821bd, 0xe8567b6f),
				"TransformPoints",
				_("Transform mesh points using input matrix"),
				"Deformation",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
};

/////////////////////////////////////////////////////////////////////////////
// transform_points_factory

k3d::iplugin_factory& transform_points_factory()
{
	return transform_points::get_factory();
}

} // namespace libk3ddeformation

