
#include <Hapy/Parser.h>
#include <Hapy/Rules.h>

#include <functional>
#include <iostream>
#include <vector>
#include <string>

// Compile as stand-alone program : `g++ -o rib rib_grammar.cpp -lHapy -Ik3d/hapy/src/include/ -Lk3d/hapy/src/libHapy.la -DHAPY_HAVE_NUMERIC_LIMITS=1 -DHAPY_HAVE_STD_ITERATOR_TYPE=1`
// Run : `./rib < sample.rib`

using namespace Hapy;


Rule rtint;
Rule rtfloat;
Rule rtstring;
Rule rtint_array;
Rule rtfloat_array;
Rule rtstring_array;

Rule rtpair;
Rule rtparameters;

Rule version;
Rule file;
Rule request_property;
Rule request_object;
Rule request;

Rule camera;
Rule frame;
Rule world;
Rule transform;
Rule attribute;
Rule solid;
Rule object;
Rule motion;
Rule shader;
Rule geometry;
Rule transformation;
Rule errors;
Rule textures;

Rule comment;
Rule comment_body;

static bool hapy_rib_parser = false;

void create_parser()
{
	if(hapy_rib_parser)
		return;

	hapy_rib_parser = true;

	rtint = !char_r('-') >> +digit_r;
	rtint.leaf(true);
	rtint.verbatim(true);
	rtint.committed(true);

	// Make sure rtfloat does not match empty string or '.' but does match 1e2
	rtfloat = !char_r('-') >> (+digit_r >> '.' >> *digit_r | !char_r('.') >> +digit_r) >> !('e' >> rtint);
	rtfloat.leaf(true);
	rtfloat.verbatim(true);
	rtfloat.committed(true);

	rtstring = quoted_r(anychar_r, '"');
	rtstring.leaf(true);
	rtstring.verbatim(true);
	rtstring.committed(true);

	rtint_array = quoted_r(rtint, '[', ']');
	rtfloat_array = quoted_r(rtfloat, '[', ']');
	rtstring_array = quoted_r(rtstring, '[', ']');

	rtpair = rtstring >> rtint
		| rtstring >> rtint_array
		| rtstring >> rtfloat
		| rtstring >> rtfloat_array
		| rtstring >> rtstring
		| rtstring >> rtstring_array
		;

	rtparameters = *rtpair;

	version = "version" >> rtfloat;

	file = *comment >> !version >> *request;


	request_property =
		"Attribute" >> rtstring >> rtparameters
		| "Bound" >> ((rtfloat >> rtfloat >> rtfloat >> rtfloat >> rtfloat >> rtfloat) | rtfloat_array)
		| "Color" >> ((rtfloat >> rtfloat >> rtfloat) | rtfloat_array)
		| "ColorSamples" >> rtfloat_array >> rtfloat_array
		| "Declare" >> rtstring >> rtstring
		| "DetailRange" >> ((rtfloat >> rtfloat >> rtfloat >> rtfloat) | rtfloat_array)
		| "Detail" >> ((rtfloat >> rtfloat >> rtfloat >> rtfloat >> rtfloat >> rtfloat) | rtfloat_array)
		| "Display" >> rtstring >> rtstring >> rtstring >> rtparameters
		| "GeometricApproximation" >> rtstring >> (rtfloat | rtfloat_array)
		| "Illuminate" >> rtint >> rtint
		| "Matte" >> rtint
		| "ObjectInstance" >> rtint
		| "Opacity" >> ((rtfloat >> rtfloat >> rtfloat) | rtfloat_array)
		| "Option" >> rtstring >> rtparameters
		| "Orientation" >> rtstring
		| "RelativeDetail" >> rtfloat
		| "ReverseOrientation"
		| "Sides" >> rtint
		| "TextureCoordinates" >> ((rtfloat >> rtfloat >> rtfloat >> rtfloat >> rtfloat >> rtfloat) | rtfloat_array)
		;

	request_object = comment
		| attribute
		| camera
		| errors
		| frame
		| geometry
		| motion
		| object
		| shader
		| solid
		| transform
		| transformation
		| world
		;

	request = request_property | request_object;
	// HAPY : cannot commit request because Points is a prefix of PointsPolygons
	// a "better" grammar would terminate every command/request with ';'
	// or equivalent so that commands can be isolated
	//request.committed(true);

	camera = "Clipping" >> rtfloat >> rtfloat
		| "CropWindow" >> rtfloat >> rtfloat >> rtfloat >> rtfloat
		| "DepthOfField" >> rtfloat >> rtfloat >> rtfloat
		| "Exposure" >> rtfloat >> rtfloat
		| "Format" >> rtint >> rtint >> rtfloat
		| "FrameAspectRatio" >> rtfloat
		| "Hider" >> rtstring >> rtparameters
		| "PixelFilter" >> rtstring >> rtfloat >> rtfloat
		| "PixelSamples" >> rtfloat >> rtfloat
		| "PixelVariance" >> rtfloat
		| "Projection" >> rtstring >> rtparameters
		| "Quantize" >> rtstring >> rtint >> rtint >> rtint >> rtfloat
		| "ScreenWindow" >> rtfloat >> rtfloat >> rtfloat >> rtfloat
		| "ShadingInterpolation" >> rtstring
		| "ShadingRate" >> rtfloat
		| "Shutter" >> rtfloat >> rtfloat
		;

	frame = "FrameBegin" >> rtint
		| "FrameEnd"
		;

	world = "WorldBegin" >> *request >> "WorldEnd"
		;

	transform = "TransformBegin" >> *request >> "TransformEnd"
		;

	attribute = "AttributeBegin" >> *request >> "AttributeEnd"
		;

	solid = "SolidBegin" >> rtstring
		| "SolidEnd"
		;

	object = "ObjectBegin" >> rtint
		| "ObjectEnd"
		;

	motion = "MotionBegin" >> rtfloat_array
		| "MotionEnd"
		;

	shader = "AreaLightSource" >> rtstring >> rtint >> rtparameters
		| "Atmosphere" >> rtstring >> rtparameters
		| "Deformation" >> rtstring >> rtparameters
		| "Displacement" >> rtstring >> rtparameters
		| "Exterior" >> rtstring >> rtparameters
		| "Imager" >> rtstring >> rtparameters
		| "Interior" >> rtstring >> rtparameters
		| "LightSource" >> rtstring >> rtint >> rtparameters
		| "Surface" >> rtstring >> rtparameters
		;

	geometry = "Polygon" >> rtparameters
		| "GeneralPolygon" >> rtint_array >> rtparameters
		| "Curves" >> rtstring >> rtint_array >> rtstring >> rtparameters
		| "Blobby" >> rtint >> rtint_array >> rtfloat_array >> rtstring_array >> rtparameters
		| "Procedural" >> rtstring >> rtstring >> rtfloat_array >> rtparameters
		| "Points" >> rtparameters
		| "PointsPolygons" >> rtint_array >> rtint_array >> rtparameters
		| "PointsGeneralPolygons" >> rtint_array >> rtint_array >> rtint_array >> rtparameters
		/** \todo RiBasis */
		| "Patch" >> rtstring >> rtparameters
		| "PatchMesh" >> rtstring >> rtint >> rtstring >> rtint >> rtstring >> rtparameters
		| "NuPatch" >> rtint >> rtint >> rtfloat_array >> rtfloat >> rtfloat >> rtint >> rtint >> rtfloat_array >> rtfloat >> rtfloat >> rtparameters
		| "TrimCurve" >> rtint_array >> rtint_array >> rtfloat_array >> rtfloat_array >> rtfloat_array >> rtint_array >> rtfloat_array >> rtfloat_array >> rtfloat_array
		| "Sphere" >> rtfloat >> rtfloat >> rtfloat >> rtfloat >> rtparameters
		| "Cone" >> rtfloat >> rtfloat >> rtfloat >> rtparameters
		| "Cylinder" >>  rtfloat >> rtfloat >> rtfloat >> rtfloat >> rtparameters
		| "Hyperboloid" >> ((rtfloat >> rtfloat >> rtfloat >> rtfloat >> rtfloat >> rtfloat >> rtfloat) | rtfloat_array) >> rtparameters
		| "Paraboloid" >> rtfloat >> rtfloat >> rtfloat >> rtfloat >> rtparameters
		| "Disk" >> rtfloat >> rtfloat >> rtfloat >> rtparameters
		| "Torus" >> rtfloat >> rtfloat >> rtfloat >> rtfloat >> rtfloat >> rtparameters
		| "Geometry" >> rtstring >> rtparameters
		| "SubdivisionMesh" >> ( rtstring >> rtint_array >> rtint_array | rtstring >> rtint_array >> rtint_array >> rtstring_array >> rtint_array >> rtint_array >> rtfloat_array) >> rtparameters
		;

	transformation = "Identity"
		| "Transform" >> rtfloat_array
		| "ConcatTransform" >> rtfloat_array
		| "Perspective" >> rtfloat
		| "Translate" >> rtfloat >> rtfloat >> rtfloat
		| "Rotate" >> rtfloat >> rtfloat >> rtfloat >> rtfloat
		| "Scale" >> rtfloat >> rtfloat >> rtfloat
		| "Skew" >> ((rtfloat >> rtfloat >> rtfloat >> rtfloat >> rtfloat >> rtfloat >> rtfloat) | rtfloat_array)
		| "CoordinateSystem" >> rtstring
		| "CoordSysTransform" >> rtstring
		| "TransformPoints"
		;

	errors = "ErrorHandler" >> rtstring
		| "ErrorIgnore"
		| "ErrorPrint"
		| "ErrorAbort"
		;

	textures = "MakeTexture" >> *rtstring >> rtfloat >> rtfloat >> rtparameters
		| "MakeBump" >> *rtstring >> rtfloat >> rtfloat >> rtparameters
		| "MakeLatLongEnvironment" >> *rtstring >> rtfloat >> rtfloat >> rtparameters
		| "MakeCubeFaceEnvironment" >> *rtstring >> rtfloat >> rtstring >> rtfloat >> rtfloat >> rtparameters
		| "MakeShadow" >> rtstring >> rtstring >> rtparameters
		;

	// Comment without backtracking
	//comment = quoted_r(anychar_r, '#', eol_r);
	comment = '#' >> comment_body >> eol_r;
	comment_body = *(anychar_r - eol_r);
	comment_body.committed(true);
	comment.verbatim(true);
}




int expressions(const Hapy::Pree& expr)
{
	const Hapy::Pree &alt = expr[0];
}

void interpret(const Hapy::Pree& result)
{
	const Hapy::Pree& exs = result[0];
	std::for_each(exs.begin(), exs.end(), std::ptr_fun(&expressions));
}


// Here's a simple example of how to use the grammar:

int main()
{
	//k3d::ri::parse();

	std::string buffer;
	std::copy(std::istreambuf_iterator<char>(std::cin), std::istreambuf_iterator<char>(), std::back_inserter(buffer));

	create_parser();

	file.trim(*Hapy::space_r);

	Hapy::Parser parser;
	parser.grammar(file);

	if(!parser.parse(buffer))
	{
		std::cerr << parser.result().location() << " -> parsing failed" << std::endl;
		exit(2);
	}

	parser.result().pree.print(std::cout << "parsing succeeded" << std::endl);

	interpret(parser.result().pree);

	return 0;
}


